/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.ui;

import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.event.DocumentAdapter;
import com.intellij.openapi.editor.event.DocumentEvent;
import com.intellij.openapi.editor.event.DocumentListener;
import com.intellij.openapi.project.Project;
import com.intellij.psi.PsiDocumentManager;
import com.intellij.psi.PsiFile;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.ui.BasicDomElementComponent;
import com.intellij.util.xml.ui.CommittablePanel;

import javax.swing.*;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

/**
 * @author peter
 */
public class AbstractCommittableTab<T extends DomElement> extends BasicDomElementComponent<T> {
  private final String myName;
  private boolean mySelected;
  private CommittablePanel myPanel;
  private final Set<Document> myDocuments = new HashSet<>();
  private final PsiDocumentManager myManager;
  private final DocumentListener myDocumentListener = new DocumentAdapter() {
    public void documentChanged(DocumentEvent e) {
      for (final Document document : myDocuments) {
        myManager.commitDocument(document);
      }
      getPanel().reset();
    }
  };

  public AbstractCommittableTab(final String name, T element) {
    super(element);
    myName = name;
    myManager = PsiDocumentManager.getInstance(element.getManager().getProject());
    ContainerUtil.addAll(myDocuments, myManager.getDocument(DomUtil.getFile(element)));
  }

  public AbstractCommittableTab(final String name, final CommittablePanel panel, final Project project) {
    super(null);
    myName = name;
    myPanel = panel;
    myManager = PsiDocumentManager.getInstance(project);
  }

  protected AbstractCommittableTab(final String name, final CommittablePanel panel, final PsiFile file) {
    this(name, panel, PsiDocumentManager.getInstance(file.getProject()).getDocument(file), file.getProject());
  }

  protected AbstractCommittableTab(final String name, final CommittablePanel panel, final Document document, final Project project) {
    super(null);
    myName = name;
    myPanel = panel;
    ContainerUtil.addAll(myDocuments, document);
    myManager = PsiDocumentManager.getInstance(project);
  }

  public final String getTabName() {
    return myName;
  }

  public final Set<Document> getWatchedDocuments() {
    return myDocuments;
  }

  public final void addWatchedElement(final DomElement domElement) {
    addWatchedDocument(myManager.getDocument(DomUtil.getFile(domElement)));
  }

  public final void removeWatchedElement(final DomElement domElement) {
    removeWatchedDocument(myManager.getDocument(DomUtil.getFile(domElement)));
  }

  public final void addWatchedDocument(final Document document) {
    stopWatching();
    myDocuments.add(document);
    startWatching();
  }

  public final void removeWatchedDocument(final Document document) {
    stopWatching();
    myDocuments.remove(document);
    startWatching();
  }

  public Icon getTabIcon() {
    return null;
  }

  public CommittablePanel getPanel() {
    return myPanel;
  }

  public void selectNotify() {
    if (mySelected) return;
    mySelected = true;
    startWatching();
    getPanel().reset();
  }

  private void startWatching() {
    if (mySelected) {
      for (final Document document : myDocuments) {
        document.addDocumentListener(myDocumentListener);
      }
    }
  }

  public final void deselectNotify() {
    if (!mySelected) return;
    stopWatching();
    mySelected = false;
    ApplicationManager.getApplication().invokeLater(() -> getPanel().commit());
  }

  private void stopWatching() {
    if (mySelected) {
      for (final Document document : myDocuments) {
        document.removeDocumentListener(myDocumentListener);
      }
    }
  }

  public JComponent getComponent() {
    return getPanel().getComponent();
  }
}
