/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.xml.beans;

import com.intellij.openapi.util.Key;
import com.intellij.psi.PsiClassType;
import com.intellij.psi.PsiType;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.DomUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

/**
 * @author Dmitry Avdeev
 */
public class TypeHolderUtil {
  private static final Key<CachedValue<List<PsiType>>> TYPE_HOLDER_TYPES = Key.create("TYPE_HOLDER_TYPES");

  private TypeHolderUtil() {
  }

  @Nullable
  public static PsiType getRequiredType(@NotNull TypeHolder typeHolder) {
    final List<PsiType> psiTypes = getRequiredTypes(typeHolder);
    return ContainerUtil.getFirstItem(psiTypes);
  }

  @NotNull
  public static List<PsiType> getRequiredTypes(@NotNull final TypeHolder typeHolder) {
    if (!DomUtil.hasXml(typeHolder) ||
        !typeHolder.isValid()) {
      return Collections.emptyList();
    }

    return CachedValuesManager.getManager(typeHolder.getManager().getProject())
      .getCachedValue(typeHolder, TYPE_HOLDER_TYPES, () -> {
        List<PsiType> requiredTypes = typeHolder.getRequiredTypes();
        return Result.create(requiredTypes, DomUtil.getFile(typeHolder));
      }, false);
  }

  @Nullable
  public static PsiClassType getRequiredClassType(@NotNull TypeHolder typeHolder) {
    final PsiType injectionType = getRequiredType(typeHolder);
    if (injectionType instanceof PsiClassType) {
      return (PsiClassType)injectionType;
    }
    return null;
  }
}
