/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils;

import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.xml.SpringModelElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringModelUtils {

  public static SpringModelUtils getInstance() {
    return ServiceManager.getService(SpringModelUtils.class);
  }

  @NotNull
  public abstract CommonSpringModel getSpringModel(@NotNull SpringModelElement modelElement);

  @NotNull
  public abstract CommonSpringModel getSpringModel(@Nullable PsiElement element);

  @NotNull
  public abstract CommonSpringModel getPsiClassSpringModel(@NotNull PsiClass psiClass);

  @NotNull
  public abstract CommonSpringModel getModuleCombinedSpringModel(@NotNull PsiElement element);

  @NotNull
  public abstract CommonSpringModel getSpringModelByBean(@Nullable CommonSpringBean springBean);

  /**
   * Determines if given configuration file is used in test context
   *
   * @param module Module to find usages in.
   * @param file   Configuration file.
   * @return {@code true} if used in test context.
   */
  public abstract boolean isTestContext(Module module, PsiFile file);

  /**
   * Determines whether given configuration file is used explicitly (fileset) or
   * implicitly (e.g. included) in any Spring model in its module (or dependent modules).
   *
   * @param configurationFile Configuration file.
   * @param checkTestFiles    Whether to check configuration files used in test contexts.
   * @return {@code true} if file is used. {@code false} in dumb mode or if file is not used.
   * @since 2016.2
   */
  public abstract boolean isUsedConfigurationFile(@NotNull PsiFile configurationFile, boolean checkTestFiles);

  /**
   * Determines if given configuration file is used explicitly (fileset) or
   * implicitly (e.g. included) in registered Spring models in its module (or dependent modules).
   *
   * @param psiFile        Configuration file.
   * @param used           if {@code true} then checks whether file is used otherwise checks whether file is not used
   * @param checkTestFiles should <code>psiFile</code> be checked if it is test file
   * @return {@code ConfigurationCheckResult} object with <code>checkResult</code> {@code true} and non-null <code>myVirtualFile</code> and
   * <code>myModule</code> if file is or not configuration according <code>used</code> parameter. {@code ConfigurationCheckResult.FALSE_RESULT} otherwise.
   * @deprecated Use {@link #isUsedConfigurationFile(PsiFile, boolean)} instead. To remove in 2016.3.
   */
  @Deprecated
  public abstract ConfigurationCheckResult isUsedOrNotConfigurationFile(@NotNull PsiFile psiFile, boolean used, boolean checkTestFiles);

  @Deprecated
  public static class ConfigurationCheckResult {
    public static final ConfigurationCheckResult FALSE_RESULT = new ConfigurationCheckResult(null, null, false);

    public final VirtualFile myVirtualFile;
    public final Module myModule;
    public final boolean checkResult;

    public ConfigurationCheckResult(VirtualFile file, Module module, boolean checkResult) {
      myVirtualFile = file;
      myModule = module;
      this.checkResult = checkResult;
    }
  }
}
