/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.utils.filters;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.PackageScope;
import com.intellij.psi.search.searches.ClassInheritorsSearch;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;

public class SpringContextIncludeAssignableFilter extends SpringContextFilter.IncludeExpression {

  public SpringContextIncludeAssignableFilter(@Nullable String expression) {
    super(expression);
  }

  @NotNull
  @Override
  public Set<SpringStereotypeElement> includeStereotypes(@NotNull Module module, @NotNull Set<PsiPackage> packages) {
    final Set<SpringStereotypeElement> components = ContainerUtil.newLinkedHashSet();

    final String fqn = getExpression();

    if (!StringUtil.isEmptyOrSpaces(fqn)) {
      final GlobalSearchScope searchScope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);
      final PsiClass assignableClass = JavaPsiFacade.getInstance(module.getProject()).findClass(fqn, searchScope);

      addCustomComponents(packages, searchScope, components, assignableClass);
    }
    return components;
  }

  public static void addCustomComponents(@NotNull Set<PsiPackage> packages,
                                         @NotNull GlobalSearchScope searchScope,
                                         @NotNull Set<SpringStereotypeElement> components,
                                         @Nullable PsiClass assignableClass) {
    if (assignableClass != null) {
      addCustomComponent(components, assignableClass);

      for (PsiPackage aPackage : packages) {
        GlobalSearchScope pkgSearchScope = searchScope.intersectWith(PackageScope.packageScope(aPackage, true));

        final Collection<PsiClass> inheritors = ClassInheritorsSearch.search(assignableClass, pkgSearchScope, true).findAll();
        for (PsiClass psiClass : inheritors) {
          addCustomComponent(components, psiClass);
        }
      }
    }
  }

  public static void addCustomComponent(@NotNull Set<SpringStereotypeElement> components, @NotNull PsiClass componentCandidateClass) {
    final PsiModifierList modifierList = componentCandidateClass.getModifierList();

    if (componentCandidateClass.isInterface()
        || componentCandidateClass.isAnnotationType()
        || (modifierList != null && modifierList.hasModifierProperty(PsiModifier.ABSTRACT))
      ) return;

    components.add(new CustomSpringComponent(componentCandidateClass));
  }
}
