/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamClassAttributeElement;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.jam.reflect.*;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiModifier;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.xml.XmlFile;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * @since 16
 */
public abstract class SpringContextConfiguration implements ContextConfiguration {

  private static final SemKey<SpringContextConfiguration> JAM_KEY =
    ContextConfiguration.CONTEXT_CONFIGURATION_JAM_KEY.subKey("SpringContextConfiguration");
  public static final JamClassMeta<SpringContextConfiguration> META =
    new JamClassMeta<>(null, SpringContextConfiguration.class, JAM_KEY);

  public static final JamAnnotationMeta ANNO_META = new JamAnnotationMeta(SpringAnnotationsConstants.CONTEXT_CONFIGURATION);

  public static final JamClassAttributeMeta.Single LOADER_ATTR_META = JamAttributeMeta.singleClass(LOADER_ATTR_NAME);

  public static final JamStringAttributeMeta.Collection<List<XmlFile>> LOCATION_ATTR_META =
    JamAttributeMeta.collectionString(LOCATIONS_ATTR_NAME, new ApplicationContextReferenceConverter());

  public static final JamStringAttributeMeta.Collection<List<XmlFile>> VALUE_ATTR_META =
    JamAttributeMeta.collectionString(VALUE_ATTR_NAME, new ApplicationContextReferenceConverter());

  protected static final JamClassAttributeMeta.Collection CLASSES_ATTR_META =
    new JamClassAttributeMeta.Collection(CLASSES_ATTR_NAME);

  protected static final JamClassAttributeMeta.Collection INITIALIZERS_ATTR_META =
    new JamClassAttributeMeta.Collection("initializers");

  static {
    META.addAnnotation(ANNO_META);
    ANNO_META.addAttribute(LOCATION_ATTR_META);
    ANNO_META.addAttribute(VALUE_ATTR_META);
    ANNO_META.addAttribute(CLASSES_ATTR_META);
    ANNO_META.addAttribute(INITIALIZERS_ATTR_META);
  }

  private final PsiClass myPsiClass;
  private final PsiElementRef<PsiAnnotation> myPsiAnnotation;


  @SuppressWarnings("unused")
  public SpringContextConfiguration(@NotNull PsiClass psiClass) {
    myPsiClass = psiClass;
    myPsiAnnotation = ANNO_META.getAnnotationRef(psiClass);
  }

  @SuppressWarnings("unused")
  public SpringContextConfiguration(PsiAnnotation annotation) {
    myPsiClass = PsiTreeUtil.getParentOfType(annotation, PsiClass.class, true);
    myPsiAnnotation = PsiElementRef.real(annotation);
  }

  @NotNull
  public List<JamStringAttributeElement<List<XmlFile>>> getLocationsAttributeElement() {
    return LOCATION_ATTR_META.getJam(myPsiAnnotation);
  }

  @NotNull
  public List<JamStringAttributeElement<List<XmlFile>>> getValueAttributeElement() {
    return VALUE_ATTR_META.getJam(myPsiAnnotation);
  }

  public boolean isInheritLocations() {
    PsiAnnotation annotation = getAnnotation();
    Boolean value = annotation == null ? null :
                    JamCommonUtil.getObjectValue(annotation.findAttributeValue("inheritLocations"), Boolean.class);

    return value == null || value.booleanValue();
  }

  @Nullable
  public PsiClass getPsiElement() {
    return myPsiClass;
  }

  @JamPsiValidity
  public abstract boolean isPsiValid();

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @NotNull
  public Set<XmlFile> getLocations(@NotNull PsiClass... contexts) {
    Set<XmlFile> locations = ContainerUtil.newLinkedHashSet();

    addFiles(contexts, locations, getValueAttributeElement());
    addFiles(contexts, locations, getLocationsAttributeElement());

    return locations;
  }

  private void addFiles(@NotNull PsiClass[] contexts,
                        @NotNull Set<XmlFile> locations,
                        @NotNull List<JamStringAttributeElement<List<XmlFile>>> elements) {
    for (JamStringAttributeElement<List<XmlFile>> stringAttributeElement : elements) {
      for (PsiClass context : contexts) {
        if (context.equals(getPsiElement())) {
          List<XmlFile> xmlFiles = stringAttributeElement.getValue();
          if (xmlFiles != null) {
            locations.addAll(xmlFiles);
          }
        }
        else {
          final String value = stringAttributeElement.getStringValue();
          if (value != null) {
            locations.addAll(ApplicationContextReferenceConverter.getApplicationContexts(value, context));
          }
        }
      }
    }
  }

  @NotNull
  public List<PsiClass> getConfigurationClasses() {
    List<PsiClass> imported = new SmartList<>();
    for (JamClassAttributeElement jamClassAttributeElement : CLASSES_ATTR_META.getJam(myPsiAnnotation)) {
      ContainerUtil.addIfNotNull(imported, jamClassAttributeElement.getValue());
    }
    if (imported.isEmpty()) {
      addDefaultConfigurations(myPsiClass, imported);
    }
    return imported;
  }

  private static void addDefaultConfigurations(@NotNull PsiClass owner, @NotNull List<PsiClass> psiClasses) {
    for (PsiClass psiClass : owner.getAllInnerClasses()) {
      if (psiClass.hasModifierProperty(PsiModifier.STATIC) &&
          SpringCommonUtils.isConfigurationOrMeta(psiClass)) {
        psiClasses.add(psiClass);
      }
    }
  }

  public Set<String> getActiveProfiles() {
    final SpringActiveProfiles profiles = getSpringActiveProfiles(getPsiElement());
    return profiles != null ? profiles.getActiveProfiles() : Collections.<String>emptySet();
  }

  @Nullable
  public static SpringActiveProfiles getSpringActiveProfiles(@NotNull PsiClass psiClass) {
    final PsiAnnotation annotation = AnnotationUtil.findAnnotation(psiClass, true, SpringAnnotationsConstants.ACTIVE_PROFILES);
    if (annotation != null) {
      return new SpringActiveProfiles(psiClass);
    }
    return null;
  }

  @Override
  public boolean hasLocationsAttribute() {
    return !getLocationsAttributeElement().isEmpty();
  }

  @Override
  public boolean hasValueAttribute() {
    return !getValueAttributeElement().isEmpty();
  }

  @Nullable
  @Override
  public PsiClass getLoaderClass() {
    return LOADER_ATTR_META.getJam(myPsiAnnotation).getValue();
  }
}
