/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.JamClassAttributeElement;
import com.intellij.jam.JamElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.*;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.PsiMember;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

public class SpringComponentScanFilter extends CommonModelElement.PsiBase implements JamElement {

  private static final JamClassAttributeMeta.Collection VALUE_ATTR_META = JamAttributeMeta.classCollection("value");
  private static final JamEnumAttributeMeta.Single<FilterType> FILTER_TYPE_ATTR_META =
    JamAttributeMeta.singleEnum("type", FilterType.class);

  public static final JamAnnotationMeta ANNOTATION_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.COMPONENT_SCAN_FILTER)
      .addAttribute(VALUE_ATTR_META)
      .addAttribute(FILTER_TYPE_ATTR_META);

  private final PsiMember myMember;
  private final PsiElementRef<PsiAnnotation> myAnnotation;

  @SuppressWarnings("unused")
  public SpringComponentScanFilter(PsiMember member) {
    myMember = member;
    myAnnotation = ANNOTATION_META.getAnnotationRef(member);
  }

  @SuppressWarnings("unused")
  public SpringComponentScanFilter(PsiAnnotation annotation) {
    myAnnotation = PsiElementRef.real(annotation);
    myMember = PsiTreeUtil.getParentOfType(annotation, PsiMember.class, true);
  }

  public static final JamClassMeta<SpringComponentScanFilter> META =
    new JamClassMeta<>(SpringComponentScanFilter.class).addAnnotation(ANNOTATION_META);

  public Set<PsiClass> getFilteredClasses() {
    Set<PsiClass> result = ContainerUtil.newLinkedHashSet();
    for (JamClassAttributeElement element : VALUE_ATTR_META.getJam(myAnnotation)) {
      final PsiClass psiClass = element.getValue();
      if (psiClass != null) {
        result.add(psiClass);
      }
    }

    return result;
  }

  public FilterType getFilterType() {
    final FilterType filterType = FILTER_TYPE_ATTR_META.getJam(myAnnotation).getValue();
    if (filterType != null) {
      return filterType;
    }
    return FilterType.ANNOTATION;
  }

  @NotNull
  @Override
  public PsiMember getPsiElement() {
    return myMember;
  }
}
