/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.qualifiers;

import com.intellij.codeInsight.AnnotationTargetUtil;
import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.ide.presentation.Presentation;
import com.intellij.jam.JamElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.psi.*;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.QualifierAttribute;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.SmartList;
import com.intellij.util.xml.NameValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

@Presentation(typeName = "@Qualifier")
public class SpringJamQualifier extends CommonModelElement.PsiBase implements JamElement, SpringQualifier {
  private final PsiAnnotation myAnno;
  protected final PsiModifierListOwner myModifierListOwner;
  private final Project myProject;

  public SpringJamQualifier(@NotNull PsiAnnotation anno, @Nullable PsiModifierListOwner modifierListOwner) {
    myAnno = anno;
    myModifierListOwner = modifierListOwner;
    myProject = anno.getProject();
  }

  public PsiAnnotation getAnnotation() {
    return myAnno;
  }

  @Nullable
  @NameValue
  public String getQualifiedName() {
    return JamCommonUtil.getObjectValue(myAnno.findAttributeValue(null), String.class);
  }

  @NotNull
  public PsiModifierListOwner getPsiElement() {
    return myModifierListOwner == null ? getType() : myModifierListOwner;
  }

  @NotNull
  private PsiClass getType() {
    final String annoQualifiedName = myAnno.getQualifiedName();

    return annoQualifiedName == null ? null : JavaPsiFacade.getInstance(myProject).findClass(annoQualifiedName, myAnno.getResolveScope());
  }

  public PsiClass getQualifierType() {
    return getType();
  }

  public String getQualifierValue() {
    return getQualifiedName();
  }

  @NotNull
  public List<? extends QualifierAttribute> getQualifierAttributes() {
    final PsiNameValuePair[] attributes = myAnno.getParameterList().getAttributes();
    final List<QualifierAttribute> list = new SmartList<>();
    for (final PsiNameValuePair pair : attributes) {
      final String name = pair.getName();
      if (name == null || name.equals("value")) {
        continue;
      }
      list.add(new QualifierAttribute() {
        public String getAttributeKey() {
          return name;
        }

        public Object getAttributeValue() {
          return JamCommonUtil.getObjectValue(pair.getValue(), Object.class);
        }
      });
    }
    return list;
  }

  public static SpringJamQualifier findSpringJamQualifier(@NotNull Module module, @NotNull PsiModifierListOwner element) {
    final JamAnnotationTypeUtil jamAnnotationTypeUtil = JamAnnotationTypeUtil.getInstance(module);
    final List<PsiClass> annotationTypeClasses = jamAnnotationTypeUtil.getQualifierAnnotationTypesWithChildren();

    for (PsiClass annotationTypeClass : annotationTypeClasses) {
      if (isElementTypeAccepted(annotationTypeClass, element)) {
        final String qname = annotationTypeClass.getQualifiedName();
        if (qname != null && !qname.equals(SpringAnnotationsConstants.NAMED)) {
          final PsiAnnotation annotation = AnnotationUtil.findAnnotation(element, true, qname);
          if (annotation != null) {
            return new SpringJamQualifier(annotation, element);
          }
        }
      }
    }
    // @Named can be used with custom qualifier annotations and @Named is annotated with javax.inject.@Qualifier,
    // @Named should be used as qualifier if there are NO other qualifier annotations
    // @Named @MyQualifier public class Foo{}
    final PsiAnnotation namedAnnotation = AnnotationUtil.findAnnotation(element, true, SpringAnnotationsConstants.NAMED);
    if (namedAnnotation != null) {
      return new SpringJamQualifier(namedAnnotation, element);
    }

    return null;
  }

  private static boolean isElementTypeAccepted(@NotNull PsiClass annotationTypeClass, @NotNull PsiModifierListOwner element) {
    if (element instanceof PsiClass) {
      return AnnotationTargetUtil.findAnnotationTarget(annotationTypeClass, PsiAnnotation.TargetType.TYPE) != null;
    }
    if (element instanceof PsiMethod) {
      return AnnotationTargetUtil.findAnnotationTarget(annotationTypeClass, PsiAnnotation.TargetType.METHOD) != null;
    }
    if (element instanceof PsiField) {
      return AnnotationTargetUtil.findAnnotationTarget(annotationTypeClass, PsiAnnotation.TargetType.FIELD) != null;
    }
    if (element instanceof PsiParameter) {
      return AnnotationTargetUtil.findAnnotationTarget(annotationTypeClass, PsiAnnotation.TargetType.PARAMETER) != null;
    }

    return false;
  }
}
