/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.profile;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.ide.presentation.Presentation;
import com.intellij.jam.JamElement;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiMember;
import com.intellij.psi.PsiModifierListOwner;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.SpringProfile;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedHashSet;
import java.util.Set;

@Presentation(typeName = "@Profile")
public class SpringJamProfile extends CommonModelElement.PsiBase implements JamElement, SpringProfile, CommonModelElement {
  private static final JamStringAttributeMeta.Collection<String> VALUE_ATTR_META = JamAttributeMeta.collectionString("value");

  private final JamAnnotationMeta myMeta;
  private final PsiMember myPsiMember;

  public SpringJamProfile(@NotNull PsiAnnotation anno,
                          @NotNull PsiMember psiMember) {
    Pair<PsiMember, JamAnnotationMeta> definingAnnoMeta = getDefiningAnnoMeta(psiMember, anno);
    if (definingAnnoMeta != null) {
      myPsiMember = definingAnnoMeta.first;
      myMeta = definingAnnoMeta.second;
    }
    else {
      myPsiMember = psiMember;
      myMeta = null;
    }
  }

  @Nullable
  private static Pair<PsiMember, JamAnnotationMeta> getDefiningAnnoMeta(PsiMember psiMember, @NotNull PsiAnnotation psiAnnotation) {
    String qualifiedName = psiAnnotation.getQualifiedName();
    while (qualifiedName != null && !SpringAnnotationsConstants.PROFILE.equals(qualifiedName)) {
      psiMember = JavaPsiFacade.getInstance(psiAnnotation.getProject()).findClass(qualifiedName, psiMember.getResolveScope());
      if (psiMember == null) break;
      PsiAnnotation annotation = AnnotationUtil.findAnnotation(psiMember, true, SpringAnnotationsConstants.PROFILE);
      if (annotation == null) break;
      qualifiedName = annotation.getQualifiedName();
    }

    return qualifiedName == null ? null : Pair.create(psiMember, new JamAnnotationMeta(qualifiedName));
  }

  @NotNull
  public Set<String> getNames() {
    Set<String> profiles = new LinkedHashSet<>();
    if (myMeta != null) {
      for (JamStringAttributeElement<String> element : myMeta.getAttribute(myPsiMember, VALUE_ATTR_META)) {
        final String value = element.getValue();
        if (StringUtil.isNotEmpty(value)) {
          for (String profile : StringUtil.split(value, ",")) {
            profiles.add(profile.trim());
          }
        }
      }
    }

    return profiles;
  }

  @NotNull
  public PsiModifierListOwner getPsiElement() {
    return myPsiMember;
  }
}