/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.java;

import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiMethod;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PropertyUtil;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.jam.JamSpringBeanPointer;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.utils.SpringModelUtils;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.DomSpringBeanPointer;
import com.intellij.spring.model.xml.beans.*;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ConcurrentMultiMap;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.MultiMap;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomManager;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericAttributeValue;
import org.jetbrains.annotations.NotNull;

import java.util.*;

/**
 * Caches Spring mappings for given PsiClass.
 *
 * @author Dmitry Avdeev
 */
public class SpringJavaClassInfo {

  private static final Key<SpringJavaClassInfo> KEY = Key.create("Spring Java Class Info");

  private final PsiClass myPsiClass;
  private final CachedValue<Boolean> myIsMapped;
  private final CachedValue<Set<SpringBeanPointer>> myBeans;
  private final CachedValue<MultiMap<String, SpringPropertyDefinition>> myProperties;
  private final CachedValue<MultiMap<PsiMethod, Pair<DomElement, SpringMethodType>>> myMethods;

  @NotNull
  public static SpringJavaClassInfo getSpringJavaClassInfo(final @NotNull PsiClass psiClass) {
    SpringJavaClassInfo info = psiClass.getUserData(KEY);
    if (info == null) {
      info = new SpringJavaClassInfo(psiClass);
      psiClass.putUserData(KEY, info);
    }
    return info;
  }

  private SpringJavaClassInfo(final PsiClass psiClass) {
    myPsiClass = psiClass;
    final Project project = psiClass.getProject();

    final CachedValuesManager cachedValuesManager = CachedValuesManager.getManager(project);
    myIsMapped = cachedValuesManager.createCachedValue(() -> {
      final CommonSpringModel model = SpringModelUtils.getInstance().getPsiClassSpringModel(myPsiClass);

      final boolean exists = SpringModelSearchers.doesBeanExist(model, myPsiClass);
      return new Result<>(exists, getDependencies(project, model));
    }, false);

    myBeans = cachedValuesManager.createCachedValue(() -> {
      final CommonSpringModel model = SpringModelUtils.getInstance().getPsiClassSpringModel(myPsiClass);
      final List<SpringBeanPointer> byInheritance =
        SpringModelSearchers.findBeans(model, SpringModelSearchParameters.byClass(myPsiClass).withInheritors().effectiveBeanTypes());
      return new Result<Set<SpringBeanPointer>>(new LinkedHashSet<>(byInheritance),
                                                getDependencies(project, model));
    }, false);

    myProperties = cachedValuesManager.createCachedValue(() -> {
      final List<DomSpringBeanPointer> list = getMappedDomBeans();
      final MultiMap<String, SpringPropertyDefinition> map = new ConcurrentMultiMap<>();
      for (DomSpringBeanPointer beanPointer : list) {
        if (!beanPointer.isValid()) continue;
        final DomSpringBean bean = beanPointer.getSpringBean();
        if (bean instanceof SpringBean) {
          final List<SpringPropertyDefinition> properties = ((SpringBean)bean).getAllProperties();
          for (SpringPropertyDefinition property : properties) {
            final String propertyName = property.getPropertyName();
            if (propertyName != null) {
              map.putValue(propertyName, property);
            }
          }
        }
      }
      return new Result<>(map, DomManager.getDomManager(project));
    }, false);
    myMethods = cachedValuesManager.createCachedValue(() -> {
      List<PsiMethod> psiMethods = Arrays.asList(psiClass.getMethods());

      final List<DomSpringBeanPointer> list = getMappedDomBeans();
      final MultiMap<PsiMethod, Pair<DomElement, SpringMethodType>> map =
        new ConcurrentMultiMap<>();

      for (DomSpringBeanPointer beanPointer : list) {
        if (!beanPointer.isValid()) continue;
        final DomSpringBean bean = beanPointer.getSpringBean();
        if (bean instanceof SpringBean) {
          final Beans beans = DomUtil.getParentOfType(bean, Beans.class, false);
          if (beans != null) {
            addSpringBeanMethods(map, psiMethods, SpringMethodType.INIT, beans.getDefaultInitMethod());
            addSpringBeanMethods(map, psiMethods, SpringMethodType.DESTROY, beans.getDefaultDestroyMethod());
          }
          final SpringBean springBean = (SpringBean)bean;

          addSpringBeanMethod(map, psiMethods, SpringMethodType.INIT, springBean.getInitMethod());
          addSpringBeanMethod(map, psiMethods, SpringMethodType.DESTROY, springBean.getDestroyMethod());

          for (LookupMethod lookupMethod : springBean.getLookupMethods()) {
            addSpringBeanMethod(map, psiMethods, SpringMethodType.LOOKUP, lookupMethod.getName());
          }
        }
      }

      final CommonSpringModel springModel = SpringModelUtils.getInstance().getPsiClassSpringModel(myPsiClass);
      for (SpringBeanPointer pointer : springModel.getAllDomBeans()) {
        if (!pointer.isValid()) continue;
        final CommonSpringBean commonSpringBean = pointer.getSpringBean();
        if (commonSpringBean instanceof SpringBean && commonSpringBean.isValid()) {
          final SpringBean springBean = (SpringBean)commonSpringBean;
          final GenericAttributeValue<PsiMethod> domFactoryMethod = springBean.getFactoryMethod();
          if (DomUtil.hasXml(domFactoryMethod)) {
            final PsiMethod factoryMethod = domFactoryMethod.getValue();
            if (factoryMethod != null && psiMethods.contains(factoryMethod)) {
              addSpringBeanMethod(map, psiMethods, SpringMethodType.FACTORY, domFactoryMethod);
            }
          }
        }
      }

      return new Result<>(map, DomManager.getDomManager(project), myPsiClass);
    }, false);
  }

  @NotNull
  private Object[] getDependencies(@NotNull Project project, @NotNull CommonSpringModel model) {
    final Set<Object> dependencies = ContainerUtil.newLinkedHashSet();
    ContainerUtil.addIfNotNull(dependencies, myPsiClass.getContainingFile());

    ContainerUtil.addAll(dependencies, SpringModificationTrackersManager.getInstance(project).getOuterModelsDependencies());
    dependencies.addAll(model.getConfigFiles());

    return ArrayUtil.toObjectArray(dependencies);
  }

  private static void addSpringBeanMethod(@NotNull MultiMap<PsiMethod, Pair<DomElement, SpringMethodType>> map,
                                          @NotNull List<PsiMethod> psiMethods,
                                          @NotNull SpringMethodType type,
                                          @NotNull GenericAttributeValue<PsiMethod> genericAttributeValue) {
    if (!DomUtil.hasXml(genericAttributeValue)) return;

    final PsiMethod psiMethod = genericAttributeValue.getValue();
    if (psiMethod != null && psiMethods.contains(psiMethod)) {
      map.putValue(psiMethod, Pair.create(genericAttributeValue, type));
    }
  }

  private static void addSpringBeanMethods(@NotNull MultiMap<PsiMethod, Pair<DomElement, SpringMethodType>> map,
                                           @NotNull List<PsiMethod> psiMethods,
                                           @NotNull SpringMethodType type,
                                           @NotNull GenericAttributeValue<Set<PsiMethod>> genericAttributeValue) {
    if (!DomUtil.hasXml(genericAttributeValue)) return;

    final Set<PsiMethod> methods = genericAttributeValue.getValue();
    if (methods != null) {
      for (PsiMethod psiMethod : methods) {
        if (psiMethods.contains(psiMethod)) {
          map.putValue(psiMethod, Pair.create(genericAttributeValue, type));
        }
      }
    }
  }

  public boolean isMapped() {
    return myIsMapped.getValue();
  }

  public boolean isMappedDomBean() {
    return isMapped() && ContainerUtil.findInstance(myBeans.getValue(), DomSpringBeanPointer.class) != null;
  }

  @NotNull
  public List<DomSpringBeanPointer> getMappedDomBeans() {
    return !isMapped()
           ? Collections.emptyList()
           : ContainerUtil.findAll(myBeans.getValue(), DomSpringBeanPointer.class);
  }

  public boolean isStereotypeJavaBean() {
    return isMapped() && ContainerUtil.findInstance(myBeans.getValue(), JamSpringBeanPointer.class) != null;
  }

  @NotNull
  public List<JamSpringBeanPointer> getStereotypeMappedBeans() {
    return !isMapped()
           ? Collections.emptyList()
           : ContainerUtil.findAll(myBeans.getValue(), JamSpringBeanPointer.class);
  }

  public boolean isAutowired() {
    final List<DomSpringBeanPointer> pointers = getMappedDomBeans();
    for (DomSpringBeanPointer pointer : pointers) {
      final DomSpringBean springBean = pointer.getSpringBean();
      if (springBean instanceof SpringBean) {
        final Autowire autowire = ((SpringBean)springBean).getBeanAutowire();
        if (autowire.isAutowired()) {
          return true;
        }
      }
    }
    return false;
  }

  @NotNull
  public Set<Autowire> getAutowires() {
    Set<Autowire> autowires = EnumSet.noneOf(Autowire.class);
    for (DomSpringBeanPointer pointer : getMappedDomBeans()) {
      final DomSpringBean springBean = pointer.getSpringBean();
      if (springBean instanceof SpringBean) {
        final Autowire autowire = ((SpringBean)springBean).getBeanAutowire();
        if (autowire.isAutowired()) {
          autowires.add(autowire);
        }
      }
    }
    return autowires;
  }

  @NotNull
  public Collection<SpringPropertyDefinition> getMappedProperties(String propertyName) {
    final MultiMap<String, SpringPropertyDefinition> value = myProperties.getValue();
    if (value == null) {
      return Collections.emptyList();
    }
    return value.get(propertyName);
  }

  @NotNull
  public Collection<Pair<DomElement, SpringMethodType>> getMethodTypes(@NotNull PsiMethod psiMethod) {
    final MultiMap<PsiMethod, Pair<DomElement, SpringMethodType>> value = myMethods.getValue();
    if (value == null) {
      return Collections.emptyList();
    }
    return value.get(psiMethod);
  }

  public boolean isMappedProperty(PsiMethod method) {
    final String propertyName = PropertyUtil.getPropertyNameBySetter(method);
    return !getMappedProperties(propertyName).isEmpty();
  }

  public enum SpringMethodType {
    INIT("init"), DESTROY("destroy"), FACTORY("factory"), LOOKUP("lookup");
    private final String myName;

    SpringMethodType(String name) {
      myName = name;
    }

    public String getName() {
      return myName;
    }

  }
}
