/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet;

import com.intellij.openapi.Disposable;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.fileTypes.StdFileTypes;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.openapi.vfs.pointers.VirtualFilePointer;
import com.intellij.openapi.vfs.pointers.VirtualFilePointerManager;
import com.intellij.psi.PsiClassOwner;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiManager;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.List;
import java.util.Set;

/**
 * @author Yann C&eacute;bron
 * @see SpringFacet#addFileSet(String, String)
 */
public abstract class SpringFileSet implements Disposable {

  private final SpringFileSetData myData;
  private final SpringFacet myFacet;
  private final List<VirtualFilePointer> myFiles = new SmartList<>();

  private boolean myAutodetected;

  protected SpringFileSet(String id, String name, SpringFacet parent) {
    this(SpringFileSetData.create(id, name), parent);
  }

  protected SpringFileSet(SpringFileSetData data, SpringFacet facet) {
    myData = data;
    myFacet = facet;

    for (String url : data.getFiles()) {
      myFiles.add(createVirtualFilePointer(url));
    }
  }

  protected SpringFileSet(SpringFileSet original) {
    myFacet = original.getFacet();
    myData = SpringFileSetData.create(original.getData());
    myFiles.addAll(original.getFiles());
    myAutodetected = original.isAutodetected();
    myData.setActiveProfiles(original.getActiveProfiles());
  }

  @NotNull
  public String getId() {
    return myData.getId();
  }

  public String getName() {
    return myData.getName();
  }

  public void setName(@NotNull final String name) {
    myData.setName(name);
  }

  public String getQualifiedName() {
    return SpringFileSetService.getInstance().getQualifiedName(this);
  }

  public abstract Icon getIcon();

  public boolean isNew() {
    return false;
  }

  public boolean isAutodetected() {
    return myAutodetected;
  }

  public void setAutodetected(final boolean autodetected) {
    myAutodetected = autodetected;
  }

  public boolean isRemoved() {
    return myData.isRemoved();
  }

  void setRemoved(final boolean removed) {
    myData.setRemoved(removed);
  }

  public SpringFacet getFacet() {
    return myFacet;
  }

  SpringFileSetData getData() {
    return myData;
  }


  // profiles

  @NotNull
  public Set<String> getActiveProfiles() {
    return myData.getActiveProfiles();
  }

  public void setActiveProfiles(@NotNull Set<String> activeProfiles) {
    myData.setActiveProfiles(activeProfiles);
  }


  // dependencies

  /**
   * @return All resolved dependency filesets (currently 0..1).
   */
  public Set<SpringFileSet> getDependencyFileSets() {
    Set<SpringFileSet> sets = ContainerUtil.newLinkedHashSet();
    for (String dependencyId : myData.getDependencies()) {
      final SpringFileSet fileSet = SpringFileSetService.getInstance().findDependencyFileSet(this, dependencyId);
      ContainerUtil.addIfNotNull(sets, fileSet);
    }
    return sets;
  }

  public void setDependencies(List<SpringFileSet> springFileSets) {
    myData.getDependencies().clear();
    for (SpringFileSet dependency : springFileSets) {
      addDependency(dependency);
    }
  }

  public void addDependency(@NotNull SpringFileSet springFileSet) {
    myData.addDependency(getDependencyIdFor(springFileSet));
  }

  public void removeDependency(@NotNull SpringFileSet springFileSet) {
    myData.getDependencies().remove(getDependencyIdFor(springFileSet));
  }

  private String getDependencyIdFor(SpringFileSet other) {
    return SpringFileSetService.getInstance().getDependencyIdFor(this, other);
  }


  // configuration files

  @NotNull
  public List<VirtualFilePointer> getFiles() {
    return myFiles;
  }

  @NotNull
  public Set<VirtualFilePointer> getXmlFiles() {
    return getConfigFiles(StdFileTypes.XML);
  }

  @NotNull
  // java, groovy, scala
  public Set<VirtualFilePointer> getCodeConfigurationFiles() {
    Set<VirtualFilePointer> filePointers = ContainerUtil.newLinkedHashSet();
    for (VirtualFilePointer virtualFilePointer : myFiles) {
      if (!virtualFilePointer.isValid()) continue;

      VirtualFile virtualFile = virtualFilePointer.getFile();
      if (virtualFile != null &&
          !isFileType(StdFileTypes.XML, virtualFile) &&
          !isFileType(StdFileTypes.PROPERTIES, virtualFile)) {
        PsiFile psiFile = PsiManager.getInstance(getFacet().getModule().getProject()).findFile(virtualFile);
        if (psiFile instanceof PsiClassOwner) {
          filePointers.add(virtualFilePointer);
        }
      }
    }
    return filePointers;
  }

  private Set<VirtualFilePointer> getConfigFiles(@NotNull final FileType fileType) {
    Set<VirtualFilePointer> filePointers = ContainerUtil.newLinkedHashSet();
    for (VirtualFilePointer virtualFilePointer : myFiles) {
      if (!virtualFilePointer.isValid()) continue;

      VirtualFile virtualFile = virtualFilePointer.getFile();
      if (virtualFile != null && isFileType(fileType, virtualFile)) {
        filePointers.add(virtualFilePointer);
      }
    }
    return filePointers;
  }

  private static boolean isFileType(@NotNull final FileType fileType, @NotNull VirtualFile file) {
    return fileType.equals(file.getFileType());
  }

  @NotNull
  public Set<VirtualFilePointer> getPropertiesFiles() {
    return getConfigFiles(StdFileTypes.PROPERTIES);
  }

  public boolean hasFile(@Nullable VirtualFile file) {
    if (file == null) {
      return false;
    }

    for (VirtualFilePointer virtualFilePointer : myFiles) {
      if (!virtualFilePointer.isValid()) continue;

      final VirtualFile virtualFile = virtualFilePointer.getFile();
      if (virtualFile != null && file.equals(virtualFile)) {
        return true;
      }
    }
    return false;
  }

  protected VirtualFilePointer createVirtualFilePointer(@NotNull String url) {
    return VirtualFilePointerManager.getInstance().create(url, this, null);
  }

  public void addFile(@NotNull final VirtualFile file) {
    addFile(file.getUrl());
  }

  public void addFile(@NonNls String url) {
    if (!StringUtil.isEmptyOrSpaces(url)) {
      myFiles.add(createVirtualFilePointer(url));
      myData.addFile(url);
    }
  }

  public void removeFile(VirtualFilePointer file) {
    myFiles.remove(file);
    myData.removeFile(file.getUrl());
  }

  public void dispose() {
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    SpringFileSet set = (SpringFileSet)o;

    if (getData() != null ? !getData().equals(set.getData()) : set.getData() != null) return false;
    if (getFacet() != null ? !getFacet().equals(set.getFacet()) : set.getFacet() != null) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = getFacet() != null ? getFacet().hashCode() : 0;
    result = 31 * result + (getData() != null ? getData().hashCode() : 0);
    return result;
  }
}
