/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring;

import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NotNullLazyKey;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.contexts.model.LocalAnnotationModel;
import com.intellij.spring.contexts.model.LocalXmlModel;
import com.intellij.spring.contexts.model.SpringModel;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Set;

/**
 * Provides access to Spring models and their dependencies.
 */
public abstract class SpringManager {

  private final static NotNullLazyKey<SpringManager, Project> INSTANCE_CACHE = ServiceManager.createLazyKey(SpringManager.class);

  public static SpringManager getInstance(Project project) {
    return INSTANCE_CACHE.getValue(project);
  }

  @NotNull
  public abstract Object[] getModelsDependencies(@NotNull Module module, Object... additional);

  // scans all modules in project. returns SpringModels or AnnotationSpringModel(if xml files are added as @ImportResource).
  // if file wasn't added in any fileset then create local model for current file (LocalXmlModel or LocalAnnotationModel)
  @NotNull
  public abstract Set<SpringModel> getSpringModelsByFile(@NotNull PsiFile file);

  @Nullable
  public abstract SpringModel getSpringModelByFile(@NotNull PsiFile file);

  @Nullable
  public abstract LocalXmlModel getLocalSpringModel(@NotNull XmlFile file);

  @Nullable
  public abstract LocalXmlModel getLocalSpringModel(@NotNull XmlFile file, @NotNull Module module);

  @Nullable
  public abstract LocalAnnotationModel getLocalSpringModel(@NotNull PsiClass psiClass);

  @Nullable
  public abstract LocalAnnotationModel getLocalSpringModel(@NotNull PsiClass psiClass, @NotNull Module module);

  /**
   * Returns all models configured in given module including all dependencies.
   *
   * @param module a module.
   * @return Models or empty set if none configured.
   * @see #getCombinedModel(Module)
   */
  @NotNull
  public abstract Set<SpringModel> getAllModels(@NotNull Module module);

  /**
   * Returns all models configured in given module without dependencies.
   * <p/>
   * Usually, you will want to use {@link #getAllModels(Module)}.
   *
   * @param module Module.
   * @return Models or empty set if none configured.
   * @since 14
   */
  public abstract Set<SpringModel> getAllModelsWithoutDependencies(@NotNull Module module);

  /**
   * Returns result of merging all models configured in given module.
   *
   * @param module a module.
   * @return Merged model.
   * @see #getAllModels(Module)
   */
  @NotNull
  public abstract SpringModel getCombinedModel(@NotNull Module module);
}
