package com.intellij.sql.psi;

import java.sql.Types;

/**
 * @author Gregory.Shrago
 */
public abstract class SqlType {

  public abstract boolean isPrimitive();

  public abstract String getDisplayName();

  public static final SqlPrimitiveType UNKNOWN = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "unknown";
    }
  };

  public static final SqlPrimitiveType DEFAULT = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "default";
    }
  };

  public static final SqlPrimitiveType INTEGER = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "number";
    }
  };

  public static final SqlPrimitiveType REAL = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "float";
    }
  };

  public static final SqlPrimitiveType STRING = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "string";
    }
  };

  public static final SqlPrimitiveType BOOLEAN = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "boolean";
    }
  };

  public static final SqlPrimitiveType DATE_TIME = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "datetime";
    }
  };

  public static final SqlPrimitiveType DATE = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "date";
    }
  };

  public static final SqlPrimitiveType TIME = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "time";
    }
  };

  public static final SqlPrimitiveType TIMESTAMP = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "timestamp";
    }
  };

  public static final SqlPrimitiveType INTERVAL = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "interval";
    }
  };

  public static final SqlPrimitiveType BYTES = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "bytes";
    }
  };

  public static final SqlPrimitiveType REFERENCE = new SqlPrimitiveType() {
    public String getDisplayName() {
      return "reference";
    }
  };

  public static final SqlPrimitiveType ARRAY = new SqlPrimitiveType() {
    @Override
    public String getDisplayName() {
      return "array";
    }
  };
  
  public static final SqlPrimitiveType COLLECTION = new SqlPrimitiveType() {
    @Override
    public String getDisplayName() {
      return "collection";
    }
  }; // not a primitive

  public static SqlPrimitiveType findByJdbcType(final int jdbcType) {
    switch (jdbcType) {
      case Types.BIT:
      case Types.TINYINT:
      case Types.SMALLINT:
      case Types.INTEGER:
      case Types.BIGINT:
      case Types.CHAR:
        return INTEGER;
      case Types.FLOAT:
      case Types.REAL:
      case Types.DOUBLE:
      case Types.NUMERIC:
      case Types.DECIMAL:
        return REAL;
      case Types.VARCHAR:
      case Types.LONGVARCHAR:
      case Types.CLOB:
        return STRING;
      case Types.DATE:
        return DATE;
      case Types.TIME:
        return TIME;
      case Types.TIMESTAMP:
        return TIMESTAMP;
      case Types.BINARY:
      case Types.VARBINARY:
      case Types.LONGVARBINARY:
      case Types.JAVA_OBJECT:
      case Types.BLOB:
        return BYTES;
        //case Types.DISTINCT:
        //case Types.STRUCT:
        //case Types.OTHER:
      case Types.ARRAY:
        return ARRAY;
        //case Types.REF:
        //case Types.DATALINK:
      case Types.BOOLEAN:
        return BOOLEAN;
      default:
        return UNKNOWN;
    }
  }

  @Override
  public String toString() {
    return getDisplayName();
  }
}
