import {ProjectResult} from "./ts-session";
import {isLogEnabled, serverLogger} from "./logger-impl";
import {HolderContainer} from "./compile-info-holder";
import {DiagnosticsContainer, extendEx} from "./util";

export function extendProjectService(TypeScriptProjectService: typeof ts.server.ProjectService,
                                     ts_impl: typeof ts,
                                     host: ts.server.ServerHost,
                                     projectEmittedWithAllFiles: HolderContainer,
                                     projectErrors: DiagnosticsContainer,
                                     isVersionTypeScript15: boolean,
                                     isVersionTypeScript16: boolean,
                                     isVersionTypeScript17: boolean,
                                     commonDefaultOptions: any) {


    extendEx(TypeScriptProjectService, "fileDeletedInFilesystem", function (this: typeof ts.server.ProjectService,
                                                                            oldFunction: (info: ts.server.ScriptInfo) => void,
                                                                            args: any[]) {
        let info: ts.server.ScriptInfo = args[0];
        let filePath = ts_impl.normalizePath(info.fileName);
        let emittedWithAllFiles = projectEmittedWithAllFiles.value;
        for (let el in emittedWithAllFiles) {
            if (emittedWithAllFiles.hasOwnProperty(el)) {
                let holder = emittedWithAllFiles[el];
                if (holder) {
                    holder.resetForFile(filePath);
                }
            }
        }
        oldFunction.apply(this, args);
    });

    extendEx(TypeScriptProjectService, "createInferredProject", function (this: ts.server.ProjectService,
                                                                          oldFunction: (root: ts.server.ScriptInfo) => ts.server.Project,
                                                                          args: any[]) {
        let project: ts.server.Project = oldFunction.apply(this, args);

        if (commonDefaultOptions && project && project.compilerService) {
            let commonSettings = project.compilerService.settings;
            let res: ts.CompilerOptions = {}
            if (commonSettings) {
                for (const id in commonSettings) {
                    res[id] = (<any>commonSettings)[id];
                }
            }
            for (const id in commonDefaultOptions) {
                res[id] = (<any>commonDefaultOptions)[id];
            }

            project.compilerService.setCompilerOptions(res);
        }

        return project;
    });

    extendEx(TypeScriptProjectService, "watchedProjectConfigFileChanged", function (this: ts.server.ProjectService,
                                                                                    oldFunction: (project: ts.server.Project) => void,
                                                                                    args: any[]) {
        let project: ts.server.Project = args[0];
        let projectFilename = project.projectFilename;
        cleanCachedData(projectFilename);

        if (isVersionTypeScript15) {
            return;
        }

        oldFunction.apply(this, args);

        serverLogger("Watcher — project changed " + (projectFilename ? projectFilename : "unnamed"), true);
    });

    extendEx(TypeScriptProjectService, "configFileToProjectOptions", function (this: ts.server.ProjectService,
                                                                               oldFunction: (configFilename: string) => {
                                                                                 succeeded: boolean;
                                                                                 projectOptions?: ts.server.ProjectOptions;
                                                                                 error?: ts.server.ProjectOpenResult;
                                                                             },
                                                                               args: any[]) {
        let configFilename: string = args[0];

        let normalizedConfigName = ts_impl.normalizePath(configFilename);
        if (normalizedConfigName) {
            if (isLogEnabled) {
                serverLogger("Parse config normalized path " + normalizedConfigName);
            }
            let value = projectErrors.value;
            value[normalizedConfigName] = null;
        }

        if (isVersionTypeScript15 || isVersionTypeScript16 || isVersionTypeScript17) {
            let result = oldFunction.apply(this, args);
            setProjectLevelError(result, configFilename);
            return result;
        }

        let configFileToProjectOptions: ProjectResult = configFileToProjectOptionsExt(configFilename);

        configFileToProjectOptions = copyOptionsWithResolvedFilesWithoutExtensions(host, configFileToProjectOptions, this, ts_impl);
        if (isLogEnabled) {
            serverLogger("Parse config result options: " + JSON.stringify(configFileToProjectOptions));
        }

        setProjectLevelError(configFileToProjectOptions, normalizedConfigName);

        return configFileToProjectOptions;
    });


    /**
     * copy of super#configFileToProjectOptions()
     */
    function configFileToProjectOptionsExt(configFilename: string): ProjectResult {

        if (isLogEnabled) {
            serverLogger("Parse config " + configFilename);
        }
        configFilename = ts_impl.normalizePath(configFilename);
        // file references will be relative to dirPath (or absolute)
        const dirPath = ts_impl.getDirectoryPath(configFilename);
        const contents = host.readFile(configFilename);
        const rawConfig: { config?: ts.server.ProjectOptions; error?: any } = ts_impl.parseConfigFileTextToJson(configFilename, contents);
        if (rawConfig.error) {
            if (isLogEnabled) {
                serverLogger("Parse config error " + JSON.stringify(rawConfig.error));
            }

            return {succeeded: false, errors: [rawConfig.error]};
        }
        else {
            let parsedJsonConfig = rawConfig.config;
            const parsedCommandLine = ts_impl.parseJsonConfigFileContent(parsedJsonConfig, host, dirPath, /*existingOptions*/ {}, configFilename);

            if (parsedCommandLine.errors && (parsedCommandLine.errors.length > 0)) {
                let result: ProjectResult = {succeeded: false, errors: parsedCommandLine.errors};
                // if (parsedCommandLine.fileNames && parsedCommandLine.options) {
                //     const projectOptions = this.createProjectOptions(parsedCommandLine, parsedJsonConfig);
                //     result.projectOptions = projectOptions;
                // }

                return result;
            }
            else {
                const projectOptions = createProjectOptionsForCommandLine(parsedCommandLine, parsedJsonConfig);

                return {succeeded: true, projectOptions};
            }
        }
    }

    function createProjectOptionsForCommandLine(parsedCommandLine: ts.ParsedCommandLine, parsedJsonConfig: ts.server.ProjectOptions): ts.server.ProjectOptions {
        const projectOptions: ts.server.ProjectOptions = {
            files: parsedCommandLine.fileNames,
            compilerOptions: parsedCommandLine.options,
        };
        if ((<any>parsedCommandLine).wildcardDirectories) {
            projectOptions.wildcardDirectories = (<any>parsedCommandLine).wildcardDirectories;
        }

        if (parsedJsonConfig && parsedJsonConfig.compileOnSave === false) {
            projectOptions.compileOnSave = false;
        }

        return projectOptions;
    }


    function cleanCachedData(projectFilename: string) {
        if (projectFilename) {
            projectEmittedWithAllFiles.reset();
            projectErrors.reset();
        }
    }


    function setProjectLevelError(configFileToProjectOptions: ProjectResult, normalizedConfigName: string) {
        if (configFileToProjectOptions.errors) {
            let errors: ts.Diagnostic[] = configFileToProjectOptions.errors;
            if (errors.length > 0) {
                let errorsForService: ts.server.protocol.Diagnostic[] = errors.map(el => {
                    return {
                        end: undefined,
                        start: undefined,
                        text: ts_impl.flattenDiagnosticMessageText(el.messageText, "\n")
                    }
                });
                let value = projectErrors.value;
                value[normalizedConfigName] = errorsForService;

                //back compatibility 1.8
                configFileToProjectOptions.error = {errorMsg: errorsForService[0].text};
            }
        } else if (configFileToProjectOptions.error) {
            let error = configFileToProjectOptions.error;
            let errorMessage = error.errorMsg ? error.errorMsg : "Error parsing tsconfig";
            let value = projectErrors.value;
            value[normalizedConfigName] = [{
                text: errorMessage,
                end: undefined,
                start: undefined,
            }];
        }
    }

}



export function copyOptionsWithResolvedFilesWithoutExtensions(host: ts.server.ServerHost, configFileToProjectOptions: ProjectResult, projectService: ts.server.ProjectService, ts_impl: any): ProjectResult {
    function getBaseFileName(path: string) {
        if (path === undefined) {
            return undefined;
        }
        let i = path.lastIndexOf(ts_impl.directorySeparator);
        return i < 0 ? path : path.substring(i + 1);
    }


    if (!configFileToProjectOptions || !configFileToProjectOptions.projectOptions) {
        return configFileToProjectOptions;
    }
    let projectOptions = configFileToProjectOptions.projectOptions;
    let files = projectOptions.files;

    if (!files) {
        return configFileToProjectOptions;
    }


    let compilerOptions = projectOptions.compilerOptions;
    let extensions: string[] = ts_impl.getSupportedExtensions(compilerOptions);
    let newFiles: string[] = [];

    let hasOverrides: boolean = false;
    l: for (let file of files) {
        let fileName = getBaseFileName(file);
        for (let extension of extensions) {
            if (fileName.lastIndexOf(extension) > 0) {
                newFiles.push(file);
                continue l;
            }
        }
        for (let extension of extensions) {
            if (host.fileExists(file + extension)) {
                hasOverrides = true;
                newFiles.push(file + extension);
                continue l;
            }
        }

        newFiles.push(file);
    }

    if (!hasOverrides) {
        return configFileToProjectOptions;
    }

    let newOptions: ProjectResult = {
        succeeded: configFileToProjectOptions.succeeded,
        projectOptions: {
            compilerOptions: compilerOptions,
            files: newFiles,
            wildcardDirectories: projectOptions.wildcardDirectories,
            compileOnSave: projectOptions.compileOnSave
        }
    }
    if (configFileToProjectOptions.error) {
        newOptions.error = configFileToProjectOptions.error;
    }
    if (configFileToProjectOptions.errors) {
        newOptions.errors = configFileToProjectOptions.errors;
    }


    return newOptions;
}
