package com.intellij.psi.css.impl.util.table;

import com.intellij.psi.PsiElement;
import com.intellij.psi.css.CssPropertyValue;
import com.intellij.psi.css.CssTerm;
import com.intellij.psi.css.CssTermType;
import com.intellij.psi.util.PsiTreeUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * @author spleaner
*/
public class CssCompletionContext {
  private final PsiElement myElement;
  private final CssCompletionContext myParent;
  private final boolean myCurrent;

  private CssCompletionContext(@Nullable final PsiElement element, @Nullable CssCompletionContext parent, boolean current) {
    myElement = element;
    myParent = parent;
    myCurrent = current;
  }

  public boolean isEmpty() {
    return myElement == null && myParent == null;
  }

  @Nullable
  public String getElementText() {
    return myElement == null ? null : myElement.getText().toLowerCase();
  }

  public int getLength() {
    int length = 0;
    CssCompletionContext c = this;
    while (c != null) {
      length += (c.getElement() == null) ? 0 : 1;
      c = c.getParent();
    }

    return length;
  }

  public boolean isValueBelongs(@NotNull final CssPropertyValue value) {
    final CssCompletionContext parent = getParent();
    return value.isValueBelongs(getElement()) || (parent != null && parent.isValueBelongs(value));
  }

  public static CssCompletionContext createContext(@Nullable final PsiElement element,
                                                   @Nullable final CssCompletionContext parent) {
    return createContext(element, parent, false);
  }

  public static CssCompletionContext createContext(@Nullable final PsiElement element,
                                                   @Nullable final CssCompletionContext parent,
                                                   boolean current) {
    return new CssCompletionContext(element, parent, current);
  }

  public PsiElement getElement() {
    return myElement;
  }

  public CssCompletionContext getParent() {
    return myParent;
  }

  public boolean matchesTermType(@Nullable CssTermType type) {
    if (type == null) {
      return true;
    }

    if (myElement == null) {
      return false;
    }

    final CssTerm cssTerm = PsiTreeUtil.getParentOfType(myElement, CssTerm.class, false);
    return cssTerm != null && type == cssTerm.getTermType();
  }

  @SuppressWarnings({"HardCodedStringLiteral"})
  public String toString() {
    final StringBuilder buffer = new StringBuilder();
    CssCompletionContext c = this;
    while (c != null) {
      buffer.insert(0, c.getElementText()).insert(0, ' ');

      c = c.getParent();
    }

    return buffer.insert(0, "Context:").toString();
  }

  public boolean isCurrent() {
    return myCurrent;
  }
}
