/*
 * Copyright 2015-2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j;

/**
 * {@code AssertionFailedError} is an <em>initial draft</em> for a common
 * base class for test-related {@link AssertionError AssertionErrors}.
 *
 * <p>In addition to a message and a cause this class stores the expected
 * and actual values of an assertion using the {@link ValueWrapper} type.
 *
 * <p><strong>WARNING</strong>: this is a <em>work in progress</em> and
 * is therefore guaranteed to undergo heavy revisions in the near future
 * based on community feedback.
 *
 * @author Sam Brannen
 * @author Marc Philipp
 * @since 1.0
 */
public class AssertionFailedError extends AssertionError {

	private static final long serialVersionUID = 1L;

	private final ValueWrapper expected;
	private final ValueWrapper actual;

	/**
	 * Constructs an {@code AssertionFailedError} with no message, no cause,
	 * and no expected/actual values.
	 */
	public AssertionFailedError() {
		this(null);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message, no cause,
	 * and no expected/actual values.
	 */
	public AssertionFailedError(String message) {
		this(message, null);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message and
	 * expected/actual values but without a cause.
	 */
	public AssertionFailedError(String message, Object expected, Object actual) {
		this(message, expected, actual, null);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message and a cause
	 * but without expected/actual values.
	 */
	public AssertionFailedError(String message, Throwable cause) {
		this(message, null, null, cause);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message,
	 * expected/actual values, and a cause.
	 */
	public AssertionFailedError(String message, Object expected, Object actual, Throwable cause) {
		this(message, ValueWrapper.create(expected), ValueWrapper.create(actual), cause);
	}

	private AssertionFailedError(String message, ValueWrapper expected, ValueWrapper actual, Throwable cause) {
		super((message == null || message.trim().length() == 0) ? "" : message);
		initCause(cause);
		this.expected = expected;
		this.actual = actual;
	}

	/**
	 * Returns {@code true} if the expected value is defined, i.e. was passed
	 * to the constructor.
	 *
	 * @see #getExpected()
	 */
	public boolean isExpectedDefined() {
		return this.expected != null;
	}

	/**
	 * Returns {@code true} if the actual value is defined, i.e. was passed
	 * to the constructor.
	 *
	 * @see #getActual()
	 */
	public boolean isActualDefined() {
		return this.actual != null;
	}

	/**
	 * Returns the wrapped expected value if it is defined; otherwise {@code null}.
	 *
	 * @see #isExpectedDefined()
	 */
	public ValueWrapper getExpected() {
		return this.expected;
	}

	/**
	 * Returns the wrapped actual value if it is defined; otherwise {@code null}.
	 *
	 * @see #isActualDefined()
	 */
	public ValueWrapper getActual() {
		return this.actual;
	}

}
