/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */


/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4,
maxerr: 50, node: true */
/*global */
(function () {
    "use strict";

    var fs              = require('fs'),
        path            = require('path'),
        mkpath          = require('mkpath'),
        CompilerManager = require('./CompilerManager');

    // makes a file in a path where directories may or may not have existed before
    function mkfile(filepath, content, callback) {
        mkpath(path.dirname(filepath), function (err) {
            if (err) {
                return callback ? callback(err) : undefined;
            }
            fs.writeFile(filepath, content, callback);
        });
    }
    
    /**
     * Creates an errObject
     * @param   {string} filename 
     * @param   {string} type     "error"/ "waring"
     * @param   {number} line     
     * @param   {number} column
     * @param   {string} err      the error description
     * @returns {object}          an object with the above fields
     */
    function createErrObj(filename, type, line, column, err) {
        return {
            type: type,
            line: line,
            column: column,
            filename: filename,
            message: err
        };
    }
    
    /**
     * Creates an errObject for compassInit
     * @param   {string} initStatus success or failure 
     * @param   {string} type     "error"/ "waring"
     * @param   {number} msg      success or error msg     
     * @returns {object}          an object { 
                                              outputType : "CompassInit",
                                              status: initStatus,
                                              message: msg 
                                            }
     */
    function createCompassInitReturnObject(initStatus, msg) {
        return {
            outputType : "CompassInit",
            status: initStatus,
            message: msg
        };
    }
    /**
     * Checks if the given path is of the extension
     * @param   {string} filePath  "c:/a.txt"
     * @param   {string} extension like "txt"
     * @returns {boolean}  [[Description]]
     */
    function _isOfExtension(filePath, extension) {
        var extName = path.extname(filePath);
        if (extName) {
            extName = extName.toLowerCase();
            if (extName === ("." + extension.toLowerCase())) {
                return true;
            }
        }
        return false;
    }
    
    function isLessFile(filePath) {
        return _isOfExtension(filePath, "less");
    }
    
    function isSassFile(filePath) {
        return _isOfExtension(filePath, "sass") || _isOfExtension(filePath, "scss");
    }
    
    function isSassPartialFile(filePath) {
        if (_isOfExtension(filePath, "sass") || _isOfExtension(filePath, "scss")) {
            return path.basename(filePath).indexOf('_') === 0;
        }
        return false;
    }
    
    
    /**
     * Removes the trailing slash(back or front) from a path, if it has one.
     *
     * @param {string} path
     * @return {string}
     */
    function stripTrailingSlash(path) {
        if (path && path[path.length - 1] === "/") {
            return path.slice(0, -1);
        } else if (path && path[path.length - 1] === "\\") {
            return path.slice(0, -1);
        } else {
            return path;
        }
    }
    
    function createCompiledFileListReturnObject(listOfFiles, outOfSiteFile) {
        if (listOfFiles && (typeof listOfFiles === "object") &&  listOfFiles.length > 0) {
            //First get the CSS file names from the input source file
            var index = 0,
                cssFilesList = [];
            
            for (index = 0; index < listOfFiles.length; ++index) {
                if (isSassFile(listOfFiles[index]) && !isSassPartialFile(listOfFiles[index])) {
                    cssFilesList.push(CompilerManager.getCSSFileName(listOfFiles[index], outOfSiteFile));
                }
            }
            
            return {
                outputType : "compileFileList",
                numberOfFilesGettingCompiled : cssFilesList.length,
                fileList : cssFilesList
            };
        }
    }
    
    exports.mkfile                          = mkfile;
    exports.isLessFile                      = isLessFile;
    exports.isSassFile                      = isSassFile;
    exports.isSassPartialFile               = isSassPartialFile;
    exports.createErrObj                    = createErrObj;
    exports.stripTrailingSlash              = stripTrailingSlash;
    exports.createCompassInitReturnObject   = createCompassInitReturnObject;
    exports.createCompiledFileListReturnObject = createCompiledFileListReturnObject;

}());
