/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 
*/

/*global require, module*/
/*jslint vars:true regexp: true*/

var URI = require('URIjs'),
    randomstring = require('randomstring'),
    requestUtils = require('./requestUtils.js'),
    config = require('../../coreConfig/config.js'),
    UAParser = require('ua-parser-js');
    
var securityDetails = {
    SECURED_PRIMARY_URL_PATHNAME: '',
    SECURED_URL_KEY: '',
    APP_URL_PATHNAME: '/preview/app/index.html',
    RANDOM_STRING_TO_URL_PARTS_MAP: {},
    URL_PARTS_MAP_TO_RANDOM_STRING: [], //array containing url parts in sequence, array has objects { original : "originalStr", replacement: "replacementString"}
    URL_CONTAINS_TRAILING_SLASH: false,
    SOCKET_CLIENT_DATA: {},
    RANDOM_STRING_LENGTH: 7
};

var getSecuredURL = function () {
    'use strict';
    return securityDetails.SECURED_PRIMARY_URL_PATHNAME;
};

var clearSecurityDetails = function () {
    'use strict';
    securityDetails.SECURED_PRIMARY_URL_PATHNAME = '';
    securityDetails.SECURED_URL_KEY = '';
    securityDetails.RANDOM_STRING_TO_URL_PARTS_MAP = {};
    securityDetails.URL_PARTS_MAP_TO_RANDOM_STRING = [];
    securityDetails.URL_CONTAINS_TRAILING_SLASH = false;
};

var secureURL = function (pathname) {
    'use strict';
    clearSecurityDetails();
    
    if (pathname) {
        var parts = pathname.split(/\//),
            securedParts = [],
            i;
        
        securityDetails.SECURED_URL_KEY = randomstring.generate(securityDetails.RANDOM_STRING_LENGTH);
        securedParts.push(securityDetails.SECURED_URL_KEY);
        for (i = 0; i < parts.length; i += 1) {
            var realPart = parts[i],
                replacement = URI.encode(randomstring.generate(securityDetails.RANDOM_STRING_LENGTH));
            
            if (realPart) {
                securityDetails.RANDOM_STRING_TO_URL_PARTS_MAP[replacement] = realPart;
                securityDetails.URL_PARTS_MAP_TO_RANDOM_STRING.push({'original': realPart, 'replacement': replacement});
                securedParts.push(replacement);
            }
        }
        
        securityDetails.SECURED_PRIMARY_URL_PATHNAME = securedParts.join('/');
        if (pathname[pathname.length - 1] === '/') {
            securityDetails.URL_CONTAINS_TRAILING_SLASH = (securedParts.length > 1); //checks if path had any content apart from / and last character is also slash
            securityDetails.SECURED_PRIMARY_URL_PATHNAME += '/';
        }
    }
    
    return securityDetails.SECURED_PRIMARY_URL_PATHNAME;
};

var secureAuxiliaryURL = function (pathname) {
    'use strict';
    
    if (pathname) {
        var parts = pathname.split(/\//),
            i,
            j,
            newURL = '',
            mismatchFound = false,
            securedParts = [];
        
        securedParts.push(securityDetails.SECURED_URL_KEY);
        for (i = 0, j = 0; i < parts.length; i += 1) {
            if (parts[i] && !mismatchFound) {
                var replaceObj = securityDetails.URL_PARTS_MAP_TO_RANDOM_STRING[j];
                if (replaceObj && replaceObj.original === parts[i] && replaceObj.replacement) {
                    securedParts.push(replaceObj.replacement);
                    j += 1;
                } else {
                    mismatchFound = true;
                }
            }
            
            if (mismatchFound) {
                securedParts.push(parts[i]);
            }
        }
        
        return securedParts.join('/');
    }
    
    return '';
};

var addSocketClientData =  function (id, data) {
    'use strict';
    securityDetails.SOCKET_CLIENT_DATA[id] = data;
};

var removeSocketClientData =  function (id) {
    'use strict';
    delete securityDetails.SOCKET_CLIENT_DATA[id];
};

var verifySecurity =  function (req, res, callback) {
    'use strict';
    //first extract the required URL parts;
    var pathparts = req.url.match(/^\/[^\/]*content\/([^\/]+)\/(.+)$/),
        filePath = '',
        socketID = pathparts[1],
        securedPath = pathparts[2],
        actualPath = '',
        success = true;
    
    //we def need the socket id, if not fail
    if (socketID) {
        //extract the client data and verify useragent and ip
        var decodedID = URI.decode(socketID),
            clientData = securityDetails.SOCKET_CLIENT_DATA[decodedID],
            reqUserAgent = req.header('User-Agent'),
            parser = new UAParser();
        
        if (clientData) {
            parser.setUA(clientData.userAgent);
            if (clientData.ip !== (req.headers['x-forwarded-for'] || req.connection.remoteAddress)) {
                success = false;
            } else if (!(reqUserAgent && (clientData.userAgent === reqUserAgent ||
                reqUserAgent.indexOf(clientData.userAgent) !== -1 || //in IOS chrome, the useragent gets appended with numbers for media, just check inclusion
                (parser.getBrowser().name.toLowerCase().indexOf('chrome') !== -1 && parser.getOS().name.toLowerCase().indexOf('android') !== -1 && reqUserAgent.toLowerCase().indexOf('stagefright') !== -1) || //in android, chrome browsers make request with user agent as stage frite
                (parser.getBrowser().name.toLowerCase().indexOf('iemobile') !== -1 && parser.getOS().name.toLowerCase().indexOf('windows phone') !== -1 && reqUserAgent.toLowerCase().indexOf('nsplayer') !== -1)))) {//in windows phone, iemobile browser makes request with user agent as NSPlayer 
                success = false;
            }
        } else {
            success = false;
        }
    } else {
        success = false;
    }
    
    //assemble parts, first part must be existant, it verifies the request is based on current url
    if (success) {
        var securedParts = securedPath.split(/\//),
            i,
            mismatchFound = false,
            parts = [];
        
        if (securedParts[0] === securityDetails.SECURED_URL_KEY) {
            for (i = 1; i < securedParts.length; i += 1) {
                if (securedParts[i] && !mismatchFound) {
                    var replacement = securityDetails.RANDOM_STRING_TO_URL_PARTS_MAP[securedParts[i]];
                    if (replacement) {
                        parts.push(replacement);
                    } else {
                        mismatchFound = true;
                    }
                }

                if (mismatchFound) {
                    parts.push(securedParts[i]);
                }
            }

            actualPath = parts.join('/');
            if (mismatchFound) {
                if (securedPath[securedPath - 1] === '/') {
                    actualPath += '/';
                }
            } else if (securityDetails.URL_CONTAINS_TRAILING_SLASH) {
                actualPath += '/';
            }
            
        } else {
            success = false;
        }
    }
    
    var isHTML = false;
    var acceptHeader = req.header('accept');
    if (acceptHeader && acceptHeader.indexOf('html') !== 0) {
        //check if we are to serve html
        isHTML = true;
    }
    
    if (success) {
        callback(null, { "path": '/' + actualPath});
    } else {
        if (isHTML) {
            res.status(301);
            res.header('location', requestUtils.createErrorUrl(config.errorMarkers.ERROR_PAGE_LOAD));
            res.end();
        }
        callback('invalidContent', '');
        requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.LOG_ERROR, config.logErrorKeys.EXTERNAL_PAGE_ACCESS);
    }
    
};

module.exports = {
    addSocketClientData: addSocketClientData,
    removeSocketClientData: removeSocketClientData,
    clearSecurityDetails: clearSecurityDetails,
    secureURL: secureURL,
    verifySecurity: verifySecurity,
    getSecuredURL: getSecuredURL,
    secureAuxiliaryURL: secureAuxiliaryURL
};