/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 
*/

/*global module, console, require*/
/*jslint regexp: true*/

var config = require('../../coreConfig/config.js'),
    URI = require('URIjs'),
    securityUtils = require('./securityUtils');

function enterRoute(location, res) {
    'use strict';
    var callback = {},
        errorMessage;
    
    callback.error = function (err) {
        if (err) {
            if (err.http_code) {
                var logMessage = err.http_code + (err.log_message ? ' ' + err.log_message : (err.http_message ? ' ' + err.http_message : ''));
                console.log(logMessage);
                
                if (typeof err.http_message === 'string' || err.http_message instanceof String) {
                    res.send(err.http_code, err.http_message);
                } else {
                    res.setHeader('Content-Type', 'application/json');
                    res.send(err.http_code, JSON.stringify(err.http_message, undefined, 2));
                }
            }
        } else {
            errorMessage = location + ': ';
            if (typeof err === 'string') {
                errorMessage += err;
            } else if (err && err.stack) {
                errorMessage += err.stack;
            } else {
                errorMessage += JSON.stringify(err);
            }
            console.log(errorMessage);
            res.send(config.coreSettings.SERVER_ERROR_STATUS_CODE, 'Internal Server Error');
        }
    };
    callback.success = function () {
    };
    return callback;
}

function getErrorString(req, callback) {
    'use strict';
    /* Ensure that URL matches <DOMAIN>/error/files/.. */
    var pathparts = req.url.match(/^\/[^\/]*error\/([^\/]+)\/(.+)$/),
        errorStr = '';
    
    if (pathparts && pathparts[1] && pathparts[2]) {
        errorStr = pathparts[2];
    }
    callback(null, {error: errorStr});
}

function purgeUrl(url, primaryURL) {
    'use strict';
    
    if (!url) {
        return '';
    }

    var urlObj = new URI(url),
        newUrl = null;
    
    newUrl = '/content/DP_CLIENT_SOCKET_ID/';
    
    if (primaryURL) {
        newUrl += securityUtils.secureURL(urlObj.pathname());
    } else {
        newUrl += securityUtils.secureAuxiliaryURL(urlObj.pathname());
    }
    
    return newUrl;
}

function getSecuredPrimaryURL() {
    'use strict';
    return '/content/DP_CLIENT_SOCKET_ID/' + securityUtils.getSecuredURL();
}

function sanitizeHTMLData(data) {
    'use strict';
    var ret = data;
    ret = ret.replace(/\\/g, '\\\\'); //escape all back slashes
    ret = ret.replace(/'/g, '\\\''); //escape single quote
    ret = ret.replace(/\n/g, '\\n'); //escape newline
    ret = ret.replace(/\//g, '\\/'); //escape forward slash
    return ret;
}

module.exports = {
    enterRoute: enterRoute,
    getErrorString: getErrorString,
    purgeUrl: purgeUrl,
    getSecuredPrimaryURL: getSecuredPrimaryURL,
    sanitizeHTMLData: sanitizeHTMLData
};