/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 
*/

/*global require, module, process, __dirname*/
/*jslint vars:true*/

var requestUtils = require('./coreUtils/requestUtils'),
    step = require('step'),
    mimeUtils = require('mime'),
    config = require('../coreConfig/config.js'),
    fs = require('fs'),
    path = require('path'),
    URI = require('URIjs'),
    assetUtils = require('../../server-main/src/assetUtils.js');

function calcContentType(contentType, assetName) {
    'use strict';
    if (contentType === 'application/unknown') {
        contentType = null;
    }
    return contentType || mimeUtils.lookup(assetName) || 'text/plain';
}


function getErrorContent(assetInfo, callback) {
    'use strict';
    
    var errorStr = assetInfo.error,
        errorContent = '',
        resp = '';
    
    switch (errorStr) {
    case config.errorMarkers.ERROR_UNSAVED_DOCUMENT:
        errorContent = config.errorStrings.ERROR_STRING_UNSAVED_DOCUMENT;
        break;
    case config.errorMarkers.ERROR_NO_HTML_DOCUMENT:
        errorContent = config.errorStrings.ERROR_STRING_NO_HTML_DOCUMENT;
        break;
    case config.errorMarkers.ERROR_NO_DOCUMENT:
        errorContent = config.errorStrings.ERROR_STRING_NO_DOCUMENT;
        break;
    case config.errorMarkers.ERROR_DW_EXITING:
        errorContent = config.errorStrings.ERROR_STRING_DW_EXITING;
        break;
    case config.errorMarkers.ERROR_RECONNECT:
        errorContent = config.errorStrings.ERROR_STRING_RECONNECT;
        break;
    case config.errorMarkers.ERROR_PAGE_LOAD:
        errorContent = config.errorStrings.ERROR_STRING_PAGE_LOAD;
        break;
    default:
        //this should never ever happen
        break;
    }

    var templatePath = __dirname + '/../../client-scripts/static/files/ErrorTemplate.html';
    fs.readFile(templatePath, function (err, data) {
        var content = assetUtils.arrayBuffer2String(data);
        if (content) {
            content = content.replace('%s', errorContent);
        }

        callback(null, content);
    });
    
}

function getLoaderContent(callback) {
    'use strict';
    var loaderPath = __dirname + '/../../client-scripts/static/files/Loader.html';
    fs.readFile(loaderPath, function (err, data) {
        callback(err, data);
    });
}


function getAssetContent(assetInfo, headOnly, reqHeaders, callback) {
    'use strict';
    
    var localPath = assetInfo.path.replace(/%7C/ig, '|'),
        options,
        responseCallback = function (err, statusCode, resp, headers, resourceInfo) {
            if (err) { callback(err); return; }

            if (statusCode === config.coreSettings.OK_STATUS_CODE) {
                if (!headOnly && !resp && (headers['content-length'] !== '0')) {
                    callback('assetsMgr.getAssetContent: empty response value from SC');
                } else {
                    callback(null, statusCode, resp, headers, resourceInfo);
                }
            } else {
                callback(requestUtils.createError(['ALLOW_NOT_FOUND'], 'assetsMgr.getAssetContent', statusCode, headers), statusCode, resp, headers, resourceInfo);
            }
        };

	//Serving From DW, If the assetInfo matches the primary URl, it implies we already have the data and we can serve immediately
    var realURL = requestUtils.constructFullURLFromPath(localPath);
    if (realURL === config.coreSettings.PRIMARY_URL && config.coreSettings.IS_BROWSING_HOME_URL) {
        var headers = {},
            resp = '',
            shouldShowSaveMessage = config.coreSettings.PROXY_HOSTNAME === '' ? config.coreSettings.IS_PRIMARY_URL_RELATED_FILES_DIRTY : (config.coreSettings.IS_PRIMARY_URL_RELATED_FILES_DIRTY || config.coreSettings.IS_PRIMARY_URL_HTML_DIRTY);
            
        headers['content-type'] = 'text/html';
        
        if (shouldShowSaveMessage) {
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.SEEK_LATEST_HTML, '');
            getErrorContent({'error': config.errorMarkers.ERROR_UNSAVED_DOCUMENT}, function (err, data) {
                responseCallback(null, config.coreSettings.OK_STATUS_CODE, data, headers, null);
            });
        } else if (config.coreSettings.PRIMARY_URL_HTML) {
            resp = config.coreSettings.PRIMARY_URL_HTML;
            responseCallback(null, config.coreSettings.OK_STATUS_CODE, resp, headers, null);
        } else {
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.SEEK_LATEST_HTML, '');
            getLoaderContent(function (err, data) {
                responseCallback(null, config.coreSettings.OK_STATUS_CODE, data, headers, null);
            });
        }
        return;
    }
    
    if (config.coreSettings.PROXY_HOSTNAME === '') {
        var urlObj = new URI(localPath);
        if (localPath && urlObj) {
            localPath = URI.decode(urlObj.pathname());
        }
        
        if (process.platform.indexOf('win') === 0 && localPath && localPath.indexOf('/') === 0) {
            localPath = localPath.slice(1); // Remove the leading '/' on Windows
        }
        
        if (localPath && localPath.indexOf('|') !== -1) {
            localPath = localPath.replace('|', ':');
        }
        
        localPath = path.resolve(localPath);
        fs.readFile(localPath, function (err, data) {
            
            var resHeaders = {};
            if (err) {
                responseCallback(requestUtils.createError(['ALLOW_NOT_FOUND'], 'assetsMgr.getAssetContent', config.coreSettings.NOT_FOUND_STATUS_CODE, resHeaders), config.coreSettings.NOT_FOUND_STATUS_CODE, data, resHeaders, null);
                return;
            }
            
            resHeaders['content-type'] = calcContentType(null, assetInfo.path);
            resHeaders['content-length'] = data.length;
            
            responseCallback(err, config.coreSettings.OK_STATUS_CODE, data, resHeaders, null);
        });
    } else {
        options = requestUtils.createHTTPOptionsObject(config.coreSettings.PROXY_HOSTNAME, config.coreSettings.PROXY_PORT, requestUtils.calcURL(assetInfo, true), headOnly ? 'HEAD' : 'GET', reqHeaders);
    
        requestUtils.executeHTTPRequest(options, true, responseCallback);
    }
}

module.exports = {
    getAssetContent: getAssetContent,
    getErrorContent: getErrorContent
};