/**
 * Copyright 2013 Adobe Systems Incorporated.  All rights reserved.
 * Purpose:
 * This Hud is for Text formatting options 
 * Bold italics and link 
 * Commit for this function happens through Text editing
 */

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global liveViewObject, dwObject, Node,KEYCODES, dwExtensionController, DW_EXTENSION_EVENT*/

/*
 *@private initTextFormattingHud 
 *Entry point for Text Formatting HUD. Set on body Onload in HTML
 *@param none
 *@return none
*/
/* All the Constants are defined here*/
/* Constants common across all the HUDS are defined in DW_CONSTANTS in dwUtility.js */
var CONSTANTS = {
    ExtensionID: "TextFormattingHud",
    LinkWidth : 140,
    FormatOptionsWidth : 76,
    FormatOptionsHeight : 28,
    TextHudOffset : 25,
    DWSpanId : "DW_SPAN_ID",
    FontStyleId : "FONT_STYLE_ID",
    FontWeightId: "Font_WEIGHT_ID",
    TextContainer : "DW_SPAN",
    filter : "*"
};
var HUD_IDS = {
    IdContainer : "container",
    IdBold : "bold",
    IdItalic : "italic",
    IdLink : "link",
    IdBrowse : "browse",
    IdInput : "linkInput",
    IdLinkContainer : "linkContainer"
};
var HUD_CLASSES = {
    SelectedBG : "selected"
};
var HEADLIGHTS = {
    ELV_TH: "ELV TH",
    Bold  : "TH Bold",
    italic: "TH Italic",
    link  : "TH LinkField",
    browse: "TH BrowseFile"
};

/* @Constructor
*/
function TextFormattingHud() {
    'use strict';
}

/* @private Initialize
 * initialize Member variabes, add event listeners
 * @param none
 */
TextFormattingHud.prototype.initialize = function () {
    'use strict';
    //Hud State
    this.m_isVisible = false;//Is Hud Visible
    this.insideTextEditing = false; // are we inside text editing session
    this.m_currentElement = null; // current DWSPAN node
    this.m_changedCode = false;
    this.m_textHUDref = null;
    this.setTextHudRef();
    //current state with respect to CurrentRange
    this.currentRange = null; // current selection
    this.boldState = false; // this is what browser tells
    this.italicState = false; // browser tells whether italic or not
    this.linkState = false; // whether the link is present on current selection
    this.isStrongPresent = false; // check if the strong or b tags present on it
    this.isItalicPresent = false; // check if the italic tag is present on it
    this.isBoldCss = false; // check if bold is set thru css
    this.isItalicCss = false; // check if italic is set thru css
    this.linkFieldState = false; //whether the link input field is shown or not
    this.currentLinkValue = null; // currentValue of Link in input field
    //Hud's width height and left position
    this.currentHudWidth = CONSTANTS.FormatOptionsWidth;
    this.currentHudHeight = CONSTANTS.FormatOptionsHeight;
    this.currentHudLeft = 0;
    //references to Hud Components
    this.m_hudContentDiv = document.getElementById(HUD_IDS.IdContainer);
    this.boldElement = document.getElementById(HUD_IDS.IdBold);
    this.italicsElement = document.getElementById(HUD_IDS.IdItalic);
    this.linkElement = document.getElementById(HUD_IDS.IdLink);
    this.linkInputElement = document.getElementById(HUD_IDS.IdInput); // link input field
    this.browseElement = document.getElementById(HUD_IDS.IdBrowse);
    //Add event listeners on browser events
    window.addEventListener("keydown", this.keyDownHandler.bind(this), false);// for escape
    //For Hud components
    this.m_hudContentDiv.addEventListener("click", this.clickHandler.bind(this), false);//set click handler
    this.linkInputElement.addEventListener("keydown", this.commitOnEnter.bind(this), true);
    this.browseElement.addEventListener("keyup", this.LaunchBrowseOnEnter.bind(this), true);// enter/space bar should bring the file selection dialog
    this.browseElement.addEventListener("keydown", this.tabOnBrowse.bind(this), true);// tab should be cyclic
    //Add Listener for events from Other HUD's.
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TEXT_EDIT_BEGIN, this.setElementAndEnableEvents, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TEXT_EDIT_END, this.destroy, this);
    this.populatelocalizedStrings();
};
/**
 * @override attachEventListeners
 * Attach Event Listeners for Selection changed and selection lost inside incontext editing
 * @param none
 */
TextFormattingHud.prototype.attachEventListeners = function () {
	'use strict';
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TEXTEDIT_SELECTION_CHANGED, this.displayHUD, this); //show HUD
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TEXTEDIT_SELECTION_LOST, this.commitAndHideHud, this); //we commit and hide the HUD 

    // from dreamweaver
    dwExtensionController.addListener(window.parent.DW_EVENTS.Textedit_selection_change, this.selectionChangedFromDw, this); //show HUD
    dwExtensionController.addListener(window.parent.DW_EVENTS.Textedit_make_bold, this.onBoldIconClicked, this); //make bold  from menu
    dwExtensionController.addListener(window.parent.DW_EVENTS.TextEdit_make_italic, this.onItalicsIconClicked, this); //make italic from menu
};
/**
 * @override detachEventListeners
 * Detach the event listeners we added previously
 * @param none
 */
TextFormattingHud.prototype.detachEventListeners = function () {
	'use strict';
    dwExtensionController.removeListener(DW_EXTENSION_EVENT.TEXTEDIT_SELECTION_CHANGED, this.displayHUD, this);
    dwExtensionController.removeListener(DW_EXTENSION_EVENT.TEXTEDIT_SELECTION_LOST, this.commitAndHideHud, this);
    dwExtensionController.removeListener(window.parent.DW_EVENTS.Textedit_selection_change, this.selectionChangedFromDw, this);
    dwExtensionController.removeListener(window.parent.DW_EVENTS.Textedit_make_bold, this.onBoldIconClicked, this);
    dwExtensionController.removeListener(window.parent.DW_EVENTS.TextEdit_make_italic, this.onItalicsIconClicked, this);
};
/* Setter Getter functions*/
/* @private setVisibility
 * Set the visibility of the HUD
 * @param visible : true or false.
 */
TextFormattingHud.prototype.setVisibility = function (visible) {
    'use strict';
    this.m_isVisible = visible;
};
/* @private getVisibility
 * get Visibility of the HUD
 * @param none
 */
TextFormattingHud.prototype.getVisibility = function () {
    'use strict';
    return this.m_isVisible;
};
/* @private setTextHudRef
 * set text hud reference
 * @param none
 */
TextFormattingHud.prototype.setTextHudRef = function () {
    'use strict';
    var ref = null;
    if (window.parent && window.parent.globalController) {
        ref = window.parent.globalController.m_txtHud;
        if (ref) {
            this.m_textHUDref = ref;
        }
    }
};
/* @private getVisibility
 * get text hud reference
 * @param none
 */
TextFormattingHud.prototype.getTextHudRef = function () {
    'use strict';
    return this.m_textHUDref;
};
/**
 * @override populatelocalizedStrings
 * populate the localization strings
 * @param none
 */
TextFormattingHud.prototype.populatelocalizedStrings = function () {
    'use strict';
    var globalController = window.parent.globalController;
    this.locStrings = {};
    this.locStrings.boldAddB = globalController.getLocalizedString('Text_Hud_AddBTag_Tooltip');
    this.locStrings.boldAddStrong = globalController.getLocalizedString('Text_Hud_AddStrongTag_Tooltip');
    this.locStrings.boldRemoveB = globalController.getLocalizedString('Text_Hud_RemoveBTag_Tooltip');
    this.locStrings.boldRemoveStrong = globalController.getLocalizedString('Text_Hud_RemoveStrongTag_Tooltip');
    this.locStrings.italicAddI = globalController.getLocalizedString('Text_Hud_AddITag_Tooltip');
    this.locStrings.italicAddEm = globalController.getLocalizedString('Text_Hud_AddEmTag_Tooltip');
    this.locStrings.italicRemoveI = globalController.getLocalizedString('Text_Hud_RemoveITag_Tooltip');
    this.locStrings.italicRemoveEm = globalController.getLocalizedString('Text_Hud_RemoveEmTag_Tooltip');
    //tooltips
    this.linkElement.title = globalController.getLocalizedString('Common_Link_ToolTip');
	this.browseElement.title = globalController.getLocalizedString('Common_Browse_Tooltip');
    //default text
	this.linkInputElement.placeholder = globalController.getLocalizedString('Text_Hud_EnterLink_Tooltip');
    //file selection related
	this.locStrings.fileDialogType = globalController.getLocalizedString('Common_DialogType');
	this.locStrings.fileTitleLink = globalController.getLocalizedString('Common_FileTitleLink');
};
/*EVENT HANDLERS*/
/**
 * @override setElementAndEnableEvents
 * setcurrentElement 
 * attach event listeners
 * add custom data id's for span nodes to distinguish between spans that get
 * added by browser as a result of unbolding/un-italic
 * @param none
 */
TextFormattingHud.prototype.setElementAndEnableEvents = function () {
	'use strict';
    this.insideTextEditing = true; // this flag is neeeded because our visibility might be false but still we are inside text editing
    if (!this.m_textHUDref) {
        this.setTextHudRef();
    }
    this.m_currentElement = this.m_textHUDref.m_elementUnderCurrentEditingSession;
    if (!this.m_currentElement) {
        return;
    }
    this.attachEventListeners(); // attach event listeners general + selection change
	this.initializeContent(this.m_currentElement); // add our id's to all the existing spantags.
};
/**
 * @override initializeContent
 * add DW_SPAN_ID to existing span nodes.
 * @param newTagName : DWSPAN element.
 */
TextFormattingHud.prototype.initializeContent = function (curElement) {
    'use strict';
    if (!curElement) {
		return;
	}
    if (curElement.nodeType === Node.ELEMENT_NODE) {
        if (curElement.tagName === "SPAN") {
            curElement.setAttribute(CONSTANTS.DWSpanId, true);
        }
        if (parent.DWTextEditUtility.hasInlineStyle(curElement, "font-style")) {
            curElement.setAttribute(CONSTANTS.FontStyleId, true);
        }
        if (parent.DWTextEditUtility.hasInlineStyle(curElement, "font-weight")) {
            curElement.setAttribute(CONSTANTS.FontWeightId, true);
        }
    }
	var childNodes = curElement.childNodes;
    var i;
    if (childNodes) {
        for (i = 0; i < childNodes.length; i += 1) {
            this.initializeContent(childNodes[i]);
        }
    }
};
/* @private keyDownHandler
 * This is attached on "keydown" event On window.
 * Majorly to capture escape key
 * @param none
 * @return none
 */
TextFormattingHud.prototype.keyDownHandler = function (evt) {
    'use strict';
    evt = evt || window.event;
    if (evt.keyCode === KEYCODES.Escape) {
        this.destroy();
        if (!this.m_textHUDref) {
            this.setTextHudRef();
        }
        this.m_textHUDref.escapeKeyPressed();
    }
};
/* @private clickHandler
 * This is attached on "click" event On HUD.
 * This takes the event and decides what should happen depending upon
 * the click point
 * @param e Event
 * @return none
 */
TextFormattingHud.prototype.clickHandler = function (e) {
	'use strict';
    if (!this.currentRange) {
        return;
    }
    var eventTarget = e.target, targetId;
    if (!eventTarget) {
		return;
    }
    targetId	= eventTarget.getAttribute('id');
	if (!targetId) {
        return;
    }
    //if the click is on a selector, expand it
    if (targetId === HUD_IDS.IdBold) {
        dwObject.logHeadlightsData(HEADLIGHTS.ELV_TH, HEADLIGHTS.Bold);
        this.onBoldIconClicked();
    } else if (targetId === HUD_IDS.IdItalic) {
        dwObject.logHeadlightsData(HEADLIGHTS.ELV_TH, HEADLIGHTS.italic);
        this.onItalicsIconClicked();
    } else if (targetId === HUD_IDS.IdLink) {
        this.onLinkIconClicked();
    } else if (targetId === HUD_IDS.IdBrowse) {
        dwObject.logHeadlightsData(HEADLIGHTS.ELV_TH, HEADLIGHTS.browse);
        this.launchFileSelection();
    }
};
/**
 * @private IsSelectionInsideInlineStyle
 * check if any of the parent has inline style applied.
 * this is required as  we dont want to edit content if they have bold or italic applied through inline css
 * @param whichStyle is it bold or italic
 */
TextFormattingHud.prototype.IsSelectionInsideInlineStyle = function (whichStyle) {
    'use strict';
    var retVal = false;
    if (!this.currentRange || !whichStyle) {
        return true;
    }
    var iterator = this.currentRange.startContainer;
    while (iterator) {
        if (iterator.nodeType === Node.ELEMENT_NODE) {
            if (iterator.tagName === CONSTANTS.TextContainer || iterator.tagName === "BODY") {
                break;
            }
            if (whichStyle === "bold" && iterator.getAttribute(CONSTANTS.FontWeightId)) {
                retVal = true;
            } else if (whichStyle === "italic" && iterator.getAttribute(CONSTANTS.FontStyleId)) {
                retVal = true;
            }
        }
        iterator = iterator.parentElement;
    }
    return retVal;
};
/**
 * @private onBoldIconClicked
 * Apply bold on the current selection 
 * @param none
 */
TextFormattingHud.prototype.onBoldIconClicked = function () {
    'use strict';
    // dont edit if the selection is inside inline style. (i.e. font-style italic / font-weight bold)
    if (this.IsSelectionInsideInlineStyle("bold")) {
        return;
    }
	// So text says it is bold and we did not find any strong tag. then surround this with B tag.
    if (this.isBoldCss && !this.boldState) {
        this.surroundSelectionWithTag("B");
        this.isStrongPresent = true;
    } else {
		// normal Workflows
        this.updateFormat("bold", null);
    }
	// if we are removing the bold tag. Check if it added a SPAN tag with font-style:noraml/font-weidht:normal
	// if yes delete it and change state to isBoldCss/isItalicCss to true and strong/italic present to false.
    if (this.boldState) {
		this.removeSurroundingSpanTag();
    }
    //update the UI so that there is no lag
    this.boldState = !this.boldState;
    this.updateBoldUI();
    this.saveSelection();
    if (this.m_currentElement) {
        this.m_currentElement.focus();
    }
    this.m_changedCode = true;
};
/**
 * @private onItalicsIconClicked
 * Apply italics on the current selection  
 * @param none
 */
TextFormattingHud.prototype.onItalicsIconClicked = function () {
    'use strict';
    // dont edit if the selection is inside inline style. (i.e. font-style italic / font-weight bold)
    if (this.IsSelectionInsideInlineStyle("italic")) {
        return;
    }
    // So text says it is italic and we did not find any I/EM tag. then surround this with I tag.
    if (this.isItalicCss && !this.italicState) {
        this.surroundSelectionWithTag("I");
        this.isItalicPresent = true;
    } else {
        this.updateFormat("italic", null);
    }
	// if we are removing the bold tag. Check if it added a SPAN tag with font-style:noraml/font-weidht:normal
	// if yes delete it and change state to isBoldCss/isItalicCss to true and strong/italic present to false.

    if (this.italicState) {
		this.removeSurroundingSpanTag();
    }
    //update the UI so that there is no lag
    this.italicState = !this.italicState;
    this.updateItalicUI();
    this.saveSelection();
    if (this.m_currentElement) {
        this.m_currentElement.focus();
    }
    this.m_changedCode = true;
};
/**
 * @private onLinkIconClicked
 * Toggle Link Input UI and reposition HUD if needed 
 * @param none
 */
TextFormattingHud.prototype.onLinkIconClicked = function () {
    'use strict';
    this.linkFieldState = !this.linkFieldState;
    this.updateLinkUI();
    if (this.linkFieldState === false) {
        this.positionHud();
    } else {
        this.linkInputElement.focus();
    }
};
/* @private commitOnEnter
 * Upon pressing enter on input box we need to commit the link value
 * @param e event
 * @return none
 */
TextFormattingHud.prototype.commitOnEnter =  function (e) {
    'use strict';
    if (e.keyCode === KEYCODES.Enter) {
        //commit the change locally and hid the text formatting hud.
        this.commitAndHideHud();
        //now actually commit the change to user document to reflect it in code.
        this.m_textHUDref.commit();
        dwObject.logHeadlightsData(HEADLIGHTS.ELV_TH, HEADLIGHTS.link);
    } else if (e.keyCode === KEYCODES.Tab) {
        this.browseElement.focus();
        e.stopPropagation();
        e.preventDefault();
    } else {
        this.handleArrowKeys(e, this.linkInputElement);
    }
};

/* @private LaunchBrowseOnEnter
 * Upon pressing enter or tab on browse buttong we shud launch file selection dialog 
 * and also if we do tab it should cycle inside the hud only.
 * @param e event
 * @return none
 */
TextFormattingHud.prototype.LaunchBrowseOnEnter =  function (e) {
    'use strict';
    if ((e.keyCode === KEYCODES.Enter && !e.ctrlKey) || e.keyCode === KEYCODES.Space) {
        if (this.getVisibility()) {
            this.launchFileSelection();
            e.preventDefault();
            e.stopPropagation();
        }
    }
};

/* @private tabOnBrowse
 * Upon pressing  tab on browse button
 * @param e event
 * @return none
 */
TextFormattingHud.prototype.tabOnBrowse =  function (e) {
    'use strict';
    if (e.keyCode === KEYCODES.Tab) {
        this.linkInputElement.focus();
        e.preventDefault();
        e.stopPropagation();
    }
};

/* @private selectionChangedFromDw
 * selection has changed from one of the dreamweaver menu items or toolbars.
 * @param none
 * @return none
 */
TextFormattingHud.prototype.selectionChangedFromDw = function () {
    'use strict';
    if (!this.m_textHUDref) {
        this.setTextHudRef();
    }

    if (!this.linkInputElement || this.linkInputElement !== document.activeElement) {
        this.m_textHUDref.selectHandler();
    }

};

/**
 * @override commitAndHideHud
 * here we dont mean to commit it completely.Just that all the changes that were done must be 
 * committed but cleanup should not happen(that changes selection and also content).
 * @param none
 */
TextFormattingHud.prototype.commitAndHideHud = function () {
    'use strict';
    if (!this.insideTextEditing) {
        return;
    }
    if (this.getVisibility() === true) {
        this.checkIfLinkChanged(); //checks if the link field is updated but not committed
        this.setVisibility(false); // sets visibility to false
        this.m_hudContentDiv.style.display = 'none';
    }
};
/* Utilities to do operations */
/**
* @private handleArrowKeys
* handles Right & Left Arrow keys for input elements
* If left arrow key is pressed and cursor is at the end of text, or
* Right arrow key is pressed and cursor is at the beginning of text,
* then nothing should occur
* @Param e:keypress event
* @Param elem:the HTML element on which key is pressed
*/
TextFormattingHud.prototype.handleArrowKeys = function (e, elem) {
    'use strict';
    if (e.keyCode === KEYCODES.Right) {
        var elemLen = elem.value.length;
        //check if cursor is at the end of elem text
        if (elem.selectionStart === elemLen) {
            e.stopPropagation();
            e.preventDefault();
        }
    } else if (e.keyCode === KEYCODES.Left) {
        //check if cursor is at the start of src text
        if (elem.selectionEnd === 0) {
            e.stopPropagation();
            e.preventDefault();
        }
    }
};

/**
 * @override getElementRect
 * get the elementRect of the current Selection. 
 * @param none
 */
TextFormattingHud.prototype.getElementRect = function () {
    'use strict';
    var elemRect = null;
    if (this.currentRange && this.currentRange.getClientRects().length > 0) {
        elemRect = this.currentRange.getClientRects()[0];
        if (elemRect.width === 0) {
            var elemRectNext = this.currentRange.getClientRects()[1];
            if (elemRectNext) {
                //When user clicks on beginning of the row, clientrect starts from end of previous row.
                elemRect = elemRectNext;
            }
        }
    }
    return elemRect;
};
/**
 * @private saveSelection
 * Save the current selection on parent window object so that we can use it later
 * @param none
 */
TextFormattingHud.prototype.saveSelection = function () {
    'use strict';
    var sel = window.parent.getSelection();
    if (sel && sel.toString() !== "" && sel.getRangeAt) {
        this.currentRange = sel.getRangeAt(0);
		var startCont = this.currentRange.startContainer,
			endCont = this.currentRange.endContainer,
			txtContent = sel.toString();

		// When the text under A tag is bold by CSS, making that text 
		// bold again with B tag would cause anchor tag lose. So we 
		// should handle it differently. 
		this.willLooseA = false;
		if (startCont && endCont && startCont === endCont) {
			if (startCont.nodeType === Node.TEXT_NODE
					&& startCont.nodeValue === txtContent
					&& startCont.parentNode.firstChild === startCont
					&& startCont.parentNode.lastChild === startCont
					&& startCont.parentNode.tagName === "A") {
				this.willLooseA = true;
			}
		}
    } else {
        this.commitAndHideHud();
    }
};
/**
 * @private restoreSelection
 * Save the restore selection on that we had saved previously
 * @param none
 */
TextFormattingHud.prototype.restoreSelection = function () {
    'use strict';
    if (this.currentRange) {
        var sel = window.parent.getSelection();
        if (sel) {
            sel.removeAllRanges();
            sel.addRange(this.currentRange);
            if (!this.m_textHUDref) {
                this.setTextHudRef();
            }
            this.m_textHUDref.updateSelection();
        }
    }
};
/**
 * @private commitLink
 * This is the logic for committing the link
 * This needs to be handled seperately as whenever we click on input field we loose focus 
 * So we should show gray feedback in the background of current Range
 * @param none
 */
TextFormattingHud.prototype.commitLink = function () {
    'use strict';
    if (!this.currentRange) {
        return;
    }
    this.restoreSelection();
    var linkValue = this.linkInputElement.value;
    if (linkValue === "") {
        this.updateFormat("UnLink", linkValue);
        this.linkState = false;
    } else if (linkValue && linkValue !== this.currentLinkValue) {
        this.updateFormat("CreateLink", linkValue);
        this.linkState = true;
        this.updateLinkUI();
        //Need not update the UI if User has deleted the contents as he might want to add new link again.
    }
    this.saveSelection();
    this.currentLinkValue = linkValue;
    this.linkInputElement.title = linkValue;
    this.m_changedCode = true;
};
/**
 * @private launchFileSelection
 * Luanching the file selection dialog and updating thel ink 
 * @param none
 */
TextFormattingHud.prototype.launchFileSelection = function () {
    'use strict';
    var argObj = {};
    argObj.filter = CONSTANTS.filter;
    argObj.operation = this.locStrings.fileDialogType;
    argObj.subOp = this.locStrings.fileTitleLink;
    argObj.callback = function (path) {
        window.parent.globalController.isFileDialogUp = false;
        if (path) {
            this.linkInputElement.value = path;
            this.linkInputElement.title = path;
            this.commitLink();
            this.linkInputElement.focus();
        } else {
            this.browseElement.focus();
        }
    }.bind(this);
        // tell globalcontroller not to listen to view lost focus
    window.parent.globalController.isFileDialogUp = true;
    if (dwObject) {
        dwObject.dwBrowseForFileURL(argObj.operation, argObj.subOp, argObj.callback, argObj.filter, false);
    }
};
/**
 * @private updateFormat
 * Function for applying the commands like making the text bold, italic or creating link
 * Creating inbuilt browser function document.execCommand. But selection needs to be proper
 * for this function to work properly
 * @param cmd Command to execute on execcommand
 * @param value value to be updated (in case of link tag only remaining times it is null
 */
TextFormattingHud.prototype.updateFormat = function (cmd, value) {
    'use strict';
    if (!cmd) {
        return;
    }
    window.parent.document.execCommand(cmd, false, value);
};
/**
 * @private removeSurroundingSpanTag
 * if we are removing the bold/italic tag. Check if it added a SPAN tag with font-style:normal/font-weidht:normal
 * if yes delete it and change state to isBoldCss/isItalicCss to true and strong/italic present to false.
 * @param none 
 * @return none
 */
TextFormattingHud.prototype.removeSurroundingSpanTag = function () {
    'use strict';
    if (!this.currentRange) {
        return;
    }
    var startCont = this.currentRange.startContainer;
    // if it is text node, get its  parent element
    if (startCont && startCont.nodeType === Node.TEXT_NODE) {
        startCont = startCont.parentElement;
    }
    if (!startCont) {
        return;
    }
    var childs = startCont.childNodes;
    var spanNode = null, i;
    // check all nodes and if any span has fontstyle is normalthen remove the attribute (Cleanup of this SPAN tag is    
    // taken care by cleanupcontent in texteditng.
    if (childs) {
        for (i = 0; i < childs.length; i++) {
			if (childs[i] && childs[i].nodeType === Node.ELEMENT_NODE && childs[i].tagName === "SPAN") {
                if (!childs[i].getAttribute(CONSTANTS.FontStyleId)) {
                    var fontStyle = parent.DWTextEditUtility.hasInlineStyle(childs[i], "font-style");
                    if (fontStyle) {
                        fontStyle = fontStyle.trim();
                        if (fontStyle.toLowerCase() === 'normal') {
                            childs[i].style.removeProperty('font-style');
                            spanNode = childs[i];
                        }
                        this.isItalicPresent = false;
                        this.isItalicCss = true;
                    }
                }
                if (!childs[i].getAttribute(CONSTANTS.FontWeightId)) {
                    var fontWeight = parent.DWTextEditUtility.hasInlineStyle(childs[i], "font-weight");
                    if (fontWeight) {
                        fontWeight = fontWeight.trim();
                        if (fontWeight.toLowerCase() === 'normal') {
                            childs[i].style.removeProperty('font-weight');
                            spanNode = childs[i];
                        }
                        this.isStrongPresent = false;
                        this.isBoldCss = true;
                    }
                }
                break;
			}
        }
    }
    // if we are able to find a span node newly introduced. Set selefction to it.
    if (spanNode) {
        this.setSelectionOnNode(spanNode);
    }
};
/**
 * @private setSelectionOnNode
 * Set selection on Given Node
 * @param node node To set selection on.
 * @return none
 */
TextFormattingHud.prototype.setSelectionOnNode = function (node) {
	'use strict';
    if (!node) {
        return;
    }
    var selection = parent.getSelection(),
        range = parent.document.createRange();
    if (selection && range) {
        range.selectNodeContents(node);
        selection.removeAllRanges();
        selection.addRange(range);
        this.saveSelection();
    }
};
/**
 * @private surroundSelectionWithTag
 * surround a given selection with tag
 * @param tagName tag name to wrap with
 * @return none
 */
TextFormattingHud.prototype.surroundSelectionWithTag = function (tagName) {
    'use strict';
	if (!tagName || !this.currentRange) {
        return;
    }
    var rangeContents = this.currentRange.cloneContents(),
        startCont = this.currentRange.startContainer;
    if (!rangeContents) {
        return;
    }
    if (startCont && startCont.nodeType === Node.TEXT_NODE) {
        startCont = startCont.parentNode;
    }
    if (!startCont || !startCont.tagName) {
        return;
    }
    var txtContent = rangeContents.textContent, newContent = null;
    var currentTag = startCont.tagName.toLowerCase();

	if ((this.willLooseA && currentTag === 'a') || ((tagName.toLowerCase() === "b" && (currentTag === "i" || currentTag === "em")) ||
			(tagName.toLowerCase() === "i" && (currentTag === "b" || currentTag === "strong")))) {
		var currHTML = startCont.outerHTML,
			newHTML = "<" + tagName + " id=\"justNowAddedDwTag\">" + txtContent + "</" + tagName + ">";
		newContent = currHTML.replace(txtContent, newHTML);
    }
    if (!newContent) {
        newContent = "<" + tagName + " id=\"justNowAddedDwTag\">" + txtContent + "</" + tagName + ">";
    }
    try {
        parent.document.execCommand("insertHTML", false, newContent);
    } catch (e) {

    }
    var nowAdded = parent.document.getElementById("justNowAddedDwTag");
    if (nowAdded) {
        nowAdded.removeAttribute("id");
        if (!this.m_textHUDref) {
            this.setTextHudRef();
        }
        parent.DWTextEditUtility.trimWhiteSpacesArroundTextElements(this.m_currentElement);
        this.setSelectionOnNode(nowAdded);
    }
};
/**
 * @private checkIfLinkChanged
 * Normally commit of this textformattinghud is taken care by text editing hud
 * But in case there is a change in link input field and it was not explicitly committed
 * We must be committing it.
 * @param none
 */
TextFormattingHud.prototype.checkIfLinkChanged = function () {
	'use strict';
    if (!this.currentLinkValue && !this.linkInputElement.value) {
        return;
    }
	if (this.linkFieldState && this.linkInputElement.value !== this.currentLinkValue) {
        this.commitLink();
    }
};
/* UI FUNCTIONS*/
/**
 * @override displayHUD
 * if hud is not visible first make sure we enable it. 
 * else if the hud is already visible then update the selection and redraw the structure.
 * @param none
 */
TextFormattingHud.prototype.displayHUD = function (selectionDetails) {
    'use strict';
    if (!this.insideTextEditing) {
        return;
    }
    if (this.getVisibility() === false) {
        // attaches event listeners and set visibility to true
		this.setVisibility(true);
    }
    this.saveSelection();
    // draw the hud
    this.clearHudFields();
    if (selectionDetails) {
        this.boldState = selectionDetails.bold;
        this.italicState = selectionDetails.italic;
        this.queryFailed = selectionDetails.failed;
    }
    this.draw();
};
/**
 * @override draw
 * populate the content of hud if not present already(taken care by generichudobj)
 * initialize the hud 
 * update Hud UI
 * position the hud(with respect to available space around the element)
 * Show the TextFormattingHud 
 * @param none
 */
TextFormattingHud.prototype.draw = function () {
    'use strict';
    var curElement = this.m_currentElement;
    if (!curElement) {
        return;
    }
    /* look at current element and decide if it is already bold/italic or link is attached */
    this.initializeHud();
    // depending state member variables show the UI (gradient/plain)
    this.updateHudUI();
     //position the hud around the element
    this.positionHud();
};
/**
 * @private initializeHud
 * Function for initializing the bold, italic and link state( if the text selected is already
 * having these attribute set, the state should be set to true
 * @param none
 */
TextFormattingHud.prototype.initializeHud = function () {
    'use strict';
    var existingLink, dwSpanParent;
    if (!this.currentRange) {
        return;
    }
    var currentSelectedElement = this.currentRange.startContainer;
    if (!currentSelectedElement) {
        return;
    }
    var iterator = currentSelectedElement;
    while (iterator && iterator.tagName !== "BODY" && iterator.tagName !== "HTML") {
        if (iterator.nodeType === Node.ELEMENT_NODE) {
            if (iterator.tagName === "A") {
                existingLink = iterator.getAttribute("href");
                this.linkState = true;
                this.linkFieldState = true;
                if (existingLink) {
                    this.currentLinkValue = existingLink;
                    this.linkInputElement.value = existingLink;
                    this.linkInputElement.title = existingLink;
                }
            } else if (iterator.tagName !== CONSTANTS.TextContainer) {
                if (iterator.tagName === "STRONG" || iterator.tagName === "B") {
                    this.isStrongPresent = true;
                } else if (iterator.tagName === "EM" || iterator.tagName === "I") {
                    this.isItalicPresent = true;
                }
            } else if (iterator.tagName ===  CONSTANTS.TextContainer) {
                dwSpanParent = iterator.parentNode;
                if (dwSpanParent && dwSpanParent.tagName === "A") {
                    existingLink = dwSpanParent.getAttribute("href");
                    this.linkState = true;
                    this.linkFieldState = true;
                    if (existingLink) {
                        this.currentLinkValue = existingLink;
                        this.linkInputElement.value = existingLink;
                        this.linkInputElement.title = existingLink;
                    }
                }
                break;
            }
        }
        iterator = iterator.parentNode;
    }
    this.isBoldCss = this.boldState && !this.isStrongPresent;
    this.isItalicCss = this.italicState && !this.isItalicPresent;
    this.boldState = this.boldState && this.isStrongPresent;
    this.italicState = this.italicState && this.isItalicPresent;
};
/**
 * @private updateHudUI
 * Function for updating the hud fields to show different icons for bold, italic and link
 * on the basis if the selected text is already having these state applied or not.
 * @param none
 */
TextFormattingHud.prototype.updateHudUI = function () {
    'use strict';
    this.updateBoldUI();
    this.updateItalicUI();
    this.updateLinkUI();
};
/**
 * @private updateBoldUI
 * Depending on boldstate reflect the UI
 * @param none
 */
TextFormattingHud.prototype.updateBoldUI = function () {
    'use strict';
    if (this.boldState === true) {
        this.boldElement.parentElement.classList.add(HUD_CLASSES.SelectedBG);
        if (parent.semanticTags) {
            this.boldElement.title =  this.locStrings.boldRemoveStrong;
        } else {
            this.boldElement.title =  this.locStrings.boldRemoveB;
        }
    } else {
        this.boldElement.parentElement.classList.remove(HUD_CLASSES.SelectedBG);
        if (parent.semanticTags) {
            this.boldElement.title =  this.locStrings.boldAddStrong;
        } else {
            this.boldElement.title =  this.locStrings.boldAddB;
        }
    }
};
/**
 * @private updateItalicUI
 * Depending on italicState reflect the UI
 * @param none
 */
TextFormattingHud.prototype.updateItalicUI = function () {
    'use strict';
    if (this.italicState === true) {
        if (parent.semanticTags) {
            this.italicsElement.title = this.locStrings.italicRemoveEm;
        } else {
            this.italicsElement.title = this.locStrings.italicRemoveI;
        }
        this.italicsElement.parentElement.classList.add(HUD_CLASSES.SelectedBG);
    } else {
        if (parent.semanticTags) {
            this.italicsElement.title = this.locStrings.italicAddEm;
        } else {
            this.italicsElement.title = this.locStrings.italicAddI;
        }
        this.italicsElement.parentElement.classList.remove(HUD_CLASSES.SelectedBG);
    }
};
/**
 * @private updateLinkUI
 * Depending on linkState and linkFieldState reflect the UI
 * @param none
 */
TextFormattingHud.prototype.updateLinkUI = function () {
    'use strict';
    var containerForLink;
    containerForLink = document.getElementById(HUD_IDS.IdLinkContainer);
    if (this.linkElement) {
        if (this.linkFieldState === true) {
            this.linkElement.parentElement.classList.add(HUD_CLASSES.SelectedBG);
        } else {
            this.linkElement.parentElement.classList.remove(HUD_CLASSES.SelectedBG);
        }
    }
    if (containerForLink) {
        if (this.linkFieldState === true) {
            containerForLink.style.display = "block";
            var hudScreenPos = this.currentHudLeft + this.currentHudWidth + CONSTANTS.TextHudOffset + CONSTANTS.LinkWidth;
            if (hudScreenPos > window.parent.innerWidth + window.parent.scrollX) {
                this.m_hudContentDiv.style.left = window.parent.innerWidth + window.parent.scrollX + this.currentHudLeft - hudScreenPos;
            }
        } else {
            containerForLink.style.display = "none";
        }
    }
};
/**
 * @private positionHud
 * Function for positioning the hud with respect to space available around element.
 * If no space on top keep it at bottom
 * @param none
 */
TextFormattingHud.prototype.positionHud = function () {
    'use strict';
    //don't do anything if we are not enabled
    if (this.getVisibility() === false) {
        return;
    }
    var elemRect = this.getElementRect();

    liveViewObject.positionExtensionById(CONSTANTS.ExtensionID, 0, 0, parent.innerWidth + parent.scrollX, parent.innerHeight + parent.scrollY);
    liveViewObject.showExtensionById(CONSTANTS.ExtensionID);
    if (elemRect) {
        //if there is no space at top, keep the HUD at bottom
        var hudTop = elemRect.top - this.currentHudHeight;
        if (hudTop < 0) {
            hudTop = elemRect.top + elemRect.height;
        }
        hudTop += window.parent.scrollY;
        //if there is no space on right, move left
        var hudLeft = elemRect.left + window.parent.scrollX;
        var hudWidth = this.currentHudWidth;
        if (this.linkFieldState === true && hudWidth < CONSTANTS.LinkWidth) {
            hudWidth = hudWidth + CONSTANTS.LinkWidth;
        }
        if (hudLeft + hudWidth + CONSTANTS.TextHudOffset >= window.parent.innerWidth + window.parent.scrollX) {
            hudLeft = window.parent.innerWidth + window.parent.scrollX - hudWidth - CONSTANTS.TextHudOffset;
        }
        //display HUD
        this.m_hudContentDiv.style.top = hudTop + "px";
        this.m_hudContentDiv.style.left = hudLeft + "px";
        this.m_hudContentDiv.style.display = 'block';
        this.currentHudLeft = hudLeft;
    }
};
/**
 * @override destroy
 * hide the TextFormattingHud and clear the hud fields
 * @param none
 */
TextFormattingHud.prototype.destroy = function () {
    'use strict';
    if (this.getVisibility() === false && this.insideTextEditing === false) {
        return;
    }
    liveViewObject.hideExtensionById(CONSTANTS.ExtensionID);
    //hide text Hud divs
    if (this.m_hudContentDiv) {
        this.m_hudContentDiv.style.display = "none";
    }
    this.browseElement.blur();
    this.clearHudFields();
    this.detachEventListeners();
    this.m_currentElement = null;
    this.m_textHUDref = null;
    this.insideTextEditing = false;
    this.m_changedCode = false;
};
/**
 * @private clearHudFields
 * Clears the HUD fields
 * @param none
 */
TextFormattingHud.prototype.clearHudFields = function () {
    'use strict';
    this.boldState = false;
    this.italicState = false;
    this.linkState = false;
    this.linkFieldState = false;
    this.isItalicPresent = false;
    this.isStrongPresent = false;
    this.isBoldCss = false;
    this.isItalicCss = false;
    this.updateHudUI();
    this.currentLinkValue = null;
    if (this.linkInputElement) {
        this.linkInputElement.value = "";
        this.linkInputElement.title = "";
    }
};
var initTextFormattingHud = function () {
    'use strict';
    window.textFormattingHudObj = new TextFormattingHud();
    window.textFormattingHudObj.initialize();
};
