/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2013-2014 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

/*jslint vars: true, node: true, sub: true */
"use strict";

var sharedCloudConfig = require('../handlers/sharedCloudConfig');
var https = require(sharedCloudConfig.getSharedCloudProtocol());
var request = require('request');
var sharedCloudUtils = require('../utils').sharedCloudUtils;
var imsServiceTokenUtils = require('../utils').imsServiceTokenUtils;
var util = require('util');
var step = require('step');
var async = require('async');
var logger = require('winston');
var users = require('../utils').sharedcloud.users;
var userFeatureUtils = require('../utils').userFeatureUtils;
var handlerUtils = require('../utils').handlerUtils;
var stepUtils = require('../utils').stepUtils;
var metricsHandlers = require('../handlers/metricsHandlers');

var awsUtils = require('../aws/awsUtils');
var url = require('url');
var path = require('path');
var fs = require('fs');

//--- begin REST call handlers
function userLogin(req, res, next) {
    var params = handlerUtils.parseQueryString(req.url);

    imsServiceTokenUtils.getUserAccessToken(params.code, function (err, sessionInfo) {
        if (err) {
            logger.verbose("getUserAccessToken Returned Error : " + err.message);
            res.writeHead(401, err);
            res.end();
        } else {
            logger.verbose("getUserAccessToken Returned sessionInfo");
            var now = new Date(),
                date = now;
            date.setTime(date.getTime() + sessionInfo.expires_in);

            var cookie = handlerUtils.getCookie(req, 'ets-metrics'),
                header = {
                    'Location': '/redirect.html',
                    'Set-Cookie': [ 'creative-cloud-v=' + sessionInfo.access_token + "; expires=" + date.toGMTString() + "; path=/; HttpOnly",
                                   'ets-metrics=; expires=' + now.toGMTString() + '; path=/; HttpOnly']
                };
            res.writeHead(302, header);
            res.end();

            // if we have any cached metrics, send those out.
            metricsHandlers.etsLogDeferredMetrics(cookie, sessionInfo.access_token);
        }
    });
}
/*jslint regexp: false*/

function userLogout(req, res, next) {
    var auth = handlerUtils.getCookie(req, 'creative-cloud-v');

    imsServiceTokenUtils.logoutAccessToken(auth, function (error, data) {
        imsServiceTokenUtils.invalidateAccessToken(auth, function (error, data) {
            var expireData = new Date(),
                header = {
                    'Location': '/',
                    'Set-Cookie': ['creative-cloud-v=; expires=' + expireData.toGMTString() + '; path=/; HttpOnly',
                                   'parfait-user-data=; expires=' + expireData.toGMTString() + '; path=/; HttpOnly']
                };
            res.cache("no-Cache");
            res.writeHead(302, header);
            res.end();
        });
    });

}

/**
    Returns a JSON structure describing all of the assets within the user's shared cloud account.
    Currently returns all assets, including Graphite data files and PSDs. Long term plan is to have
    this API return just PSDs.
**/
function listAssetsHandler(req, res, next) {
    var requuid = stepUtils.generateUUID();
    var logName = "listAssetsHandler(" + requuid + ")";
    var asyncCallStartTime = Date.now();
    var authorization = handlerUtils.getAuthHeader(req);
    res.cache("no-Cache");
    logger.verbose('entering %s', logName);

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function (err, localServiceToken) {
                if (err) { throw err; }

                sharedCloudUtils.scListAssets(err, localServiceToken, authorization, requuid, this);
            },
            function returnAssetListResults(err, statusCode, responseText) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, logName);
                    return;
                }
                logger.debug("%s success. Elapsed time= %d ms", logName, Date.now() - asyncCallStartTime);

                if (200 !== statusCode) {
                    res.statusCode = statusCode;
                    res.send(responseText);
                    return;
                }
                res.writeHead(200, {'Content-Type': 'application/json'});
                res.end(responseText);
            }
        );

    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, logName);
    }
}

/**
    Retrieve the shared cloud asset metatadata for a specified PSD asset.

    Parameters
        :id - the GUID of the PSD in question
**/
function getAssetInfo(req, res, next) {
    var asyncCallStartTime = Date.now(),
        authorization = handlerUtils.getAuthHeader(req),
        requuid = stepUtils.generateUUID(),
        logName = "getAssetInfo(" + requuid + ")";
    logger.verbose('entering %s', logName);

    res.cache("no-Cache");
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function (err, localServiceToken) {
                if (err) { throw err; }

                sharedCloudUtils.scGetAssetMetadata(err, localServiceToken, authorization, req.params.id, requuid, this);
            },
            function returnAssetListResults(err, statusCode, responseText) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, logName);
                    return;
                }
                logger.debug("%s success. Elapsed time= %d ms", logName, Date.now() - asyncCallStartTime);

                if (200 !== statusCode) {
                    res.statusCode = statusCode;
                    res.send(responseText);
                    return;
                }
                res.writeHead(200, {'Content-Type': 'application/json'});
                res.end(responseText);
            }
        );

    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, logName);
    }
}

function getDefaultRendition(req, res, next) {
    var authorization = handlerUtils.getAuthHeader(req),
        requuid = stepUtils.generateUUID(),
        theGraphiteDataEtag = req.headers['if-none-match'];

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function (err, localServiceToken) {
                var x,
                    options = {
                        uri : sharedCloudConfig.getSharedCloudURI() + "/api/v1/assets/" + req.params.id + "/renditions/png/full",
                        method: 'GET',
                        headers: {
                            'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(localServiceToken),
                            'X-User-Token': sharedCloudConfig.getUserHeaderValue(authorization),
                            'x-request-id': 'Parfait-GDR-' + requuid
                        }
                    };

                if (theGraphiteDataEtag) {
                    options.headers['If-None-Match'] = theGraphiteDataEtag;
                }

                x = request(options);
                x.pipe(res);
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getDefaultRendition");
    }
}

/**
    Retrieve a thumbnail of the entire PSD asset; primarily used in the selection UI for the Graphite web app.

    Parameters
        :id - the GUID of the PSD in question
**/
function getThumbnailHandler(req, res, next) {
    var authorization = handlerUtils.getAuthHeader(req);
    var asyncCallStartTime = Date.now();
    var requuid = stepUtils.generateUUID();
    var logName = "getThumbnailHandler(" + requuid + ")";
    logger.verbose('entering %s for %s', logName, req.params.id);

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function getThumbnailHandlerImpl(err, localServiceToken) {
                if (err) { throw err; }

                sharedCloudUtils.scGetMacaronThumbnail(err, localServiceToken, authorization, req.params.id, req.headers['if-none-match'], res, requuid, this);
            },
            function reportTimeAndErr(err) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, logName);
                }
                logger.debug("%s success. Elapsed time= %d ms", logName, Date.now() - asyncCallStartTime);
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, logName);
    }
}

/**
    A high level administrative call; resets all user content. Removes all uploaded PSDs as well as generated JSON data and PNG proxy assets.
**/
function resetGraphiteAcct(req, res, next) {
    var authorization = handlerUtils.getAuthHeader(req),
        requuid = stepUtils.generateUUID(),
        logName = "resetGraphiteAcct(" + requuid + ")";

    logger.verbose('entering %s', logName);
    res.cache("no-Cache");
    if (null === authorization) {
        res.statusCode = 400;
        res.send("Unauthorized");
        return;
    }
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    sharedCloudUtils.scResetGraphiteAcct(null, authorization, requuid, function () {
        res.statusCode = 200;
        res.send("Graphite account reset");
        logger.verbose('leaving %s', logName);
    });
}

/**
    Delete a particular PSD from the shared cloud, along with all of it's associated Graphite data structures and associated "ASSETS" folder

    Parameters
        :id - the GUID of the PSD in question
**/
function deleteAssetHandler(req, res, next) {
    var authorization = handlerUtils.getAuthHeader(req);
    var serviceToken;
    var requuid = stepUtils.generateUUID();
    var theAssetId = req.params.id;
    var psdName;
    var aCollectionETag;
    var aCollectionId;

    res.cache("no-Cache");
    if (null === authorization) {
        res.statusCode = 400;
        res.send("Unauthorized");
        return;
    }
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        logger.info("deleteAssetHandler assertId=" + req.params.id);
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function getMetadata(err, localServiceToken) {
                //initialize the global serviceToken - used by scDeleteCollection and  deleteAssetQueueAction below
                serviceToken = localServiceToken;
                //get the name of the psd
                sharedCloudUtils.scGetAssetMetadata(err, serviceToken, authorization, theAssetId, requuid, this);
            },
            function returnAssetMetadataResults(err, statusCode, responseText) {
                if (err) { throw err; }

                if (200 !== statusCode) {
                    res.statusCode = statusCode;
                    res.send(responseText);
                    return;
                }

                var responseJSON = JSON.parse(responseText);
                psdName = responseJSON.asset.fileName;

                var extIndex = psdName.lastIndexOf(".");
                if (extIndex !== -1) {
                    psdName = psdName.substr(0, extIndex);
                }
                sharedCloudUtils.scListCollection(err, serviceToken, authorization, requuid, this);
            },
            function returnCollectionResults(err, statusCode, responseText) {
                if (err) { throw err; }

                if (200 !== statusCode) {
                    res.statusCode = statusCode;
                    res.send(responseText);
                    return;
                }

                //traverse the existing collections to see if the psdName-assets folder already exist or not
                var responseJSON = JSON.parse(responseText);
                var folderNameToSearch = psdName + "_assets";
                var i,
                    folderExists;
                for (i in responseJSON) {
                    if (responseJSON.hasOwnProperty(i)) {
                        if (responseJSON[i].metadata.collection.collection_name === folderNameToSearch) {
                            folderExists = true;
                            aCollectionId = responseJSON[i].id;
                            aCollectionETag = responseJSON[i].etag;
                            break;
                        }
                    }
                }

                if (folderExists) {
                    //delete the "psd_assets" collection/folder if existing
                    sharedCloudUtils.scDeleteCollection(null, serviceToken, authorization, aCollectionId, aCollectionETag, requuid, this);
                } else {
                    var that = this;
                    that(null, statusCode, responseText);
                }
            },
            function deleteAsset(err, statusCode, responseText) {
                if (err) { throw err; }
                sharedCloudUtils.deleteAssetQueueAction(serviceToken, authorization, req.params.id, requuid, this);
            },
            function returnDeleteAssetResults(err, statusCode, responseText) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "deleteAssetHandler");
                }

                res.statusCode = 200;
                res.writeHead(200, {'Content-Type': 'application/json'});
                res.end(responseText || "{}");
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "deleteAssetHandler");
    }
}


/**
 Delete a particular derived asset(png, jpg, svg) from the shared cloud

 Parameters
 :id - the GUID of the derived asset in question
 **/
function deleteDerivedAssetHandler(req, res, next) {
    var authorization = handlerUtils.getAuthHeader(req);
    var requuid = stepUtils.generateUUID();
    var theAssetId = req.params.id;

    res.cache("no-Cache");
    if (null === authorization) {
        res.statusCode = 400;
        res.send("Unauthorized");
        return;
    }
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        logger.info("deleteDerivedAssetHandler assertId=" + theAssetId);
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function deleteAsset(err, localServiceToken) {
                if (err) { throw err; }

                sharedCloudUtils.deleteAssetQueueAction(localServiceToken, authorization, theAssetId, requuid, this);
            },
            function returnDeleteAssetResults(err, statusCode, responseText) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "deleteDerivedAssetHandler");
                }

                if (statusCode !== 200) {
                    res.send(statusCode, responseText);
                    return;
                }

                res.statusCode = 200;
                res.writeHead(200, {'Content-Type': 'application/json'});
                res.end(responseText || "{}");
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "deleteDerivedAssetHandler");
    }
}


/**
    Retrieve the contents of an asset within the Shared Cloud by its GUID.

    Parameters
        :sharedCloudGUID - the GUID of the asset
**/
function getGenericAssetHandler(req, res, next) {
    logger.verbose('entering getGenericAssetHandler');
    var asyncCallStartTime = Date.now();
    var theSharedCloudGUID = req.params.sharedCloudGuid;
    var authHeader = handlerUtils.getAuthHeader(req);

    var theGraphiteDataEtag = req.headers['if-none-match'];
    var requuid = stepUtils.generateUUID();
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function getGraphiteAsset(err, localServiceToken) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "getGenericAssetHandler");
                    return;
                }

                var options = {
                        uri : sharedCloudConfig.getSharedCloudURI() + "/api/v1/assets/" + theSharedCloudGUID,
                        method: 'GET',
                        headers: {
                            'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(localServiceToken),
                            'X-User-Token': sharedCloudConfig.getUserHeaderValue(authHeader),
                            'x-request-id' : 'Parfait-GGA-' + requuid
                        }
                    };

                if (theGraphiteDataEtag) {
                    options.headers['If-None-Match'] = theGraphiteDataEtag;
                }

                var x = request(options);
                x.on('end', function () { logger.debug("getGenericAsset success. Elapsed time= %d ms", Date.now() - asyncCallStartTime); });
                x.pipe(res);
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getGenericAssetHandler");
    }
}

/**
    Retrieve a spritesheet containing all or some of the PSD layer proxies.  Note that for each
    layer in the Graphite JSON data, a sprite sheet id and bounds is specified.  There could be
    1 or more spritesheets associated with a PSD, given the dimensions and numbers of layers
    contained in the PSD.

    Parameters:
        :id - the GUID of the PSD in question
        :sheetId- the ID of the spritesheet (usually 1..N)
**/
function getSpriteSheetRendition(req, res, next) {
    var authHeader = null;
    var theAssetId = req.params.id;
    var asyncCallStartTime = Date.now();
    var requuid = stepUtils.generateUUID();
    var layerCompId = req.params.layerCompId;
    var layerCompInfo = layerCompId ? "-lcid" + layerCompId : "";
    var logName = "getSpriteSheetRendition(" + requuid + ")";
    logger.verbose('entering %s', logName);

    var theGraphiteDataEtag = req.headers['if-none-match'];
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function (err, localServiceToken) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, logName);
                    return;
                }

                authHeader = handlerUtils.getAuthHeader(req);
                var options = {
                    uri : sharedCloudConfig.getSharedCloudURI() + "/api/v1/assets/" + theAssetId + "/renditions/graphite/SpriteSheet-" + req.params.sheetId + layerCompInfo,
                    method: 'GET',
                    headers: {
                        'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(localServiceToken),
                        'X-User-Token': sharedCloudConfig.getUserHeaderValue(authHeader),
                        'x-request-id': 'Parfait-GSR-' + requuid
                    }
                };

                if (theGraphiteDataEtag) {
                    options.headers['If-None-Match'] = theGraphiteDataEtag;
                }

                var x = request(options);
                x.on('end', function () { logger.debug("%s success. Elapsed time= %d ms", logName, Date.now() - asyncCallStartTime); });
                x.pipe(res);
            }
        );

    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, logName);
    }

}

/**
    Create a new PSD asset within the Graphite environment and starts the PSD processing worker
    once the asset is successfully stored in the cloud. The GUID for the newly created asset
    is returned in the response's Location HTTP header.
**/
function uploadAssetHandler(req, res, next) {
    var localReq = req;
    var authorization = handlerUtils.getAuthHeader(req);
    var requuid = stepUtils.generateUUID();
    var params = handlerUtils.parseQueryString(req.url);
    var logName = "uploadAssetHandler(" + requuid + ")";
    res.cache("no-Cache");

    logger.verbose('entering %s', logName);

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    function uploadResult(err, statusCode, result) {
        if (err) {
            if (statusCode === 401) {
                res.send(statusCode, err);
            } else {
                handlerUtils.genericErrorMessage(err, res, "uploadResult");
            }
        } else {
            res.send(statusCode, result);
        }
        logger.verbose('leaving %s', logName);
    }
    sharedCloudUtils.scUploadAsset(null, authorization, localReq, requuid, params, uploadResult);
}

/**
    Create a new PSD asset within the Graphite environment by pulling the publicly accessible URL
    specified in the form. Once the PSD is imported, the PSD processing worker is started. The GUID
    for the newly created asset is returned in the response's Location HTTP header.
**/
function uploadURLassetHandler(req, res, next) {
    logger.verbose('entering uploadURLassetHandler');
    var asyncCallStartTime = Date.now();

    var localReq = req;
    var authorization = handlerUtils.getAuthHeader(req);
    var requuid = stepUtils.generateUUID();
    res.cache("no-Cache");
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function processFormPost(err, localServiceToken) {
                stepUtils.processFormInRequest(err, localReq, localServiceToken, this);
            },
            function downloadURL(err, files, fields, localServiceToken) {
                if (err) { throw err; }

                var that = this;

                if (!fields.hasOwnProperty("urlToLoad")) {
                    that('bad form data');
                }

                var filename = path.basename(url.parse(fields.urlToLoad).pathname);

                // otherwise return that we've accepted the request
                awsUtils.setUploadProgress(requuid, 0, "uploading", "", "", function (err, data) {
                    res.statusCode = 202;
                    res.setHeader("Location", '/api/v1/uploadProgress/' + requuid);
                    res.end();
                });

                function updateUploadProgress(data) {
                    awsUtils.setUploadProgress(requuid, data.percent, "uploading", "", "", stepUtils.noopCallback);
                }

                sharedCloudUtils.scPostURLRequest(null, localServiceToken, authorization, filename, request(fields.urlToLoad), requuid, this, stepUtils.updateUploadProgress);
            },
            function filePosted(err, location, localServiceToken) {
                function noopCallback(err) { }

                awsUtils.setUploadProgress(requuid, 100, "uploading", "", "", noopCallback);

                if (err) { throw err; }
                logger.debug("filePosted success. Elapsed time= %d ms", Date.now() - asyncCallStartTime);

                sharedCloudUtils.scCreateGraphiteWorker(err, localServiceToken, authorization, stepUtils.getAssetIdFromLocation(location), null, requuid, {}, this);
            },
            function handleResult(err, statusCode, assetId, workerId, bodyText) {
                awsUtils.setUploadProgress(requuid, 100, "working", assetId, workerId, stepUtils.noopCallback);
                stepUtils.handleResultFromWorker(err, null, statusCode, assetId, null, workerId, bodyText);
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "uploadURLassetHandler");
    }
}

function getUploadProgress(req, res, next) {
    var uploadID = req.params.id;
    res.cache("no-Cache");

    awsUtils.getUploadProgress(uploadID, function (err, result) {
        if (err) {
            handlerUtils.genericErrorMessage(err, res, "getUploadProgress");
        } else {
            res.send(200, result);
        }
    });

}

function getMultiUploadProgress(req, res, next) {
    var fullBody = '',
        k;
    res.cache("no-Cache");

    req.on('data', function (chunk) {
        // append the current chunk of data to the fullBody variable
        fullBody += chunk.toString();
    });

    req.on('end', function () {
        var uploadIDs = JSON.parse(fullBody),
            localRes = { };

        awsUtils.getUploadProgress(uploadIDs, function (err, result) {
            // we can get called any number of times depending on which node the key is
            // stored on so copy the results into a local result until we've got all the requested
            // data
            var k;
            for (k in result) {
                if (result.hasOwnProperty(k)) {
                    localRes[k] = result[k];
                }
            }

            if (Object.keys(localRes).length === uploadIDs.length) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "getMultiUploadProgress");
                } else {
                    res.send(200, JSON.stringify(localRes));
                }
            }
        });
    });
}

function deleteUploadProgress(req, res, next) {
    var uploadID = req.params.id;
    res.cache("no-Cache");
    awsUtils.deleteUploadProgress(uploadID, function (err, data) {
        if (err) {
            handlerUtils.genericErrorMessage(err, res, "deleteUploadProgress");
        } else {
            res.send(200, "OK");
        }
    });
}

/**
    Return an JSON object containing an array of assets that were generated from a specified PSD

    Parameters
        :id - the GUID of the PSD
**/
function getDerivedAssets(req, res, next) {
    logger.verbose('entering getDerivedAssets');
    var authorization = handlerUtils.getAuthHeader(req);
    var serviceToken = null;
    var theAssetId = req.params.id;
    var requuid = stepUtils.generateUUID();
    var psdName = null;
    var parentCollectionUrl = null;
    var parentCollectionId = null;

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function psdName(err, localServiceToken) {
                //get the name of the psd
                serviceToken = localServiceToken;
                sharedCloudUtils.scGetAssetMetadata(err, serviceToken, authorization, theAssetId, requuid, this);
            },
            function returnAssetMetadataResults(err, statusCode, responseText) {
                if (err) { throw err; }

                if (statusCode !== 200) {
                    res.send(statusCode, responseText);
                    return;
                }

                var responseJSON = JSON.parse(responseText);
                psdName = responseJSON.asset.fileName;

                var extIndex = psdName.lastIndexOf(".");
                if (extIndex !== -1) {
                    psdName = psdName.substr(0, extIndex);
                }
                sharedCloudUtils.scListCollection(err, serviceToken, authorization, requuid, this);
            },
            function returnCollectionResults(err, statusCode, responseText) {
                if (err) { throw err; }

                if (200 !== statusCode) {
                    if (statusCode === 404) {
                        res.statusCode = 200;
                        res.send([]);
                    } else {
                        res.statusCode = statusCode;
                        res.send(responseText);
                    }
                    return;
                }

                //traverse the existing collections to see if the psdName-assets folder already exist or not
                var responseJSON = JSON.parse(responseText);
                var folderNameToSearch = psdName + "_assets";
                var i,
                    folderAlreadyExists;
                for (i in responseJSON) {
                    if (responseJSON.hasOwnProperty(i)) {
                        if (responseJSON[i].metadata.collection.collection_name === folderNameToSearch) {
                            folderAlreadyExists = true;
                            parentCollectionUrl = responseJSON[i].url;
                            parentCollectionId = responseJSON[i].id;
                            break;
                        }
                    }
                }

                sharedCloudUtils.scListAssetsInCollection(err, serviceToken, authorization, parentCollectionId, requuid, this);
            },
            function handleQueryResults(err,  statusCode, responseText) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "getDerivedAssets");
                    return;
                }

                if (200 !== statusCode) {
                    if (statusCode === 404) {
                        res.statusCode = 200;
                        res.send([]);
                    } else {
                        res.statusCode = statusCode;
                        res.send(responseText);
                    }
                    return;
                }

                var assetsToReturn = [],
                    responseObj = JSON.parse(responseText),
                    i;
                for (i in responseObj.assets) {
                    if (responseObj.assets.hasOwnProperty(i)) {
                        var theAsset = responseObj.assets[i];
                        assetsToReturn.push(theAsset);
                    }
                }

                res.statusCode = 200;
                res.send(assetsToReturn);
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getDerivedAssets");
    }
}

function getMimeByExt(ext) {
    var mimeTypes = {
        'png' : 'image/png',
        'svg' : 'image/svg+xml',
        'jpg' : 'image/jpeg'
    };

    return mimeTypes[ext] || false;

}

/*
 * validates the parameters for extract asset job and returns corresponding status code
 *  Parameters
        :jobParams - job parameters for createAsset command.
        :res - a node response object.
*/
function validateJobParams(jobParams, res) {
    var validEncodingTypes = {
        'png8' : {
            fileExtension: 'png',
            mimeType: getMimeByExt('png'),
            encodingType: 'png8'
        },
        'png32' : {
            fileExtension: 'png',
            mimeType: getMimeByExt('png'),
            encodingType: 'png'
        },
        'svg' : {
            fileExtension: 'svg',
            mimeType: getMimeByExt('svg'),
            encodingType: 'svg'
        },
        'jpeg' : {
            fileExtension: 'jpg',
            mimeType: getMimeByExt('jpg'),
            encodingType: 'jpeg'
        }
    };

    //check if the parameters are defined , if not return error code
    if ((!jobParams) || (!jobParams.layerIds) || (!jobParams.encodingType)) {
        res.statusCode = 400;
        res.send("Bad URL params");
        return false;
    }

    if (!validEncodingTypes.hasOwnProperty(jobParams.encodingType)) {
        res.statusCode = 406;
        res.send("Unsupported encoding");
        return false;
    } else {
        //default to encodingQualityFactor -1
        if (!jobParams.encodingQualityFactor) {
            jobParams.encodingQualityFactor = -1;
        }

        //input params are valid.
        res.statusCode = 200;

        return validEncodingTypes[jobParams.encodingType];
    }
}

/**
    For a given PSD, create a new, derived asset from one or more of the layers contained within
    the document. A serialized JSON object specifying the parameters is the payload of the POST request.
    The properties of the un-named object must include:
        layerIds – an array of numbers, denoting the layer ids to include
        encoding – one of "png" or "jpeg". Will be adding GIF shortly
        quality - one of "low", "medium" or "high"
        psdGuid – the guid of the PSD
        assetName – what you want to name it

    Parameters
        :psdGuid - the GUID of the PSD in question

**/
function extractAsset(req, res, attachment) {
    logger.verbose('entering extractAsset');
    var authorization = handlerUtils.getAuthHeader(req);
    var serviceToken = null;
    var generatedAssetId = null;
    var generatedAssetUrl = null;
    var requuid = stepUtils.generateUUID();
    var theAssetId = req.params.id;
    var layerCompId = req.params.layerCompId;

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    //get the query parameters
    var url_parts = url.parse(req.url, true),
        jobParams = url_parts.query,
        extractAssetParams = validateJobParams(jobParams, res);

    //if input params are not valid return response statusCode
    if (res.statusCode !== 200) {
        return;
    }

    //set the assetId
    jobParams.psdGuid = theAssetId;

    if (layerCompId) {
        jobParams.layerCompId = layerCompId;
    }

    //get the layerId
    //convert the string into array of numbers
    var sLayerIds = [];
    jobParams.layerIds.split(',').forEach(function (aLayer) {
        sLayerIds.push(parseInt(aLayer, 10));
    });
    jobParams.layerIds = sLayerIds;


    //generate assetName/layerName to download to the browser client , it is what appears in download area inside the browser
    //append the first layerId to base prefix "layer"
    var layerName = "layer";
    if (jobParams.layerIds.length > 0) {
        layerName += jobParams.layerIds[0];
    }

    //Normalize the encoding type based on what came back from the validate function
    //We do this because sharedCloud expects png instead of png32
    jobParams.encodingType = extractAssetParams.encodingType;

    //generated layername based on layerId and fileExtension
    jobParams.assetName = layerName;
    layerName += ".";
    layerName += extractAssetParams.fileExtension;
    var encodedFileName = encodeURIComponent(layerName);

    //set the response header for file download
    if (attachment) {
        res.setHeader('Content-disposition', 'attachment; filename="' + encodedFileName + '";' + " filename*=UTF-8''" + encodedFileName);

    }

    res.setHeader('Content-type', extractAssetParams.mimeType);
    var fullBody = '';
    try {
        req.on('data', function (chunk) {
            // append the current chunk of data to the fullBody variable
            fullBody += chunk.toString();
        });

        req.on('end', function () {
            step(
                imsServiceTokenUtils.asyncGetServiceToken,
                function postDerivedAssetWorker(err, localServiceToken) {
                    if (err) { throw err; }

                    serviceToken = localServiceToken;
                    sharedCloudUtils.scCreateDerivedAssetWorker(err, localServiceToken, authorization, jobParams, requuid, !attachment, this);
                },
                function waitForDerivedAssetWorker(err, statusCode, psdGuid, workerId) {

                    if (err) { throw err; }

                    if (statusCode === 201 && workerId) {
                        sharedCloudUtils.waitForWorkerToComplete(serviceToken, authorization, workerId, requuid, this);
                    } else if (statusCode === 401) {
                        res.send(statusCode, "Error creating derived asset worker");
                    } else {
                        throw fullBody;
                    }
                },
                function handleDerivedAssetResults(err, workerResults, workerOutput) {
                    if (err) { throw err; }

                    if (!workerOutput || !workerOutput[0]) {
                        throw "No worker output specified";
                    }

                    var idx = workerOutput[0].lastIndexOf("/");
                    if (-1 === idx) {
                        throw "Worker output URI malformed";
                    }

                    generatedAssetUrl = workerOutput[0];
                    generatedAssetId = workerOutput[0].substring(idx + 1);

                    sharedCloudUtils.scGetAssetMetadata(err, serviceToken, authorization, generatedAssetId, requuid, this);
                },
                function returnCreatedAsset(err, metadata) {
                    if (err) {
                        handlerUtils.genericErrorMessage(err, res, "extractAsset");
                        return;
                    }

                    var options = {
                        uri : generatedAssetUrl,
                        method: 'GET',
                        headers: {
                            'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(serviceToken),
                            'X-User-Token': sharedCloudConfig.getUserHeaderValue(authorization),
                            'x-request-id': 'Parfait-GDR-' + requuid
                        }
                    };

                    var req = request(options);
                    res.statusCode = 200;
                    req.pipe(res);
                }
            );
        });
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "extractAsset");
    }
}

function deriveSpriteRendition(req, res) {
    extractAsset(req, res, false);
}

function deriveAsset(req, res) {
    extractAsset(req, res, true);
}

function saveDeriveAssetSC(req, res, next) {
    logger.verbose('entering deriveAsset');

    var authorization = handlerUtils.getAuthHeader(req);
    var serviceToken = null;
    var generatedAssetId = null;
    var generatedAssetUrl = null;
    var psdName = null;
    var parentCollectionUrl = null;
    var parentCollectionId = null;
    var requuid = stepUtils.generateUUID();
    var theAssetId = req.params.id;
    var existingAssetId = null;

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }
    var fullBody = '';

    try {
        req.on('data', function (chunk) {
            // append the current chunk of data to the fullBody variable
            fullBody += chunk.toString();
        });

        req.on('end', function () {
            var jobParams = JSON.parse(fullBody),
                extractAssetParams = validateJobParams(jobParams, res);

            //if input params are not valid return response statusCode (error code)
            if (res.statusCode !== 200) {
                return;
            }

            //set the assetId
            jobParams.psdGuid = theAssetId;

            //Normalize the encoding type based on what came back from the validate function
            //We do this because sharedCloud expects png instead of png32
            jobParams.encodingType = extractAssetParams.encodingType;

            //append the file extension to assetname which defaults to user specified input
            jobParams.assetName += ".";
            jobParams.assetName += extractAssetParams.fileExtension;

            step(
                imsServiceTokenUtils.asyncGetServiceToken,
                function psdName(err, localServiceToken) {
                    //get the name of the psd
                    serviceToken = localServiceToken;
                    sharedCloudUtils.scGetAssetMetadata(err, serviceToken, authorization, theAssetId, requuid, this);
                },
                function returnAssetMetadataResults(err, statusCode, responseText) {
                    if (err) { throw err; }

                    if (200 !== statusCode) {
                        res.send(statusCode, responseText);
                        return;
                    }

                    var responseJSON = JSON.parse(responseText);
                    psdName = responseJSON.asset.fileName;

                    var extIndex = psdName.lastIndexOf(".");
                    if (extIndex !== -1) {
                        psdName = psdName.substr(0, extIndex);
                    }
                    sharedCloudUtils.scListCollection(err, serviceToken, authorization, requuid, this);
                },
                function returnCollectionResults(err, statusCode, responseText) {
                    if (err) { throw err; }

                    if (200 !== statusCode) {
                        res.statusCode = statusCode;
                        res.send(responseText);
                        return;
                    }

                    //traverse the existing collections to see if the psdName-assets folder already exist or not

                    var responseJSON = JSON.parse(responseText);
                    var folderNameToSearch = psdName + "_assets";
                    var i,
                        folderAlreadyExists;
                    for (i in responseJSON) {
                        if (responseJSON.hasOwnProperty(i)) {
                            if (responseJSON[i].metadata.collection.collection_name === folderNameToSearch) {
                                folderAlreadyExists = true;
                                parentCollectionUrl = responseJSON[i].url;
                                parentCollectionId = responseJSON[i].id;
                                break;
                            }
                        }
                    }

                    if (!folderAlreadyExists) {
                        //if not found - create one
                        sharedCloudUtils.scCreateCollection(err, serviceToken, authorization, folderNameToSearch, requuid, this);
                    } else {
                        var that = this;
                        that(null, parentCollectionId, parentCollectionUrl);
                    }
                },
                function listAssetsInCollection(err, parentId, parentUrl) {
                    if (err) { throw err; }

                    parentCollectionUrl = parentUrl;
                    parentCollectionId = parentId;
                    sharedCloudUtils.scListAssetsInCollection(err, serviceToken, authorization, parentCollectionId, requuid, this);
                },
                function returnAssetsInCollection(err, statusCode, responseText) {
                    if (err) {
                        handlerUtils.genericErrorMessage(err, res, "saveDeriveAssetSC");
                        return;
                    }

                    if (200 !== statusCode) {
                        res.statusCode = statusCode;
                        res.send(responseText);
                        return;
                    }

                    //traverse the existing collections to see if the layername/assetname already exist or not
                    var responseJSON = JSON.parse(responseText);
                    var i;
                    for (i in responseJSON.assets) {
                        if (responseJSON.assets.hasOwnProperty(i)) {
                            if (responseJSON.assets[i].name === jobParams.assetName) {
                                existingAssetId =  responseJSON.assets[i].id;
                                break;
                            }
                        }
                    }

                    if (existingAssetId) {
                        //delete the existing asset with same layername/assetname
                        sharedCloudUtils.deleteAssetQueueAction(serviceToken, authorization, existingAssetId, requuid, this);
                    } else {
                        var that = this;
                        that(null, statusCode, responseText);
                    }
                },
                function returnDeleteAssetResults(err, statusCode, responseText) {
                    if (err) {
                        handlerUtils.genericErrorMessage(err, res, "saveDeriveAssetSC");
                        return;
                    }

                    //extract layer and save in SC
                    var that = this;
                    that(null, serviceToken);
                },
                function postDerivedAssetWorker(err, localServiceToken) {
                    if (err) { throw err; }

                    sharedCloudUtils.scCreateDerivedAssetWorker(err, serviceToken, authorization, jobParams, requuid, false, this);
                },
                function waitForDerivedAssetWorker(err, statusCode, psdGuid, workerId) {
                    if (err) { throw err; }

                    if (statusCode === 201 && workerId) {
                        sharedCloudUtils.waitForWorkerToComplete(serviceToken, authorization, workerId, requuid, this);
                    } else {
                        throw fullBody;
                    }
                },
                function handleDerivedAssetResults(err, workerResults, workerOutput) {
                    if (err) { throw err; }

                    if (!workerOutput || !workerOutput[0]) {
                        throw "No worker output specified";
                    }

                    var idx = workerOutput[0].lastIndexOf("/");
                    if (-1 === idx) {
                        throw "Worker output URI malformed";
                    }

                    generatedAssetUrl = workerOutput[0];
                    generatedAssetId = workerOutput[0].substring(idx + 1);

                    sharedCloudUtils.scMoveGeneratedAsset(err, serviceToken, authorization, jobParams.assetName, generatedAssetUrl, parentCollectionUrl, requuid, this);
                },
                function returnMoveGeneratedAssetResults(err, statusCode) {

                    if (err) {
                        res.send(statusCode || 500, err);
                        res.end();
                        return;
                    }

                    sharedCloudUtils.scGetAssetMetadata(err, serviceToken, authorization, generatedAssetId, requuid, this);
                },
                function returnDerivedAssetMetadata(err, statusCode,  metadata) {
                    if (err) {
                        res.send(statusCode || 500, err);
                        res.end();
                        return;
                    }

                    var ret = {};

                    ret.id = generatedAssetId;
                    ret.url = generatedAssetUrl;
                    ret.type = "asset";
                    ret.metadata = JSON.parse(metadata);

                    res.cache("no-Cache");
                    res.writeHead(200, {'Content-Type': 'application/json'});
                    res.end(JSON.stringify(ret));
                    return;
                }
            );
        });
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "deriveAsset");
    }
}

function downloadDerivedAsset(req, res, next) {
    logger.verbose('entering downloadDerivedAsset');

    var authorization = handlerUtils.getAuthHeader(req);
    var serviceToken = null;
    var requuid = stepUtils.generateUUID();
    var theDerivedAssetId = req.params.id;
    var theDerivedAssetUrl = null;
    var theDerivedAssetName = null;
    var theDerivedAssetMimeType = null;

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function contentType(err, localSeriviceToken) {
                //get the content type for the derived asset being downloaded
                serviceToken = localSeriviceToken;
                sharedCloudUtils.scGetAssetMetadata(err, serviceToken, authorization, theDerivedAssetId, requuid, this);
            },
            function returnAssetMetadataResults(err, statusCode, responseText) {
                if (err) { throw err; }

                var responseJSON = JSON.parse(responseText);
                theDerivedAssetName = responseJSON.asset.fileName;
                theDerivedAssetUrl = sharedCloudConfig.getSharedCloudURI() + "/api/v1/assets/" + theDerivedAssetId;

                var fileExtension;
                var extIndex = theDerivedAssetName.lastIndexOf(".");
                if (extIndex !== -1) {
                    fileExtension = theDerivedAssetName.substr(extIndex + 1);
                }

                //look up mimetype based on file extension
                theDerivedAssetMimeType = getMimeByExt(fileExtension);

                //set the content type
                var encodedFileName = encodeURIComponent(theDerivedAssetName);
                res.setHeader('Content-disposition', 'attachment; filename="' + encodedFileName + '";' + " filename*=UTF-8''" + encodedFileName);
                res.setHeader('Content-type', theDerivedAssetMimeType);


                var options = {
                    uri : theDerivedAssetUrl,
                    method: 'GET',
                    headers: {
                        'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(serviceToken),
                        'X-User-Token': sharedCloudConfig.getUserHeaderValue(authorization),
                        'x-request-id': 'Parfait-DDA-' + requuid
                    }
                };

                var req = request(options);
                res.statusCode = 200;
                req.pipe(res);
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "downloadDerivedAsset");
    }
}

/**
    Create a new Graphite extraction worker in the shared cloud. The GUID of the resulting worker,
    if it's successfully created, is returned in the content of the HTTP response.

    Parameters
        :id - the GUID of the PSD to queue for processing
        :layerCompId - Optional layer comp id.
**/
function createJSONWorkerHandler(req, res, next) {
    logger.verbose('entering createJSONWorkerHandler');
    var authorization = handlerUtils.getAuthHeader(req);
    var requuid = stepUtils.generateUUID();
    var layerCompId = req.params.layerCompId;

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function postWorker(err, localServiceToken) {
                if (err) { throw err; }
                sharedCloudUtils.scCreateGraphiteWorker(err, localServiceToken, authorization, req.params.id, req.params.layerCompId, requuid, {}, this);
            },
            function handleResult(err, statusCode, assetId, workerId, bodyText) {
                stepUtils.handleResultFromWorker(err, res, statusCode, assetId, layerCompId, workerId, bodyText);
            }
        );

    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "createJSONWorkerHandler");
    }
}

function getGraphiteWorkerResultsImpl(req, res, inWorkerId) {
    logger.verbose('entering getGraphiteWorkerResultsImpl');
    var authorization = handlerUtils.getAuthHeader(req);
    var requuid = stepUtils.generateUUID();

    res.cache("no-Cache");
    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    if (null === inWorkerId) {
        // There is no worker
        res.statusCode = 204;
        res.send("No Worker");
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function (err, localServiceToken) {
                // Get the workerId from the assetId
                var that = this;
                that(err, localServiceToken, inWorkerId);

            },

            function (err, localServiceToken, workerId) {
                if (workerId !== null) {
                    var options = {
                        host: sharedCloudConfig.getSharedCloudHost(),
                        port: sharedCloudConfig.getSharedCloudPort(),
                        path: "/api/v1/jobs/" + workerId,
                        method: 'GET',
                        headers: {
                            'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(localServiceToken),
                            'X-User-Token': sharedCloudConfig.getUserHeaderValue(authorization),
                            'x-request-id': 'Parfait-GWR-' + requuid
                        }
                    };

                    var bodyarr = [];
                    var thatResp = null;

                    var myreq = https.request(options, function (response) {
                        response.setEncoding('utf8');
                        thatResp = response;
                        response.on('data', function (chunk) {
                            bodyarr.push(chunk);
                        });

                        response.on('end', function () {
                            var respText = bodyarr.join('');
                            if (response.statusCode === 404) { // If it's a 404, remove the workerId from the AWS.
                                awsUtils.setWorkerIdForAsset(req.params.id, '');
                                res.statusCode = 204;
                                respText = "No Worker";
                            } else if (response.statusCode !== 200) {
                                res.statusCode = response.statusCode;
                            }
                            res.send(respText);
                        });
                    });

                    myreq.on('error', function (e) {
                        logger.error('problem with request: %s', e.message);
                    });
                    myreq.end();
                } else {
                    // There is no worker
                    res.statusCode = 204;
                    res.send("No Worker");
                }
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getGraphiteWorkerResultsImpl");
    }
}

/**
    Retrieve the results of the last queued Graphite worker for an asset. The format of
    the data that is returned is JSON, and is a simple passthrough of the data generated
    by the shared cloud. Note that this API is often used to check on the status of the
    processing (whether it has started, progress, etc).

    Parameters
        :id - the GUID of the PSD to queue for processing
**/

function getGraphiteJSONWorkerResultsForGenericJob(req, res, next) {
    getGraphiteWorkerResultsImpl(req, res, req.params.workerId);
}

function putCustomMetadataOnAsset(req, res, next) {
    logger.verbose('entering putCustomMetadataOnAsset');
    var authorization = handlerUtils.getAuthHeader(req);
    var requuid = stepUtils.generateUUID();

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    var fullBody = '';
    try {
        req.on('data', function (chunk) {
            // append the current chunk of data to the fullBody variable
            fullBody += chunk.toString();
        });
        req.on('end', function () {
            var metaData = JSON.parse(fullBody);

            step(
                imsServiceTokenUtils.asyncGetServiceToken,
                function (err, localServiceToken) {
                    sharedCloudUtils.scSetMetadataOnAsset(authorization, localServiceToken, req.params.id, metaData, requuid, this);
                },
                function (err, result) {
                    res.send(200, "ok");
                }
            );
        });

    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "putCustomMetadataOnAsset");
    }

}

/**
    Retrieve the generated Graphite JSON metadata structure for a specific PSD. This format is documented at Graphite JSON data description.

    Parameters
        :id - the GUID of the PSD in question
**/
function getGraphiteDataForAsset(req, res, next) {
    logger.verbose('entering getGraphiteDataForAsset');
    var asyncCallStartTime = Date.now();
    var myServiceToken = null;
    var authHeader = null;
    var theAssetId = req.params.id;
    var layerCompId = req.params.layerCompId;
    var layerCompInfo = layerCompId ? "-lcid" + layerCompId : "";
    var requuid = stepUtils.generateUUID();

    var theGraphiteDataEtag = req.headers['if-none-match'];

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function (err, localServiceToken) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "getGraphiteDataForAsset");
                    return;
                }

                asyncCallStartTime = Date.now();

                authHeader = handlerUtils.getAuthHeader(req);
                myServiceToken = localServiceToken;
                var options = {
                    host: sharedCloudConfig.getSharedCloudHost(),
                    port: sharedCloudConfig.getSharedCloudPort(),
                    path: "/api/v1/assets/" + theAssetId + "/renditions/graphite/json_data" + layerCompInfo,
                    method: 'GET',
                    headers: {
                        'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(localServiceToken),
                        'X-User-Token': sharedCloudConfig.getUserHeaderValue(authHeader),
                        'x-request-id': 'Parfait-GGD-' + requuid
                    }
                };

                if (theGraphiteDataEtag) {
                    options.headers['If-None-Match'] = theGraphiteDataEtag;
                }

                var bodyarr = [],
                    that = this;
                var myreq = https.request(options, function (response) {
                    response.on('data',
                        function (chunk) {
                            bodyarr.push(chunk);
                        });
                    response.on('end',
                        function () {
                            logger.debug("getGraphiteDataForAsset success. Elapsed time= %d ms", Date.now() - asyncCallStartTime);
                            var respText = bodyarr.join('');
                            res.setHeader('etag', response.headers['etag']);
                            res.statusCode = response.statusCode;
                            if (res.statusCode === 200 && respText !== "") {
                                try {
                                    var assetData = JSON.parse(respText);
                                    res.send(assetData);
                                } catch (err) {
                                    res.send(respText);
                                }
                            } else {
                                res.send(respText);
                            }
                        });
                });
                myreq.on('error', function (e) {
                    that(e, null);
                });
                myreq.end();

            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getGraphiteDataForAsset");
    }

}

/**
    Returns the version information of the worker deployed in the Shared Cloud environment utilized by the graphite server.
**/
function getGraphiteVersion(req, res, next) {
    var authHeader = null;
    var authorization = handlerUtils.getAuthHeader(req);
    res.cache("no-Cache");
    var requuid = stepUtils.generateUUID();

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function postVerCheckingWorker(err, localServiceToken) {
                if (err) { throw err; }

                var postDataObj = {};
                postDataObj.parameters = {};
                postDataObj.parameters.cmd = "workerInfo";
                postDataObj.inputs = [];
                var that = this;

                var callback = function (err, statusCode, psdGuid, workerId) {
                    if (err) {
                        that(err);
                    } else if (statusCode === 201 && workerId) {
                        that(null, localServiceToken, workerId);
                    } else {
                        that('Error: could not create worker');
                    }
                };
                sharedCloudUtils.scCreateGenericWorker(err, localServiceToken, authorization, postDataObj, null, requuid, callback);
            },
            function getWorkerVersionFromResult(err, localServiceToken, workerId) {
                if (err) { throw err; }

                logger.verbose('entering getWorkerVersionFromResult for %s', workerId);

                sharedCloudUtils.waitForWorkerToComplete(localServiceToken, authorization, workerId, requuid, this);
            },
            function handleWorkerResults(err, workerResults) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "getGraphiteVersion");
                } else {
                    res.statusCode = 200;
                    res.send(workerResults.graphite_workerInfo);
                }
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getGraphiteVersion");
    }
}

function getUserQuota(req, res, next) {
    var authHeader = null;
    var authorization = handlerUtils.getAuthHeader(req);
    res.cache("no-Cache");

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function getUserQuotaImpl(err, localServiceToken) {
                if (err) { throw err; }
                users.getUserQuota(null, localServiceToken, authorization, this);
            },
            function returnUserQuota(err, data) {
                if (err) {
                    if (data === 401) {
                        res.send(data, "Error retrieving user quota");
                    } else {
                        handlerUtils.genericErrorMessage(err, res, "getUserQuota");
                    }
                } else {
                    res.statusCode = 200;
                    res.send(data);
                }
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getUserQuota");
    }
}

function getAssetETag(req, res, next) {
    var assetId = req.params.id;
    var requuid = stepUtils.generateUUID();
    var authorization = handlerUtils.getAuthHeader(req);
    res.cache("no-Cache");

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,
            function getUserQuotaImpl(err, localServiceToken) {
                if (err) { throw err; }
                sharedCloudUtils.scGetAssetEtag(err, localServiceToken, authorization, assetId, requuid, this);
            },
            function returnETag(err, data, response) {
                if (err) {
                    handlerUtils.genericErrorMessage(err, res, "getAssetETag");
                } else {
                    if (response.statusCode === 401) {
                        res.send(401, "Error retrieving etag");
                    } else {
                        res.send(response.statusCode, data);
                    }
                }
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getAssetETag");
    }
}

function getUserData(req, res, next) {
    var authorization = handlerUtils.getAuthHeader(req),
        adobeData,
        userData;

    res.cache("no-Cache");

    try {
        if (!authorization) {
            throw "no authorization";
        }

        step(
            function getAdobeId() {
                imsServiceTokenUtils.getAdobeID(authorization, this);
            },
            function getUserData(err, data) {
                if (err) {
                    throw err;
                }
                adobeData = data;
                userFeatureUtils.getUserData(data.email, true, this);
            },
            function saltUser(err, data) {
                if (err) {
                    res.send(200, JSON.stringify({adobeID: null, user_data: null, error: err}));
                } else {
                    userData = data;
                    userData.email = adobeData.email;
                    userFeatureUtils.getUserKey(adobeData.email, this);
                }
            },
            function returnData(err, saltedUser) {
                if (err) {
                    res.send(200, JSON.stringify({adobeID: null, user_data: null, error: err}));
                } else {
                    var header = {'Content-Type': 'application/json',
                                  'Set-Cookie': ['parfait-user-data=' + userFeatureUtils.convertToCookieString(userData) + "; path=/; HttpOnly"]};
                    res.writeHead(200, header);
                    res.end(JSON.stringify({adobeID: saltedUser.toString('hex'), user_data: userData}));
                }
            }
        );
    } catch (err) {
        res.send(200, JSON.stringify({adobeID: null, user_data: null}));
    }
}

function getUserStats(req, res, next) {
    res.cache("no-Cache");

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    awsUtils.countEnabledUsers(function (err, data) {
        res.writeHead(200, {'Content-Type': 'application/json'});
        res.end(JSON.stringify(data));
    });
}

function getUserFeatureFlags(req, res, next) {
    var userID;
    var newData;
    res.cache("no-Cache");

    if (handlerUtils.userNotAllowed(req, res)) {
        return;
    }

    try {
        step(
            function getUserData() {
                userFeatureUtils.getUserData(req.params.id, false, this);
            },
            function returnResult(err, data) {
                if (err) {
                    data = {error: err};
                }

                if (!data) {
                    data = {error: "User is not registered"};
                }

                res.writeHead(200, {'Content-Type': 'application/json'});
                res.end(JSON.stringify(data));
            }
        );
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "getUserFeatureFlags");
    }
}

function setAnalyticsOptIn(req, res, next) {
    logger.verbose('setAnalyticsOptIn');

    var authorization = handlerUtils.getAuthHeader(req);
    var ret = {};
    var fullBody = '';
    try {
        req.on('data', function (chunk) {
            // append the current chunk of data to the fullBody variable
            fullBody += chunk.toString();
        });
        req.on('end', function () {
            var analyticsParams = JSON.parse(fullBody);
            var imsData;

            step(
                function getAdobeId() {
                    imsServiceTokenUtils.getAdobeID(authorization, this);
                },
                function getUserData(err, data) {
                    if (err) {
                        throw err;
                    }
                    imsData = data;
                    //retrive the record
                    userFeatureUtils.getUserData(data.email, false, this);
                },
                function updateAnalyticsOptIn(err, data) {
                    if (err) {
                        throw err;
                    }
                    //update the record optin/opt out option
                    data.analyticsOptIn = analyticsParams.analyticsOptIn;
                    userFeatureUtils.putUserData(imsData.email, data, this);
                },
                function returnResult(err, data) {
                    if (err) {
                        ret = {error: err};
                    }
                    res.cache("no-Cache");
                    res.writeHead(200, {'Content-Type': 'application/json'});
                    res.end(JSON.stringify(ret));
                }
            );
        });
    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "setAnalyticsOptIn");
    }
}

function getEnvironmentData(req, res, next) {
    var ret = {};
    ret.SHAREDCLOUD_HOST = sharedCloudConfig.getSharedCloudHost();
    ret.IMS_HOST = sharedCloudConfig.getIMSHost();
    ret.LOGOUT_HOST = sharedCloudConfig.getServicesHost();
    ret.ENVIRONMENT = sharedCloudConfig.getCurrentEnvironment();
    ret.IMAGE_COMPARE_TESTING = process.env.IMAGE_COMPARE_TESTING;

    res.cache("no-Cache");
    res.writeHead(200, {'Content-Type': 'application/json'});
    res.end(JSON.stringify(ret));
}

function deleteGraphiteDataHandler(req, res, next) {
    var theAssetId = req.params.id;
    logger.verbose('entering deleteGraphiteDataHandler for assetID= %s', theAssetId);
    var authorization = handlerUtils.getAuthHeader(req);
    var theGraphiteDataFileName = handlerUtils.generateGrDataAssetName(theAssetId);
    var requuid = stepUtils.generateUUID();
    res.cache("no-Cache");

    try {
        step(
            imsServiceTokenUtils.asyncGetServiceToken,

            function (err, localServiceToken) {
                sharedCloudUtils.scListAssets(err, localServiceToken, authorization, requuid, this);
            },
            function getGraphiteAsset(err, statusCode, responseText, localServiceToken) {
                if (err) { throw err; }

                var theGraphiteDataGuid = null;
                var theGraphiteDataEtag = null;

                if (200 !== statusCode) {
                    res.statusCode = statusCode;
                    res.send(responseText);
                    return;
                }

                var foo = JSON.parse(responseText),
                    i;
                for (i in foo) {
                    if (foo.hasOwnProperty(i)) {
                        if (foo[i].metadata.asset.fileName === theGraphiteDataFileName) {
                            theGraphiteDataGuid = foo[i].id;
                            break;
                        }
                    }
                }

                var that = this;
                var options = {
                    hostname: sharedCloudConfig.getSharedCloudHost(),
                    port: sharedCloudConfig.getSharedCloudPort(),
                    path: "/api/v1/assets/" + theGraphiteDataGuid,
                    method: 'HEAD',
                    headers: {
                        'Authorization': sharedCloudConfig.getAuthorizationHeaderValue(localServiceToken),
                        'X-User-Token': sharedCloudConfig.getUserHeaderValue(authorization),
                        'x-request-id': 'Parfait-GGA-' + requuid
                    }
                };

                logger.debug("SC-API request: %s - %s", options.method, options.path);

                var myreq = https.request(options, function (response) {
                    res.on('data', function (chunk) {
                    });
                });

                myreq.on('response', function (response) {
                    theGraphiteDataEtag = response.headers.etag;
                    that(null, localServiceToken, theGraphiteDataGuid, theGraphiteDataEtag);
                });

                myreq.on('error', function (e) {
                    that(e, null);
                });
                myreq.end();
            },

            function deleteGraphiteAsset(err, localServiceToken, graphiteDataGuid, graphiteDataEtag) {
                sharedCloudUtils.scDeleteAsset(err, localServiceToken, authorization, graphiteDataGuid, graphiteDataEtag, requuid, this);
            },

            function endDelete(err, statusCode, respText, localServiceToken) {
                if (statusCode === 200) {
                    res.statusCode = 200;
                } else {
                    res.statusCode = statusCode;
                    if (respText !== "" && respText !== null && respText !== undefined) {
                        res.send(respText);
                        return;
                    }
                }
                res.end();
            }
        );

    } catch (err) {
        handlerUtils.genericErrorMessage(err, res, "deleteGraphiteDataHandler");
    }
}

// this is only used for the compositeTest route for downloading images for image compare testing
function getTestUserLogin(req, res, next) {
    if (process.env.GRAPHITE_TESTUSER && process.env.GRAPHITE_TESTPASSWORD) {
        res.writeHead(200, {'Content-Type': 'application/json'});
        res.end(JSON.stringify({adobeID: process.env.GRAPHITE_TESTUSER, password: process.env.GRAPHITE_TESTPASSWORD}));
    } else {
        res.send(500, "Could not get test user info");
    }
}

function installV1Handlers(server) {
    server.get('/api/v1/login', userLogin);
    server.get('/api/v1/logout', userLogout);
    server.post('/api/v1/psd', uploadAssetHandler); //Documented
    // DPO - This appears to be an unused endpoint.  Removing for now.
    // server.post('/api/v1/psdURL', uploadURLassetHandler); //Documented

    server.get('/api/v1/uploadProgress/:id', getUploadProgress);
    server.post('/api/v1/uploadProgress/', getMultiUploadProgress);
    server.del('/api/v1/uploadProgress/:id', deleteUploadProgress);

    server.get('/api/v1/psd', listAssetsHandler); //Documented
    server.get('/api/v1/psd/:id/info', getAssetInfo); //Documented
    server.get('/api/v1/psd/:id/thumbnail', getThumbnailHandler); //Documented
    server.get('/api/v1/psd/:id/spriteSheet/:sheetId', getSpriteSheetRendition); //Documented
    server.get('/api/v1/psd/:id/:layerCompId/spriteSheet/:sheetId', getSpriteSheetRendition); //Documented
    server.get('/api/v1/psd/:id/sprite/derived', deriveSpriteRendition);
    server.get('/api/v1/psd/:id/:layerCompId/sprite/derived', deriveSpriteRendition);
    server.get('/api/v1/psd/:id/asset/derived', deriveAsset);
    server.get('/api/v1/psd/:id/:layerCompId/asset/derived', deriveAsset);
    server.get('/api/v1/psd/:id/derived', getDerivedAssets); //Documented
    server.post('/api/v1/psd/:id/derived', saveDeriveAssetSC); //Documented

    server.get('/api/v1/psd/:id/graphitejson', getGraphiteDataForAsset); //Documented
    server.get('/api/v1/psd/:id/:layerCompId/graphitejson', getGraphiteDataForAsset);
    server.post('/api/v1/psd/:id/graphitejson', createJSONWorkerHandler); //Documented
    server.post('/api/v1/psd/:id/:layerCompId/graphitejson', createJSONWorkerHandler);
    server.del('/api/v1/psd/:id/delete', deleteAssetHandler); //Documented
    server.del('/api/v1/psd/:id/derived/delete', deleteDerivedAssetHandler);
    server.get('/api/v1/psd/:id', getDefaultRendition);
    server.put('/api/v1/psd/:id/metadata', putCustomMetadataOnAsset);

    server.get('/api/v1/asset/:sharedCloudGuid', getGenericAssetHandler); //Documented
    server.get('/api/v1/asset/:id/thumbnail', getThumbnailHandler);
    server.get('/api/v1/asset/:id/download', downloadDerivedAsset); //Documented
    server.get('/api/v1/asset/:id/etag', getAssetETag); //Documented

    server.get('/api/v1/worker/:workerId', getGraphiteJSONWorkerResultsForGenericJob);

    server.get('/api/v1/environment', getEnvironmentData);

    server.get('/api/v1/user/quota', getUserQuota);
    server.get('/api/v1/user/data', getUserData);
    server.get('/api/v1/user/stats', getUserStats);
    server.get('/api/v1/user/:id', getUserFeatureFlags);
    server.post('/api/v1/user/analyticsOptIn', setAnalyticsOptIn);

    /*
        DPO - Commenting out the resetAcct method to protect from CSRF.  Before we turn this back on we should:
            1 - Change from Get to Post
            2 - Re-add the reset button to public/js/views/ViewTemplates.html Admin template
            3 - Change button to submit a form as a post containing a temporary token.  Example: http://shiflett.org/articles/cross-site-request-forgeries
            4 - Move this handler to the admin handlers.
    server.get('/api/v1/admin/resetAcct', resetGraphiteAcct);  //Documented
    */
    // DPO - These look admin API methods. TODO: It should be moved to the Admin handlers.
    server.get('/api/v1/admin/graphiteVersion', getGraphiteVersion); //Documented
    server.post('/api/v1/admin/urlAsset', uploadURLassetHandler);
    server.get('/api/v1/testUser', getTestUserLogin);
}


function installHandlers(server) {
    installV1Handlers(server);
}


module.exports = {
    installHandlers : installHandlers,
    resetGraphiteAcct : resetGraphiteAcct
};
