/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2013-2014 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true */
/*global define: true, graphite: true, Modernizr: true, window: true, clearTimeout: true, setTimeout: true, FileReader: true*/

define([
    'jquery',
    'underscore',
    'backbone',
    'plugin-dependencies',
    'views/landing/AssetView',
    'controllers/UploadController',
    'controllers/AuthController',
    'controllers/AssetListController',
    'utils/TemplateUtil',
    'parfait/AssetListView',
    'breadcrumb',
    'models/AssetModel',
    'dreamweaver/utils/CommonUtil',
    'DWPSDExtractManager'
], function ($, _, Backbone, deps, AssetView, UploadController,
    AuthController, AssetListController, TemplateUtil, AssetListView, breadcrumb, AssetModel, CommonUtil, DWPSDExtractManager) {

    "use strict";

    var DwAssetListView = AssetListView.extend({
        
        initialize: function (authModel, path) {
            AssetListView.prototype.initialize.apply(this, arguments);
            _.bindAll(this, "resizeHandler");
            $(window).resize(this.resizeHandler);
        },
        
        addHandlers: function () {
            AssetListView.prototype.addHandlers.apply(this, arguments);
            window.graphite.events.on('loadMoreAssets', this.loadMoreAssets, this);
            window.graphite.events.on('loadingAssetsComplete', this.assetLoadingComplete, this);
        },
        
        assetLoadingComplete: function () {
            var uploadPSDElelment = $('.grid article.card-asset .uploadAssetContainer');
            if (uploadPSDElelment && uploadPSDElelment.length > 0 && window.dwData.shouldShowUploadDialogOnReload === "true") {
                DWPSDExtractManager.setPrefToShowUploadDialogOnReload(false);
                window.dwData.shouldShowUploadDialogOnReload = "false";
                uploadPSDElelment.trigger('click');
            }
        },
        
        loadMoreAssets: function (incrementalFetchObj) {
            this.collection = AssetListController.getAssetList(
                this.onAssetCollectionSuccess,
                this.onAssetCollectionError,
                this,
                incrementalFetchObj.path,
                incrementalFetchObj.xChildrenNextStart
            );
        },
        
        resizeHandler: function () {
            // Calculate bounds and lay out cards dynamically
            var cardMinWidth = 168,
                cardWidth = 200,
                $cardHolder = $('#card-holder')[0],
                containerWidth = $cardHolder ? $cardHolder.getBoundingClientRect().width : 0,
                fittingNumber = Math.floor(containerWidth / cardWidth),
                margin = Math.round((containerWidth - (fittingNumber * cardMinWidth)) / (fittingNumber + 1)),
                timer = null,
                that = this;
            if ($cardHolder) {
                $('.card-asset').css('margin-left', margin + 'px');
            }
            
            if (timer) {
                clearTimeout(timer);
            }

            timer = setTimeout(function () {
                that.$el.find("#breadcrumbs").rcrumbs({
                    nbFixedCrumbs: 1
                });
            }, 100);
        },
        
        inputFileChangeHandler: function (e) {
            var self = e.data.self;
            
            self.showWelcomePageIfNeeded();
            
            e.preventDefault();

            var files = this.files,
                filesToUpload = [],
                i,
                headersLoaded = 0,
                thisContext = this;

            _.each(files, function (file) {
                var blob = file.slice(0, 4, 'UTF-8');
                var reader = new FileReader();
                var mimetype = file.type;
                var currFile = file;
                var ext = file.name.substr((~-file.name.lastIndexOf(".") >>> 0) + 2).toLowerCase();

                reader.addEventListener("loadend", function () {
                    if (this.result === "8BPS") { // http://www.adobe.com/devnet-apps/photoshop/fileformatashtml/#50577409_pgfId-1055726
                        filesToUpload.push(currFile);
                    }

                    headersLoaded = headersLoaded + 1;
                    if (headersLoaded === files.length) {
                        if (filesToUpload.length > 0) {
                            graphite.events.trigger("upload-init", { files: filesToUpload });
                        }
                        if (filesToUpload.length !== files.length) {
                            graphite.events.trigger('load-alert', deps.translate("You can only upload PSD files. " + (files.length - filesToUpload.length) + " of " + files.length + " files were not uploaded."));
                            graphite.events.trigger('bad-file-type-dragdrop', {extension: ext, type: mimetype});
                            window.graphite.events.trigger("bad-file-type-upload", 'badFileUpload', (files.length - filesToUpload.length), files.length);
                        }
                        $(thisContext).val('');
                    }
                });
                reader.readAsText(blob, 'UTF-8');
            });
            
            return false;
        },
        
        isPSDInView: function (psdName) {
            var foundModels = _.filter(this.collection.models, function (model) {
                return (CommonUtil.typeMatchSupportedPSD(model.get('type')) && model.get('name') === psdName);
            });
            if (foundModels && foundModels.length > 0) {
                return true;
            } else {
                var uploadQueue = UploadController.uploadList.concat(UploadController.uploadQueue);
                if (uploadQueue && uploadQueue.length > 0) {
                    var foundInProcessingQueue = _.filter(uploadQueue, function (entry) {
                        return entry.asset.get('name') === psdName;
                    });
                    if (foundInProcessingQueue && foundInProcessingQueue.length > 0) {
                        return true;
                    }
                }
            }
            return false;
        },
            
        uploadFiles: function (data) {
            if (AuthController.isLoggedIn()) {
                var files = data.files,
                    i,
                    counter = 1,
                    newFileFullName = '',
                    fileExt = '',
                    fileOldNameWoExt = '';
                            
                for (i = 0; i < files.length; i++) {
                    newFileFullName = files[i].name;
                    fileExt = window.dwPSDExtractManager.extractFileExt(newFileFullName);
                    fileOldNameWoExt = window.dwPSDExtractManager.extractFileNameWithoutExt(newFileFullName);

                    while (this.isPSDInView(newFileFullName)) {
                        var fileNewNameWoExt = fileOldNameWoExt + '_' +  counter++;
                        newFileFullName = fileNewNameWoExt + '.' + fileExt;
                    }
                    var item = UploadController.upload(files[i], newFileFullName);

                    this.collection.add(item);
                    $("#upload_files_here").hide();
                    newFileFullName = '';
                    counter = 1; //Reset the counter for the next file.
                }
            } else {
                // DPO - I can't work out when or how this method gets fired.
                // There was an alert here before. This would never get shown, because it's not an alert.
                graphite.events.trigger('load-alert', deps.translate("NO_LONGER_LOGGED_IN_LOGIN_AGAIN"));
                window.location.href = window.location;
            }
        },

        //------------------------------------------------
        // Check for Folder Sharing
        //------------------------------------------------

        checkFolderSharing: function (e) {
            if (CommonUtil.typeMatchDirectory(e.type)) {
                var folderShared = window.graphite.routers.main.folderShared;
                
                if (e.collaboration === "INCOMING" || e.collaboration === "OUTGOING") {
                    folderShared["#" + Backbone.history.fragment] = true;
                } else {
                    var currentURLs = Backbone.history.fragment.split('/'),
                        urlIndex = 0,
                        urlInfo = { urls: [] },
                        that = this;

                    for (urlIndex; urlIndex < currentURLs.length; ++urlIndex) {
                        var newURLInfo = {};
                        newURLInfo.label = decodeURI(currentURLs[urlIndex]);
                        if (urlIndex === 0) {
                            if (currentURLs[urlIndex] === '') {
                                currentURLs[urlIndex] = 'files';
                            }
                            currentURLs[urlIndex] = '#' + currentURLs[urlIndex];
                        } else {
                            currentURLs[urlIndex] = currentURLs[urlIndex - 1] + '/' + currentURLs[urlIndex];
                        }
                        newURLInfo.href = currentURLs[urlIndex];
                        urlInfo.urls.push(newURLInfo);
                    }

                    folderShared[urlInfo.urls[urlInfo.urls.length - 1].href] = false;

                    for (urlIndex = currentURLs.length - 2; urlIndex > 0; --urlIndex) {
                        if (folderShared[urlInfo.urls[urlIndex].href] === true) {
                            folderShared[urlInfo.urls[urlInfo.urls.length - 1].href] = true;
                            break;
                        }
                    }
                }
            }

        },
        
        //------------------------------------------------
        // Render
        //------------------------------------------------
        
        renderUploadedAssetView: function () {
            if (this.assetUploadModel) {
                this.assetUploadModel.destroy();
                this.assetUploadModel = null;
            }
            this.assetUploadModel = new AssetModel({type: 'image/vnd.adobe.photoshop/upload'});
            if (this.assetUploadView) {
                this.assetUploadView.remove();
                this.assetUploadView = null;
            }
            this.assetUploadView = new AssetView({
                model: this.assetUploadModel
            });
            
            // Calculate bounds and lay out cards dynamically
            var cardMinWidth = 168,
                cardWidth = 200,
                containerWidth = $('#card-holder')[0].getBoundingClientRect().width,
                fittingNumber = Math.floor(containerWidth / cardWidth),
                margin = Math.round((containerWidth - (fittingNumber * cardMinWidth)) / (fittingNumber + 1));
            
            this.assetUploadView.$el.css('margin-left', margin + 'px');
            $("#card-holder").prepend(this.assetUploadView.el);
        },
        
        renderNewAsset: function (item) {
            //metadata is empty on new uploads, so we need to skip mimetype checks on those
            var mimeType = item ? item.attributes.type : null,
                view;

            if (!mimeType || CommonUtil.typeMatchSupportedPSD(mimeType) || CommonUtil.typeMatchDirectory(mimeType)) {

                var currentCCPath = Backbone.history.fragment || "files";
                if (item.get('ccPath') && item.get('ccPath') !== currentCCPath) {
                    return;
                } else {
                    view = new AssetView({
                        model: item
                    });
                    
                    // Calculate bounds and lay out cards dynamically
                    var cardMinWidth = 168,
                        cardWidth = 200,
                        containerWidth = $('#card-holder')[0].getBoundingClientRect().width,
                        fittingNumber = Math.floor(containerWidth / cardWidth),
                        margin = Math.round((containerWidth - (fittingNumber * cardMinWidth)) / (fittingNumber + 1));
                    
                    view.$el.css('margin-left', margin + 'px');
                    if (item.get("fileToLoad") !== undefined) {
                        //If view is of recently uploaded PSD, append it right after the view of 'Upload New PSD'
                        $(view.render().el).insertAfter($("#card-holder").children().first());
                    } else {
                        //All other asset views should be appended to the list.
                        $("#card-holder").append(view.el);
                    }
                    
                    this.childViews.push(view);
                }
            }
        },
        
        showWelcomePageIfNeeded: function () {
            var argsObj = {};
            argsObj.callback = function () {
                window.graphite.extractHiddenWelcomePage = false;
            };
            
            if (window.graphite.extractHiddenWelcomePage === true) {
                window.dwPSDExtractManager.showWelcomePageIfNeeded(argsObj);
            }
            
            window.document.body.onfocus = null;
        },
        
        render: function () {
            var currentURLs = Backbone.history.fragment.split('/'),
                urlIndex = 0,
                urlInfo = { urls: [] },
                that = this;

            for (urlIndex; urlIndex < currentURLs.length; ++urlIndex) {
                var newURLInfo = {};
                newURLInfo.label = decodeURI(currentURLs[urlIndex]);
                if (urlIndex === 0) {
                    if (currentURLs[urlIndex] === '') {
                        currentURLs[urlIndex] = 'files';
                    }
                    currentURLs[urlIndex] = '#' + currentURLs[urlIndex];
                } else {
                    currentURLs[urlIndex] = currentURLs[urlIndex - 1] + '/' + currentURLs[urlIndex];
                }
                newURLInfo.href = currentURLs[urlIndex];
                urlInfo.urls.push(newURLInfo);
            }
            
            //First label will always be 'Creative Cloud Assets'
            if (urlInfo.urls.length > 0) {
                // Creative cloud assets is the feature name and will not be localized
                urlInfo.urls[0].label = "Creative Cloud Assets";
            }
            
            var tmpl = TemplateUtil.createTemplate("#asset-list-view-template"),
                header = TemplateUtil.createTemplate("#thumbnail-header-template", urlInfo),
                noFiles = TemplateUtil.createTemplate("#upload-files-here-template"),
                dragDrop = TemplateUtil.createTemplate("#drag-drop-file-capture-template");

            if (!this.$el[0]) {
                this.setElement("#results");
            }
            this.$el.find(".thumbnail-header").remove();
            this.$el.html(header + tmpl);
            this.renderUploadedAssetView();
                    
            this.$el.append(noFiles);
            this.$el.append(dragDrop);

            this.$el.find("#breadcrumbs").rcrumbs({
                nbFixedCrumbs: 1
            });
            
            this.$el.find('input[name="uploadPSD"]').off('change', this.inputFileChangeHandler).on('change', { self : this }, this.inputFileChangeHandler);
            
            window.dwPSDExtractManager.setPSDAvailable(false);
            
            // By default the "Files" anchor leads to the root(#), but we want to root our application default 
            // route which shows all the file thumbnails
            this.$el.find(".thumbnail-header .back-to-files")
                .click(function () {
                    if (navigator.onLine) {
                        var backHref = this.getAttribute("data-href");
                        /* Refresh the view in case BackHref is same as current BackBone URL. */
                        if (backHref === '#' + encodeURI(Backbone.history.fragment)) {
                            var newFragment = Backbone.history.fragment;
                            Backbone.history.fragment = null;
                            Backbone.history.navigate(newFragment, true);
                        } else {
                            // Navigate to clicked folder
                            //'backHref' is already URI encoded.
                            Backbone.history.navigate(decodeURI(backHref), true);
                        }
                    }
                    return false;
                });
            
            if (deps.utils.getCurrentPage() !== undefined) {
                this.$el.find(".thumbnail-header .back-to-files").css("visibility", "visible");
                this.$el.find(".thumbnail-header .header-separator").css("visibility", "visible");
            } else {
                this.$el.find(".thumbnail-header .back-to-files").css("visibility", "hidden");
                this.$el.find(".thumbnail-header .header-separator").css("visibility", "hidden");
            }

            if (this.loading === false) {
                $("#loadingSpinner").hide();
                this.showUploadGuidance();
            }

            if (this.scErrorText !== undefined) {
                graphite.events.trigger("creative-cloud-access-error", "creativeCloudAccessError");
            }

            if (this.collection.models.length === 1 && this.collection.models[0].id === undefined) {
                return this;
            }
                   
            _.each(this.collection.models, function (assetModel) {
                if (CommonUtil.typeMatchSupportedPSD(assetModel.attributes.type) || CommonUtil.typeMatchDirectory(assetModel.attributes.type)) {
                    that.renderAsset(assetModel);
                }
            }, this);

            var dropzone = $('#dropzone');

            if (typeof (Modernizr) !== "undefined" && Modernizr.draganddrop) {
                var $dropTarget = $('#drag_drop_file_capture'),
                    $uploadFiles = $("#upload_files_here");
                that.$el[0].ondragover = function (e) {
                    if (!$.contains(that.$el[0], $dropTarget) && $("#draganddropmessage").is(':visible')) {
                        $dropTarget.show();
                        $("#upload_files_here").hide();
                    }
                    return false;
                };

                $dropTarget[0].ondragleave = function (e) {
                    if ($.contains(that.$el[0], $dropTarget[0])) {
                        $dropTarget.hide();
                        that.showUploadGuidance();
                    }
                };
                $dropTarget[0].ondragend = function (e) {
                    if ($.contains(that.$el[0], $dropTarget[0])) {
                        $dropTarget.hide();
                        that.showUploadGuidance();
                    }
                };
                $dropTarget[0].ondrop = function (e) {
                    if ($.contains(that.$el[0], $dropTarget[0])) {
                        $dropTarget.hide();
                        that.showUploadGuidance();
                    }
                    e.preventDefault();

                    var files = e.dataTransfer.files,
                        filesToUpload = [],
                        i,
                        headersLoaded = 0;

                    _.each(files, function (file) {
                        var blob = file.slice(0, 4, 'UTF-8');
                        var reader = new FileReader();
                        var mimetype = file.type;
                        var currFile = file;
                        var ext = file.name.substr((~-file.name.lastIndexOf(".") >>> 0) + 2).toLowerCase();

                        reader.addEventListener("loadend", function () {
                            if (this.result === "8BPS") { // http://www.adobe.com/devnet-apps/photoshop/fileformatashtml/#50577409_pgfId-1055726
                                filesToUpload.push(currFile);
                            }

                            headersLoaded = headersLoaded + 1;
                            if (headersLoaded === files.length) {
                                if (filesToUpload.length > 0) {
                                    graphite.events.trigger("upload-init", { files: filesToUpload });
                                }
                                if (filesToUpload.length !== files.length) {
                                    graphite.events.trigger('load-alert', deps.translate("CAN_ONLY_UPLOAD_PSD_FILES_COUNT_FILES_NOT_UPLOADED", (files.length - filesToUpload.length), files.length));
                                    graphite.events.trigger('bad-file-type-dragdrop', {extension: ext, type: mimetype});
                                }
                            }
                        });
                        reader.readAsText(blob, 'UTF-8');
                    });

                    return false;
                };
            } else {
                dropzone.hide();
            }
            return this;
        },
        
        remove: function () {
            if (this.assetUploadModel) {
                this.assetUploadModel.destroy();
                this.assetUploadModel = null;
            }
            if (this.assetUploadView) {
                this.assetUploadView.remove();
                this.assetUploadView = null;
            }
            AssetListView.prototype.remove.apply(this, arguments);
        }
        
    });
    
    return DwAssetListView;
});
