/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4,
maxerr: 50, node: true */
/*global */
(function () {
    
    "use strict";
    
    
    var fs          = require('fs'),
        path        = require('path'),
        HTMLHint    = require("htmlhint").HTMLHint,
        JSHINT      = require("jshint").JSHINT,
        CSSLint     = require("csslint").CSSLint,
		errorString = "error",
		warningString = "warning",
        moreErrors = "moreerrors";
    
    var MAX_ERR_COUNT = 50;
		
    
    //load the rules from config file
    function loadrules(configFile, callback) {
        var configrules;
        if (!configFile) {
            callback(null, configrules);
        }
        
        fs.readFile(configFile, function (err, buffer) {
            if (err) {
                return callback(err);
            }
            var content = buffer.toString();
            
            try {
                configrules = JSON.parse(content);
            } catch (e) {
                err = "config_file_error";
                configrules = "";
            }
            callback(err, configrules);
        });
        
        
    }
    
    //load the configrules file first and then the file to be linted
    function linthtml(content, configFile, callback) {
        loadrules(configFile, function (err, configrules) {
            if (err) {
                return callback(err);
            }
            
            var results;
            try {
                results = HTMLHint.verify(content, configrules);
            } catch (e) {
                return callback(e);
            }
            callback(null, results);
        });
    }
    
    //load the configrules file first and then the file to be linted
    function lintjs(content, configFile, callback) {
        loadrules(configFile, function (err, configrules) {
            if (err) {
                return callback(err);
            }
            
            var result, retvalue;
            try {
                result = JSHINT(content, configrules);
            } catch (e) {
                return callback(e);
            }
            if (!result) {
                retvalue = JSHINT.errors;
            } else {
                retvalue = [];
            }
            callback(null, retvalue);
        });
    }
    
    //load the configrules file first and then the file to be linted
    function lintcss(content, configFile, callback) {
        loadrules(configFile, function (err, configrules) {
            if (err) {
                return callback(err);
            }
            
            var result;
            try {
                result = CSSLint.verify(content, configrules);
            } catch (e) {
                return callback(e);
            }
            callback(null, result);
        });
    }
    
    //format html lint output
    function formathtmllintoutput(lintFile, results, output) {
       
        if (results.length) {
            var index, len;
            for (index = 0, len = results.length; index < len; index++) {

                var messageOb = results[index];
                if (!messageOb.hasOwnProperty("line") &&
                        !messageOb.hasOwnProperty("type") &&
                        !messageOb.hasOwnProperty("col") &&
                        !messageOb.hasOwnProperty("message")) {
                    return;
                }
                if (messageOb.line) {
                    //info message type will be added as warnings
                    var errorType = warningString;
                    if (messageOb.type === errorString) {
                        errorType = errorString;
                    }

                    output.push({
                        filename : lintFile,
                        type : errorType,
                        message : messageOb.message,
                        line : messageOb.line,
                        col : messageOb.col
                    });
                }
                    
            }
        }
    }
    
   //format js lint output
    function formatjslintoutput(lintFile, results, output) {
       
        if (results.length) {
            var index, len;
            for (index = 0, len = results.length; index < len; index++) {

                var messageOb = results[index];
				if (messageOb) {
                    
                    if (!messageOb.hasOwnProperty("line") &&
                            !messageOb.hasOwnProperty("id") &&
                            !messageOb.hasOwnProperty("character") &&
                            !messageOb.hasOwnProperty("reason")) {
                        return;
                    }

                    if (messageOb.line) {

                        var type = messageOb.id;
                        if (type !== undefined) {
                            type = type.replace("(", "");
                            type = type.replace(")", "");
                        }

                        var errorType = warningString;
                        if (type === errorString) {
                            errorType = errorString;
                        }

                        output.push({
                            filename : lintFile,
                            type : errorType,
                            message : messageOb.reason,
                            line : messageOb.line,
                            col : messageOb.character
                        });
                    }
                }
                    
            }
        }
    }
    
    //format css lint output
    function formatcsslintoutput(lintFile, results, output) {
        if (results.messages.length) {
            var index, len;
            for (index = 0, len = results.messages.length; index < len; ++index) {

                var messageOb = results.messages[index];
                if (!messageOb.hasOwnProperty("line") &&
                        !messageOb.hasOwnProperty("type") &&
                        !messageOb.hasOwnProperty("message") &&
                        !messageOb.hasOwnProperty("col")) {
                    return;
                }
                if (messageOb.line) {
                    var errorType = warningString;
                    if (messageOb.type === errorString) {
                        errorType = errorString;
                    }

                    output.push({
                        filename : lintFile,
                        type : errorType,
                        message : messageOb.message,
                        line : messageOb.line,
                        col : messageOb.col
                    });
                }
                    
            }
        }
    }
	
	function sortByKey(array, key1, key2) {
    return array.sort(function(a, b) {
		var aKey1 = a[key1];
		var bKey1 = b[key1];
		var aKey2 = a[key2];
		var bKey2 = b[key2];
		
		if(aKey1 == bKey1)
		{
			return (aKey2 < bKey2) ? -1 : (aKey2 > bKey2) ? 1 : 0;
		}
		else
		{
			return (aKey1 < bKey1) ? -1 : 1;
		}
		});
	}
    
    function limitoutput(lintFile, output) {
        var updatedoutput = [];
        var errorsOnly = output.filter(function (elem) {
            return elem.type === errorString;
        });
		errorsOnly = sortByKey(errorsOnly, "line", "col");//sort by line no, then col no
        
		var warningsOnly = output.filter(function (elem) {
            return elem.type === warningString;
        });
		warningsOnly = sortByKey(warningsOnly, "line", "col");//sort by line no, then col no
        
        var shouldAppendMoreErrorsString = false;
        var sliceStartIndex = 0;
        var sliceEndIndex = MAX_ERR_COUNT;
        if (output.length <= MAX_ERR_COUNT) {
            sliceEndIndex = output.length;
        } else {
            shouldAppendMoreErrorsString = true;
        }
            
        updatedoutput = (errorsOnly.concat(warningsOnly)).slice(sliceStartIndex, sliceEndIndex);
        
        if (shouldAppendMoreErrorsString) {
            updatedoutput.push({
                filename : lintFile,
                type : moreErrors,
                message : moreErrors,
                line : -1,
                col : -1
            });
        }
        // in ideal case this should never happen but just being safe
        if (output.length > 0 && updatedoutput.length === 0) {
            updatedoutput = output;
        }
        
        return updatedoutput;
        
    }
    
    // lint the file
    function lint(lintFile, configFile, callback) {
        if (!lintFile || !configFile) {
            return callback(null, null);
        }
        
        lintFile = decodeURIComponent(lintFile);
        configFile = decodeURIComponent(configFile);
        var output = [];
        if(!fs.existsSync(lintFile)) {
            var errMsg = lintFile + "not found";
            output.push({
                filename : lintFile,
                type : errMsg,
                message : errMsg,
                line : 0,
                col : 0
            });
            return callback(null, output);
        }
        if(!fs.existsSync(configFile)) {
            var msg = configFile + "not found";
            output.push({
                filename : configFile,
                type : msg,
                message : msg,
                line : 0,
                col : 0
            });
            return callback(null, output);
        }
        fs.readFile(lintFile, function (err, buffer) {
             if (err) {
                return callback(null, output);
            }
	      
            var content = buffer.toString();
            if (content.length === 0) {
                return callback(null, output);
            }
            
            var fileextension = path.extname(lintFile);
            fileextension = fileextension.toLowerCase();
            
            
            if (fileextension === ".html" || fileextension === ".htm") {
                linthtml(content, configFile, function (err, results) {
                    if (err) {
                        return callback(err);
                    }
                    formathtmllintoutput(lintFile, results, output);
                    var updatedOutput = limitoutput(lintFile, output);
                    callback(null, updatedOutput);
                });
            } else if (fileextension === ".js") {
                lintjs(content, configFile, function (err, results) {
                    if (err) {
                        return callback(err);
                    }
                    formatjslintoutput(lintFile, results, output);
                    var updatedOutput = limitoutput(lintFile, output);
                    callback(null, updatedOutput);
                });
            } else if (fileextension === ".css") {
                lintcss(content, configFile, function (err, results) {
                    if (err) {
                        return callback(err);
                    }
                    formatcsslintoutput(lintFile, results, output);
                    var updatedOutput = limitoutput(lintFile, output);
                    callback(null, updatedOutput);
                });
            }
        });
                    
    }

    function init(DWServiceManager) {
        
        if (!DWServiceManager.hasService('Linters')) {
            DWServiceManager.registerService('Linters', { major: 1, minor: 0 });
        }
        
        DWServiceManager.registerCommand(
            'Linters', // service name
            'lint', // command name
            lint, // command handler function
            true, // this command is asynchronous
            'Lint a file',
            ['lintFilePath', 'configFilePath'], // path parameters
            null
        );
    }

    exports.init = init;

}());
