/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */


/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4,
maxerr: 50, node: true */
/*global */
(function () {
    "use strict";
 
    var sass            = require('node-sass'),
        rubySass        = require('./RubySass'),
        fs              = require('fs'),
        path            = require('path'),
        Utils           = require('./Utils');

    function compileNodeSass(parameters, callback) {
        var sassFile = "";
        if (parameters.sourceFile && parameters.sourceFile !== "" && typeof parameters.sourceFile === "string") {
            sassFile = parameters.sourceFile;
        }
        sassFile = decodeURIComponent(sassFile);
        var sassPath = path.dirname(sassFile),
            cssFile = "",
            options,
            outputStyle = "nested",
            createSourceMap = false,
            createSourceComments = false;

        if (parameters.options && typeof parameters.options !== "undefined") {
            options = parameters.options;
        }
        
        if (parameters.cssFile && parameters.cssFile !== "" && typeof parameters.cssFile === "string") {
            cssFile = parameters.cssFile;
        }
        cssFile = decodeURIComponent(cssFile);
        
        if (options) {
            if (options.outputStyle && options.outputStyle !== "" && typeof options.outputStyle === "string") {
                outputStyle = options.outputStyle;
            }
            
            if (options.sourceMap && 
                        ((typeof options.sourceMap === "string" && options.sourceMap === "true") ||
                        (typeof options.sourceMap === "boolean" && options.sourceMap === true) ||
                        typeof options.sourceMap === "object")) {
                options.sourceMap = {};
                createSourceMap = true;
            } else {
                createSourceMap = false;
            }
            
            if (options.sourceComments && options.sourceComments !== "" && typeof options.sourceComments === "string") {
                if (options.sourceComments === "true") {
                    createSourceComments = true;
                } else {
                    createSourceComments = false;
                }
            }
        }

        sass.render({
            file: sassFile,
            outFile: cssFile,
            outputStyle: outputStyle,
            sourceMap: createSourceMap,
            sourceComments: createSourceComments,
            success: function (result) {
                var css = result.css;
                // add version tag
                if (!options.compress && !options.cleanCss) {
                    css = '/* Generated by node-sass */\n' + css;
                }

                // write output
                Utils.mkfile(cssFile, css, function (err) {
                    if (err) {
                        return callback(err);
                    }

                    // write source map
                    if (result.map && options.sourceMapFilename) {
                        Utils.mkfile(options.sourceMapFilename, result.map, function (err) {
                            if (err) {
                                return callback(err);
                            }
                            callback(null, { filename: path.resolve(sassFile), compileFile: path.resolve(cssFile) });
                        });
                    } else {
                        callback(null, { filename: path.resolve(sassFile), compileFile: path.resolve(cssFile) });
                    }
                });
            },
            error: function (err) {
                var errFile = "";
                if (err.file) {
                    errFile = err.file;
                } else if (callback && callback.sourceFile) {
                    errFile = callback.sourceFile;
                }
                
                return callback({
                    type: "Parse",
                    filename: path.resolve(errFile),
                    line: err.line,
                    column: err.column - 1,
                    message: err.message
                });
            }
        });
    }

    function compile(parameters, callback) {
        var sassFile = "";
        if (parameters.sourceFile && parameters.sourceFile !== "" && typeof parameters.sourceFile === "string") {
            sassFile = parameters.sourceFile;
        }
        sassFile = decodeURIComponent(sassFile);
        var sassPath = path.dirname(sassFile),
            cssFile = "",
            options,
            siteDetails,
            changedSource = (parameters.optionalDetails && parameters.optionalDetails.changedSource) ? path.normalize(parameters.optionalDetails.changedSource) : path.normalize(sassFile),
            outputStyle = "nested",
            createSourceMap = false,
            createSourceComments = false,
            includeLibraryPathList = parameters.includeLibraryPathList || [],
            compassOptions = parameters.siteDetails.compassoptions;

        if (parameters.cssFile && parameters.cssFile !== "" && typeof parameters.cssFile === "string") {
            cssFile = parameters.cssFile;
        }
        cssFile = decodeURIComponent(cssFile);
        
        
        if (parameters.options && typeof parameters.options !== "undefined") {
            options = parameters.options;
        }
        if (parameters.siteDetails && typeof parameters.siteDetails !== "undefined") {
            siteDetails = parameters.siteDetails;
        }

        if (options) {
            if (options.outputStyle && options.outputStyle !== "" && typeof options.outputStyle === "string") {
                outputStyle = options.outputStyle;
            }
            
            if (options.sourceMap && 
                        ((typeof options.sourceMap === "string" && options.sourceMap === "true") ||
                        (typeof options.sourceMap === "boolean" && options.sourceMap === true) ||
                        typeof options.sourceMap === "object")) {
                options.sourceMap = {};
                createSourceMap = true;
            } else {
                createSourceMap = false;
            }
            
            if (options.sourceComments && options.sourceComments !== "" && typeof options.sourceComments === "string") {
                if (options.sourceComments === "true") {
                    createSourceComments = true;
                } else {
                    createSourceComments = false;
                }
            }
        }
        
        options = {
            file: sassFile,
            outFile: cssFile,
            outputStyle: outputStyle,
            sourceMap: createSourceMap,
            sourceComments: createSourceComments,
            rubyPath : options.rubyPath,
            rubyGemPath : options.rubyGemPath,
            sassCachePath : siteDetails.siteCache,
            includeLibraryPathList: includeLibraryPathList,
            compassOptions : compassOptions
        };
        
        var success = function (cssFile) {
            callback(
                null,
                {
                    changedSource: changedSource,
                    compiledSource : path.resolve(sassFile),
                    filename: path.resolve(sassFile),
                    compiledFile: path.resolve(cssFile),
                    cssFileBeforeCompile: parameters && parameters.backedupCssFile ? path.resolve(parameters.backedupCssFile) : ""
                }
            );
        };
        
        var error = function (sourceFile, errors) {
                return callback({
                    changedSource: changedSource,
                    compiledSource : path.resolve(sourceFile),
                    cssFileBeforeCompile: parameters && parameters.backedupCssFile ? path.resolve(parameters.backedupCssFile) : "",
                    errors : errors
                });
            };
        
        rubySass.compile(options, success, error);
    }
    
    /**
     * Will Invoke the ruby compiler to perform compass Init
     * @param {object}   options   rubyPath: path where ruby recides
                                    rubyGemPath: path of ruby gems
                                    siteroot: site root where compass init is to be initiated
     * @param {function} successcb(outputType,status,message)
     * @param {function} errorrcb(outputType,status,message)
     */
    function compassInit(options, successCallback, failureCallback) {
        
        var success = function (msg) {
            successCallback(msg);
        };
        
        var error = function (msg) {
                failureCallback(msg);
            };
        
        rubySass.compassInit(options, success, error);
        
    }
    

    exports.compile          = compile;
    exports.compileNodeSass  = compileNodeSass;
    exports.compassInit      = compassInit;
    
}());
