/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */


/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4,
maxerr: 50, node: true */
/*global */
(function () {
    "use strict";
    
    var exec                = require('child_process').exec,
        fs                  = require('fs'),
        path                = require('path'),
        Utils               = require('./Utils'),
        CompilerManager     = require('./CompilerManager');
    
    /**
     * Creates the folder if required synchronously
     * @param {string} path The path to create
     */
    var _mkdirSync = function (path) {
        try {
            fs.mkdirSync(path);
        } catch (e) {
            if (e.code !== 'EEXIST') {
                console.log(e);
            }
        }
    };
    
    /**
     * Return true if the given line is the starting of an error description of the form 
     * "Error: File to import not found or unreadable: compass/reset."
     * @param   {string}  line 
     * @returns {boolean} true if it is an error line
     */
    var _isErrorStartingLine = function (line) {
        if (line.trim().toLowerCase().indexOf("error:") === 0) {
            return true;
        }
        return false;
    };
    
    /**
     * Return true if the given line is the starting of a warning description of the form 
     * "WARNING on line 6 of C:\Users\name\Documents\Unnamed Site 2\compass-example-master\sass\sassFile2.sass:"
     * "WARNING: 'rotator-black.png' was not found (or cannot be read) in ./images/Assets"
     * "DEPRECATION WARNING on line 4, column 8 of /path/to/_sass/_blog.scss:"
     * @param   {string}  line 
     * @returns {boolean} true if it is a warning line
     */
    var _isWarningStartingLine = function (line) {
        if (line.trim().toLowerCase().indexOf("warning on line") === 0 ||
                line.trim().toLowerCase().indexOf("deprecation warning on line") === 0 ||
                line.trim().toLowerCase().indexOf("warning:") === 0) {
            return true;
        }
        return false;
    };
    
    /**
     * Finds the error line number line of the form for the current error line index
     * "on line 6 of C:\Users\someone\Documents\site\compass-example-master\sass\screen.scss"
     * the scan will be done till the above line is found, or another error/warning header is found- in which case
     * we return -1 as further position is not for the error index under consideration.
     * @param   {Array}  errArray          string with all errors and warnings
     * @param   {number} startIndexInArray The starting location from which the scan needs to be performed
     * @returns {string} the line we are looking for if found, or empty string
     */
    var _findErrorLocationLine = function (errArray, startIndexInArray) {
        var i = startIndexInArray, errorLine = "";
        for (i = startIndexInArray; i < errArray.length; i++) {
            errorLine = errArray[i].trim();
            if (errorLine.toLowerCase().indexOf("on line") === 0) {
                return errorLine;
            } else if (_isErrorStartingLine(errorLine) || _isWarningStartingLine(errorLine)) {
                return "";
            }
        }
        return "";
    };

    /**
     * Checks if a string repesents a number without decimal places
     * @param   {string}  n 
     * @returns {boolean} true if Integer
     */
    function _isInteger(n) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    }

    /**
     * Processes a single error returned by ruby
     * @param   {Array}        errArray    The array of error+warn lines returned by ruby
     * @param   {number}       startIndexInArray  the index in the array from which the error string needs to be decoded
     * @param   {string}       optionalCompiledFileName the file that was compiled that resulted in the error
     * @returns {type: "error",line: 0,filename: "",message: "some error"} The decoded error message
     */
    var _processError = function (errArray, startIndexInArray, optionalCompiledFileName) {
        var errObj = Utils.createErrObj("", "error", 0, 0, "An error occurred.");

        //line 1 of the form Error: File to import not found or unreadable: compass/reset.
        errObj.message = errArray.length > startIndexInArray ? errArray[startIndexInArray] : errObj.message;
        // line 2+ of the form on line 6 of C:\Users\someone\Documents\site\compass-example-master\sass\screen.scss
        var errorLine = _findErrorLocationLine(errArray, startIndexInArray + 1);
        if (errorLine) {
            var errDetailsArray = errorLine.split(" "),
                fileNameIndex = errorLine.toLowerCase().indexOf("of ");
            if (fileNameIndex >= 0) {
                fileNameIndex += 3;
                errObj.filename = path.normalize(errorLine.slice(fileNameIndex));
            }
            errObj.line = errDetailsArray.length > 3 ? parseInt(errDetailsArray[2], 10) : 0;
            errObj.line = _isInteger(errObj.line) ? errObj.line : 0;

        }
        
        if (!errObj.filename) {
            errObj.filename = optionalCompiledFileName;
        }
        
        return errObj;
    };
    /**
     * Processes a single warning returned by ruby
     * @param   {Array}        errArray    The array of error+warn lines returned by ruby
     * @param   {number}       startIndexInArray  the index in the array from which the error string needs to be decoded
     * @param   {string}       optionalCompiledFileName the file that was compiled that resulted in the error
     * @returns {type: "warning",line: 0,filename: "",message: "some warning"} The decoded warning message
     */
    var _processWarning = function (errArray, startIndexInArray, optionalCompiledFileName) {
        var warnObj = Utils.createErrObj("", "warning", 0, 0, "A warning occurred."),
            deprecationWarning = false,
            lineNumberOffset = 3;

        if (errArray[startIndexInArray].toLowerCase().indexOf("warning:") === 0) {
            // single line warning of the form WARNING: 'rotator-black.png' was not found (or cannot be read) in ./images/Assets
            warnObj.message = errArray[startIndexInArray];
            warnObj.filename = optionalCompiledFileName;
            return warnObj;
        }
        
        //line 2 is the warning message of the form "This selector doesn't have any properties and will not be rendered."
        warnObj.message = errArray.length > (startIndexInArray + 1) ? errArray[startIndexInArray + 1] : warnObj.message;
        // line 1 of the form "WARNING on line 6 of C:\Users\name\Documents\sass\sassFile2.sass:"  or
        //                    "DEPRECATION WARNING on line 4, column 8 of /path/to/_sass/_blog.scss:"
        var warningLine = errArray.length > startIndexInArray ? errArray[startIndexInArray].trim() : "";
        if (warningLine.toLowerCase().indexOf("deprecation warning on line") === 0) {
            deprecationWarning = true;
            lineNumberOffset = 4;
            warnObj.message = "DEPRECATED: " + warnObj.message;
        }
        if (warningLine.toLowerCase().indexOf("warning on line") === 0 || deprecationWarning) {
            var warningDetailsArray = warningLine.split(" "),
                fileNameIndex = warningLine.toLowerCase().indexOf("of ");
            if (fileNameIndex >= 0) {
                fileNameIndex += 3;
                warnObj.filename = warningLine.slice(fileNameIndex);
                warnObj.filename = path.normalize(warnObj.filename.trim().substr(0, warnObj.filename.length - 1)); // the last char is a : char in warning
            }
            warnObj.line = warningDetailsArray.length > lineNumberOffset ? parseInt(warningDetailsArray[lineNumberOffset], 10) : 0;
            warnObj.line = _isInteger(warnObj.line) ? warnObj.line : 0;
        }
        
        if (!warnObj.filename) {
            warnObj.filename = optionalCompiledFileName;
        }
        
        return warnObj;
    };

    /**
     * Processes the errors and warnings returned by ruby
     * @param {string} errStr The error string of the form 
     *                        WARNING on line 6 of C:\Users\name\Documents\Unnamed Site 2\compass-example-master\sass\sassFile2.sass:
     *                        This selector doesn't have any properties and will not be rendered.
     *                        WARNING: 'rotator-black.png' was not found (or cannot be read) in ./images/Assets
     *                        Error: File to import not found or unreadable: compass/reset.
     *                           on line 6 of C:\Users\someone\Documents\site\compass-example-master\sass\screen.scss
     *                           Use --trace for backtrace.
     * @param {string} optionalCompiledFileName the file that was compiled that resulted in the error
     * @returns { array of errors and warnings }
     */
    var _processErrorWarnings = function (errStr, optionalCompiledFileName) {
        var arr = errStr.split("\n"),
            i = 0,
            errorsAndWarnings = [];

        for (i = 0; i < arr.length; i++) {
            if (_isErrorStartingLine(arr[i])) {
                errorsAndWarnings.push(_processError(arr, i, optionalCompiledFileName));
            } else if (_isWarningStartingLine(arr[i])) {
                errorsAndWarnings.push(_processWarning(arr, i, optionalCompiledFileName));
            }
        }

        return errorsAndWarnings;
    };
    
    /**
     * Process the compass output and converts it to the standard error format for creation of Error Object.
     * @private
     * @param   {object}   options      file: sassFile,
                                        outFile: cssFile,
                                        outputStyle: outputStyle,
                                        sourceMap: createSourceMap,
                                        sourceComments: createSourceComments 
     * @param   {string}   resultLine   The first line of the output generated by compass. 
     *                                  Contains all relevant information
     * @returns {string}
     */
    var _processCompassError = function (options, resultLine) {
        var processedError = resultLine;
        var isError = (resultLine.split(" ")[0] === "error") ? true : false;
        var filePath = "";
        var lineInfo = "";
        
        if (isError) {
            var ptStart = resultLine.indexOf("(");
            var ptEnd = resultLine.lastIndexOf(")");
            var rawErrorString = resultLine.substr(ptStart + 1, ptEnd - ptStart - 1);
            var possibleLineAndFileInfo = rawErrorString.substr(0, rawErrorString.indexOf(":"));
            var properErrorString = rawErrorString.substr(rawErrorString.indexOf(":") + 1).trim();
            
            if (possibleLineAndFileInfo.indexOf("of") !== -1) {
                lineInfo = possibleLineAndFileInfo.split("of")[0].trim().toLowerCase();
                filePath = path.normalize(options.compassOptions.siteroot + "/" + possibleLineAndFileInfo.split("of")[1].trim());
            } else {
                lineInfo = possibleLineAndFileInfo.toLowerCase();
                filePath = path.normalize(options.compassOptions.siteroot + "/" + resultLine.split(" ")[1]);
            }
            
            processedError = "Error: " + properErrorString + "\n" + "        on " + lineInfo + " of " + filePath + "\n";
        }
        return processedError;
    };
    
    
    /**
     * Create output filepath in case of compass compilation. 
     * @private
     * @param   {object}    options     file: sassFile,
                                        outFile: cssFile, 
                                        outputStyle: outputStyle,
                                        sourceMap: createSourceMap,
                                        sourceComments: createSourceComments 
     * @param   {string}    currentPath The current css output file path, created manually independent of config.rb
     *                                  Will work in out of site case where we are not using compass.
     * @param   {string}    resultLine  The output generated by compass on successful compilation,
     *                                  Contains path of generated css relative to config.rb
     * @returns {string} 
     */
    var _processOutputFilePath = function (options, currentPath, resultLine) {
        var processedPath = currentPath;
        if (resultLine !== "") {
            var isWrite = (resultLine.split(" ")[0] === "write") ? true : false;
            if (isWrite) {
                var filePath = resultLine.split("\n")[0].split("write")[1].trim();
                processedPath = path.normalize(options.compassOptions.siteroot + "/" + filePath);
            }
        }
        return processedPath;
    };
    
    /**
     * Creates the directory for outputPath if not present. If outputPath is a file
     * creates the parent directory
     * @param {string} outputPath The output file path or directory
     */
    var _createOutputFolder = function (outputFilePath) {
        _mkdirSync(path.dirname(outputFilePath));
    };
    
    /**
    * Given a list of libraryPaths this generates a commandline option 
    * which can be used to inform compiler about include locations
    * to search for
    * @param {array} include library paths
    */
    var _createIncludeLibDirCmdOption = function (fileListArray) {
        //Assuming fileListArray has length > 0 
        var includeOption = ' -I ';
        var cmdOption = '';
        var index;
        for (index = 0; index < fileListArray.length; ++index) {
            var normalizedPath = path.normalize(fileListArray[index]);
            cmdOption += includeOption + '"' + Utils.stripTrailingSlash(normalizedPath) + '" ';
        }
        return cmdOption;
    };
    
    /**
     * Sets up the sass command to compile the file
     * @param {object}   options   file: sassFile,
                                   outFile: cssFile,
                                   outputStyle: outputStyle,
                                   sourceMap: createSourceMap,
                                   sourceComments: createSourceComments
    * @returns compileCommand string
    */
    var _setUpSassCommand = function (options) {
        var sassCommand;
        
        if (options) {
            // There is a bug in ruby sass where if sass cache is enabled, then warnings are not printed on compile, if the file has not been
            // modified. This will be particularly problematic when an imported file that has warnings didn't change. disabling ruby sass cache
            // As our workflow is more to do with development.
            options.sassCachePath = null;
        
            var rubyBin = options.rubyPath || "ruby",
                sassBin = options.rubyGemPath ? (' "' + options.rubyGemPath + 'sass" ') : " sass ",
                styleOption = options.outputStyle ? " --style " + options.outputStyle + " " : "",
                sassCacheOption = options.sassCachePath ? ' --cache-location "' + path.join(options.sassCachePath, 'sassCache') + '" ' : " --no-cache ",
                sourceMapAutoOption = ' --sourcemap=auto ',
                sourceMapNoneOption = ' --sourcemap=none ',
                lineCommentsOption = options.sourceComments ? ' --line-comments ' : '',
                includeLibDir = '';

            if (options.includeLibraryPathList && options.includeLibraryPathList.length > 0) {
                //We have an array of lib directories to be included
                includeLibDir = _createIncludeLibDirCmdOption(options.includeLibraryPathList);
            }
            
            _createOutputFolder(options.outFile);
            sassCommand = "\"" + rubyBin + "\"" + sassBin + includeLibDir + sassCacheOption + styleOption + lineCommentsOption;
            if (options.sourceMap &&
                    ((typeof options.sourceMap === "string" && options.sourceMap === "true") ||
                    (typeof options.sourceMap === "boolean" && options.sourceMap === true) ||
                    typeof options.sourceMap === "object")) {
                sassCommand += sourceMapAutoOption;
            } else {
                sassCommand += sourceMapNoneOption;
            }
            sassCommand += ' "' + options.file + '" "' + options.outFile + '"';
        }
        
        return sassCommand;
    };
    
    /**
     * Sets up the compass command to compile the file
     * @param {object}   options   file: sassFile,
                                   outFile: cssFile,
                                   outputStyle: outputStyle,
                                   sourceMap: createSourceMap,
                                   sourceComments: createSourceComments
    * @returns compileCommand string
    */
    var _setUpCompassCommand = function (options) {
        var compassCommand;
        
        if (options) {
            var rubyBin = options.rubyPath || "ruby",
                compassBin = options.rubyGemPath ? (' "' + options.rubyGemPath + 'compass" ') : " compass ",
                compileOption = ' compile ',
                compassOptions = options.compassOptions;

            var useCompass = false,
                configfileOption,
                httpOptions,
                imagesOptions,
                fontsOptions,
                generatedImagesOptions,
                outputDirOption,
                inputDirOption,
                relativeAssetsOption,
                lineCommentsOption = options.sourceComments ? '' : ' --no-line-comments ',
                changeCurrentDirCmd = ' ',
                cmdSeparator = ' ',
                platform = process.platform;


            if (compassOptions) {
                outputDirOption = ' --css-dir ' + ' "' + compassOptions.outputfolder + '" ';
                inputDirOption = ' --sass-dir ' + ' "' + compassOptions.inputfolder + '" ';

                if (compassOptions.configfilepath) {
                    useCompass = true;
                    configfileOption = ' --config ' + ' "' + compassOptions.configfilepath + '" ';
                } else if (compassOptions.manualconfigpathlist) {
                    useCompass = true;
                    httpOptions = ' --http-path "' + compassOptions.manualconfigpathlist.htmlpath + '" ';
                    imagesOptions = ' --images-dir "' + compassOptions.manualconfigpathlist.imagespath + '" ';
                    fontsOptions = ' --fonts-dir "' + compassOptions.manualconfigpathlist.fontspath + '" ';
                    generatedImagesOptions = ' --generated-images-path "' + compassOptions.manualconfigpathlist.generatedimagespath + '" ';
                    if (compassOptions.relativeassets === "true") {
                        relativeAssetsOption = ' --relative-assets ';
                    }
                }
            }
            
            if (useCompass) {
            
                var sourceMapOption;
                if (options.sourceMap &&
                        ((typeof options.sourceMap === "string" && options.sourceMap === "true") ||
                        (typeof options.sourceMap === "boolean" && options.sourceMap === true) ||
                        typeof options.sourceMap === "object")) {
                    sourceMapOption = " --sourcemap ";
                } else {
                    sourceMapOption = " --no-sourcemap ";
                }
            
                compassCommand = "\"" + rubyBin + "\"" + compassBin + compileOption + sourceMapOption + ' "' + options.file + '" ';
                if (configfileOption) {
                    compassCommand += configfileOption;
                } else {
                    compassCommand += imagesOptions + fontsOptions + generatedImagesOptions + outputDirOption + inputDirOption + httpOptions;
                    if (relativeAssetsOption) {
                        compassCommand += relativeAssetsOption;
                    }
                }
                
                compassCommand += lineCommentsOption;
        
                if (platform === 'win32') {
                    changeCurrentDirCmd = 'cd /d ';
                    cmdSeparator = ' & ';
                } else {
                    changeCurrentDirCmd = 'cd ';
                    cmdSeparator = ' ; ';
                }
                
                compassCommand = changeCurrentDirCmd + ' "' + compassOptions.siteroot + '" ' + cmdSeparator + compassCommand;
                compassCommand += " --force --boring";//To force file overwrite and to escape from useless color codes in stdout and stderr in Mac 
            }
        }
        
        return compassCommand;
    };
    
    /**
     * Determine which output stream contains the error output generated by compass
     * @private
     * @param   {string}   outputStream String reprensenting the stdout stream 
     * @param   {string}   errorStream  String reprensenting the stdout stream
     * @returns {string} 
     */
    var _getResultString = function (outputStream, errorStream) {
        return (outputStream !== "") ? outputStream : errorStream;
    };
    
    /**
     * Invokes the ruby compiler to compile the file
     * @param {object}   options   file: sassFile,
                                   outFile: cssFile,
                                   outputStyle: outputStyle,
                                   sourceMap: createSourceMap,
                                   sourceComments: createSourceComments
     * @param {function} successcb(filePath,cssGenerated)
     * @param {function} errorrcb(filePath,line,column,errorString)
     */
    var compile = function (options, successcb, errorrcb) {
        
        if (!successcb || !errorrcb) {
            return;
        }
        
        
        if (!options.file || !options.outFile) {
            errorrcb(options.file, [Utils.createErrObj("", "error", 0, 0, "Invalid file name.")]);
            return;
        }
        
        var compileCommand = _setUpCompassCommand(options) || _setUpSassCommand(options);
        if (compileCommand) {
            var child  = exec(compileCommand, function (error, stdout, stderr) {
                var errObj;
                var compilationResult = _getResultString(stdout, stderr);
                //Null check so that nothing breaks if stdout and stderr are undefined in any case. 
                if (compilationResult === undefined) {
                    compilationResult = "";
                }
                if (error) {
                    errObj = _processErrorWarnings(compilationResult, options.file);
                    if (errObj.length === 0) {
                        //Try compass specific parsing
                        var mainResultLine = compilationResult.split(/\r?\n/)[0].trim(); //We're interested in the first line only
                        var processedOutput = _processCompassError(options, mainResultLine);
                        
                        errObj = _processErrorWarnings(processedOutput, options.file);
                        if (errObj.length === 0) {
                            // there was some error, but we couldnt parse it properly, show the error output from the ruby compiler asis
                            errObj = [Utils.createErrObj(options.file, "error", 0, 0, mainResultLine)];
                        }
                    }
                    errorrcb(options.file, errObj);
                } else {
                    successcb(_processOutputFilePath(options, options.outFile, compilationResult.trim()));
                }
            });
        }
    };
    
    /**
     * Invokes the ruby compiler to Initialize a site with compass framework( Compass --init)
     * @param {object}   options   rubyPath: path where ruby recides
                                    rubyGemPath: path of ruby gems
                                    siteroot: site root where compass init is to be initiated
     * @param {function} successcb(outputType,status,message)
     * @param {function} errorrcb(outputType,status,message)
     */
    var compassInit = function (options, successcb, errorcb) {
        
        var rubyBin = options.rubyPath || "ruby",
            compassBin = options.rubyGemPath ? (' "' + options.rubyGemPath + 'compass" ') : " compass ",
            compassInitOption = " init --force ", /*This will overwrite the existing files if they are not identical*/
            cmdSeparator = ' ',
            changeDirCmd = ' ',
            platform = process.platform,
            compassInitCommand = "\"" + rubyBin + "\"" + compassBin + compassInitOption;
        
        if (platform === 'win32') {
            changeDirCmd = 'cd /d ';
            cmdSeparator = ' & ';
        } else {
            changeDirCmd = 'cd ';
            cmdSeparator = ' ; ';
        }
        
        compassInitCommand = changeDirCmd + ' "' + options.siteroot + '" ' + cmdSeparator + compassInitCommand;
        
        if (compassInitCommand) {
            CompilerManager.setCompassInitInProgress(); //Notify Compiler Manager saying that compass init has started
            var child  = exec(compassInitCommand),
                output = "",
                error  = "";

            child.stdout.on('data', function (data) {
                output += data;
            });

            child.stderr.on('data', function (data) {
                error += data;
            });

            child.on('close', function (code) {
                if (error) {
                    errorcb(Utils.createCompassInitReturnObject(false, error));
                } else if (code === 0) {
                    successcb(Utils.createCompassInitReturnObject(true, "Install File success"));
                } else {
                    errorcb(Utils.createCompassInitReturnObject("failure", error));
                }
                child.stdin.end();
            });
        }
    };

    exports.compile     = compile;
    exports.compassInit = compassInit;
}());