/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */


/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4,
maxerr: 50, node: true */
/*global */
(function () {
    "use strict";
    
    var DWFileWatcher      = require('../../DWFileWatch/DWFileWatcher'),
        path               = require('path'),
        fs                 = require('fs'),
        crypto             = require('crypto'),
        Utils              = require('./Utils'),
        LessCompiler       = require('./LessCompiler'),
        SassCompiler       = require('./SassCompiler'),
        DependencyTracker  = require('./DependencyTracker'),
        DWEvents           = require('../../DWEvents');
        
    
    var EXTENSIONS_TO_WATCH = ["less", "sass", "scss"];
    
    var lessStyleOptions = {
            outputDir: "",
            sourceMap: "false",
            strictMath: "false",
            strictUnits: "false",
            relativeUrls: "false"
        },
        sassStyleOptions = {
            outputDir: "",
            sourceMap: "false",
            outputStyle: "nested",
            rubyPath: "ruby",
            rubyGemPath: "",
            sourceComments: "false"
        },
        defaultLessOptions = lessStyleOptions,
        defaultSassOptions = sassStyleOptions,
        currentSiteDetails = {
            root: "",
            output: "",
            watch: "true",
            siteCache: ""
        },
        includeLibraryPathList = [],
        watcherReady = false,
        compassConfigFileName = 'config.rb',
        compassInitInProgress = false,
        compileRequestQueue = [],
        compilationEnabled = false,
        userConfigurationTempPath = "";
    
    /**
     * Sets the default options for the LESS compiler to use.
     * @param {object} lessOptions {
                "outputDir": "",
                "sourceMap": "false",
                "strictMath": "false",
                "strictUnits": "false",
                "relativeUrls": "false"
            }
     */
    function setDefaultLessOptions(lessOptions) {
        defaultLessOptions = lessOptions;
    }
    
    /**
     * Sets the default options for the SASS/SCSS compiler to use.
     * @param {object} SassOptions {
                "outputDir": "",
                "sourceMap": "false",
                "outputStyle": "nested",
                "rubyPath": "path_to_bin",
                "rubyGemPath": "path_gem",
                "sourceComments": "false"
            }
     */
    function setDefaultSASSOptions(SassOptions) {
        defaultSassOptions = SassOptions;
    }
    
    /**
     * Checks if the given file is part of the site root
     * @param {string} filePath 
     */
    function isFileInSite(filePath) {
        var normPath = path.normalize(filePath);
        if (currentSiteDetails && currentSiteDetails.root) {
            if (normPath.indexOf(currentSiteDetails.root) === 0) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Constructs output file path (CSS/source map) by looking up the
     * site details.
     * @private
     * @param   {string}   filePath        Preprocessor file (LESS/Sass)
     * @param   {boolean}  fileOutsideSite Boolean flag to check whether or not it is out of site scenario.
     * @returns {string}   Path of the CSS/map file deduced.
     */
    function _getOutputFilePath(filePath, fileOutsideSite) {
        var fileInSite  = isFileInSite(filePath),
            currSite    = currentSiteDetails,
            userOption  = currSite.outputUserOption,
            outputDir   = null;
        
        if (fileOutsideSite === true) {
            outputDir = path.dirname(filePath);
        } else if (!isFileInSite || userOption === "sameFolder") {
            outputDir = path.dirname(filePath);
        } else if (userOption === "specificFolder") {
            outputDir = currSite.output;
            outputDir = decodeURIComponent(outputDir);
        } else if (userOption === "segmentPath") {
            var fromPath            = currSite.segmentFrom,
                toPath              = currSite.segmentTo,
                sassPath            = path.dirname(filePath),
                relativePathSrc     = sassPath.substr(currSite.root.length, sassPath.length),
                relativePathDest    = relativePathSrc.replace(fromPath, toPath);
            
            outputDir = path.join(currSite.root, relativePathDest);
        }
        
        return outputDir;
    }
    
    function _trimFileNameWithoutExtension(filePath) {
        var filename = path.basename(filePath);
        filename = filename.substr(0, filename.length - path.extname(filename).length);
        return filename;
    }

    function _constructOutputFileName(filePath, extension, fileOutsideSite) {
        var resolvedFilePath    = path.resolve(filePath),
            outputDir           = _getOutputFilePath(resolvedFilePath, fileOutsideSite),
            fileName            = _trimFileNameWithoutExtension(resolvedFilePath) + extension;

        fileName = path.join(outputDir, fileName);
        return fileName;
    }
    
    /**
     * Deduce the filepath for the css file to be generated based on the filepath.
     * 
     * @param   {string}  filePath        The filepath of the preprocessor file.
     * @param   {boolean} fileOutsideSite Boolean flag to check whether or not it is out of site scenario.
     * @returns {string}                  The complete filepath of the css file that will be generated.
     */
    function getCSSFileName(filePath, fileOutsideSite) {
        return _constructOutputFileName(filePath, '.css', fileOutsideSite);
    }
    
    /**
     * Deduce the filepath for the source map to be generated based on the filepath.
     * 
     * @param   {string}  filePath        The filepath of the preprocessor file.
     * @param   {boolean} fileOutsideSite Boolean flag to check whether or not it is out of site scenario.
     * @returns {string}                  The complete filepath of the source map file that will be generated.
     */
    function getSourceMapFileName(filePath, fileOutsideSite) {
        return _constructOutputFileName(filePath, '.map', fileOutsideSite);
    }
    
    /**
     * Create dummy site details for compilation of out of site files
     * @param {string} filePath      The filepath of the preprocessor file.
     */
    function _createPseudoSiteDetails(filepath) {
        var dummySiteDetails    = {},
            dummyCompassOptions = {};

        var fileDirectory = path.dirname(filepath);

        dummySiteDetails.includeLibraryPathList = currentSiteDetails.includeLibraryPathList;
        dummySiteDetails.maintainHierarchy = "false";
        dummySiteDetails.output = "";
        dummySiteDetails.outputUserOption = "";
        dummySiteDetails.root = fileDirectory + "\\";
        dummySiteDetails.input = encodeURIComponent(fileDirectory);
        dummySiteDetails.siteCache = undefined;
        dummySiteDetails.watch = "false";

        dummyCompassOptions.inputfolder = fileDirectory;
        dummyCompassOptions.outputfolder = "";
        dummyCompassOptions.siteroot = fileDirectory;

        var createdConfigfile = dummySiteDetails.root + compassConfigFileName;
        if (createdConfigfile && fs.existsSync(createdConfigfile)) {
            dummyCompassOptions.configfilepath = createdConfigfile;
        }

        dummySiteDetails.compassoptions = dummyCompassOptions;

        return dummySiteDetails;
    }
    
    /**
     * Sets the Compilation state which signifies whether it is
     * enabled / disabled
     * @param {bool} true or false
     */
    function setCompilationState(compilationState) {
        if (compilationState !== undefined) {
            compilationEnabled = compilationState;
        }
    }
    
    function setUserConfigTempPath(userConfigPath) {
        userConfigurationTempPath = decodeURIComponent(userConfigPath);
        userConfigurationTempPath = path.resolve(userConfigurationTempPath);
    }
    
    /**
     * Sets the current site details and setups the wacther(if enabled) to watch the new site
     * @param {object} siteDetails {
                "root": "site_root",
                "output": "siteOutput_folder"
                "watch" : "true"
            }
     */
    function switchSite(siteDetails) {
        if (siteDetails.root) {
            siteDetails.root = path.normalize(siteDetails.root);
        }
                
        if (siteDetails.includeLibraryPathList) {
            includeLibraryPathList = siteDetails.includeLibraryPathList;
        }
        
        if (siteDetails.root && currentSiteDetails.root && siteDetails.root === currentSiteDetails.root) {
            currentSiteDetails = siteDetails;
            if (currentSiteDetails.watch === "false") {
                DWFileWatcher.unwatchFileOrFolder(siteDetails.root);
                DependencyTracker.resetTrackedFiles();
                DependencyTracker.buildDependencyTree(siteDetails.root);
            } else {
                DependencyTracker.resetTrackedFiles();
                DWFileWatcher.watchFileOrFolder(siteDetails.root, EXTENSIONS_TO_WATCH);
                DependencyTracker.buildDependencyTree(siteDetails.root);
            }
            return;
        }
        
        if (siteDetails.root) {
            DWFileWatcher.unwatchFileOrFolder(currentSiteDetails.root);
            if (siteDetails.watch === "true") {
                DependencyTracker.resetTrackedFiles();
                DWFileWatcher.watchFileOrFolder(siteDetails.root, EXTENSIONS_TO_WATCH);
                DependencyTracker.buildDependencyTree(siteDetails.root);
            } else {
                DependencyTracker.resetTrackedFiles();
                DependencyTracker.buildDependencyTree(siteDetails.root);
            }
        }
        
        currentSiteDetails = siteDetails;
    }
    
    // compile the given less/sass file
    function compile(parameters, callback) {
        var sourceFile  = "",
            cssFile     = "";
        
        if (parameters.sourceFile && parameters.sourceFile !== "" && parameters.sourceFile !== "undefined") {
            sourceFile = parameters.sourceFile;
        }
        sourceFile = decodeURIComponent(sourceFile);

		var compileNow = function () {
			if (parameters.options &&
					parameters.options.sourceMap &&
					parameters.options.sourceMap !== "" &&
					parameters.options.sourceMap !== "undefined" &&
					parameters.options.sourceMap === "true") {
				parameters.options.sourceMapFilename = getSourceMapFileName(sourceFile,
                                                                            (parameters.optionalDetails ?
                                                                            parameters.optionalDetails.fileOutsideSite :
                                                                            false));
			}

			var extName = path.extname(sourceFile);
			if (callback) {
				callback.sourceFile = sourceFile;
			}

			if (extName) {
				extName = extName.toLowerCase();
				if (parameters.optionalDetails && parameters.optionalDetails.fileOutsideSite) {
                    parameters.siteDetails = _createPseudoSiteDetails(sourceFile);
				} else {
					parameters.siteDetails = currentSiteDetails;
				}
				parameters.includeLibraryPathList = includeLibraryPathList;
				if (extName === ".less") {
					LessCompiler.compile(parameters, callback);
				} else if (extName === ".sass" || extName === ".scss") {
					if (compassInitInProgress) {
						var storedObj = {
							parameters: parameters,
							callback: callback
						};
						compileRequestQueue.push(storedObj); //Queue all the request as init has not yet finshed
					} else {
						SassCompiler.compile(parameters, callback);
					}
				}
			}
		};
		//The nDocDir parameter here is only for when the compilation is triggered to compile
        // bootstrap from New Document Dialog. In all other cases it should go to the else case.
		if (parameters.options.nDocDir && parameters.options.nDocDir !== "" &&
				parameters.options.nDocDir !== "undefined") {
			//if nDocDir is specified, then this is used as just compile service
			var cssPath = parameters.options.nDocDir;
			cssPath = decodeURIComponent(cssPath);
			cssPath = path.resolve(cssPath) + "/";
			
			var cssFilename = path.basename(sourceFile);
            cssFilename = cssFilename.substr(0, cssFilename.length - path.extname(cssFilename).length) + '.css';
						
			parameters.cssFile = cssPath + cssFilename;
			compileNow();
		} else {
			// compute the CSS output file name
			cssFile = getCSSFileName(sourceFile,
                                     (parameters.optionalDetails ?
                                      parameters.optionalDetails.fileOutsideSite :
                                      false));
            
			parameters.cssFile = cssFile;
			
			fs.exists(cssFile, function (exists) {
				if (exists) {
					var tempPrefix = crypto.randomBytes(10).readUInt32LE(0);
					var cssFileBaseName = path.basename(cssFile);
					var backedUpFilePath = path.join(userConfigurationTempPath, tempPrefix + cssFileBaseName);
					fs.readFile(cssFile, function (err, buffer) {
						var content = buffer.toString();
						
						if (err) {
							return;
						} else {
							Utils.mkfile(backedUpFilePath, content, function (err) {
								if (!err) {
									parameters.backedupCssFile = backedUpFilePath;
								}

								compileNow();
							});
						}
					});
				} else {
					compileNow();
				}
			});
		}
    }
    
    /**
     * Compiles the specified less/sass file and raises events on dw to show in output panel
     * @param {string}   filePath        
     * @param {object} optionalDetails  you want to supply to the compilers, or out of site flag
     *                {changedSource:"the dependant path that caused this file to be compiled"}                  
     */
    function compileFile(filePath, optionalDetails) {
        if (compilationEnabled === true) {
            
            var parameters = {
                sourceFile : filePath,
                optionalDetails : optionalDetails
            };
            var compileCallBack = function (para, para1) {
                var dataObj = para || para1;
                DWEvents.raiseEventOnDW('preprocessorOutputSessionData', dataObj);
            };
            var overrideOptions = false;
            if (parameters.optionalDetails && parameters.optionalDetails.fileOutsideSite) {
                overrideOptions = true;
            }
            if (Utils.isLessFile(filePath)) {
                parameters.options = overrideOptions ? lessStyleOptions : defaultLessOptions;
                compile(parameters, compileCallBack);
            } else if (Utils.isSassFile(filePath)) {
                parameters.options = overrideOptions ? sassStyleOptions : defaultSassOptions;
                parameters.options.rubyPath =  defaultSassOptions.rubyPath;
                parameters.options.rubyGemPath = defaultSassOptions.rubyGemPath;
                compile(parameters, compileCallBack);
            }
        }
    }
    
    /**
     * Compile the filePath and informing the compiler that the compile was initiated by a change
     * in the dependantPath
     * @param {string} filePath      to compile
     * @param {string} dependantPath the file which changed and triggered this compile
     */
    function compileDependentFile(filePath, dependantPath) {
        compileFile(filePath, {
            changedSource : dependantPath
        });
    }
    
    /**
     * trigger less/sass compilation when we detected that 
     * @param {string} changedFilePath 
     */
    function _watchedFileChanged(changedFilePath) {
        if (isFileInSite(changedFilePath)) {
            DependencyTracker.processTrackedFile(changedFilePath);
            DependencyTracker.compileAllDependants(changedFilePath);
        }
    }
    
    function _watchedFileAdded(changedFilePath) {
        if (isFileInSite(changedFilePath)) {
            DependencyTracker.processTrackedFile(changedFilePath);
            if (watcherReady) {
                DependencyTracker.compileAllDependants(changedFilePath);
            }
        }
    }
    
    function _watchedFileRemoved(changedFilePath) {
        if (isFileInSite(changedFilePath)) {
            DependencyTracker.removeTrackedFile(changedFilePath);
        }
    }
    
    function _watcherNotReady() {
        watcherReady = false;
    }
    
    function _watcherReady() {
        watcherReady = true;
    }
    
    function compileAllDependents(fileName) {
        DependencyTracker.compileAllDependants(fileName);
    }
    
    function compileFolder(folderPath) {
        DependencyTracker.compileFolder(folderPath);
    }
    
    /**
     * Sets a flag specifying that compass init has started and it is under progress
     * This needs to be reset once we recieve a success or error message from compiler      
     */
    function setCompassInitInProgress() {
        compassInitInProgress = true;
    }
    
    function _resetCompassInitInProgress() {
        compassInitInProgress = false;
    }
    
    function _compileCompassInitQueuedRequest(createdConfigfile) {
        var index;
        //ToDo: Change this fs.existsSync to fs.existsAsync , as that will not block.
        var addConfigFilePath = createdConfigfile && fs.existsSync(createdConfigfile);
        if (compileRequestQueue) {
            for (index = 0; index < compileRequestQueue.length; ++index) {
                var obj = compileRequestQueue[index];
                if (obj && obj.parameters && obj.callback) {
                    
                    if (addConfigFilePath && obj.parameters.siteDetails && obj.parameters.siteDetails.compassoptions &&
                            !obj.parameters.siteDetails.compassoptions.configfilepath) {
                        //Add the config file path for compilation , as currently SiteDetails dosen't have the same
                        obj.parameters.siteDetails.compassoptions.configfilepath = createdConfigfile;
                    }
                    compile(obj.parameters, obj.callback);
                }
            }
            compileRequestQueue.splice(0); //delete the entries in the queue as we have processed them
        }
    }
    
    function compassInit(compassInitParams) {
        
        var initSuccessCallBack = function (msg) {
            var createdConfigfile;
            if (currentSiteDetails && currentSiteDetails.root) {
                createdConfigfile = currentSiteDetails.root + compassConfigFileName;
                if (createdConfigfile && fs.existsSync(createdConfigfile)) {
                    if (currentSiteDetails.compassoptions && currentSiteDetails.compassoptions.configfilepath) {
                        currentSiteDetails.compassoptions.configfilepath = createdConfigfile;
                    }
                }
            }
            _resetCompassInitInProgress(); //Reset marking that compass init has finished execution
            //Now if any compilation request was queued trigger them now 
            _compileCompassInitQueuedRequest(createdConfigfile);
            DWEvents.raiseEventOnDW('preprocessorOutputSessionData', msg);
        };
        var initFailureCallBack = function (msg) {
            _resetCompassInitInProgress(); //Reset marking that compass init has finished execution
            DWEvents.raiseEventOnDW('preprocessorOutputSessionData', msg);
        };
        var options = {
            rubyPath: defaultSassOptions.rubyPath,
            rubyGemPath: defaultSassOptions.rubyGemPath,
            siteroot: compassInitParams.siteRoot
        };
        
        SassCompiler.compassInit(options, initSuccessCallBack, initFailureCallBack);
    }
    
    function notifyDWFileListToBeCompiled(fileList, outOfSiteFile) {
        if (compilationEnabled && !compassInitInProgress && fileList) {
            var listOfFiles = [],
                index = 0;
            if (typeof fileList === "string") {
                //there is only 1 file , which is getting compiled
                listOfFiles.push(fileList);
            } else if (typeof fileList === "object") {
                for (index = 0; index < fileList.length; ++index) {
                    if (!Utils.isSassPartialFile(fileList[index]) && Utils.isSassFile(fileList[index])) {
                        listOfFiles.push(fileList[index]);
                    }
                }
            }
            if (listOfFiles.length > 0) {
                var msg = Utils.createCompiledFileListReturnObject(listOfFiles, outOfSiteFile);
                if (msg) {
                    DWEvents.raiseEventOnDW('preCompileData', msg);
                }
            }
        }
    }
    
    function init() {
        DWFileWatcher.on(DWFileWatcher.FILE_CHANGED_EVENT, _watchedFileChanged);
        DWFileWatcher.on(DWFileWatcher.FILE_ADDED_EVENT, _watchedFileAdded);
        DWFileWatcher.on(DWFileWatcher.FILE_REMOVED_EVENT, _watchedFileRemoved);
        DWFileWatcher.on(DWFileWatcher.FILE_WATCH_NOT_READY, _watcherNotReady);
        DWFileWatcher.on(DWFileWatcher.FILE_WATCH_READY, _watcherReady);
    }
    
    exports.init                      = init;
    exports.setDefaultLessOptions     = setDefaultLessOptions;
    exports.setDefaultSASSOptions     = setDefaultSASSOptions;
    exports.setUserConfigTempPath     = setUserConfigTempPath;
    exports.switchSite                = switchSite;
    exports.compile                   = compile;
    exports.compileFile               = compileFile;
    exports.compileDependentFile      = compileDependentFile;
    exports.compileAllDependents      = compileAllDependents;
    exports.compileFolder             = compileFolder;
    exports.isFileInSite              = isFileInSite;
    exports.compassInit               = compassInit;
    exports.setCompassInitInProgress  = setCompassInitInProgress;
    exports.setCompilationState       = setCompilationState;
    exports.notifyDWFileListToBeCompiled = notifyDWFileListToBeCompiled;
    exports.getCSSFileName            = getCSSFileName;
}());
