/*
 * Copyright (c) 2013 Adobe Systems Incorporated. All rights reserved.
 *  
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"), 
 * to deal in the Software without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, 
 * and/or sell copies of the Software, and to permit persons to whom the 
 * Software is furnished to do so, subject to the following conditions:
 *  
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *  
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 * 
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4,
maxerr: 50, node: true */
/*global */

(function () {
    "use strict";
        
    var util              = require("util");

    /**
     * @constructor
     * DWServiceManager is a module/class that handles the loading, registration,
     * and execution of all commands and events. It is a singleton, and is passed
     * to a service in its init() method.
     */
    var self = exports;
    
    /**
     * @private
     * @type {object}
     * Map of all the registered services
     */
    var _services = {};

    /**
     * @private
     * @type {Array.<Module>}
     * Array of all modules we have loaded. Used for avoiding duplicate loading.
     */
    var _initializedServiceModules = [];

    /**
     * @private
     * @type {number}
     * Used for generating unique IDs for events.
     */
    var _eventCount = 1;

    /**
     * @private
     * @type {Array}
     * JSON.stringify-able Array of the current API. In the format of
     * Inspector.json. This is a cache that we invalidate every time the
     * API changes.
     */
    var _cachedServiceDescriptions = null;
    
    /**
     * Returns whether a service with the specified name exists or not.
     * @param {string} serviceName The service name.
     * @return {boolean} Whether the service exists
     */
    function hasService(serviceName) {
        return !!_services[serviceName];
    }
    
    /**
     * Returns a new empty service. Throws error if the service already exists.
     * @param {string} serviceName The service name.
     * @param {{major: number, minor: number}} version The service version.
     *   The version has a format like {major: 1, minor: 2}. It is reported
     *   in the API spec, but serves no other purpose on the server. The client
     *   can make use of this.
     */
    function registerService(serviceName, version) {
        if (!hasService(serviceName)) {
            // invalidate the cache
            _cachedServiceDescriptions = null;
            
            _services[serviceName] = {version: version, commands: {}, events: {}};
        } else {
            console.error("[DWServiceManager] service " + serviceName + " already registered");
        }
    }
    
    /**
     * Registers a new command with the specified service. If the service does
     * not yet exist, it registers the service with a null version.
     * @param {string} serviceName The service name.
     * @param {string} commandName The command name.
     * @param {Function} commandFunction The callback handler for the function.
     *    The function is called with the arguments specified by the client in the
     *    command message. Additionally, if the command is asynchronous (isAsync
     *    parameter is true), the function is called with an automatically-
     *    constructed callback function of the form cb(err, result). The function
     *    can then use this to send a response to the client asynchronously.
     * @param {boolean} isAsync See explanation for commandFunction param
     * @param {?string} description Used in the API documentation
     * @param {?Array.<{name: string, type: string, description:string}>} parameters
     *    Used in the API documentation.
     * @param {?Array.<{name: string, type: string, description:string}>} returns
     *    Used in the API documentation.
     */
    function registerCommand(serviceName, commandName, commandFunction, isAsync,
        description, parameters, returns) {
        // invalidate the cache
        _cachedServiceDescriptions = null;
        
        if (!hasService(serviceName)) {
            registerService(serviceName, null);
        }

        if (!_services[serviceName].commands[commandName]) {
            _services[serviceName].commands[commandName] = {
                commandFunction: commandFunction,
                isAsync: isAsync,
                description: description,
                parameters: parameters,
                returns: returns
            };
        } else {
            throw new Error("Command " + serviceName + "." +
                commandName + " already registered");
        }
    }

    function lookup(serviceNameToLookup, commandName) {
        var serviceNames = Object.keys(_services);
        var found = false;

        serviceNames.some(function (serviceName) {
            if (serviceNameToLookup === serviceName) {
                var service = _services[serviceName];
                var commandNames = Object.keys(_services[serviceName].commands);
                commandNames.forEach(function (commandHandlerName) {
                    if (commandName === commandHandlerName) {
                        found = true;
                    }
                });
            }
        });

        return found;
    }
    
    function executeCommand(body, res) {
        var serviceName  = body.service;
        var commandName = body.command;
        var parameters  = body.parameters;

        var found = lookup(serviceName, commandName);

        if (!found) {
            res.statusCode = 400;
            res.end(JSON.stringify({status: 'error'}));
        } else if (_services[serviceName] &&
                _services[serviceName].commands[commandName]) {
            var command = _services[serviceName].commands[commandName];
            if (command.isAsync) {
                var callback = function (err, result) {
                    if (err) {
                        res.statusCode = 500;
                        res.end(JSON.stringify(err));
                    } else {
                        res.statusCode = 200;
                        res.end(JSON.stringify(result));
                    }
                };
                parameters.push(callback);
                command.commandFunction.apply(null, parameters);
            } else { // synchronous command
                try {
                    res.statusCode = 200;
                    res.end(JSON.stringify(command.commandFunction.apply(null, parameters)));
                } catch (e) {
                    res.statusCode = 500;
                    res.end(JSON.stringify({"status": "error"}));
                }
            }
        } else {
            res.statusCode = 500;
            res.end(JSON.stringify({"status": "error"}));
        }
    }

    /**
     * Registers an event service and name.
     * @param {string} serviceName The service name.
     * @param {string} eventName The event name.
     * @param {?Array.<{name: string, type: string, description:string}>} parameters
     *    Used in the API documentation.
     */
    function registerEvent(serviceName, eventName, parameters) {
        // invalidate the cache
        _cachedServiceDescriptions = null;
        
        if (!hasService(serviceName)) {
            registerService(serviceName, null);
        }

        if (!_services[serviceName].events[eventName]) {
            _services[serviceName].events[eventName] = {
                parameters: parameters
            };
        } else {
            console.error("[DWServiceManager] Event " + serviceName + "." +
                eventName + " already registered");
        }
    }

    /**
     * Loads and initializes service modules using the specified paths. Checks to
     * make sure that a module is not loaded/initialized more than once.
     *
     * @param {Array.<string>} paths The paths to load. The paths can be relative
     *    to the DWServiceManager or absolute. However, modules that aren't in core
     *    won't know where the DWServiceManager module is, so in general, all paths
     *    should be absolute.
     * @return {boolean} Whether loading succeded. (Failure will throw an exception).
     */
    function loadServiceModulesFromPaths(paths) {
        var pathArray = paths;
        if (!util.isArray(paths)) {
            pathArray = [paths];
        }
        pathArray.forEach(function (path) {
            var m = require(path);
            if (m && m.init && _initializedServiceModules.indexOf(m) < 0) {
                m.init(self);
                _initializedServiceModules.push(m); // don't init more than once
            }
        });
        return true; // if we fail, an exception will be thrown
    }
    
    /**
     * Returns a description of all registered services in the format of WebKit's
     * Inspector.json. Used for sending API documentation to clients.
     *
     * @return {Array} Array describing all services.
     */
    function getServiceDescriptions() {
        if (!_cachedServiceDescriptions) {
            _cachedServiceDescriptions = [];
            
            var serviceNames = Object.keys(_services);
            serviceNames.forEach(function (serviceName) {
                var d = {
                    service: serviceName,
                    version: _services[serviceName].version,
                    commands: [],
                    events: []
                };
                var commandNames = Object.keys(_services[serviceName].commands);
                commandNames.forEach(function (commandName) {
                    var c = _services[serviceName].commands[commandName];
                    d.commands.push({
                        name: commandName,
                        description: c.description,
                        parameters: c.parameters,
                        returns: c.returns
                    });
                });
                var eventNames = Object.keys(_services[serviceName].events);
                eventNames.forEach(function (eventName) {
                    d.events.push({
                        name: eventName,
                        parameters: _services[serviceName].events[eventName].parameters
                    });
                });
                _cachedServiceDescriptions.push(d);
            });
        }

        var descriptions = {
            services: []
        };

        descriptions.services.push.apply(descriptions.services, _cachedServiceDescriptions);

        return descriptions;
    }
    
    exports.hasService                  = hasService;
    exports.registerService             = registerService;
    exports.registerCommand             = registerCommand;
    exports.lookup                      = lookup;
    exports.executeCommand              = executeCommand;
    exports.registerEvent               = registerEvent;
    exports.loadServiceModulesFromPaths = loadServiceModulesFromPaths;
    exports.getServiceDescriptions      = getServiceDescriptions;
}());
