/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 
*/

/*global require, console, module, Buffer, setInterval, clearInterval */

var config = require('../config/config.js'),
    socketIO = require('socket.io'),
    SocketAuth = require("./socketAuth.js"),
    requestUtils = require('../server-core/src/coreUtils/requestUtils.js');

var getControllerSocket = function (socketGroup) {
    'use strict';
    var clients = module.exports.io.sockets.connected,
        clientId,
        socket;
    for (clientId in clients) {
        if (clients.hasOwnProperty(clientId)) {
            socket = module.exports.io.sockets.connected[clientId];
            if (socket.request.controller === true) {
                return socket;
            }
        }
    }
    return null;
};

var socketHandlers = function (socket) {
    "use strict";
    
    socket.on('change:url', function (data) {
        var UrlCommand = require('./changeUrlCommand.js'),
            command = new UrlCommand.ChangeUrlCommand(socket, data);
        command.execute();
    });
 
    socket.on('update:proxy:url', function (data) {
        var UrlCommand = require('./changeUrlCommand.js'),
            command = new UrlCommand.UpdateProxyUrlCommand(socket, data);
        command.execute();
    });
    
    socket.on('update:preview:service:url', function (data) {
        var UrlCommand = require('./changeUrlCommand.js'),
            command = new UrlCommand.UpdatePreviewServiceUrlCommand(socket, data);
        command.execute();
    });
    
    socket.on('check:authorization', function (data) {
        var CheckAuthCommand = require('./checkAuthCommand.js'),
            command = new CheckAuthCommand(socket, data);
        command.execute();
    });
		
	socket.on("mouse:hover:in", function (data) {
        var MouseMoveCommand = require('./mouseMoveCommand.js'),
            command = new MouseMoveCommand.MouseHoverInCommand(socket, data);
        command.execute();
    });
    
    socket.on("mouse:hover:out", function (data) {
        var MouseMoveCommand = require('./mouseMoveCommand.js'),
            command = new MouseMoveCommand.MouseHoverOutCommand(socket, data);
        command.execute();
    });
    
	socket.on("join:room", function (data) {
        var JoinRoomCommand = require('./joinRoomCommand.js'),
            command = new JoinRoomCommand(socket, data);
        command.execute();
    });

    socket.on('disconnect', function () {
        var DisconnectCommand = require('./disconnectCommand.js'),
            command = new DisconnectCommand(socket);
        
        command.execute();
    });
	
    socket.on("partial:refresh:delete:rule", function (rule) {
        var PartialRefreshCommand = require('./partialRefreshCommand.js'),
            command = new PartialRefreshCommand.deleteRule(socket, rule);
        command.execute();
    });
	
    socket.on("partial:refresh:insert:rule", function (rule) {
        var PartialRefreshCommand = require('./partialRefreshCommand.js'),
            command = new PartialRefreshCommand.insertRule(socket, rule);
        command.execute();
    });
    
    socket.on('dw:log:event', function (eventParams) {
        var socketGroup = socket.request.group,
            controllerSocket;
        if (socketGroup && socketGroup !== '') {
            controllerSocket = module.exports.getControllerSocket(socketGroup);
        
            if (controllerSocket && controllerSocket.id !== socket.id) {
                module.exports.io.to(controllerSocket.id).emit('dw:log:event', eventParams);
            }
        }
    });
    
    socket.on('query:device:viewport:dw', function (socketId) {
        module.exports.io.to(socketId).emit('query:device:viewport');
    });
    
    socket.on('query:device:viewport:cb', function (viewportSize) {
        var socketGroup = socket.request.group,
            controllerSocket;
        if (socketGroup && socketGroup !== '') {
            controllerSocket = module.exports.getControllerSocket(socketGroup);
        
            if (controllerSocket && controllerSocket.id !== socket.id) {
                module.exports.io.to(controllerSocket.id).emit('query:device:viewport:dw:cb', viewportSize);
            }
        }
    });
    
    socket.on('inspect:dw', function (inspectParams) {
        var socketId = inspectParams.socketId,
            liveId = inspectParams.liveId,
            socketGroup;
        
        if (socketId) {
            module.exports.io.to(socketId).emit('inspect:dw', inspectParams);
        } else {
            socketGroup = socket.request.group;
            socket.broadcast.to(socketGroup).emit('inspect:dw', inspectParams);
        }
    });
    
    socket.on('disconnect:yourself', function (params) {
        var errorURL = requestUtils.createErrorUrl(params),
            socketGroup = socket.request.group;
    
        socket.broadcast.to(socketGroup).emit('disconnect:yourself', errorURL);
    });
    
    socket.on('inspect:off:dw', function (socketId) {
        var socketGroup;
        if (socketId) {
            module.exports.io.to(socketId).emit('inspect:off:dw');
        } else {
            socketGroup = socket.request.group;
            socket.broadcast.to(socketGroup).emit('inspect:off:dw');
        }
    });
};

module.exports.init = function (server) {
    "use strict";
    var io = null,
        socketAuthObject;
    
    if (config.GlobalConf.disableLogging) {
        console.log = function () {};
    }
    
    io = module.exports.io = socketIO.listen(server, config.SocketIOConf);
    
    socketAuthObject = new SocketAuth();
    socketAuthObject.setupAuthHandlers();
    io.sockets.on('connection', socketHandlers);
};

module.exports.getControllerSocket = getControllerSocket;