/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 
*/

/*global require, module*/
/*jslint regexp: true, nomen: true, vars: true*/

var serverCore = require('../../server-core'),
    step = require('step'),
    validate = serverCore.validate,
    routeUtils = serverCore.routeUtils,
    assetsMgr = serverCore.assetsMgr,
    serverInfo = serverCore.serverInfo,
    serverCoreSettings = serverCore.coreSettings,
    assetUtils = require('./assetUtils.js'),
    securityUtils = require('../../server-core/src/coreUtils/securityUtils.js');

function _getContent(req, res, isHead, callback) {
    'use strict';
    var err404,
        contentInfo = {};
    
    step(
        function () {
            securityUtils.verifySecurity(req, res, this);
        },
        function (err, _contentInfo) {
            if (err) { callback(err); return; }
            contentInfo = _contentInfo;
            assetsMgr.getAssetContent(contentInfo, isHead, req.headers, this);
        },
        function (err, status, _contents, respHeaders, resourceInfo) {
            if (err && err.http_code === 404) {
                err404 = err;
            }
            if (err && !err404) { callback(err); return; }
            
            callback(err || err404, status, contentInfo, _contents, respHeaders);
        }
    );
}

function _applyUTF8Charset(contentString) {
    'use strict';
    var headOpenTagRegex = /<head[\s\S]*?>/i,
        match = headOpenTagRegex.exec(contentString),
        metaString = '<meta charset="utf-8">',
        retVal = contentString;
    
    if (contentString && match && match.length > 0) {
        var index = match.index + match[0].length;
        retVal = [contentString.slice(0, index), metaString, contentString.slice(index)].join('');
    }
    
    return retVal;
}

function _modifyContent(contents, contentType, callback) {
    'use strict';
    if (contentType.toLowerCase().indexOf('text/html') !== -1) {
        var contentString = (typeof contents === 'string' || contents instanceof String) ? contents : assetUtils.arrayBuffer2StringWithEncoding(contentType, contents, 'utf-8'),
            additionalScript = '',
            scriptPath = '';
        
        contentString = _applyUTF8Charset(contentString);
        
        additionalScript += '<dw-container-div>\n';
        scriptPath = '/static/files/deviceClientScript.js';
        additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
        scriptPath = '/static/files/IOSScrolling.js';
        additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
        
        var liveEditString = serverCore.coreSettings.PROXY_PROTOCOL.toLowerCase() === 'file' ? serverCore.coreSettings.PRIMARY_URL_HTML : serverCore.coreSettings.PRIMARY_URL_PARTIAL_REFRESH_HTML;
        //if we are viewing file ptotcol url, then it implies there is no diff between rendered html and html required for initialising
        //partial refrrsh, in case of testing server, partial refresh gets initialised on html containing server tags
        if (liveEditString) {
            scriptPath = '/static/files/HTMLDOMDiff.js';
            additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
            scriptPath = '/static/files/HTMLSimpleDOM.js';
            additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
            scriptPath = '/static/files/HTMLTokenizer.js';
            additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
            scriptPath = '/static/files/murmurhash3_gc.js';
            additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
            scriptPath = '/static/files/RemoteFunctions.js';
            additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
            scriptPath = '/static/files/LiveUpdateTools.js';
            additionalScript += ('<script src=\"' + scriptPath + '\"></script>\n');
            additionalScript += ('<script>$dwJQuery = window.parent.$; window.DevicePreview_RenderSuccess = true; DW_LiveEdit_SetupDoc(\'' + routeUtils.sanitizeHTMLData(liveEditString) + '\');</script>\n');
        }
        additionalScript += '</dw-container-div>';
        
        var index = -1;
        if ((index = contentString.lastIndexOf('</body>')) !== -1) {
            //we want to insert our scripts in the very end of body tag
            contentString = contentString.slice(0, index) + additionalScript + contentString.slice(index);
        } else {
            //we dint find body tag's end!!!!!!!!!!!!!!!!! just append it and pray it works!
            contentString += additionalScript;
        }
        
        callback(null, contentString);
    } else {
        callback(null, contents);
    }
}

function getAssetContent(req, res, next) {
    'use strict';
    var callback = routeUtils.enterRoute('assetRoutes.getAssetContent' + req.url, res),
        isHead = req.method === 'HEAD',
        statusCode = 200,
        headers = {};
    
    step(
        function () {
            _getContent(req, res, isHead, this);
        },
        function (err, status, contentInfo, contents, resHeaders) {
            if (err) { callback.error(err); return; }
            
            statusCode = status;
            headers = resHeaders;
            
            var contentType = headers['content-type'] || '';
            if (contentType && contentType.toLowerCase().indexOf('text/html') !== -1) { /*we will add some stuff to html pages, hence dont set this*/
                delete headers['content-length'];
            }
            
            if (req.query.download) {
                headers['content-disposition'] = 'attachment; filename="' +  req.query.download + '"';
            }
            
            _modifyContent(contents, contentType, this);
        },
        function (err, contents) {
            if (err) { callback.error(err); return; }
            
            var range = req.headers.range;
            if (serverCoreSettings.PROXY_HOSTNAME === '' && range && headers['content-length']) {
                //our approach is quite rudimentary as we keep reading the video for every partial request, we must improve this
                var total = contents.length,
                    positions = range.replace(/bytes=/, "").split("-"),
                    start = parseInt(positions[0], 10),
                    end = positions[1] && parseInt(positions[1], 10) < total - 1 ? parseInt(positions[1], 10) : total - 1,
                    chunksize = (end - start) + 1;
 
                headers["content-range"] = "bytes " + start + "-" + end + "/" + total;
                headers["accept-ranges"] = "bytes";
                headers["content-length"] = chunksize;
                res.writeHead(206, headers);
                res.end(contents.slice(start, end + 1), "binary");
            } else {

                res.writeHead(statusCode, headers);
                if (contents) {
                    res.end(contents);
                } else {
                    res.end();
                }
            }
            
            callback.success();
        }
    );
}


function _getErrorContent(req, callback) {
    'use strict';
    var contentInfo = {};
    
    step(
        function () {
            routeUtils.getErrorString(req, this);
        },
        function (err, _contentInfo) {
            if (err) { callback(err); return; }
            contentInfo = _contentInfo;
            assetsMgr.getErrorContent(contentInfo, this);
        },
        function (err, _contents) {
            callback(err, _contents);
        }
    );
}


function getErrorContent(req, res, next) {
    'use strict';
    var callback = routeUtils.enterRoute('assetRoutes.getErrorContent' + req.url, res),
        headers = {};
    
    step(
        function () {
            _getErrorContent(req, this);
        },
        function (err, contents) {
            if (err) { callback.error(err); return; }
            
            headers['Content-Type'] = 'text/html';
            if (req.query.download) {
                headers['Content-Disposition'] = 'attachment; filename="' +  req.query.download + '"';
            }
            
            res.writeHead(200, headers);
            
            if (contents) {
                res.end(contents);
            } else {
                res.end();
            }
            
            callback.success();
        }
    );
}

function getPreviewServiceUrl(req, res, next) {
    'use strict';
    var callback = routeUtils.enterRoute('assetRoutes.getAssetContent' + req.url, res);
    res.end(serverCoreSettings.PREVIEW_SERVICE_URL);
    callback.success();
}

function getLocalServiceUrl(req, res, next) {
    'use strict';
    var callback = routeUtils.enterRoute('assetRoutes.getLocalServiceUrl' + req.url, res);
    res.end(serverInfo.LOCAL_IP_ADDRESS);
    callback.success();
}

function attachHandlers(server) {
    'use strict';
    validate.defineRoute(server, 'GET', /^\/content\/.+$/, getAssetContent, {});
    validate.defineRoute(server, 'GET', /^\/previewapp\/url/, getPreviewServiceUrl, {});
    validate.defineRoute(server, 'GET', /^\/previewapp\/localurl/, getLocalServiceUrl, {});
    validate.defineStaticRoute(server, 'GET', /^\/static\/.+$/);
    validate.defineDeviceAppRoute(server, 'GET', /^\/preview\/.+$/);
    validate.defineRoute(server, 'GET', /^\/error\/.+$/, getErrorContent, {});
}

module.exports = {
    attachHandlers: attachHandlers
};