/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 
*/

/*global require, console, process, module*/
/*jslint nomen: true, plusplus: true, vars: true*/

var restify = require('restify'),
    socketUtils = require('../../socketUtils'),
    config = require('../coreConfig/config.js'),
    URI = require('URIjs'),
    routeUtils = require('./coreUtils/routeUtils.js'),
    requestUtils = require('./coreUtils/requestUtils.js'),
    securityUtils = require('./coreUtils/securityUtils.js');

/**
 * @private
 * @type{number} unique IDs for messages to parent process
 */
var _commandCount = 1;

function extendHandler(handler) {
    'use strict';
    return function (req, res, next) {
        return handler(req, res, next);
    };
}

function DevicePreviewServer(moduleList, sStaticContentPath) {
    'use strict';
    
    var server = restify.createServer(),
        loadedModules = false,
        self = this;
    
    /* create websocket layer */
    socketUtils.init(server.server);
    
    [
        'get',
        'head',
        'opts',
        'post',
        'put',
        'patch'
    ].forEach(function (method) {
        self[method] = function () {
            var newArguments = [],
                i,
                arg;
            for (i = 0; i < arguments.length; i++) {
                arg = arguments[i];
                if (i > 0 && typeof arg === 'function') {
                    newArguments.push(extendHandler(arg));
                } else {
                    newArguments.push(arg);
                }
            }
            
            return server[method].apply(server, newArguments);
        };
    });
    
    /**
     * Stops the server and does appropriate cleanup.
     * Emits an "end" event when shutdown is complete.
     */
    this.stop = function () {
        console.log("[Server] stopping");

        if (server) {
            try {
                server.close();
            } catch (err) { }
        }
        console.log("[Server] stopped");
        process.exit();
    };
    
    this.ProcessKeyValuePair = function (key, value) {
        switch (key) {
        case config.incomingIPCCommands.UPDATE_HTML:
            config.coreSettings.PRIMARY_URL_HTML = value;
            config.coreSettings.PRIMARY_URL_PARTIAL_REFRESH_HTML = '';
            break;
        case config.incomingIPCCommands.UPDATE_PARTIAL_REFRESH_HTML:
            config.coreSettings.PRIMARY_URL_PARTIAL_REFRESH_HTML = value;
            break;
        case config.incomingIPCCommands.UPDATE_EXTERNAL_URL:
            config.coreSettings.IS_BROWSING_HOME_URL = false;
            this.HandleMarkHTMLDirty(false);
            this.HandleUpdateURL(value, true);
            break;
        case config.incomingIPCCommands.UPDATE_URL_HOME_WITH_DATA:
            config.coreSettings.IS_BROWSING_HOME_URL = true;
            config.coreSettings.IS_PRIMARY_URL_HTML_DIRTY = false;
            config.coreSettings.IS_PRIMARY_URL_RELATED_FILES_DIRTY = false;
            this.HandleUpdateURL(value, true);
            break;
        case config.incomingIPCCommands.UPDATE_URL_HOME_WITHOUT_DATA:
            config.coreSettings.IS_BROWSING_HOME_URL = true;
            this.HandleMarkHTMLDirty(false);
            this.HandleUpdateURL(value, false);
            break;
        case config.incomingIPCCommands.SCROLL_COMMAND:
            this.HandleScrollCommand(value);
			break;
        case config.incomingIPCCommands.SCROLL_ELEMENT_TO_VIEW:
            this.HandleScrollElementToView(value);
			break;
        case config.incomingIPCCommands.PARTIAL_REFRESH_HTML:
            this.HandlePartialRefreshHTML(value);
            this.HandleMarkHTMLDirty(true, true);
            break;
        case config.incomingIPCCommands.PARTIAL_REFRESH_CSS_DELETE:
            this.HandlePartialRefreshCSS(value, true);
            this.HandleMarkHTMLDirty(true, false);
            break;
        case config.incomingIPCCommands.PARTIAL_REFRESH_CSS_INSERT:
            this.HandlePartialRefreshCSS(value, false);
            this.HandleMarkHTMLDirty(true, false);
            break;
        case config.incomingIPCCommands.MARK_HTML_DIRTY:
            this.HandleMarkHTMLDirty(true, false);
            break;
        case config.incomingIPCCommands.COMMAND_ERROR_STRING_UNSAVED_DOCUMENT:
            config.errorStrings.ERROR_STRING_UNSAVED_DOCUMENT = value;
            break;
        case config.incomingIPCCommands.COMMAND_ERROR_STRING_NO_DOCUMENT:
            config.errorStrings.ERROR_STRING_NO_DOCUMENT = value;
            break;
        case config.incomingIPCCommands.COMMAND_ERROR_STRING_NO_HTML_DOCUMENT:
            config.errorStrings.ERROR_STRING_NO_HTML_DOCUMENT = value;
            break;
        case config.incomingIPCCommands.COMMAND_ERROR_STRING_DW_EXITING:
            config.errorStrings.ERROR_STRING_DW_EXITING = value;
            break;
        case config.incomingIPCCommands.COMMAND_ERROR_STRING_RECONNECT:
            config.errorStrings.ERROR_STRING_RECONNECT = value;
            break;
        case config.incomingIPCCommands.COMMAND_OVERLAY_STRING_OS:
            config.overlayStrings.OVERLAY_STRING_OS = value;
            break;
        case config.incomingIPCCommands.COMMAND_OVERLAY_STRING_VIEWPORT_SIZE:
            config.overlayStrings.OVERLAY_STRING_VIEWPORT_SIZE = value;
            break;
        case config.incomingIPCCommands.COMMAND_OVERLAY_STRING_DEVICE_PIXEL_RATIO:
            config.overlayStrings.OVERLAY_STRING_DEVICE_PIXEL_RATIO = value;
            break;
        case config.incomingIPCCommands.COMMAND_OVERLAY_STRING_USER_AGENT:
            config.overlayStrings.OVERLAY_STRING_USER_AGENT = value;
            break;
        case config.incomingIPCCommands.COMMAND_ERROR_STRING_PAGE_LOAD:
            config.errorStrings.ERROR_STRING_PAGE_LOAD = value;
            break;
        case config.incomingIPCCommands.DIRTY_DOCUMENT_PREVIEW:
            this.SendErrorUrl(config.errorMarkers.ERROR_UNSAVED_DOCUMENT);
            break;
        case config.incomingIPCCommands.NO_DOCUMENT_PREVIEW:
            this.SendErrorUrl(config.errorMarkers.ERROR_NO_DOCUMENT);
            break;
        case config.incomingIPCCommands.NON_HTML_DOCUMENT_PREVIEW:
            this.SendErrorUrl(config.errorMarkers.ERROR_NO_HTML_DOCUMENT);
            break;
        case config.incomingIPCCommands.RETRIEVE_IP:
            config.serverInfo.IP_ADDRESSES = requestUtils.getExternalIPList();
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.IP_PORT, config.serverInfo.IP_ADDRESSES + ':' + config.serverInfo.PORT);
            break;
        case config.incomingIPCCommands.RETRIEVE_LOCAL_IP:
            config.serverInfo.LOCAL_IP_ADDRESS = requestUtils.getLocalIP();
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.LOCAL_IP_PORT, config.serverInfo.LOCAL_IP_ADDRESS + ':' + config.serverInfo.PORT);
            break;
        case config.incomingIPCCommands.DW_EXITING:
            this.HandleDWQuitting();
            break;
        case config.incomingIPCCommands.SERVER_URL:
            config.coreSettings.SERVER_URL = value;
            break;
        default:
            break;
        }
    };
    
    this.SendErrorUrl = function (error) {
        if (!error) {
            return;
        }
        
        var newURL = requestUtils.createErrorUrl(error);
        
        var data      = {};
        data.url      = newURL;
        data.isServer = false;
        socketUtils.IPCToSocketBridger(config.socketCommands.CHANGE_URL, data);
    };
    
    this.HandleScrollElementToView = function (value) {
        var data = {},
            liveId = '',
            selectorString = '',
            splitIndex = -1;
        
        splitIndex = value.indexOf(':');
        if (splitIndex !== -1) {
            data.liveId = value.slice(0, splitIndex);
            data.selectorString = value.slice(splitIndex + 1);
            socketUtils.IPCToSocketBridger(config.socketCommands.SCROLL_ELEMENT_TO_VIEW, data);
        }
    };
    
    this.HandleDWQuitting = function () {
        socketUtils.IPCToSocketBridger(config.socketCommands.DISCONNECT_DEVICE, requestUtils.createErrorUrl(config.errorMarkers.ERROR_DW_EXITING));
    };
    
    this.HandleMarkHTMLDirty = function (dirty, isHTML) { /*isHTML is applicable only when dirty is true*/
        config.coreSettings.PRIMARY_URL_HTML = '';
        config.coreSettings.PRIMARY_URL_PARTIAL_REFRESH_HTML = '';
        if (dirty) {
            if (isHTML) {
                config.coreSettings.IS_PRIMARY_URL_HTML_DIRTY = true;
            } else {
                config.coreSettings.IS_PRIMARY_URL_RELATED_FILES_DIRTY = true;
            }
        } else {
            config.coreSettings.IS_PRIMARY_URL_HTML_DIRTY = false;
            config.coreSettings.IS_PRIMARY_URL_RELATED_FILES_DIRTY = false;
        }
    };
    
    this.HandlePartialRefreshCSS = function (argStr, isDelete) {
        var command = isDelete ? config.socketCommands.PARTIAL_REFRESH_CSS_DELETE : config.socketCommands.PARTIAL_REFRESH_CSS_INSERT;
        var inputRegex = /([\s\S]*?)(?:,(?!\\))([\s\S]*?)(?:,(?!\\))([\s\S]*)/g, result = null;
        //the commands will arrive as {cssText},{mediatext},{path} incase either of three items have comma in it
        //it will be succeded by a backslash, so we need to find two commas without succeding backslash
        if ((result = inputRegex.exec(argStr)) !== null) {
            var data = {};
            data.cssText = result[1].replace(/,\\/g, ',');
            data.mediaText = result[2].replace(/,\\/g, ',');
            var path = result[3].replace(/,\\/g, ','), pathIndex = -1;
            if (config.coreSettings.PROXY_PROTOCOL.toLowerCase() === 'file' && (pathIndex = path.lastIndexOf('file://')) > 0) {
                //here is a tragedy of the system, in DW we screw up the path name in mac in case of file protocol, essentially it can either 
                //be regular posix path or some pseudo hfs pth with /. the path recieved here may sometimes conatin two paths
                //appended to each other, we need to separate them out and purge them separately
                var path1 = path.slice(0, pathIndex);
                var path2 = path.slice(pathIndex);
                data.pathArray = [routeUtils.purgeUrl(path1, false), routeUtils.purgeUrl(path2, false)];
            } else {
                data.pathArray = [routeUtils.purgeUrl(path, false)];
            }
            socketUtils.IPCToSocketBridger(command, data);
        }
    };
    
    this.HandleUpdateURL = function (url, shouldBroadcast) {
        config.coreSettings.PRIMARY_URL = url;
        var newURL = '';
        if (url) {
            var uriObj = new URI(url);
            
            config.coreSettings.PROXY_PROTOCOL = uriObj.protocol();
            config.coreSettings.PROXY_HOSTNAME = uriObj.hostname();
            config.coreSettings.PROXY_PORT = uriObj.port();
            
            newURL = routeUtils.purgeUrl(url, true);
        } else {
            config.coreSettings.PROXY_PROTOCOL = '';
            config.coreSettings.PROXY_HOSTNAME = '';
            config.coreSettings.PROXY_PORT = '';
            securityUtils.clearSecurityDetails();
        }
        
        var data = {};
        data.url      = newURL;
        data.isServer = false;
        
        if (shouldBroadcast) {
            if (config.coreSettings.SERVER_URL.length > 0) {
                data.url      = config.coreSettings.SERVER_URL;
                data.isServer = true;
            }
            socketUtils.IPCToSocketBridger(config.socketCommands.CHANGE_URL, data);
        }
        
    };
    
    this.HandlePartialRefreshHTML = function (value) {
        var data = {},
            startOffsetIndex = -1,
            endOffsetIndex = -1,
            shouldOnlyUpdateIndex = -1,
            fullRefreshIfFailedIndex = -1;
            
        startOffsetIndex = value.indexOf(',');
        endOffsetIndex = value.indexOf(',', startOffsetIndex + 1);
        fullRefreshIfFailedIndex = value.lastIndexOf(',');
        shouldOnlyUpdateIndex = value.lastIndexOf(',', fullRefreshIfFailedIndex - 1);
        
        if ((startOffsetIndex < endOffsetIndex) && (endOffsetIndex < shouldOnlyUpdateIndex) && (shouldOnlyUpdateIndex < fullRefreshIfFailedIndex)) {
            data.startOffset = parseInt(value.slice(0, startOffsetIndex), 10);
            data.endOffset = parseInt(value.slice(startOffsetIndex + 1, endOffsetIndex), 10);
            data.editString = value.slice(endOffsetIndex + 1, shouldOnlyUpdateIndex);
            data.shouldOnlyUpdate = (value.slice(shouldOnlyUpdateIndex + 1, fullRefreshIfFailedIndex).toLowerCase() === 'true');
            data.fullRefreshIfFailed = (value.slice(fullRefreshIfFailedIndex + 1).toLowerCase() === 'true');
        
            socketUtils.IPCToSocketBridger(config.socketCommands.PARTIAL_REFRESH_HTML, data);
        }
    };
    
    this.HandleScrollCommand = function (scrollParams) {
        if (scrollParams) {
            var scrollxy = scrollParams.split(":"),
                scrollParamsObj = {
                    'xperc': parseInt(scrollxy[0], 10),
                    'yperc': parseInt(scrollxy[1], 10)
                };
            socketUtils.IPCToSocketBridger(config.socketCommands.SCROLL_COMMAND, scrollParamsObj);
        }
    };
    
    this.setupStdin = function () {
        
        var buffer = "", self = this;
        // re-enable getting events from stdin
        try {
            process.stdin.resume();
            process.stdin.setEncoding("utf8");
        } catch (e) {
            // Couldn't resume stdin, so something is terribly wrong
            self.stop();
        }
        
        var sanitizeString = function (str) {
           //replace \\ wid \ and |\ with |
            if (str) {
                str = str.replace(/\\\\/g, '\\');
                str = str.replace(/\|\\/g, '|');
            }
            return str;
        };
        
        // set up event handlers for stdin
        process.stdin.on("data", function (data) {
            buffer += data;
            var inputRegex = /\n\n\d+\|([\S\s]*?)(?:\|(?!\\)|\|(?=\\\\\\))([\S\s]*?)\|\n\n/g, result = null;
            //the message sent to/from will be of the form \n\n1|ip:port|1.1.1.1:1234|\n\n with key being "ip:port" and value being "1.1.1.1:1234"
            //if the key or value contains | character, a \ will be added in front of it, same goes for a valid \. Hence in the message there should never be a single \.
            
            var lastIndex = 0;
            while ((result = inputRegex.exec(buffer)) !== null) {
                var key = result[1], value = result[2];
                key = sanitizeString(key);
                value = sanitizeString(value);
                self.ProcessKeyValuePair(key, value);
                lastIndex = inputRegex.lastIndex;
            }
            buffer = buffer.substr(lastIndex);
        });

        process.stdin.on("end", function receiveStdInClose() {
            self.stop();
        });
    };
    
    this.getAndSetExternalAndLocalHostIPList = function () {
        var newIP = requestUtils.getExternalIPList();
        if (newIP !== config.serverInfo.IP_ADDRESSES) {
            config.serverInfo.IP_ADDRESSES = newIP;
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.IP_PORT, config.serverInfo.IP_ADDRESSES + ':' + config.serverInfo.PORT);
        }
        
        var newLocalIP = requestUtils.getLocalIP();
        if (newLocalIP !== config.serverInfo.LOCAL_IP_ADDRESS) {
            config.serverInfo.LOCAL_IP_ADDRESS = newLocalIP;
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.LOCAL_IP_PORT, config.serverInfo.LOCAL_IP_ADDRESS + ':' + config.serverInfo.PORT);
        }
    };
    
    this.setupStdout = function () {
        // Routinely check if stdout is closed. Stdout will close when our
        // parent process closes (either expectedly or unexpectedly) so this
        // is our signal to shutdown to prevent process abandonment.
        //
        // We need to continually ping because that's the only way to actually
        // check if the pipe is closed in a robust way (writable may only get
        // set to false after trying to write a ping to a closed pipe).
        var self = this;
        setInterval(function () {
            if (!process.stdout.writable) {
                // If stdout closes, our parent process has terminated or
                // has explicitly closed it. Either way, we should exit.
                console.log("[Server] stopping because stdout closed");
                self.stop();
            } else {
                try {
                    requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.PING, '');
                } catch (e) {
                    console.log("[Server] stopping because stdout was not writable");
                    self.stop();
                }
            }
        }, config.coreSettings.PING_DELAY);
        
        
        //check if the ip has changed due to switching of networks
        setInterval(this.getAndSetExternalAndLocalHostIPList.bind(self), config.coreSettings.IP_CHECK_INTERVAL);
    };
    
    function loadModules() {
        if (!loadedModules) {
            server.use(function (req, res, next) {
                res.header('Access-Control-Allow-Credentials', true);
                res.header('Cache-Control', "no-cache, no-store, must-revalidate");
                next();
            });
            
            server.use(restify.fullResponse());
            server.use(restify.queryParser({mapParams: false}));
            server.use(restify.bodyParser({mapParams: false}));
            
            moduleList.forEach(function (module) {
                var moduleName = (module && module.name) ? module.name : 'Unknown Module';
                if (module && module.loaded && module.loaded.attachHandlers) {
                    module.loaded.attachHandlers(self);
                } else {
                    console.log('Unable to load module: ' + moduleName);
                    process.exit(1);
                }
            });
            loadedModules = true;
        }
    }
    
    this.run = function (port) {
        var self = this;
        
        loadModules();
        
        this.setupStdout();
        this.setupStdin();
        
        server.listen(port, function () {
            config.serverInfo.IP_ADDRESSES = requestUtils.getExternalIPList();
            config.serverInfo.PORT = server.address().port;
            config.serverInfo.LOCAL_IP_ADDRESS = requestUtils.getLocalIP();
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.IP_PORT, config.serverInfo.IP_ADDRESSES + ':' + config.serverInfo.PORT);
            requestUtils.sendCommandToParentProcess(config.outgoingIPCCommands.LOCAL_IP_PORT, config.serverInfo.LOCAL_IP_ADDRESS + ':' + config.serverInfo.PORT);
        });
        
        server.on('error', function (err) {
            if (err.message === 'listen EADDRINUSE') {
                console.log('Cannot start the server because port ' + port + ' is already in use');
            } else {
                console.log('Server Error: ' + err.message);
            }
        });
        
        server.on('uncaughtException', function (req, res, next) {
            if (!req.failedWithError) {
                req.failedWithError = true;
                res.send(config.coreSettings.SERVER_ERROR_STATUS_CODE, 'Internal Server Error');
            }
        });
    };
}

module.exports = DevicePreviewServer;
