
/*	
 Copyright 2014 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
  File which has the functions to draw the VMQ UI.
*/

/*jslint vars: true, plusplus: true*/
/*global dw, attachEventListeners, removeEventListeners, VisualMediaQueryViewEventHandler, redrawRuler, $, DW_RULER_CONSTANTS, VMQ_CONSTANTS, window, document, getComputedStyle */



/* @Constructor
*/
function VisualMediaQueryView() {
    'use strict';
    /* Member variables */
    this.mediaQueryString = "";
    this.isLoading = false;
    this.visualMediaQueryViewEventObj = new VisualMediaQueryViewEventHandler();
    this.locStrings = {};
    this.zoomFactor = 1.0;
    this.minwidthList = [];
    this.maxwidthList = [];
    this.minmaxwidthList = [];
    
    this.minwidthNonPxList = [];
    this.maxwidthNonPxList = [];
    this.minmaxwidthNonPxList = [];
    this.curShowState = true;
    this.refreshPending = false; //if some UI refresh is pending..
    this.pendingMqString = ""; //Pending mq string to be processed to refresh UI.
    this.isEditable = true;    //Editable for normal pages. Non-Editable for browsing URI..
    this.showAddButton = true; //current state of add button.
    this.scrollXVal = -1; //current scroll position..
}

/*
 * Function that will init localization. All new strings needs to be added here.
 */
VisualMediaQueryView.prototype.initLocalization = function () {
    "use strict";
	var urlStr = document.location.href;
	var urlParams = (function (params) {
        var paramsObj = {};
		var i = 0;
        for (i = 0; i < params.length; ++i) {
            var temp = params[i].split('=', 2);
            if (temp[1]) {
				temp[1] = decodeURIComponent(temp[1].replace(/\+/g, " "));
			}
            paramsObj[temp[0]] = temp[1];
        }
        return paramsObj;
    }((urlStr.split('?'))[1].split('&')));
	
	var appLang = "en_US";
	if (urlParams.dwlang) {
		appLang = urlParams.dwlang;
	}
	
    this.locStrings.nomqString = window.DWVmqUtility.getLocalisedNoMQString(appLang);
};

/*
 * Function that will get the Media query JSON from DW and 
 * draw each of the divs.
 */
VisualMediaQueryView.prototype.drawVisualMediaQueryDivs = function () {
    "use strict";
    
    window.DWVmqUtility.sortMinWidthArray(this.minwidthList);
    window.DWVmqUtility.sortMaxWidthArray(this.maxwidthList);
    window.DWVmqUtility.sortMinMaxWidthArray(this.minmaxwidthList);
     
    this.drawVisualMaxWidthMediaQueryDivs();
    this.drawVisualMinMaxWidthMediaQueryDivs();
    this.drawVisualMinWidthMediaQueryDivs();
    
    this.visualMediaQueryViewEventObj.attachEventListeners(this.maxwidthList, this.minwidthList, this.minmaxwidthList);
    
    //Show and hide divs according to mq data.
    var waitingNode = document.getElementById("waiting");
    var mqcontainernode = document.getElementById("mqcontainer");
    var nomqnode = document.getElementById("nomqdiv");
             
    if ((this.minwidthList.length === 0) && (this.maxwidthList.length === 0) && (this.minmaxwidthList.length === 0)) {
        
        if (waitingNode) {
            waitingNode.style.display = "none";
        }
        
        if (nomqnode) {
            nomqnode.style.display = "block";
        }
        
        var nodemqtext = document.getElementById("nomqtext");
        if (nodemqtext) {
            nodemqtext.style.display = "block";
            var nomqVal = this.locStrings.nomqString;
            if (nomqVal.indexOf("+") !== -1) {
                nomqVal = nomqVal.replace("+", DW_RULER_CONSTANTS.plusImgTag);
            }
            nodemqtext.innerHTML = nomqVal;
        }
        
        if (mqcontainernode) {
            mqcontainernode.style.display = "none";
        }
        
    } else {
        
        if (waitingNode) {
            waitingNode.style.display = "none";
        }
        
        if (mqcontainernode) {
            mqcontainernode.style.display = "block";
        }
        
        if (nomqnode) {
            nomqnode.style.display = "none";
        }
    }

    this.visualMediaQueryViewEventObj.setSelectedState(this.maxwidthList, this.minwidthList, this.minmaxwidthList);
    this.setNoMqTextMargin();

    var argsObj = {
        callback: handleLiveViewSize
    };
    if (window.getLiveViewSize) {
        window.getLiveViewSize(argsObj);
    }

};


/*
 * Function which draws the max-width divs.
 */
VisualMediaQueryView.prototype.drawVisualMaxWidthMediaQueryDivs = function () {
    "use strict";
    var maxvmqCount = 0;
   
    var node = document.getElementById("maxwidthrow");
    if (node) {
        node.innerHTML = "";
    }
    
    while (maxvmqCount < this.maxwidthList.length) {
        
        
        var maxstr = this.maxwidthList[maxvmqCount].maxpxvalue;
        var maxres = maxstr.substring(0, maxstr.length - 2);
        
        if (window.DWVmqUtility.isNumber(maxres)) {
            
            var obj = document.createElement('div');
        
            obj.setAttribute("class", "maxwidth");
            obj.setAttribute("id", "maxwidth" + this.maxwidthList[maxvmqCount].maxpxvalue);
            obj.setAttribute("data-mediaStr", this.maxwidthList[maxvmqCount].mediaStr);
              
            obj.style.width = ((maxres * this.zoomFactor) - VMQ_CONSTANTS.adjustPixel) + "px"; //adjusting 2 pixels for border
            
            obj.title = this.maxwidthList[maxvmqCount].maxpxstr;
            if (maxvmqCount !== 0) {
                obj.style.float = "none";
            }

            var egripObj = document.createElement('div');
            egripObj.setAttribute("class", "egrip");
            obj.appendChild(egripObj);
           
            var dispString = this.maxwidthList[maxvmqCount].maxpxstr.match(/\d+([\w\W]d*)\w+/g);
           
            if (dispString) {
                //Span tag which holds the value.
                var spanRightObj = document.createElement('span');
                spanRightObj.setAttribute("class", "contentrightmaxwidth");
                spanRightObj.innerHTML = parseFloat(dispString[0]);
                spanRightObj.setAttribute("id", VMQ_CONSTANTS.rightSpanTag + this.maxwidthList[maxvmqCount].maxpxstr);
                
                //Span tag which holds the unit.
                var spanRightPxObj = document.createElement('span');
                spanRightPxObj.setAttribute("class", "contentrightmaxwidth");
                var spanRightPxObjStr = dispString[0].match(/[(a-z)|(A-Z)]+/g);
                if (spanRightPxObjStr) {
                    spanRightPxObj.innerHTML = spanRightPxObjStr[0];
                    spanRightPxObj.setAttribute("id", VMQ_CONSTANTS.rightSpanPxTag + this.maxwidthList[maxvmqCount].maxpxstr);
                    obj.appendChild(spanRightPxObj);
                }
                
                obj.appendChild(spanRightObj);
            }
            var maxwidthrowdiv = document.getElementById("maxwidthrow");
            maxwidthrowdiv.appendChild(obj);
        }
        
        maxvmqCount++;
    }
    
};

/*
 * Function which draws the min-width & max-width divs.
 */
VisualMediaQueryView.prototype.drawVisualMinMaxWidthMediaQueryDivs = function () {
    "use strict";
    
    var minmaxvmqCount = 0;
    
    var node = document.getElementById("minmaxwidthrow");
    if (node) {
        node.innerHTML = "";
    }
    
    while (minmaxvmqCount < this.minmaxwidthList.length) {
        
        var minstr = this.minmaxwidthList[minmaxvmqCount].minpxvalue;
        var minres = minstr.substring(0, minstr.length - 2);
    
        var maxstr = this.minmaxwidthList[minmaxvmqCount].maxpxvalue;
        var maxres = maxstr.substring(0, maxstr.length - 2);
    
        if (window.DWVmqUtility.isNumber(maxres) && window.DWVmqUtility.isNumber(minres)) {
            
            var obj = document.createElement('div');
            obj.setAttribute("class", "minmaxwidth");
            obj.setAttribute("id", "minmaxwidth" + this.minmaxwidthList[minmaxvmqCount].minpxvalue  + "maxvalue" +  this.minmaxwidthList[minmaxvmqCount].maxpxvalue);
            obj.setAttribute("data-mediaStr", this.minmaxwidthList[minmaxvmqCount].mediaStr);
            
            var widthres = maxres - minres;
            obj.style.width = ((widthres * this.zoomFactor) - VMQ_CONSTANTS.adjustPixel) + "px";//adjusting 2 pixels for border
            obj.style.marginLeft = (minres * this.zoomFactor)  + "px";

            //Do not display divs with max < min.
            if (widthres < 0) {
                obj.style.display = "none";
            } else if (widthres < VMQ_CONSTANTS.minWidthMQ) {
                obj.style.width = (VMQ_CONSTANTS.minRangeVal * this.zoomFactor) + "px";
            }
                
            if (minmaxvmqCount !== 0) {
                obj.style.float = "none";
            }
            obj.title = this.minmaxwidthList[minmaxvmqCount].minpxstr + ", " + this.minmaxwidthList[minmaxvmqCount].maxpxstr;

            var egripObj = document.createElement('div');
            egripObj.setAttribute("class", "egrip");
            if (widthres < VMQ_CONSTANTS.minHandleWidth) {
                egripObj.style.display = "none";
            }
            obj.appendChild(egripObj);
            
            var wgripObj = document.createElement('div');
            wgripObj.setAttribute("class", "wgrip");
            if (widthres < VMQ_CONSTANTS.minHandleWidth) {
                wgripObj.style.display = "none";
            }
            obj.appendChild(wgripObj);
            
            var dispMinString = this.minmaxwidthList[minmaxvmqCount].minpxstr.match(/\d+([\w\W]d*)\w+/g);
            if (dispMinString) {
                //Span tag which holds the value.
                var spanLeftObj = document.createElement('span');
                spanLeftObj.setAttribute("class", "contentleftminmaxwidth");
                spanLeftObj.innerHTML = parseFloat(dispMinString[0]);
                
                spanLeftObj.setAttribute("id", VMQ_CONSTANTS.rangeStr + VMQ_CONSTANTS.leftSpanTag +  this.minmaxwidthList[minmaxvmqCount].minpxstr);
                obj.appendChild(spanLeftObj);
                
                //Span tag which holds the unit.
                var spanLeftPxObj = document.createElement('span');
                spanLeftPxObj.setAttribute("class", "contentleftminmaxwidth");
                var spanLeftPxObjStr = dispMinString[0].match(/[(a-z)|(A-Z)]+/g);
                if (spanLeftPxObjStr) {
                    spanLeftPxObj.innerHTML = spanLeftPxObjStr[0];
                    spanLeftPxObj.setAttribute("id", VMQ_CONSTANTS.rangeStr + VMQ_CONSTANTS.leftSpanPxTag + this.minmaxwidthList[minmaxvmqCount].minpxstr);
                    obj.appendChild(spanLeftPxObj);
                }
                if (widthres < VMQ_CONSTANTS.minContentWidth) {
                    spanLeftObj.style.display = "none";
                    spanLeftPxObj.style.display = "none";
                }
            }
            
            var dispMaxString = this.minmaxwidthList[minmaxvmqCount].maxpxstr.match(/\d+([\w\W]d*)\w+/g);
            if (dispMaxString) {
                //Span tag which holds the value.
                var spanRightObj = document.createElement('span');
                spanRightObj.setAttribute("class", "contentrightminmaxwidth");
                spanRightObj.innerHTML = parseFloat(dispMaxString[0]);
                spanRightObj.setAttribute("id", VMQ_CONSTANTS.rangeStr + VMQ_CONSTANTS.rightSpanTag + this.minmaxwidthList[minmaxvmqCount].maxpxstr);
                
                //Span tag which holds the unit.
                var spanRightPxObj = document.createElement('span');
                spanRightPxObj.setAttribute("class", "contentrightminmaxwidth");
                var spanRightPxObjStr = dispMaxString[0].match(/[(a-z)|(A-Z)]+/g);
                if (spanRightPxObjStr) {
                    spanRightPxObj.innerHTML = spanRightPxObjStr[0];
                    spanRightPxObj.setAttribute("id", VMQ_CONSTANTS.rangeStr + VMQ_CONSTANTS.rightSpanPxTag + this.minmaxwidthList[minmaxvmqCount].maxpxstr);
                    obj.appendChild(spanRightPxObj);
                }
                
                if (widthres < VMQ_CONSTANTS.minContentWidth) {
                    spanRightObj.style.display = "none";
                    spanRightPxObj.style.display = "none";
                }
                obj.appendChild(spanRightObj);
            }
            
            var maxwidthrowdiv = document.getElementById("minmaxwidthrow");
            maxwidthrowdiv.appendChild(obj);
        
        }
        minmaxvmqCount++;
    }
};

/*
 * Function which draws the min-width divs.
 */
VisualMediaQueryView.prototype.drawVisualMinWidthMediaQueryDivs = function () {
    "use strict";
    
    var minvmqCount = 0;
    
    var node = document.getElementById("minwidthrow");
    if (node) {
        node.innerHTML = "";
    }
    
    while (minvmqCount < this.minwidthList.length) {
        
        var minstr = this.minwidthList[minvmqCount].minpxvalue;
        var minres = minstr.substring(0, minstr.length - 2);
        
        if (window.DWVmqUtility.isNumber(minres)) {
            
            var obj = document.createElement('div');
        
            obj.setAttribute("class", "minwidth");
            obj.setAttribute("id", "minwidth" + this.minwidthList[minvmqCount].minpxvalue);
            obj.setAttribute("data-mediaStr", this.minwidthList[minvmqCount].mediaStr);
            
            //Do not display divs with a large minwidth
            if (minres > VMQ_CONSTANTS.maxWidthMQ) {
                obj.style.display = "none";
            } else {
                obj.style.marginLeft = (minres * this.zoomFactor) + "px";
            }
            
            if (minvmqCount !== 0) {
                obj.style.float = "none";
            }
            obj.title = this.minwidthList[minvmqCount].minpxstr;

            var wgripObj = document.createElement('div');
            wgripObj.setAttribute("class", "wgrip");
            obj.appendChild(wgripObj);
            
            var dispMinString = this.minwidthList[minvmqCount].minpxstr.match(/\d+([\w\W]d*)\w+/g);
           
            if (dispMinString) {
                //Span tag which holds the value.
                var spanLeftObj = document.createElement('span');
                spanLeftObj.setAttribute("class", "contentleftminwidth");
                spanLeftObj.innerHTML = parseFloat(dispMinString[0]);
                spanLeftObj.setAttribute("id", VMQ_CONSTANTS.leftSpanTag +  this.minwidthList[minvmqCount].minpxstr);
                obj.appendChild(spanLeftObj);
                
                //Span tag which holds the unit.
                var spanLeftPxObj = document.createElement('span');
                spanLeftPxObj.setAttribute("class", "contentleftminwidth");
                var spanLeftPxObjStr = dispMinString[0].match(/[(a-z)|(A-Z)]+/g);
                if (spanLeftPxObjStr) {
                    spanLeftPxObj.innerHTML = spanLeftPxObjStr[0];
                    spanLeftPxObj.setAttribute("id", VMQ_CONSTANTS.leftSpanPxTag + this.minwidthList[minvmqCount].minpxstr);
                    obj.appendChild(spanLeftPxObj);
                }
            }
            
            var minwidthrowdiv = document.getElementById("minwidthrow");
            minwidthrowdiv.appendChild(obj);
        
        }
        
        minvmqCount++;
    }
};

/*
 * Handle pending refresh requests.
 */
VisualMediaQueryView.prototype.handlePendingRequests = function () {
    "use strict";
    if (this.refreshPending === true) {
        this.refreshPending = false;
        window.refreshMediaQueryView(this.pendingMqString, this.isEditable);
    }
};

/*
 * Populate min max values from Mq string for values other than px.
 * Note that for px we can directly set the width value. 
 * For other units except % pass the value to DW and get the converted value.
 */
VisualMediaQueryView.prototype.populateMinMaxValues = function () {
    "use strict";
    
    var i;
    var mqarray = [];
    var mqindex = 0;
    
    if (this.maxwidthNonPxList.length > 0 || this.minwidthNonPxList.length > 0 || this.minmaxwidthNonPxList.length > 0) {
        
        for (i = 0; i < this.maxwidthNonPxList.length; i++) {
            mqarray[mqindex++] = this.maxwidthNonPxList[i].maxpxstr;
        }

        for (i = 0; i < this.minwidthNonPxList.length; i++) {
            mqarray[mqindex++] = this.minwidthNonPxList[i].minpxstr;
        }

        for (i = 0; i < this.minmaxwidthNonPxList.length; i++) {
            mqarray[mqindex++] = this.minmaxwidthNonPxList[i].maxpxstr;
            mqarray[mqindex++] = this.minmaxwidthNonPxList[i].minpxstr;
        }

        var argObject = {};
        argObject.callback = function (value) {

            var index = 0;
            if (value) {
                if (value.length === this.maxwidthNonPxList.length + this.minwidthNonPxList.length + (2 * this.minmaxwidthNonPxList.length)) {

                    var j = this.maxwidthList.length;
                    var mqval;

                    for (i = 0; i < this.maxwidthNonPxList.length; i++) {
                        mqval = value[index++];
                        if (mqval !== VMQ_CONSTANTS.zeropxstr) {
                            this.maxwidthList[j] = this.maxwidthNonPxList[i];
                            this.maxwidthList[j].maxpxvalue = mqval;
                            j++;
                        }
                    }

                    j = this.minwidthList.length;
                    for (i = 0; i < this.minwidthNonPxList.length; i++) {
                        mqval = value[index++];
                        if (mqval !== VMQ_CONSTANTS.zeropxstr) {
                            this.minwidthList[j] = this.minwidthNonPxList[i];
                            this.minwidthList[j].minpxvalue = mqval;
                            j++;
                        }
                    }

                    j = this.minmaxwidthList.length;
                    for (i = 0; i < this.minmaxwidthNonPxList.length; i++) {
                        mqval = value[index++];
                        var minmqval = value[index++];
                        if (mqval !== VMQ_CONSTANTS.zeropxstr && minmqval !== VMQ_CONSTANTS.zeropxstr) {
                            this.minmaxwidthList[j] = this.minmaxwidthNonPxList[i];
                            this.minmaxwidthList[j].maxpxvalue = mqval;
                            this.minmaxwidthList[j].minpxvalue = minmqval;

                            j++;
                        }
                    }
                }
            }
            this.ProcessPendingReqAndUpdateView();

        }.bind(this);

        argObject.mqArray = mqarray;

        window.dwGetViewWidthFromMqText(argObject);
    
    } else {
        
        this.ProcessPendingReqAndUpdateView();
    }
    
};

/*
 * Function which draws the view and handles any pending requests if any.
 */
VisualMediaQueryView.prototype.ProcessPendingReqAndUpdateView = function () {
    "use strict";
    
    this.applyCurrentShowState(this.curShowState);
    this.visualMediaQueryViewEventObj.isEditable = this.isEditable;
    var plusButtonThrow = document.getElementById("plusButton");
    
    var minMqAddGuide = document.getElementById("minMqAddGuide");
    var maxMqAddGuide = document.getElementById("maxMqAddGuide");
    var minMaxMqAddGuide = document.getElementById("minMaxMqAddGuide");
    
    if (plusButtonThrow) {
        if (this.isEditable) {
            plusButtonThrow.style.display = 'block';
        } else {
            plusButtonThrow.style.display = 'none';
        }
    }
    
    if (minMqAddGuide && maxMqAddGuide && minMaxMqAddGuide) {
        minMqAddGuide.style.display = 'none';
        maxMqAddGuide.style.display = 'none';
        minMaxMqAddGuide.style.display = 'none';
        
    }
    this.isLoading = false;
    this.handlePendingRequests();
    
};

/* 
 * Function which parses the JSON MQ list and forms three objects
 * 1. List of only min-width values.
 * 2. List of only max-width values.
 * 3. List of both min-width and max-width values.
 
 {mediaQueryList:[{restrictor : '',mediaType : '',mediaFeatures : [{feature:'width',comparisonType:'min',value:'12px'},
																  {feature:'width',comparisonType:'max',value:'56px'}]},
				 {restrictor : '',mediaType : '',mediaFeatures : [{feature:'width',comparisonType:'max',value:'123px'}]},
				 {restrictor : '',mediaType : '',mediaFeatures : [{feature:'width',comparisonType:'min',value:'30px'}]}], 
				 errorStr:''}
                 
 {mediaQueryList:[{restrictor : '',mediaType : '',mediaFeatures : [{feature:'device-width',comparisonType:'max',value:'480px'},     
                                                                   {feature:'orientation'  ,comparisonType:'exact',value:'landscape'}]}], 
                errorStr:''}
*/

VisualMediaQueryView.prototype.constructMediaQueryLists = function (strJSON) {
    
    "use strict";
    var objJSON = eval("(function(){return " + strJSON + ";})()");
    var mediaQueryList, mediaFeatures, mediaprop, i;
    var minCounter = 0;
    var maxCounter = 0;
    var minmaxCounter = 0;
    
    this.visualMediaQueryViewEventObj.removeEventListeners(this.maxwidthList, this.minwidthList, this.minmaxwidthList);
    
    this.minwidthList = [];
    this.maxwidthList = [];
    this.minmaxwidthList = [];
    
    this.minwidthNonPxList = [];
    this.maxwidthNonPxList = [];
    this.minmaxwidthNonPxList = [];
    var minNonPxCounter = 0;
    var maxNonPxCounter = 0;
    var minmaxNonPxCounter = 0;
    
    //Using hasOwnProperty while iterating so that the inherited properties are ignored.
    if (!objJSON.errorStr) {
        
        for (mediaQueryList in objJSON) { //Iterating through mediaQueryList, errorStr. 
            
            if (objJSON.hasOwnProperty(mediaQueryList)) {
               
                var mqlist = objJSON[mediaQueryList];

                for (mediaFeatures in mqlist) { //Iterating through the list of {restrictor, mediaType, mediaFeatures.}, {restrictor, mediaType, mediaFeatures.}
                    
                    if (mqlist.hasOwnProperty(mediaFeatures)) {
                        
                        var mediaF = mqlist[mediaFeatures];
                     
                        for (mediaprop in mediaF) { //Iterating through restrictor, mediaType, mediaFeatures.
                            
                            if (mediaF.hasOwnProperty(mediaprop)) {
                                
                                var mediaFeaturesProp = mediaF[mediaprop];
                                
                                if (mediaprop === 'mediaFeatures') {
                                         
                                    var minW = 0;
                                    var maxW = 0;
                                    
                                    //Iterate through {feature:'width',comparisonType:'max',value:'123px'}
                                    for (i = 0; i < mediaFeaturesProp.length; i++) {
                                        
                                        if (mediaFeaturesProp[i].feature === 'width'  && mediaFeaturesProp[i].comparisonType === 'min') {
                                            minW = mediaFeaturesProp[i].value;
                                        }

                                        if (mediaFeaturesProp[i].feature === 'width' && mediaFeaturesProp[i].comparisonType === 'max') {
                                            maxW = mediaFeaturesProp[i].value;
                                        }
                                    }
                                    
                                    //Populate each of the arrays now.
                                    if (minW !== 0 && maxW !== 0) {
                                        
                                        if (window.DWVmqUtility.isValidVmqUnit(minW) && window.DWVmqUtility.isValidVmqUnit(maxW)) {
                                         
                                            var newminmaxobj = {};
                                            newminmaxobj.minpxstr = "(min-width : " + minW + ")";
                                            newminmaxobj.maxpxstr = "(max-width : " + maxW + ")";
                                            
                                            if (minW.match(/px/g) && maxW.match(/px/g)) {
                                                newminmaxobj.minpxvalue = minW;
                                                newminmaxobj.maxpxvalue = maxW;
                                                this.minmaxwidthList[minmaxCounter++] = newminmaxobj;
                                            } else {
                                                this.minmaxwidthNonPxList[minmaxNonPxCounter++] = newminmaxobj;
                                            }
                                            
                                        }
                                        
                                    } else if (minW !== 0) {
                                        
                                        if (window.DWVmqUtility.isValidVmqUnit(minW)) {
                                            
                                            var newminobj = {};
                                            newminobj.minpxstr = "(min-width : " + minW + ")";
                                            
                                            if (minW.match(/px/g)) {
                                                newminobj.minpxvalue = minW;
                                                this.minwidthList[minCounter++] = newminobj;
                                            
                                            } else {
                                                this.minwidthNonPxList[minNonPxCounter++] = newminobj;
                                            }
                            
                                        }
                                                                                
                                    } else if (maxW !== 0) {
                                        
                                        if (window.DWVmqUtility.isValidVmqUnit(maxW)) {
                                            
                                            var newmaxobj = {};
                                            newmaxobj.maxpxstr = "(max-width : " + maxW + ")";
                                            
                                            if (maxW.match(/px/g)) {
                                                newmaxobj.maxpxvalue = maxW;
                                                this.maxwidthList[maxCounter++] = newmaxobj;
                                            
                                            } else {
                                                this.maxwidthNonPxList[maxNonPxCounter++] = newmaxobj;
                                            }
                                        }
                                                                               
                                    } // if (maxW !== 0) 
                                    
                                }// if (mediaprop === 'mediaFeatures')
                                
                            }//if (mediaF.hasOwnProperty(mediaprop))
                            
                        }//for (mediaprop in mediaF)
                        
                    }//if (mqlist.hasOwnProperty(mediaFeatures))
                    
                }//for (mediaFeatures in mqlist)

            }//if (objJSON.hasOwnProperty(mediaQueryList)) 
            
        }//for (mediaQueryList in objJSON)
        
    } //if (!objJSON.errorStr)
    
    this.populateMinMaxValues();
};


/*
 * Function which will retreive media query JSON from a string.
 */
VisualMediaQueryView.prototype.retreiveMediaQueryJSONfromMqStr = function () {
    "use strict";
    
    var argObject = {};
    argObject.callback = function (value) {
        this.constructMediaQueryLists(value);
    }.bind(this);

    argObject.mediastr = this.mediaQueryString;
    window.dwMediaQueryListToJSON(argObject);
};


/*
 * Function which sets the MQ view.
 */
VisualMediaQueryView.prototype.setUpMediaQueryView = function (mediaStr, zoomfactor, mediaJSON, guidePos) {
    "use strict";
  
    if (zoomfactor > 5) {
        zoomfactor = 5.0;
    } else if (zoomfactor < 0.25 && zoomfactor > 0) {
        zoomfactor = 0.25;
    } else if (zoomfactor <= 0) {
        zoomfactor = 1.0;
    }
    
    this.zoomFactor = zoomfactor;
    this.mediaQueryString = mediaStr;
    if (this.zoomFactor !== 1.0) {
        redrawRuler();
    }
    this.constructMediaQueryLists(mediaJSON);
           
    this.drawMQGuide(guidePos, true);
    this.drawPlusButton(guidePos);
    this.displayAddGuides(true);
};



/*
 * Draw guides at a specified position
 */
VisualMediaQueryView.prototype.drawMQGuide = function (pos, show) {
    "use strict";
     
    if (window.DWVmqUtility.isNumber(pos)) {
        var mqguide = document.getElementById("mqguide");
        if (mqguide) {
            if (show) {
                pos = pos + "px";
                mqguide.style.marginLeft = pos;
                mqguide.style.display = "block";
            } else {
                mqguide.style.display = "none";
            }
        }
    }
};


/*
 * Check if clicked element is an MQ div.
 */
VisualMediaQueryView.prototype.isMqDiv = function (clickedObj) {
    "use strict";
    
    var i;
    for (i = 0; i < this.maxwidthList.length; i++) {
        var maxwidthrowdiv = document.getElementById("maxwidth" + this.maxwidthList[i].maxpxvalue);
        if (maxwidthrowdiv === clickedObj) {
            return true;
        }
    }
    
    for (i = 0; i < this.minwidthList.length; i++) {
        var minwidthrowdiv = document.getElementById("minwidth" + this.minwidthList[i].minpxvalue);
        if (minwidthrowdiv === clickedObj) {
            return true;
        }
    }
    
    for (i = 0; i < this.minmaxwidthList.length; i++) {
        var minmaxwidthrowdiv = document.getElementById("minmaxwidth" + this.minmaxwidthList[i].minpxvalue  + "maxvalue" +  this.minmaxwidthList[i].maxpxvalue);
        if (minmaxwidthrowdiv === clickedObj) {
            return true;
        }
    }

    return false;
};

/*
 * Align controls to the current document width.
 */
VisualMediaQueryView.prototype.AlignControls = function () {
    "use strict";
    
    var docWidth =  $(document).width();
    
    var mqcontainerdiv = document.getElementById("mqcontainer");
    if (mqcontainerdiv) {
        mqcontainerdiv.style.width = docWidth + "px";
    }

    var rulerdivelem = document.getElementById("rulerdiv");
    if (rulerdivelem) {
        rulerdivelem.style.width = docWidth + "px";
    }
    
    var nomqdivelem = document.getElementById("nomqdiv");
    if (nomqdivelem) {
        nomqdivelem.style.width = docWidth + "px";
    }
    
    this.setNoMqTextMargin();
    
    var i;
    for (i = 0; i < this.minwidthList.length; i++) {
        var minwidthrowdiv = document.getElementById("minwidth" + this.minwidthList[i].minpxvalue);
        if (minwidthrowdiv) {
            minwidthrowdiv.style.marginRight = docWidth + "px";
        }
    }

};


/*
 * Draw plusButton at a specified position
 */
VisualMediaQueryView.prototype.drawPlusButton = function (pos) {
    "use strict";
     
    if (window.DWVmqUtility.isNumber(pos)) {
        
        var plusButtonThrow = document.getElementById("plusButton");
        if (plusButtonThrow) {
            plusButtonThrow.style.marginLeft = (pos - (DW_RULER_CONSTANTS.plusButtonWidth / 2) - 1) + "px";
           
            if (this.visualMediaQueryViewEventObj && this.visualMediaQueryViewEventObj.isEditable) {
                plusButtonThrow.style.display = 'block';
            } else {
                plusButtonThrow.style.display = 'none';
            }
    
        }
       
    }
};

VisualMediaQueryView.prototype.displayAddGuides = function (show) {
    "use strict";
    
    var plusButtonThrow = document.getElementById("plusButton");
    var minMqAddGuide = document.getElementById("minMqAddGuide");
    var maxMqAddGuide = document.getElementById("maxMqAddGuide");
    var minMaxMqAddGuide = document.getElementById("minMaxMqAddGuide");
    
    if (plusButtonThrow && minMqAddGuide && maxMqAddGuide) {
        if (show && this.visualMediaQueryViewEventObj && this.visualMediaQueryViewEventObj.isEditable) {
            plusButtonThrow.style.display = 'block';
        } else {
            plusButtonThrow.style.display = 'none';
        }
        minMqAddGuide.style.display = 'none';
        maxMqAddGuide.style.display = 'none';
        minMaxMqAddGuide.style.display = 'none';
    }
};

/*
 * Show vmq bar based on the current hidden state.      
 */
VisualMediaQueryView.prototype.applyCurrentShowState = function (show) {
    "use strict";
    
    if (show) {
        this.drawVisualMediaQueryDivs();
    } else {
        var waiting = document.getElementById("waiting");
        if (waiting) {
            waiting.style.display = "none";
        }
        var mqcontainer = document.getElementById("mqcontainer");
        if (mqcontainer) {
            mqcontainer.style.display = "none";
        }
        var nomqdiv = document.getElementById("nomqdiv");
        if (nomqdiv) {
            nomqdiv.style.display = "none";
        }
    }
};


/*
 * draw guide at selected width
 */
VisualMediaQueryView.prototype.drawGuideAtSelectedWidth = function (width) {
    "use strict";
    width = parseFloat(width);
    
    if (width < 0) {
        return;
    }
    
    if (width < 20) {
        width = 20;
    }
    
    window.drawGuide(width, true, true);
    
};


/*
 * set the left margin of the no mq text span element
 */
VisualMediaQueryView.prototype.setNoMqTextMargin = function () {
    "use strict";
    
    var nomqtextelem = document.getElementById("nomqtext");
    if (nomqtextelem) {
        var hidden = $('#dummyspan');
        if (hidden) {
            hidden.html(nomqtextelem.innerHTML);
            var nomqtextouterelem = document.getElementById("nomqtextouter");
            if (nomqtextouterelem) {
                nomqtextouterelem.style.marginLeft = (($(window).width() / 2) - (hidden.width() / 2)) + "px";
            }
        }
    }
};
VisualMediaQueryView.changeTheme = function (val) {
    "use strict";
    document.getElementById("theme").className = "";
	if (val === 'dark') {
        document.getElementById("theme").classList.add("dark");
    } else if (val === 'midDark') {
        document.getElementById("theme").classList.add("midDark");
    } else if (val === 'midLight') {
        document.getElementById("theme").classList.add("midLight");
    } else {
        document.getElementById("theme").classList.add("light");
    }
};