// Copyright 2012-2013 Adobe Systems Incorporated.  All rights reserved.

//--------------------------------------------------------------------
// CLASS:
//   JQuery.DesignTime.Widget.Button
//
// DESCRIPTION:
//   This class is used by the Button property inspector to manage
//   and manipulate the design-time state of the widget's markup in the
//   design view.
//--------------------------------------------------------------------

//--------------------------------------------------------------------
// FUNCTION:
//   Button
//
// DESCRIPTION:
//	 This is the constructor method for this design time Button object.
//	 It calls the base widget constructor which helps us inherit all base methods
//	 like addClass, removeClass, ensureValidElements etc
//	 
//   It also manages the Button's widget markup in the design view. This constructor
//   is registered in the JQuery Widget translator found at:
//
//       Configuration/Translators/jQueryWidget.htm
//	 
//
// ARGUMENTS:
//   dom - object - The DOM used by the document in design view.
//   element - object - The top-level DOM element for our widget markup.
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button = function(dom, element, consArgs)
{
  JQuery.DesignTime.Widget.Base.call(this, dom, element);

	this.opts = {};
	this.opts = consArgs;
	this.init(element);
};

// JQuery.DesignTime.Widget.Button derives from our JQuery.DesignTime.Widget.Base
// class, so create a Base object and use it as our prototype so we inherit all of its
// methods.
JQuery.DesignTime.Widget.Button.prototype = new JQuery.DesignTime.Widget.Base();

// Reset the constructor property back to Button for our newly created prototype
// object so callers know that our object was created with the Button constructor.
JQuery.DesignTime.Widget.Button.prototype.constructor = JQuery.DesignTime.Widget.Button;

//--------------------------------------------------------------------
// FUNCTION:
//  	initIconValueArray
//
// DESCRIPTION:
//  	Initializes the iconValue array with predefined values.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------
JQuery.DesignTime.Widget.Button.initIconValueArray = function()
{
	var noneString = dw.loadString('jquery/widgets/button/none');
	ICON_ARRAY.push(noneString);
	ICON_ARRAY.push("ui-icon-search");
	ICON_ARRAY.push("ui-icon-wrench");
	ICON_ARRAY.push("ui-icon-gear");
	ICON_ARRAY.push("ui-icon-heart");
	ICON_ARRAY.push("ui-icon-star");
	ICON_ARRAY.push("ui-icon-link");
	ICON_ARRAY.push("ui-icon-cancel");
	ICON_ARRAY.push("ui-icon-plus");	
}

ICON_ARRAY = [];
JQuery.DesignTime.Widget.Button.initIconValueArray();

//--------------------------------------------------------------------
// FUNCTION:
//   init
//
// DESCRIPTION:
//   Initializes the design-time object's properties. This  method is
//   called from the constructor and refresh() methods.
//
// ARGUMENTS:
//   element - object - The widget's top-level DOM element.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------
JQuery.DesignTime.Widget.Button.prototype.init = function(element)
{
	this.element = this.getElement(element);
	
	this.canRefresh = true;
};

//--------------------------------------------------------------------
// FUNCTION:
//  	getLabelValue
//
// DESCRIPTION:
//  	This method returns the value of the Label attribute in the widget constructor.
//		If user changes the Label value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the Label attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.getLabelValue = function()
{
	if( this.opts && this.opts.label != null && typeof this.opts.label != 'undefined' )
	{
		return this.opts.label;
	}
	return "";
}

JQuery.DesignTime.Widget.Button.prototype.recalculateOpts = function()
{
	this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	this.opts = consArgs;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setLabelValue
//
// DESCRIPTION:
//  	When a new value for laebl is attained for the label attribute from the PI
//		we have to update our opts object with this new label value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	labelValue: The value of the label attribute that needs to be set
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.setLabelValue = function(labelValue)
{
	if( labelValue == null || typeof labelValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldLabelValue = this.opts.label
	
	if( labelValue == "" ) 
		this.opts.label = null;
	else	
		this.opts.label = labelValue;
	
	if( this.opts.label != oldLabelValue )
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getDisabledValue
//
// DESCRIPTION:
//  	This method returns the value of the disabled attribute in the widget constructor.
//		If user changes the disabled value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the disabled attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.getDisabledValue = function()
{	
	if( this.opts && this.opts.disabled != null && typeof this.opts.disabled != 'undefined' )
	{
		return (this.opts.disabled == true);
	}
	return false;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setDisabledValue
//
// DESCRIPTION:
//  	When a new value is attained for the disabled attribute from the PI
//		we have to update our opts object with this new disabled value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.setDisabledValue = function(disabledValue)
{	
	if( disabledValue == null || typeof disabledValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
		
	var oldDisabledValue = this.opts.disabled	
	if( disabledValue == true )
	{
		this.opts.disabled = true;
	}	
	else
	{
		this.opts.disabled = null;
	}
	
	if( oldDisabledValue != this.opts.disabled )
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getPrimaryIconIndex
//
// DESCRIPTION:
//  	This method will return the index of the primary list that should be displayed in
//		the PI. In case user has entered a new value in code himself, it may also
//		return the value of the newly entered number for animation
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of primary icon list or a new string value for the icon class
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.getPrimaryIconIndex = function()
{
	if( this.opts && this.opts.icons != null && typeof this.opts.icons != 'undefined' )
	{
		if( this.opts.icons.primary && typeof this.opts.icons.primary != 'undefined')
		{
			for( var i = 0; i < ICON_ARRAY.length; i++ )
			{
				if( this.opts.icons.primary.toString() == ICON_ARRAY[i] )
					return i;
			}
			if( this.opts.icons.primary.toString() != "" )
			{
				ICON_ARRAY.push(this.opts.icons.primary.toString());
				return ICON_ARRAY.length-1;
			}
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setPrimaryIconIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the primary icon
//		list.
//
// ARGUMENTS:
//  	index of primary icon list
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.setPrimaryIconIndex = function(listIndex)
{
	if( listIndex == null || typeof listIndex == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.icons)
		this.opts.icons = {};
	
	var oldPrimaryValue = this.opts.icons.primary;
	
	if( listIndex == 0 )
		this.opts.icons.primary = null;
	else
		this.opts.icons.primary = ICON_ARRAY[listIndex];	
		
	if( this.opts.icons && !this.opts.icons.primary && !this.opts.icons.secondary )
	{
		this.opts.icons = null;	
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.icons.primary != oldPrimaryValue )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setPrimaryIconValue
//
// DESCRIPTION:
//  	Here we set the opts object from the new value that user has 'typed' 
//		in the select list. We will not find this value in our list of values
//		It must be entered seperately
//
// ARGUMENTS:
//  	New value of primary icon class
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.setPrimaryIconValue = function(iconValue)
{
	if( iconValue == null || typeof iconValue == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.icons)
		this.opts.icons = {};
	
	var oldPrimaryValue = this.opts.icons.primary;
	var noneString = dw.loadString('jquery/widgets/button/none');
	
	if( iconValue == "" || iconValue == noneString )
		this.opts.icons.primary = null;
	else
		this.opts.icons.primary = iconValue;	
		
	if( this.opts.icons && !this.opts.icons.primary && !this.opts.icons.secondary )
	{
		this.opts.icons = null;	
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.icons.primary != oldPrimaryValue )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getSecondaryIconIndex
//
// DESCRIPTION:
//  	This method will return the index of the secondary list that should be displayed in
//		the PI. In case user has entered a new value in code himself, it may also
//		return the value of the newly entered number for animation
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of secondary icon list or a new string value for the icon class
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.getSecondaryIconIndex = function()
{
	if( this.opts && this.opts.icons != null && typeof this.opts.icons != 'undefined' )
	{
		if( this.opts.icons.secondary && typeof this.opts.icons.secondary != 'undefined')
		{
			for( var i = 0; i < ICON_ARRAY.length; i++ )
			{
				if( this.opts.icons.secondary.toString() == ICON_ARRAY[i] )
					return i;
			}
			if( this.opts.icons.secondary.toString() != "" )
			{
				ICON_ARRAY.push(this.opts.icons.secondary.toString());
				return ICON_ARRAY.length-1;
			}
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setSecondaryIconIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the secondary icon
//		list.
//
// ARGUMENTS:
//  	index of secondary icon list
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.setSecondaryIconIndex = function(listIndex)
{
	if( listIndex == null || typeof listIndex == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.icons)
		this.opts.icons = {};
	
	var oldSecondaryValue = this.opts.icons.secondary;
	
	if( listIndex == 0 || listIndex >= ICON_ARRAY.length )
		this.opts.icons.secondary = null;
	else
		this.opts.icons.secondary = ICON_ARRAY[listIndex];	
	
	if( this.opts.icons && !this.opts.icons.primary && !this.opts.icons.secondary )
	{
		this.opts.icons = null;
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.icons.secondary != oldSecondaryValue )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setSecondaryIconValue
//
// DESCRIPTION:
//  	Here we set the opts object from the new value that user has 'typed' 
//		in the select list. We will not find this value in our list of values
//		It must be entered seperately
//
// ARGUMENTS:
//  	New value of secondary icon class
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.setSecondaryIconValue = function(iconValue)
{
	if( iconValue == null || typeof iconValue == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.icons)
		this.opts.icons = {};
	
	var oldSecondaryValue = this.opts.icons.secondary;
	var noneString = dw.loadString('jquery/widgets/button/none');
	
	if( iconValue == "" || iconValue == noneString )
		this.opts.icons.secondary = null;
	else
		this.opts.icons.secondary = iconValue;	
		
	if( this.opts.icons && !this.opts.icons.primary && !this.opts.icons.secondary )
	{
		this.opts.icons = null;
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.icons.secondary != oldSecondaryValue )	
		this.updateOptions(this.widgetType, this.opts);	
}



//--------------------------------------------------------------------
// FUNCTION:
//   getAssets
//
// DESCRIPTION:
//   Static function that returns the assets to be applied to page
//
// ARGUMENTS:
//   None
//
// RETURNS:
//   (array of objects)
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.getAssets = function()
{
	var assets = new Array();
	
	assetObject = new Object();
	assetObject.fullPath = jQButtonImagesPath;
	assetObject.file = jQButtonImagesFile;
	assetObject.type = "";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQCoreCssPath;
	assetObject.file = jQCoreCssFile;
	assetObject.type = "link";
	assets.push(assetObject);

	assetObject = new Object();
	assetObject.fullPath = jQCssThemePath;
	assetObject.file = jQCssThemeFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQButtonCssWidgetPath;
	assetObject.file = jQButtonCssWidgetFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQMainPath;
	assetObject.file = jQMainFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQButtonJsPath;
	assetObject.file = jQButtonJsFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	return (assets);
	
};

//--------------------------------------------------------------------
// FUNCTION:
//   refresh
//
// DESCRIPTION:
//   None
//
// ARGUMENTS:
//   Syncs up the internal state of the design-time widget object
//   with the markup that currently exists in the design view.
//
//   This method gets called from the translator if a design-time
//   object already exists for a specific widget ID, and from various
//   methods in the Button property inspector.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.refresh = function()
{
  if (!this.canRefresh)
	return;
	
	if (!this.opts)
		this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	
	this.opts = consArgs; 
  this.init(this.element_id);  
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Button.getNewButtonSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string containing the
//   markup for a complete Button.
//
// ARGUMENTS:
//  id - string - The string to use as the widget's id attribute.
//
// RETURNS:
//   String that is the HTML markup fragment for the panel.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.getNewButtonSnippet = function(id)
{
  var dpSnippet = '<button id="' + id + '">Button</button>';
  return dpSnippet;
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Button.getNewButtonConstructorSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string that contains the
//   constructor snippet for creating a JQuery widget with the specified id.
//
// ARGUMENTS:
//  id - string - The id of the widget markup to insert into the
//                constructor snippet.
//
// RETURNS:
//   String containing JS that is the constructor call to create the
//   widget.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.getNewButtonConstructorSnippet = function(id)
{
  return ' $(function() {\n\t$( "#' + id + '" ).button(); \n});';
 /* $(function() {
		$( "#Button1" ).Button();
	});*/
};


//--------------------------------------------------------------------
// FUNCTION:
//  	getIconValueArray
//
// DESCRIPTION:
//  	Return the iconValue Array containing all possible values for 'icon' primary and secondary
//		attribute of the constructor object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
JQuery.DesignTime.Widget.Button.prototype.getIconValueArray = function()
{
	return ICON_ARRAY;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getTextValue
//
// DESCRIPTION:
//  	This method returns the value of the text attribute in the widget constructor.
//		If user changes the text value in the code, This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the text attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.getTextValue = function()
{	
	if( this.opts && this.opts.text != null && typeof this.opts.text != 'undefined' )
	{
		return (this.opts.text == true);
	}
	return true;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setTextValue
//
// DESCRIPTION:
//  	When a new value is attained for the text attribute from the PI
//		we have to update our opts object with this new text value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Button.prototype.setTextValue = function(textValue)
{	
	if( textValue == null || typeof textValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
		
	var oldTextValue = this.opts.text	
	if( textValue == false )
	{
		this.opts.text = false;
	}	
	else
	{
		this.opts.text = null;
	}
	
	if( oldTextValue != this.opts.text )
		this.updateOptions(this.widgetType, this.opts);
}

