/*	
 Copyright 2014 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file has the implementation of utility methods called from Text editing functionality in Live Edit view

AdobePatentID="4273US01"
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global globalController, DWTextEditUtility, DW_LIBRARY_ITEM, ENTER_POST_PROCESSING_ACTIONS, CustomEvent, DW_LIVE_TEXT_EDIT, DW_LIVEEDIT_CONSTANTS, DW_LIVEEDIT_EVENTS, DW_EVENTS, DW_ICE_HEADLIGHTS, window, Node, TextEditHud, GenericHUDObject, DW_EDITABLE_REGION_CONSTS, document,dwExtensionController, liveViewExtensionsObject, DW_EXTENSION_EVENT*/

function UtilityHelpers() {
    'use strict';

	/**
	 * hasInlineStyle
	 * check if the given style is in given element's inline style.
	 * @param elem : obj element needs to be scanned
	 * @param tag  : style stylename
	 * @Return stylevalue or null if not present.
	 */
	this.hasInlineStyle = function (obj, style) {
		if (!obj || !style) {
			return null;
		}
		var attrs = obj.getAttribute('style'), x;
		if (attrs === null) {
			return null;
		}
		if (typeof attrs === 'object') {
			attrs = attrs.cssText;
		}
		var styles = attrs.split(';');
		for (x = 0; x < styles.length; x++) {
			var attr = styles[x].split(':')[0].replace(/^\s+|\s+$/g, "").toLowerCase();
			if (attr === style) {
				return styles[x].split(':')[1];
			}
		}
		return null;
	};


	/**
	 * isInlineTextTag
	 * Check weather the give tag name is in line tag 
	 * @param tagName : Elements tag name
	 * @Return true if the tag is inline, false otherwise
	 */
	this.isInlineTextTag = function (tagName) {


		if (DW_LIVE_TEXT_EDIT.InlineTextElements.indexOf(tagName) >= 0) {
			return true;
		}
		return false;
	};


	/**
	 * wrapChildrenWithTag
	 * wrap the children of given element with the given tag name 
	 * @param elem : Current Element 
	 * @param tag  : tag with which we need to wrap children with.
	 * @Return parentElement
	 */
	this.wrapChildrenWithTag = function (elem, tag) {

		if (!elem || !tag) {
			return;
		}
		var newNode = document.createElement(tag);
		if (!newNode) {
			return;
		}
		var childNodes = elem.childNodes;
		var i;
		if (childNodes) {
			for (i = 0; i < childNodes.length; i += 1) {
				var cloneNode = childNodes[i].cloneNode(true);
				newNode.appendChild(cloneNode);
				elem.removeChild(childNodes[i]);
			}
		}
		elem.appendChild(newNode);
		return elem;
	};



	/**
	 * cleanUpContent
	 * Depending on user preference. use b/strong, i/em tags
	 * remove extra span tags attached by browser.
	 * recursive call(MUST BE BOTTOM UP because we try to remove some nodes) that starts from dw_span and goes to all children of it.
	 * @param curElement : Current Element 
	 * @to-do need to respect the preference setting from dreamweaver
	 */
	this.cleanUpContent = function (curElement) {

		//do bottom up clearning for all nodes.
		if (!curElement) {
			return;
		}
		if (curElement.nodeType === Node.ELEMENT_NODE && curElement.tagName !== DW_LIVEEDIT_CONSTANTS.TextContainer) {
			var fontStyle = null, fontWeight = null;
			if (!curElement.getAttribute(DW_LIVEEDIT_CONSTANTS.FontStyleId) && !curElement.parentNode.getAttribute(DW_LIVEEDIT_CONSTANTS.FontStyleId)) {
				fontStyle = this.hasInlineStyle(curElement, "font-style");
				if (fontStyle) {
					fontStyle = fontStyle.trim();
					if (fontStyle === 'italic') {
						curElement.style.removeProperty('font-style');
						curElement = this.wrapChildrenWithTag(curElement, "i");
					} else if (fontStyle === 'normal') {
						curElement.style.removeProperty('font-style');
					}
				}
			} else {
				fontStyle = this.hasInlineStyle(curElement, "font-style");
				if (!fontStyle) {
					var parentFont = this.hasInlineStyle(curElement.parentNode, "font-style");
					if (!parentFont) {
						curElement.style["font-style"] = 'italic';
					}
				}
			}
			if (!curElement.getAttribute(DW_LIVEEDIT_CONSTANTS.FontWeightId) && !curElement.parentNode.getAttribute(DW_LIVEEDIT_CONSTANTS.FontWeightId)) {
				fontWeight = this.hasInlineStyle(curElement, "font-weight");
				if (fontWeight) {
					fontWeight = fontWeight.trim();
					if (fontWeight === 'bold') {
						curElement.style.removeProperty('font-weight');
						curElement = this.wrapChildrenWithTag(curElement, "b");
					} else if (fontWeight === 'normal') {
						curElement.style.removeProperty('font-weight');
					}
				}
			} else {
				fontWeight = this.hasInlineStyle(curElement, "font-weight");
				if (!fontWeight) {
					var parentWeight = this.hasInlineStyle(curElement.parentNode, "font-weight");
					if (!parentWeight) {
						curElement.style["font-weight"] = 'bold';
					}
				}
			}

			if (fontStyle || fontWeight) {
				var style = curElement.getAttribute("Style");
				if (style.trim() === "") {
					curElement.removeAttribute("Style");
				}
			}
		}
		var childNodes = curElement.childNodes;
		var i;
		if (childNodes) {
			for (i = 0; i < childNodes.length; i += 1) {
				this.cleanUpContent(childNodes[i]);
			}
		}

		if (curElement.nodeType === Node.ELEMENT_NODE) {
			curElement.removeAttribute(DW_LIVEEDIT_CONSTANTS.FontStyleId);
			curElement.removeAttribute(DW_LIVEEDIT_CONSTANTS.FontWeightId);
			if ((curElement.tagName === "B") && window.parent.semanticTags) {
				//replace with strong
				this.replaceTheTagName(curElement, "STRONG");
			} else if ((curElement.tagName === "I") && window.parent.semanticTags) {
				this.replaceTheTagName(curElement, "EM");
				//replace with em
			} else if (curElement.tagName === "SPAN") {
				var isExisting = curElement.getAttribute("DW_SPAN_ID");
				if (!isExisting) {
					//remove span tag
					this.removeNodeKeepChildren(curElement);
				} else {
					curElement.removeAttribute("DW_SPAN_ID");
				}
			}
		}
	};


	/**
	 * replaceTheTagName
	 * replace the given element's tag name with the newTagName
	 * this is done mainly for the purpose of b/strong ,i/em tags.
	 * browser inserts b tag but semantically strong tag is correct.
	 * @param element : Current Element 
	 * @param newTagName : Tag name with which we need to replace.
	 */
	this.replaceTheTagName = function (element, newTagName) {

		if (!element || !newTagName) {
			return;
		}
		var elemParent = element.parentNode;
		if (!elemParent) {
			return;
		}
		var semanticElement = window.parent.document.createElement(newTagName),
			attrs = element.attributes,
			i;
		if (attrs) {
			for (i = 0; i < attrs.length; i++) {
				semanticElement.setAttribute(attrs[i].nodeName, attrs[i].nodeValue);
			}
		}
		while (element.firstChild) {
			semanticElement.appendChild(element.firstChild);
		}
		elemParent.replaceChild(semanticElement, element);
	};



	/**
	 * removeNodeKeepChildren
	 * Remove the node  and attach it's children to it's parent. 
	 * @param node : Node to be removed 
	 */
	this.removeNodeKeepChildren = function (node) {

		if (!node || !node.parentElement) {
			return;
		}
		while (node.firstChild) {
			node.parentElement.insertBefore(node.firstChild, node);
		}
		node.parentElement.removeChild(node);
	};



	/**
	 * @private removeDynamicAttribute
	 * Removes all dynamically populated attributes from element
	 * @param elem : input element
	 * attrMap: map that has static attributes of element
	 */
	this.removeDynamicAttribute = function (elem, attrMap) {


		this.logDebugMsg("removeDynamicAttribute from TextEditHud ");
		if (!elem) {
			return false;
		}

		if (elem.nodeType !== Node.ELEMENT_NODE) {
			return true;
		}

		var i,
			dwID = elem.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);

		if (dwID) {
			var map = attrMap !== null ? attrMap[dwID] : null;
			var removeList = [],
				setList = [];
			// Find all attributes that needs to be removed or retained
			for (i = 0; i < elem.attributes.length; i++) {
				var a = elem.attributes[i];
				if (a.name !== DW_LIVEEDIT_CONSTANTS.DWTempId) {
					if (map && map[a.name]) {
						setList.push(a.name);
					} else {
						removeList.push(a.name);
					}
				}
			}
			// Remove all dynamically populated attributes
			for (i = 0; i < removeList.length; i++) {
				elem.removeAttribute(removeList[i]);
			}
			if (map) {
				// Set all static attributes with original values
				for (i = 0; i < setList.length; i++) {
					elem.setAttribute(setList[i], map[setList[i]]);
				}
			}
		}

		// Remove dynamic attributes in child nodes as well.
		var childNodes = elem.childNodes;
		if (childNodes) {
			for (i = 0; i < childNodes.length; i += 1) {
				this.removeDynamicAttribute(childNodes[i], attrMap);
			}
		}
		return true;
	};



	/**
	 * @cleanUpArtifactsOfEnterKeyPress
	 * Goes over all text nodes of the given element and trims 
	 * white-spaces around each text elements.
	 * @param node : Parent element node whose child textnodes are 
	 * looked for trimming the text nodes.
	 */
	// Find first text node in the inside incoming node.
	this.cleanUpArtifactsOfEnterKeyPress = function (node) {


		if (!node) {
			return;
		}
		this.logDebugMsg("cleanUpArtifactsOfEnterKeyPress:" + node);

		this.trimWhiteSpacesArroundTextElements(node);
		this.removeDuplicateIdAttributes(node);
	};



	/**
	 * @trimWhiteSpacesArroundTextElements
	 * Goes over all text nodes of the given element and trims 
	 * white-spaces around each text elements.
	 * @param node : Parent element node whose child textnodes are 
	 * looked for trimming the text nodes.
	 */
	// Find first text node in the inside incoming node.
	this.trimWhiteSpacesArroundTextElements = function (node) {


		if (!node) {
			return;
		}

		this.logDebugMsg("trimWhiteSpacesArroundTextElements:" + node);

		if (node.nodeType === Node.TEXT_NODE && node.nodeValue && node.nodeValue.length > 1) {
			var parentHasOnlyWhitespaceAsItsContent =  node.parentNode ? DWTextEditUtility.hasOnlyWhitespaceAsContent(node.parentNode) : false;
			if (node.nodeValue.charCodeAt(0) === DW_LIVEEDIT_CONSTANTS.NBSPCharCode && !parentHasOnlyWhitespaceAsItsContent) {
				node.nodeValue = " " + node.nodeValue.substr(1, node.nodeValue.length);
			}
			if (node.nodeValue.charCodeAt(node.nodeValue.length - 1) === DW_LIVEEDIT_CONSTANTS.NBSPCharCode && !parentHasOnlyWhitespaceAsItsContent) {
				node.nodeValue =  node.nodeValue.substr(0, node.nodeValue.length - 1) + " ";
			}

			return;
		}

		if (node.nodeType === Node.ELEMENT_NODE) {
			var chidNodes = node.childNodes;

			if (chidNodes) {
				var i, tempNode = chidNodes[0];
				for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
					this.trimWhiteSpacesArroundTextElements(tempNode);
				}
			}
		}
	};



	/**
	 * @removeDuplicateIdAttributes
	 * Goes over all elements nodes and removes all duplicate ids except the first node
	 * @param node : Parent element node whose child element nodes to be looked for ID attribute
	 * looked for trimming the text nodes.
	 */
	// Find first text node in the inside incoming node.
	this.removeDuplicateIdAttributes = function (node) {


		if (!node) {
			return;
		}

		this.logDebugMsg("removeDuplicateIdAttributes:" + node);

		var chidNodes = node.childNodes;
		if (chidNodes) {
			var i = 0,
				curNode = null,
				removeIdsFromNewlyAddedNodes = false;
			for (curNode = chidNodes[i]; curNode && removeIdsFromNewlyAddedNodes === false; i += 1, curNode = chidNodes[i]) {
				if (curNode.nodeType === Node.ELEMENT_NODE && curNode.hasAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID)) {
					var curNodesIdAttribute = curNode.getAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID);
					var nodeForward = null;
					for (nodeForward = curNode.nextSibling; nodeForward; nodeForward = nodeForward.nextSibling) {
						if (nodeForward.nodeType === Node.ELEMENT_NODE && nodeForward.hasAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID) && (nodeForward.getAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID) === curNodesIdAttribute)) {
							nodeForward.removeAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID);
						}
					}

					if (nodeForward === null) {
						removeIdsFromNewlyAddedNodes = true;
					}
				}
			}
		}
	};


	/**
	 * @removeUnwantedBRTag
	 * When you delete all text from editing session, <br> tag gets added automatically. This function is to clean that.
	 * @param node :dwSpanElement inside which <br> tag should be looked at.
	 */
	this.removeUnwantedBRTag = function (dwSpanElement) {

		if (!dwSpanElement) {
			return;
		}

		// When user selects all the text and deletes it, browser inserts <br>. 
		// If the text element which is under editing is only child for parent,
		// then we inserts &nbsp; to keep the tag editable in ELV.
		if (dwSpanElement.parentNode.firstChild === dwSpanElement.parentNode.lastChild) {
			if (dwSpanElement.innerHTML === "<br>" || dwSpanElement.innerHTML.length === 0) {
				dwSpanElement.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
			}

			if ((dwSpanElement.parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TableDataTagName ||
				 dwSpanElement.parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TableHeaderTagName) &&
					(dwSpanElement.innerHTML === "<br>" || dwSpanElement.innerHTML === DW_LIVEEDIT_CONSTANTS.NonBreakingSpace)) {
				dwSpanElement.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
			}
		} else if (dwSpanElement.innerHTML === "<br>") {
			// #bug=3731578
			// We reach here when we select the complete content of hanging text, delete and commit the changes.
			// In this case of hanging text under edit, dwspan will not be only the child of its parent. 
			// The siblings could be new line elements or other valid tags itself.

			// we don't add nbsp; here, we delete the complete content.
			dwSpanElement.innerHTML = "";
		}

		// if single block child and its content is just '<br>', replace it
		// with "&nbsp;".
		var firstChild =  dwSpanElement.firstChild;
		if (firstChild && firstChild.nodeType === Node.ELEMENT_NODE &&
				firstChild === dwSpanElement.lastChild &&
				DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(firstChild.tagName) >= 0 &&
				firstChild.innerHTML === "<br>") {
			firstChild.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
		}

		// Still we have only <br>, remove it.
		if (dwSpanElement.innerHTML === "<br>") {
			dwSpanElement.innerHTML = "";
		}
	};


	/**
	 * @getTextNodeFromClick
	 * Get the text node that is under the mouse click 
	 * @param event : mouse event from which element need to be figured out.
	 */
	this.getTextNodeFromClick = function (event) {

		if (!event) {
			return null;
		}

		// Check if clicked element is text
		// As event.targetNode is always block level element, we need to
		// use display rect to findow out exact text element clicked.
		var range,
			x = event.clientX,
			y = event.clientY,
			elementOnClick = document.elementFromPoint(x, y);

		if (elementOnClick && elementOnClick.childNodes) {
			var i,
				foundTextNode = false,
				chidNodes = elementOnClick.childNodes,
				node = chidNodes[0];

			range = document.createRange();

			for (i = 0; node; i += 1, node = chidNodes[i]) {
				if (node.nodeType === Node.TEXT_NODE) {
					range.selectNode(node);
					var j,
						rects = range.getClientRects(),
						rect = rects[0];

					for (j = 0; rect; j += 1, rect = rects[j]) {
						if (x > rect.left && x < rect.right && y > rect.top && y < rect.bottom) {
							elementOnClick = node;
							foundTextNode = true;
							break;
						}
					}
				}
				if (foundTextNode) {
					break;
				}
			}
		}

		// If we found text node, set it
		if (elementOnClick.nodeType === Node.TEXT_NODE) {
			return elementOnClick;
		}

		return null;
	};


	/**
	 * @isIgnorableText
	 * checks whether given text node is dummy text which can be ignored during sibling scanning
	 * @param node : text node inside which white-spaces are looked at.
	 */
	this.isIgnorableText = function (node) {


		if (!node || node.nodeType !== Node.TEXT_NODE) {
			return false;
		}

		// Is it ignorable Text?
		var i,
			ignorable = true,
			str = node.nodeValue,
			len = node.nodeValue.length;

		for (i = 0; i < len; i += 1) {
			if (DW_LIVE_TEXT_EDIT.WhiteSpaceChars.indexOf(str.charCodeAt(i)) < 0) {
				ignorable = false;
				break;
			}
		}

		return ignorable;
	};

	/**
	 * @IsTextContainerDirectlyAroundText
	 * returns true when the text container (dwpsan) directly around the text
	 * @param element : dwpsan element
	 */
	this.IsTextContainerDirectlyAroundText = function (element) {
		if (!element) {
			return false;
		}

		this.logDebugMsg("function begin : TextEditHud : IsDwSpanDirectlyAroundText" + element);

		if (element.firstChild && element.firstChild.nodeType === Node.ELEMENT_NODE) {

			// If the grouping is block level, dwspan will always be around the tags (from the list 
			// BlockInclusionList), return false in this case.
			if (DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(element.firstChild.tagName) >= 0) {
				return false;
			}

			// If there was only &nbsp; in the container tags which are part of TagsThatCreatesPTagsOnEnter
			// return false;
			if (DW_LIVE_TEXT_EDIT.TagsThatCreatesPTagsOnEnter.indexOf(element.firstChild.tagName) >= 0) {
				return false;
			}
		}

		// if we don't fall under any of above two cases, dwspan will be directly around the text,
		return true;
	};

	/**
	 * @findAnchorNodeParentForGivenNode
	 * returns anchor node reference if the incoming node is child has an anchor node
	 * @param node : text node inside which anchor tags are looked at.
	 */
	this.findAnchorNodeParentForGivenNode = function (node) {


		if (!node) {
			return null;
		}

		this.logDebugMsg("function begin :  findAnchorNodeParentForGivenNode:" + node);

		//Check if the incoming node is child has an anchor node.
		var anchorNode = null,
			nodeIter = node;
		while (nodeIter && nodeIter.tagName !== 'BODY') {
			if (nodeIter.tagName === 'A') {
				anchorNode = nodeIter;
				break;
			}
			nodeIter = nodeIter.parentNode;
		}

		//Check if the incoming node has direct and only child which is anchor.
		if (anchorNode === null) {
			if (node.firstChild && node.firstChild.tagName === 'A' && node.lastChild &&  node.firstChild === node.lastChild) {
				anchorNode = node.firstChild;
			}
		}

		return anchorNode;
	};



	/**
	 * @getTextIdentifierString
	 * Streams the textIdentifier properties as string and returns it.  Used for debugging purpose.
	 * @param textIdentifier : textIdentifier object
	 */
	this.getTextIdentifierString = function (textIdentifier) {


		if (!textIdentifier) {
			return DW_LIVEEDIT_CONSTANTS.none;
		}

		var str = "";
		str += textIdentifier.nodeType;
		if (textIdentifier.nodeType === Node.TEXT_NODE) {
			str += " - ";
			str += textIdentifier.parentNodeID;
			str += " - ";
			str += textIdentifier.preNodeID;
			str += " - ";
			str += textIdentifier.succNodeID;
			str += " - ";
			str += textIdentifier.noOfBeforeTexts;
			str += " - ";
			str += textIdentifier.noOfAfterTexts;
			str += " - ";
			str += textIdentifier.editableRegion;
		} else {
			str += " - ";
			str += textIdentifier.dwId;
		}

		return str;
	};



	/**
	 * @isInLibraryItem
	 * isInLibraryItem - checks whether the node is inside of Library Item.
	 * @param nodeobj : node which need to be checked for library items.
	 */
	this.isInLibraryItem = function (nodeobj) {


		this.logDebugMsg("function begin :  isInLibraryItem");

		if (!nodeobj) {
			return false;
		}

		var inLibrary = false,
			currNode = nodeobj;
		while (currNode) {
			if (currNode.nodeType === Node.COMMENT_NODE && currNode.nodeValue.indexOf(DW_LIBRARY_ITEM.Begin) === 0) {
				inLibrary = true;
				break;
			}

			if (currNode.nodeType === Node.COMMENT_NODE && currNode.nodeValue.indexOf(DW_LIBRARY_ITEM.End) === 0) {
				break;
			}

			if (currNode.previousSibling) {
				currNode = currNode.previousSibling;
			} else {
				currNode = currNode.parentNode;
			}
		}

		return inLibrary;
	};



	/**
	 * @isFragmentHasLiteralSpaceAtBeginning
	 * true if the incoming fragment starts with literal space characters
	 *		   : false if the fragment starts with any non-literal space (nbsp, br, newline or any other alphanumeric chars)
	 * @param fragment : fragment inside which spaces are looked at.
	 */
	this.isFragmentHasLiteralSpaceAtBeginning = function (fragment) {


		if (!fragment) {
			return false;
		}

		this.logDebugMsg("function begin :  isFragmentHasLiteralSpaceAtBeginning" + fragment);

		var elementChildNodes = fragment.childNodes;
		if (elementChildNodes && elementChildNodes.length >= 1) {
			var nodeValueOfFirstChild = null;
			if (elementChildNodes[0].nodeType === Node.TEXT_NODE) {
				nodeValueOfFirstChild = elementChildNodes[0].nodeValue;
			} else if (elementChildNodes[0].nodeType === Node.ELEMENT_NODE) {
				nodeValueOfFirstChild = elementChildNodes[0].innerText;
			}

			if (nodeValueOfFirstChild && nodeValueOfFirstChild.charCodeAt(0) === DW_LIVEEDIT_CONSTANTS.SpaceKeyCode) {
				return true;
			}
		}

		return false;
	};



	/**
	 * @hasSingleElementWhoseContentIsWhitespace
	 * This function takes the fragment as input argument and checks its has single tag and the tag contains only white space
	*  as its content. White space may includes any number of new line, br tags and nbsps;
	 * @param fragment : fragment inside which spaces are looked at.
	 */
	this.hasSingleElementWhoseContentIsWhitespace = function (fragment) {


		if (!fragment) {
			return false;
		}

		this.logDebugMsg("function begin :  hasSingleElementWhoseContentIsWhitespace" + fragment);

		var elementChildNodes = fragment.childNodes;
		if (elementChildNodes && elementChildNodes.length === 1) {
			return this.hasOnlyWhitespaceAsContent(elementChildNodes[0]);
		}

		return false;
	};



	/**
	 * @hasOnlyWhitespaceAsContent
	 * This function takes the element as input argument and checks that element contains only white space
	 * as its content. White space may includes any number of new line, br tags and nbsps;
	 * @param fragment : fragment inside which spaces are looked at.
	 */
	this.hasOnlyWhitespaceAsContent = function (element) {


		if (!element) {
			return false;
		}

		this.logDebugMsg("function begin :  hasOnlyWhitespaceAsContent" + element);

		var nodeValueOfElement = null;
		if (element.nodeType === Node.TEXT_NODE) {
			nodeValueOfElement = element.nodeValue;
		} else if (element.nodeType === Node.ELEMENT_NODE) {
			nodeValueOfElement = element.innerText;
		}


		if (nodeValueOfElement) {
			nodeValueOfElement = nodeValueOfElement.trim();
			if (nodeValueOfElement.length === 0) {
				return true;
			}
		}

		return false;
	};



	/**
	 * @isFragmentHasOnlyOneBRTagAsItsContent
	 * This function takes the element as input argument and checks that element contains only <br> tag
	 * as its content. White space may includes any number of new line, br tags and nbsps;
	 * @param fragment : fragment inside <br> tags are looked at.
	 */
	this.isFragmentHasOnlyOneBRTagAsItsContent = function (fragment) {


		if (!fragment) {
			return false;
		}

		this.logDebugMsg("function begin :  isFragmentHasOnlyOneBRTagAsItsContent" + fragment);

		var elementChildNodes = fragment.childNodes,
			elementInnerHTML = null;
		if (elementChildNodes.length === 1) {
			var nodeValueOfFirstChild = null;
			if (elementChildNodes[0].nodeType === Node.TEXT_NODE) {
				nodeValueOfFirstChild = elementChildNodes[0].nodeValue;
			} else if (elementChildNodes[0].nodeType === Node.ELEMENT_NODE) {
				nodeValueOfFirstChild = elementChildNodes[0].innerText;
				elementInnerHTML = elementChildNodes[0].innerHTML;
			}

			if (nodeValueOfFirstChild) {
				// TODO - handle the case when fragment directly contains <br>
				if (nodeValueOfFirstChild.length === 1 && elementInnerHTML === "<br>") {
					return true;
				}
			}
		}

		return false;
	};



	/**
	 * @isCaretAtBeginningOfTextNode
	 * Identifies whether the ibeam is at the beginning of the incoming text node.
	 * returns true if the ibeam is at the beginning and false otherwise.
	 * @param fragment : fragment inside caret position is looked at.
	 */
	this.isCaretAtBeginningOfTextNode = function (currentTextNode) {


		if (!currentTextNode) {
			return false;
		}

		this.logDebugMsg("function begin :  isCaretAtBeginningOfTextNode :" + currentTextNode);

		var at_start = false,
			range = window.getSelection().getRangeAt(0),
			pre_range = document.createRange();

		if (!range) {
			return false;
		}

		pre_range.selectNodeContents(currentTextNode);
		pre_range.setEnd(range.startContainer, range.startOffset);

		var prior_text = pre_range.cloneContents();

		if (!prior_text) {
			return false;
		}

		// If the prior_text.textContent has literal space characters, they are not rendered in browser preview, hence we need to ignore
		// while calculating caret position.

		// If the prior_text.textContent has non-breaking space (&nbsp;), they are rendered in preview, hence we should not ignore them.
		var doTrim = this.isFragmentHasLiteralSpaceAtBeginning(prior_text);

		if (doTrim) {
			prior_text.textContent = prior_text.textContent.trim();
		}

		// If the text's length is 0, we're at the start of the text node.
		at_start = prior_text.textContent.length === 0;

		return at_start;
	};



	/**
	 * @isCaretAtEndOfTextNode
	 * Identifies whether the ibeam is at the end of the incoming text node.
	 * returns true if the ibeam is at the end and false otherwise.
	 * @param fragment : ffragment inside caret position is looked at.
	 */
	this.isCaretAtEndOfTextNode = function (currentTextNode) {


		if (!currentTextNode) {
			return false;
		}

		this.logDebugMsg("function begin :  isCaretAtEndOfTextNode:" + currentTextNode);

		var at_end = false,
			range = window.getSelection().getRangeAt(0),
			post_range = document.createRange();

		if (!range) {
			return false;
		}

		post_range.selectNodeContents(currentTextNode);
		post_range.setStart(range.endContainer, range.endOffset);
		var next_text = post_range.cloneContents();
		next_text.textContent = next_text.textContent.trim();

		// If the text's length is 0, we're at the end of the text node.
		at_end = next_text.textContent.length === 0;
		return at_end;
	};


	/**
	 * @getFirstBlockLevelParent
	 * Utility method - the returns the first block level parent in the hierarchy for given text node.
	 * @param node : node whose first block level parent is searched for.
	 */
	this.getElementUnderIP = function () {
		this.logDebugMsg("function begin getElementUnderIP");

		if (!document.getSelection()) {
			return;
		}

		var node = document.getSelection().anchorNode;
		if (!node) {
			return;
		}

		var nodeActual =  (node.nodeType === Node.TEXT_NODE ? node.parentNode : node);
		if (!nodeActual) {
			return;
		}

		this.logDebugMsg("function begin getElementUnderIP" + nodeActual);
		return nodeActual;
	};


	/**
	 * @getFirstBlockLevelParent
	 * Utility method - the returns the first block level parent in the hierarchy for given text node.
	 * @param node : node whose first block level parent is searched for.
	 */
	this.getFirstBlockLevelParent = function (node) {


		if (!node || node.nodeType !== Node.ELEMENT_NODE) {
			return;
		}

		this.logDebugMsg("function begin :  getFirstBlockLevelParent:" + node);

		var nodeIter = node,
			blockLevelParent = null;
		while (nodeIter) {
			if (DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(nodeIter.tagName) >= 0 || nodeIter.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
				break;
			} else {
				nodeIter = nodeIter.parentNode;
			}
		}

		if (nodeIter) {
			blockLevelParent = nodeIter;
		}

		return blockLevelParent;
	};



	/**
	 * @elementContainTheTag
	 * Utility method - the check whether a given Tag as child?
	 * @param node : node whose first block level parent is searched for.
	 */
	this.elementContainTheTag = function (node, tagName) {

		if (node && node.nodeType === Node.ELEMENT_NODE) {
			if (node.tagName === tagName) {
				return true;
			}

			var chidNodes = node.childNodes;
			if (chidNodes) {
				var i, tempNode = chidNodes[0];
				for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
					if (this.elementContainTheTag(tempNode, tagName)) {
						return true;
					}
				}
			}
		}
		return false;
	};



	/**
	 * @insertHtmlAtCaretPosition
	 * Utility method that inserts incoming 'html' at the cursor point. Currently this is called for adding <br> tag on Shift+Enter key press.
	 * @param html : html which is currently getting inserted in.
	 */
	this.insertHtmlAtCaretPosition = function (html) {


		if (!html) {
			return;
		}

		this.logDebugMsg("function begin :  insertHtmlAtCaretPosition:" + html);

		var sel = window.getSelection(),
			range;

		if (!sel) {
			return;
		}

		if (sel.getRangeAt && sel.rangeCount) {
			range = sel.getRangeAt(0);
			range.deleteContents();

			//Create a temp div to contain the incoming html
			var el = document.createElement("div");
			el.innerHTML = html;

			//Create a temp fragment and insert at the range
			//TODO - currently this function is used for inserting <br> tag, that's why we are interested 
			// only in first child. Improve this if this method needs to work for multiple nodes html.
			var frag = document.createDocumentFragment(),
				node = el.firstChild,
				lastNode;

			if (node) {
				lastNode = frag.appendChild(node);
			}

			// Insert the fragment which contains incoming html at the range.
			range.insertNode(frag);

			// Set the selection to the end of newly added html
			if (lastNode) {
				range = range.cloneRange();
				range.setStartAfter(lastNode);
				range.collapse(true);
				sel.removeAllRanges();
				sel.addRange(range);
			}
		}
	};


	/**
	 * @findAnyTextNode
	 * Utility method - Find first text node in the inside incoming node.
	 * @param node : element node inside which text node is searched for.
	 */
	this.findAnyTextNode = function (node) {


		if (!node) {
			return null;
		}

		if (node.nodeType === Node.TEXT_NODE && !this.isIgnorableText(node) && !(node.nodeValue.length === 1 && node.nodeValue.charCodeAt(0) === DW_LIVEEDIT_CONSTANTS.NBSPCharCode)) {
			return node;
		}

		if (node.nodeType === Node.ELEMENT_NODE) {
			var chidNodes = node.childNodes,
				textNode = null;

			if (chidNodes) {
				var i, tempNode = chidNodes[0];
				for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
					textNode = this.findAnyTextNode(tempNode);
					if (textNode) {
						return textNode;
					}
				}
			}
		}

		return null;
	};


	/**
	 * @findAnyTextNode
	 * Utility method - Find any <br> node in the inside incoming node.
	 * @param node : element node inside which text node is searched for.
	 */
	this.findAnyBRNode = function (node) {


		if (!node || node.nodeType !== Node.ELEMENT_NODE) {
			return null;
		}

		if (node.tagName === "BR") {
			return node;
		}

		var chidNodes = node.childNodes,
			brNode = null;

		if (chidNodes) {
			var i, tempNode = chidNodes[0];
			for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
				brNode = this.findAnyBRNode(tempNode);
				if (brNode) {
					return brNode;
				}
			}
		}

		return null;
	};


	/**
	 * @findAnyTextNode
	 * Utility method - Find last text node in the inside incoming node.
	 * @param node : element node inside which text node is searched for.
	 */
	this.findLastTextNode = function (node) {


		if (!node) {
			return null;
		}

		if (node.nodeType === Node.TEXT_NODE) {
			return node;
		}

		if (node.nodeType === Node.ELEMENT_NODE) {
			var lastChild = node.lastChild,
				textNode = null;

			while (lastChild) {
				textNode = this.findLastTextNode(lastChild);

				if (textNode) {
					break;
				}

				lastChild = lastChild.previousSibling;
			}

			return textNode;
		}

		return null;
	};


	/**
	 * @getCharacterOffsetWithin
	 * Utility method - returns the offsets from start of the text node till the current caret position.
	 * @param element : element node inside which the offset is computed.
	 */
	this.getCharacterOffsetWithin = function (element) {

		this.logDebugMsg("function begin :  getCharacterOffsetWithin");
		var caretOffset = 0,
			selection = window.getSelection();

		if (selection && element) {
			var preCaretRange = selection.getRangeAt(0),
				range = preCaretRange.cloneRange();

			range.selectNodeContents(element);
			range.setEnd(preCaretRange.endContainer, preCaretRange.endOffset);

			caretOffset = range.toString().length;
		}

		return caretOffset;
	};
    
    /**
	 * @dontPropogateWhiteSpace
	 * dontPropogateWhiteSpace - During Live edit of certain elements in carousel's heading and caption, the space keydown was propogated to the parents which ignored whitespaces. So, using this function will prevent propogation.
     * Ref WatsonBug# 3986399
	 */
    
    this.dontPropogateWhiteSpace = function (evt) {
        
        if (evt.keyCode === DW_LIVEEDIT_CONSTANTS.SpaceKeyCode) {
            evt.stopPropagation();
        }
    };

	/**
	 * @addContentEditableAttribute
	 * addContentEditableAttribute - adds content editable attribute to the input element &
	 *								disables the default UI of content editable attribute
	 * @param element : element for which the content editable flag is added.
	 */
	this.addContentEditableAttribute = function (element) {
        
        this.logDebugMsg("function begin :  addContentEditableAttribute");
		if (element) {
			element.setAttribute("contenteditable", "true");
            //adding this to prevent propogation of space event for bootstrap's carousel element
            element.addEventListener("keydown", this.dontPropogateWhiteSpace, true);
			// spl handling for PRE tag. In case we know more cases we can make it an array and traverse
			if (element.parentNode && element.parentNode.tagName === "PRE") {
				element.style.display = "inline-block";
			}
		} else {
			this.logDebugMsg("error :  addContentEditableAttribute - failed");
		}
	};



	/**
	 * @removeContentEditableAttribute
	 * removeContentEditableAttribute - removes content editable attribute to the input element &
									removes the default UI of content editable attribute
	 * @param element : element for which the content editable flag is removed.
	 */
	this.removeContentEditableAttribute = function (element) {


		this.logDebugMsg("function begin :  removeContentEditableAttribute");

		if (element) {
			element.removeAttribute("contenteditable");
            element.removeEventListener("keydown", this.dontPropogateWhiteSpace, true);
			element.removeAttribute("style");
			element.blur();
		} else {
			this.logDebugMsg("error :  removeContentEditableAttribute - failed");
		}
	};


	/**
	 * @logDebugMsg
	 * logDebugMsg - utility method that logs given log string to log file.
	 * @param message : message which need to be logged.
	 */
	this.logDebugMsg = function (message) {
		if (!message || message === "") {
			return;
		}

		if (globalController) {
			globalController.logDebugMsg(message);
		}
	};


	/**
	 * @logHeadlightsData
	 * logHeadlightsData - removes content editable attribute to the input element &
									removes the default UI of content editable attribute
	 * @param section : section in the headlight database.
	 * @param message : message in the headlight database.
	 */
	this.logHeadlightsData = function (section, message) {


		if (!section || !message) {
			return;
		}

		if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
			window.liveViewExtensionsObject.dwObject.logHeadlightsData(section, message);
		}
	};

	/**
	 * isEditableElement
	 *   Check weather the give node is editable. That means static element.
	 *   As well, it checks whether all the child elments also static element (editable). 
	 * @param node : Incomming node.
	 * @Return true if the node is static element, false if it contains any dynamic element.
	 */
	this.isEditableElement = function (node) {
		if (!node) {
			return false;
		}

		if (node.nodeType === Node.TEXT_NODE) {
			// Not worrying about the text node as it needs to be
			// identified always relative to Element node.
			return true;
		}

		if (node.nodeType !== Node.ELEMENT_NODE) {
			// We don't allow editing for nodes that 
			// are not of type ELEMENT_NODE or TEXT_NODE.
			return false;
		}

		// If no DW unique ID, it is not editable.
		if (!node.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId)) {
			return false;
		}

		// Make sure all the children are editable.
		var i, childNodes = node.childNodes;
		if (childNodes) {
			for (i = 0; i < childNodes.length; i += 1) {
				if (!this.isEditableElement(childNodes[i])) {
					return false;
				}
			}
		}

		// All is verified. It should be editable now.
		return true;
	};
}

/**
 * @Utility object declaration.
 * Global utility object that will be available to every one from the point this file evaled.
 */
window.DWTextEditUtility = new UtilityHelpers();