/*	
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file has the implementation of Template Instance Editable Region Hilighting functionality in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global globalController, DW_LIVE_TEXT_EDIT, DW_LIVEEDIT_CONSTANTS, DW_LIVEEDIT_EVENTS, window, Node, TextEditHud, GenericHUDObject, DW_EDITABLE_REGION_CONSTS, document, EditableRegion */

EditableRegion.prototype = new GenericHUDObject();
EditableRegion.prototype.constructor = EditableRegion;
EditableRegion.prototype.baseClass = GenericHUDObject.prototype.constructor;

function EditableRegion(liveEditEvt) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : constructor");
    this.setHudName(DW_LIVEEDIT_CONSTANTS.HudEditableRegion);
    this.initialize(liveEditEvt);

    // We need to listen both single click and double click to hide EditableRgion HUD for
    // the region that is under selection.
    this.m_liveEditEvent.addListener(DW_LIVEEDIT_EVENTS.SingleClick, this.enableHud, this);
    this.m_liveEditEvent.addListener(DW_LIVEEDIT_EVENTS.DoubleClick, this.enableHud, this);
    this.m_liveEditEvent.addListener(DW_LIVEEDIT_EVENTS.POSTEscape, this.showAllER, this);

    // When changes happens from other HUD / Text Editing, we need to reposition all the region
    // HUDs that follows the current region.
    this.m_liveEditEvent.addListener(DW_LIVEEDIT_EVENTS.ElementChanged, this.repositionAllER, this);

    // Other huds will send this event when they cancel editing to bring back ER.
    this.m_liveEditEvent.addListener(DW_LIVEEDIT_EVENTS.EditOpFailed, this.ResetCurrentER, this);

    // Other huds will send this event when they enter into editing.
    this.m_liveEditEvent.addListener(DW_LIVEEDIT_EVENTS.EditingText, this.HideAllER, this);

    // All the region details and current editing region index
    this.m_editableRegions = null;
    this.m_currentRegionIndex = -1;
	this.m_allERHidden = false;
	this.m_isERPostionsDirty = false;
	this.m_allERHiddenForELVDisable = false;

    // Cache for remembering x, y coordinates of click and corresponding region index
    this.m_cache = {
        x: -1,
        y: -1,
        index: -1
    };
}

/*
Overrided functions from base class
*/
EditableRegion.prototype.commit = function () {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : commit");

    if (this.getVisibility() === false) {
        return;
    }

    GenericHUDObject.prototype.commit.call(this);

};
EditableRegion.prototype.draw = function () {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : draw");

    if (this.getVisibility() === false) {
        return;
    }

    GenericHUDObject.prototype.draw.call(this);

	if (!globalController.m_liveEditDisabled) {

		if (this.m_allERHiddenForELVDisable) {
			this.m_allERHiddenForELVDisable = false;
			this.showAllER();
		}

	    // Just hide the ER HUD for the region under Edit.
		if (this.m_currentRegionIndex !== -1 && this.m_editableRegions && this.m_currentRegionIndex < this.m_editableRegions.length) {
			var region = this.m_editableRegions[this.m_currentRegionIndex];

			//hide overlay
			region.overlayDiv.style.display = 'none';

			//hide HUD
			region.hudDiv.style.display = 'none';
		}
	}
};
EditableRegion.prototype.destroy = function (args) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : destroy");

	if (globalController.m_liveEditDisabled) {
		this.m_allERHiddenForELVDisable = true;
		this.HideAllER();
	}

    if (this.getVisibility() === false) {
        return;
    }

	if (!globalController.m_liveEditDisabled) {
		// Reset current editing region and bring back the overlay and ER HUD.
		this.ResetCurrentER();

		if (this.m_allERHidden && (!args || args.click !== DW_LIVEEDIT_EVENTS.DoubleClick)) {
			this.showAllER();
		}
	}

    GenericHUDObject.prototype.destroy.call(this);

};
EditableRegion.prototype.escapeKeyPressed = function () {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : escapeKeyPressed");

    // Reset current editing region and bring back the overlay and ER HUD.
    this.ResetCurrentER();
};
EditableRegion.prototype.setCurrentElement = function (args) {
    'use strict';

    // Validate arguments.
    this.logDebugMsg("function begin : EditableRegion : setCurrentElement");

    // Set / reset member's state.
    this.m_currentRegionIndex = -1;

    if (!(args && args.element && args.event)) {
        return;
    }

    // Get index from cache that we populated in isClickInEditableRegion
    if (this.m_cache && this.m_cache.x === args.event.clientX && this.m_cache.y === args.event.clientY) {
        this.m_currentRegionIndex = this.m_cache.index;
    } else {
        this.m_currentRegionIndex = this.getRegionForPoint(args.event.clientX, args.event.clientY);
    }

    this.logDebugMsg("function END : EditableRegion : setCurrentElement");
};
/*
End of overrided functions from base class
*/

//getEditableRegionForElement - returns region index if element is under ER or returns false
EditableRegion.prototype.getEditableRegionForElement = function (element) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : getEditableRegionForElement");
    if (!element) {
        return -1;
    }

    var curNode = element,
		erBegin = null;
    while (curNode) {
        if (curNode.nodeType === Node.COMMENT_NODE) {
            // if we reach ER End first, then element is outside of ER.
            if (curNode.nodeValue.indexOf(DW_EDITABLE_REGION_CONSTS.EditableRegionEnd) === 0) {
                break;
            }

            // if we reach ER Begin first, then element is inside ER.
            if (curNode.nodeValue.indexOf(DW_EDITABLE_REGION_CONSTS.EditableRegionBegin) === 0) {
                erBegin = curNode;
				break;
            }
        }

        // If no previous sibling, then take the parent and search
        if (curNode.previousSibling) {
            curNode = curNode.previousSibling;
        } else {
            curNode = curNode.parentNode;
        }
    }

	if (erBegin) {
		var i,
			regionIndex = -1,
			length = this.m_editableRegions.length;

		for (i = 0; i < length; i += 1) {
			if (this.m_editableRegions[i].begin === erBegin) {
				regionIndex = i;
				break;
			}
		}

		return regionIndex;
	}

    // if we didn't find any ER markers, then return -1.
    return -1;
};

//isClickInEditableRegion - returns true if click event target is under ER or returns false
EditableRegion.prototype.isClickInEditableRegion = function (event, clicktype) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : isClickInEditableRegion");

    if (!this.m_editableRegions || !event || !clicktype) {
        return false;
    }

    var clickRgnIndex = this.getRegionForPoint(event.clientX, event.clientY),
		targetRgnIndex = this.getEditableRegionForElement(event.target);

	if (clickRgnIndex === -1) {
		clickRgnIndex = targetRgnIndex;
	}

    // remember the index for coordinates
    this.m_cache = {
        x: event.clientX,
        y: event.clientY,
        index: clickRgnIndex
    };

    // If it is single click, then check for the event target (block level element)
    // is comming under the editable region.
    if (clicktype === DW_LIVEEDIT_EVENTS.SingleClick) {
        return (targetRgnIndex !== -1);
    }

    this.logDebugMsg("EditableRegion : Click is in Region index - " + clickRgnIndex);

    return (clickRgnIndex !== -1);
};

// HideAllER - hides all the ER HUD and Overlays.
EditableRegion.prototype.HideAllER = function () {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : HideAllER");

	// Set ER hidden flag
	this.m_allERHidden = true;

    // Hide all regions 
    if (this.m_editableRegions) {
        var i;
        for (i = 0; i < this.m_editableRegions.length; i += 1) {
            var region = this.m_editableRegions[i];

            //Hide overlay
            region.overlayDiv.style.display = 'none';

            //Hide HUD
            region.hudDiv.style.display = 'none';
        }
    }
};

// repositionAllER - enables dirty flag before showing ERs so that postions are recalcuated again.
EditableRegion.prototype.repositionAllER = function (args) {
	'use strict';
	this.logDebugMsg("function begin : EditableRegion : repositionAllER");

	this.m_isERPostionsDirty = true;
	if (!this.m_allERHidden) {
		this.showAllER(args);
	}
};

// showAllER - show all the regions.
EditableRegion.prototype.showAllER = function (args) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : showAllER");

	if (globalController.m_liveEditDisabled) {
		return;
	}

    if (this.m_editableRegions) {

        // Reposition all regions
        var i;
        for (i = 0; i < this.m_editableRegions.length; i += 1) {
            var region = this.m_editableRegions[i];

            // reset rect so that position will be recalculated
			if (this.m_isERPostionsDirty) {
				region.mergedRect = null;
			}

            if (i === this.m_currentRegionIndex && args && args.element) {
                // Reset the rect for current region.
                this.refreshRegionBoundry(this.m_editableRegions[this.m_currentRegionIndex], args.element);
            } else {
                // show overlay recalculates the position
                this.showEROverLay(region);
                this.showERHud(region);
            }
        }

		this.m_isERPostionsDirty = false;
    }

	// Reset ER hidden flag
	this.m_allERHidden = false;
};

// ResetCurrentER - reset the state (makes visible) of the current ER.
EditableRegion.prototype.ResetCurrentER = function () {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : ResetCurrentER");

    // Bring back the ER HUD and overlay
    if (this.m_currentRegionIndex !== -1 && this.m_editableRegions && this.m_currentRegionIndex < this.m_editableRegions.length) {
        var region = this.m_editableRegions[this.m_currentRegionIndex];

        this.showEROverLay(region);
        this.showERHud(region);
    }

    // Reset index
    this.m_currentRegionIndex = -1;
};

// initER - Initializes the ER regions. It scans the complete source for 
// ER markes and creates the regions object. It is invoked from Controller.
EditableRegion.prototype.initER = function () {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : initER");

    // CSS is initialized? (see LiveEditBridgingObject.js)
    if (!window.liveEditableRegionCss) {
        this.logDebugMsg("EditableRegion HUD: CSS content not found");
        return;
    }

    //Create style tag for HUD. 
    var style = document.createElement('style');
    style.type = 'text/css';
    style.innerHTML = window.liveEditableRegionCss;
    this.m_hudContainer.appendChild(style);

    // Find all editable regions and initialize overlay for all
    this.logDebugMsg("Find and Prepare HUD for EditableRegions");
    var result = this.findAllEditableRegions(null);
    if (result && !result.error && result.regions) {
        this.m_editableRegions = result.regions;
        this.createHudAndOverlayDivs(result.regions);
        this.highlightEditableRegions(result.regions);
    }
    this.logDebugMsg("function end : EditableRegion : initER");
};

// highlightEditableRegions - makes all the regions visible.
EditableRegion.prototype.highlightEditableRegions = function (regions) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : highlightEditableRegions");
    if (!regions) {
        return;
    }

    // Show all region's HUD and Overlay
    var i;
    for (i = 0; i < regions.length; i += 1) {
        this.showEROverLay(regions[i]);
        this.showERHud(regions[i]);
    }
};

// createHudAndOverlayDivs - creates and initilializes the overlay Div and
// HUD Div for all the regions.
EditableRegion.prototype.createHudAndOverlayDivs = function (regions) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : createHudAndOverlayDivs");
    if (!regions) {
        return;
    }

    var i;
    for (i = 0; i < regions.length; i += 1) {
        var region = regions[i];

        //add HUD div
        region.hudDiv = document.createElement(DW_LIVEEDIT_CONSTANTS.ContainerDiv);
        region.hudDiv.setAttribute('class', DW_EDITABLE_REGION_CONSTS.EditableRegionHudCss);
        region.hudDiv.innerHTML = this.getEditableRegionAttributeVal(region.begin, DW_EDITABLE_REGION_CONSTS.EditableRgnName);
        this.m_hudContainer.appendChild(region.hudDiv);

        // add Overlay div
        region.overlayDiv = document.createElement(DW_LIVEEDIT_CONSTANTS.ContainerDiv);
        region.overlayDiv.setAttribute('class', DW_EDITABLE_REGION_CONSTS.EditableRgnOverlayCss);
        this.m_hudContainer.appendChild(region.overlayDiv);
    }
};

// showEROverLay - makes the ER HUD visible for given region.
EditableRegion.prototype.showEROverLay = function (region) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : showEROverLay");

    if (!region) {
        return;
    }

    //display overlay
    region.overlayDiv.style.display = 'block';

    // position overlay
    this.positionOverlay(region);
};

// showERHud - makes the ER HUD visible for given region.
EditableRegion.prototype.showERHud = function (region) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : showERHud");

    if (!region) {
        return;
    }

    //display HUD and then position it
    region.hudDiv.style.display = 'block';

    //position it on overlay
    this.positionHud(region);
};

// positionOverlay - positions the overly DIV for given region.
EditableRegion.prototype.positionOverlay = function (region) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : positionOverlay");

    if (!region || !region.overlayDiv) {
        this.logDebugMsg("EditableRegion : positionOverlay - Error in region init");
        return;
    }

    var rect = this.getEditableRegionRect(region);
    if (!rect) {
        this.logDebugMsg("EditableRegion : positionOverlay - overlay rect is missing");
        region.overlayDiv.style.display = 'none';
        return;
    }

    if (rect.width <= 0 || rect.height <= 0) {
        this.logDebugMsg("EditableRegion Overlay: region height or width is zero or not set");
        region.overlayDiv.style.display = 'none';
        return;
    }

    region.overlayDiv.style.top = rect.top + 'px';
    region.overlayDiv.style.left = rect.left + 'px';

    //we should reduce the width and height to accomodate the border
    //of the overlay div. so reduce 2*border_width	
    region.overlayDiv.style.width = (rect.width - 2 * DW_LIVEEDIT_CONSTANTS.OverlayBorderWidth) + 'px';
    region.overlayDiv.style.height = (rect.height - 2 * DW_LIVEEDIT_CONSTANTS.OverlayBorderWidth) + 'px';
};

// positionHud - postions the ER HUD over the given region's overlay div. 
EditableRegion.prototype.positionHud = function (region) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : positionHud");

    if (!region || !region.hudDiv) {
        this.logDebugMsg("EditableRegion : positionHud - Error in region init");
        return;
    }

    var rect = this.getEditableRegionRect(region);
    if (!rect) {
        this.logDebugMsg("EditableRegion : positionHud - overlay rect is missing");
        region.hudDiv.style.display = 'none';
        return;
    }

    if (rect.width <= 0 || rect.height <= 0) {
        this.logDebugMsg("EditableRegion HUD: region height or width is zero or not set");
        region.hudDiv.style.display = 'none';
        return;
    }

    //if there is no space at top, keep the HUD at bottom
    var hudTop = rect.top - region.hudDiv.offsetHeight;
    if (hudTop < 0) {
        hudTop = rect.top + rect.height;
    }

    //if there is no space on right, move left
    var hudLeft = rect.left;
    if (hudLeft + region.hudDiv.offsetWidth > document.documentElement.offsetWidth) {
        hudLeft = rect.left + rect.width - region.hudDiv.offsetWidth;
    }

    region.hudDiv.style.top = hudTop + 'px';
    region.hudDiv.style.left = hudLeft + 'px';
};

//findEditableRegionEnd - finds the end node of the ER for given starting node.
EditableRegion.prototype.findEditableRegionEnd = function (startingChild) {
    'use strict';

    if (!startingChild) {
        return null;
    }

    // The End marker of editable region must be of sibling to this node. 
    var curNode = startingChild;
    while (curNode) {
        if (curNode.nodeType === Node.COMMENT_NODE) {
            if (curNode.nodeValue.indexOf(DW_EDITABLE_REGION_CONSTS.EditableRegionEnd) === 0) {
                return curNode;
            }
        }
        curNode = curNode.nextSibling;
    }

    return null;
};

//findAllEditableRegions - finds all the editable region that are under
// the given node. If node is null, then document's body will be taken.
EditableRegion.prototype.findAllEditableRegions = function (node) {
    'use strict';

    //this.logDebugMsg("function begin : EditableRegion : findAllEditableRegions");

    // If node is not passed, search entire body for editable regions.
    if (!node) {
        node = document.getElementsByTagName("body")[0];
    }

    // if no node or node type is not element, return from here.
    if (!node || node.nodeType !== Node.ELEMENT_NODE) {
        return {
            error: false,
            regions: null
        };
    }

    // Find editable regions
    var regions = [],
        curNode = node.firstChild;
    while (curNode) {
        if (curNode.nodeType === Node.COMMENT_NODE && curNode.nodeValue.indexOf(DW_EDITABLE_REGION_CONSTS.EditableRegionBegin) === 0) {
            // Found the begining marker of editable region. The end marker
            // should be sibling to this node. Find it.
            var regionBegin = curNode;
            var regionEnd = this.findEditableRegionEnd(curNode.nextSibling);

            // If no end marker, then it is malformed template insatance
            // and return error.
            if (!regionEnd) {
                return {
                    error: true,
                    regions: null
                };
            }

            // Push the region to array and continue.
            regions.push({
                begin: regionBegin,
                end: regionEnd
            });

            curNode = regionEnd.nextSibling;
        } else {
            // If this is element then search in its children.
            if (curNode.nodeType === Node.ELEMENT_NODE) {
                var result = this.findAllEditableRegions(curNode);

                if (!result || result.error) {
                    return {
                        error: true,
                        regions: null
                    };
                }

                // concat editable regions
                regions.push.apply(regions, result.regions);
            }

            curNode = curNode.nextSibling;
        }
    }

    if (regions.length > 0) {
        return {
            error: false,
            regions: regions
        };
    }

    return {
        error: false,
        regions: null
    };
};

// refreshRegionBoundry - refresh the region values and it's boundry using affected element.
EditableRegion.prototype.refreshRegionBoundry = function (region, affectedElement) {
    'use strict';

    if (!region || !affectedElement) {
        return;
    }

    var curNode = affectedElement.previousSibling,
        foundRegion = false;
    while (curNode) {
        if (curNode.nodeType === Node.COMMENT_NODE) {
            var value = curNode.nodeValue;
            if (value.indexOf(DW_EDITABLE_REGION_CONSTS.EditableRegionBegin) === 0) {
                // Found the begining marker of editable region. The end marker
                // should be sibling to this node. Find it.
                var regionBegin = curNode,
                    regionEnd = this.findEditableRegionEnd(curNode.nextSibling);

                // Error
                if (!regionEnd) {
                    return;
                }

                foundRegion = true;

                // Reset region boundry.
                region.begin = regionBegin;
                region.end = regionEnd;

                break;
            }
        }

        curNode = curNode.previousSibling;
    }

    if (!foundRegion) {
        this.refreshRegionBoundry(region, affectedElement.parentNode);
    }
};

// getEditableRegionRect - gives the rect object of given ER region object
EditableRegion.prototype.getEditableRegionRect = function (region) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : getEditableRegionRect");

	if (!region) {
		return null;
	}

    if (region.mergedRect) {
        return region.mergedRect;
    }

    // Calculate rect
    var range = document.createRange();
    range.setStartAfter(region.begin);
    range.setEndBefore(region.end);

    // Set the region rect
    var rect = range.getBoundingClientRect();
	
	// rect is read-only object, we have to create new object
	// to do scroll postion adjustment.
	var resultRect = {};
	resultRect.top = rect.top + window.scrollY;
    resultRect.left = rect.left + window.scrollX;
    resultRect.width = rect.width;
    resultRect.height = rect.height;
	
	region.mergedRect = resultRect;
   
    return resultRect;
};

// getRegionForPoint - checks the given point with each ER's region rectangel
// and returns the matching ER.
EditableRegion.prototype.getRegionForPoint = function (x, y) {
    'use strict';

    this.logDebugMsg("function begin : EditableRegion : getRegionForPoint");

    if (!this.m_editableRegions || !x || !y) {
        return -1;
    }

    var i,
		regionIndex = -1,
		lenght = this.m_editableRegions.length;

    for (i = 0; i < lenght && regionIndex === -1; i += 1) {
        var region = this.m_editableRegions[i],
            range = document.createRange();

        range.setStartAfter(region.begin);
        range.setEndBefore(region.end);

        var j, rect, rects = range.getClientRects();
        for (j = 0; j < rects.length; j += 1) {
            rect = rects[j];
            if (!rect) {
                break;
            }

            if (x > rect.left && x < rect.right && y > rect.top && y < rect.bottom) {
                regionIndex = i;
                break;
            }
        }
    }

    return regionIndex;
};
