/*
Copyright 2013 Adobe Systems Incorporated.  All rights reserved. 

Purpose:

LiveView Briding base class for All HUDS
Things that are common across the HUDS needs to be added here.

AdobePatentID="4273US01"
*/
/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global dw, DWfile, DWLE_FILEPATHS, window, DW_LIVEEDIT_CONSTANTS, globalController, Node*/

function GenericLiveEditBridger() {
    'use strict';
    this.m_documentDOM = null;
}
// Checks if the object is equal to {}
GenericLiveEditBridger.prototype.isEmpty = function (obj) {
    for(var prop in obj) {
        if(obj.hasOwnProperty(prop))
            return false;
    }
    return true;
};
// utility method to print any nested object 
GenericLiveEditBridger.prototype.printObj = function ( o) 
{
    var str='';

    for(var p in o){
        if(typeof o[p] == 'string'){
            str+= p + ': ' + o[p]+';';
        }else{
            str+= p + ': { ' + this.printObj(o[p]) + '}';
        }
    }

    return str;
};
GenericLiveEditBridger.prototype.setDocumentDOM = function (documentDOM) {
    'use strict';
    this.m_documentDOM = documentDOM;
};

GenericLiveEditBridger.prototype.getDocumentDOM = function () {
    'use strict';
    this.logDebugMsg("GenericLiveEditBridger.prototype.getDocumentDOM() : documentDOM = " + this.m_documentDOM);
    return this.m_documentDOM;
};

GenericLiveEditBridger.prototype.logDebugMsg = function (content) {
    'use strict';
	if (DW_LIVEEDIT_DEBUGGING === true) {
        dw.logDebugMsg(content);
	}
};

/*
function:clearLiveHighlight
(Internal function on spider monkey side)
Arguments:
        none
Description: 
Clear any live Highligthts ( Reverse Inspect ) which might be there in Live View 
*/
GenericLiveEditBridger.prototype.clearLiveHighlight = function () {
    'use strict';
    dw.clearLiveHighlight();
};

GenericLiveEditBridger.prototype.invokeNFWHint = function (argObj) {
    'use strict';
    if (argObj) {
        dw.invokeNFWHint(argObj.type, argObj.left, argObj.top, argObj.width, argObj.height, argObj.feature);
   }
};

/*
function:getElementByDWId
(Internal function on spider monkey side)
Arguments:
    
    dwid - Value of data_liveedit_tagid attribute that we searching for.

Description: 
Search element by dreamweaver data attribute (data_liveedit_tagid) in the document.
*/

GenericLiveEditBridger.prototype.getElementByDWId = function (dwid) {
    'use strict';
    if (!dwid) {
        return null;
    }

    var element = null;
    var theDOM = this.getDocumentDOM();
    if (theDOM) {
        element = theDOM.getElementByDwId(dwid);
    }

    return element;
};


/*
function:DWSMLaunchHelp

Arguments:
    
    id - id of the url to be opened as help

Return Obj : Nothing

Description: 
    to update an element's attribute value.
*/

GenericLiveEditBridger.prototype.DWSMLaunchHelp = function (argObj) {
    'use strict';
    this.logDebugMsg("GenericLiveEditBridger  DWSMLaunchHelp");
    dw.openHelpURL(argObj.id);
};

/*
function:updateDWDocElementAttr

Arguments:
    
    uniqId - Value of data_liveedit_tagid attribute that we searching for
    attrName - what attribute needs to be changed
    attrValue - new value of the attribute

Return Obj : Nothing

Description: 
    to update an element's attribute value.
*/

GenericLiveEditBridger.prototype.updateDWDocElementAttr = function (argObj) {
    'use strict';
    this.logDebugMsg("updateDWDocElementAttr from GenericLiveEditBridger");

    var elementId = argObj.uniqId;
    var attrName = argObj.attrName;
    var attrVal = argObj.attrValue;

    if (elementId && elementId.length && attrName && attrName.length) {
        var userDOMElement = this.getElementByDWId(elementId);

        if (userDOMElement) {

            if (attrName.toLowerCase() === DW_LIVEEDIT_CONSTANTS.AltTagName || (attrVal && attrVal.length)) {
                userDOMElement.setAttribute(attrName, attrVal);
            } else {
                userDOMElement.removeAttribute(attrName);
            }
        } else {
            this.logDebugMsg("ERROR: updateDWDocElementAttr: Can not find the User DOM element");
        }
    } else {
        this.logDebugMsg("ERROR: updateDWDocElementAttr: arguments not Valid");
    }
    if (argObj.callback) {
        argObj.callback();
    }

    if (argObj.selectionId) {
        var selElement = this.getElementByDWId(argObj.selectionId);
        if (selElement) {
            var theDOM = this.getDocumentDOM();
            var offsets = theDOM.nodeToOffsets(selElement);
            theDOM.setSelection(offsets[0], offsets[1]);
        }
    }
};
/*
 Wrapper written for TextEditing alone to watch out for the cases where the parent is too big to process (example body with 500 children)
 Depending on Previous sibling and next sibling
*/
GenericLiveEditBridger.prototype.populateNewIDObjectInfoWrapperForTextEdit = function (prevSibId,nextSibId, newIdObj) {
    'use strict';
    this.logDebugMsg("populateNewIDObjectInfoWrapperForTextEdit prevSibId ");
    var prevTagNode = null, succTagNode = null, currentNode = null;
    if (prevSibId !== "") {
        prevTagNode = this.getElementByDWId(prevSibId);
    } 
    if (nextSibId !== "") {
        succTagNode = this.getElementByDWId(nextSibId);
    } 
    if(prevTagNode !== null) {
        var tagIterPrev = prevTagNode.nextSibling;
        while (tagIterPrev && tagIterPrev !== succTagNode) {
            this.populateNewIDObjectInfo(tagIterPrev,newIdObj);
            tagIterPrev = tagIterPrev.nextSibling;
        }
       
     } else if (succTagNode !== null) {
        var tagIterNext = succTagNode.previousSibling;
        while (tagIterNext  && tagIterNext  !== prevTagNode) {
            this.populateNewIDObjectInfo(tagIterNext,newIdObj);
            tagIterNext = tagIterNext.previousSibling;
        }
    }
};

/*
Populate new data_liveedit_tagid values for all temp ids
Args:
        node - node to start with.
        newIDObj - object which needs to be populated with new ID info.
*/
GenericLiveEditBridger.prototype.populateNewIDObjectInfo = function (node, newIdObj) {
    'use strict';
    this.logDebugMsg("populateNewIDObjectInfo from GenericLiveEditBridger");

    if (!node) {
        return;
    }
    if (node.nodeType !== Node.ELEMENT_NODE) {
        return;
    }
    if (node[DW_LIVEEDIT_CONSTANTS.DWTempId]) {

        this.logDebugMsg("populateNewIDObjectInfo from GenericLiveEditBridger" + node[DW_LIVEEDIT_CONSTANTS.DWTempId] + "   " + node[DW_LIVEEDIT_CONSTANTS.DWUniqueId]);
        newIdObj[node.data_temp_dwid] = node[DW_LIVEEDIT_CONSTANTS.DWUniqueId];
        node.removeAttribute(DW_LIVEEDIT_CONSTANTS.DWTempId);
        node.removeAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
    }

    var childNodes = node.childNodes;
    var i;
    for (i = 0; i < childNodes.length; i += 1) {
        this.populateNewIDObjectInfo(childNodes[i], newIdObj);
    }
};

/*
Checks whether the given element is Editable Region Comment Node or not.
Args:
        element - input element 
*/
GenericLiveEditBridger.prototype.isElementERComment = function (element) {
	'use strict';

	if (!element) {
		return false;
	}

	var theDOM = this.m_documentDOM,
		regions = theDOM.getEditableRegionList(true),
		regionIndex = theDOM.getSelectedEditableRegion(true),
		isERComment = false;

	if (regions && regionIndex < regions.length && regionIndex >= 0) {
		isERComment = (element === regions[regionIndex]);
	}

	// Sometimes, current selected editable region is not set due to 
	// selection sync issue between Live View and DW. In this case we have to 
	// search through all the editable regions to find the one that needs 
	// to be updated.
	if (!isERComment && regions) {
		var i;
		for (i = 0; i < regions.length; i += 1) {
			if (regions[i] === element) {
				isERComment = true;
				break;
			}
		}
	}

	this.logDebugMsg("isElementERComment from GenericLiveEditBridger : " + isERComment);

	return isERComment;
};

/*
Get all data_liveedit_tagid values using temp ID info.
Args:
        idInfoObj - info of parent node to start with and 
            call back function of Live View.
*/
GenericLiveEditBridger.prototype.getDWIDsForTempIDs = function (idInfoObj,isTextEdit) {

    'use strict';
    this.logDebugMsg("getDWIDsForTempIDs from GenericLiveEditBridger Parent ID:" + idInfoObj.parentId);

    var parentNode = this.getElementByDWId(idInfoObj.parentId),
		theDOM = this.getDocumentDOM();

	// Default to <body> or document if parentNode is not specified.
	// It happens for dynamic pages like PHP where <body>
	// itself is created dynamically.
	if (!parentNode && theDOM) {
		parentNode = theDOM.getElementsByTagName('body')[0];
		if (!parentNode) {
			parentNode = theDOM;
		}
	}

    if (parentNode) {
        var newIdObj = {};
        if (isTextEdit === true && (idInfoObj.prevSibId !== "" || idInfoObj.nextSibId !== "")) {
            this.populateNewIDObjectInfoWrapperForTextEdit(idInfoObj.prevSibId,idInfoObj.nextSibId, newIdObj);
        } else {
            this.populateNewIDObjectInfo(parentNode, newIdObj);
        }
        newIdObj.parentId = idInfoObj.parentId;
		newIdObj.parentSwapped = false;

		if (idInfoObj.lastTempId && !newIdObj[idInfoObj.lastTempId]) {
			this.logDebugMsg("getDWIDsForTempIDs - ERROR");
			throw DW_LIVEEDIT_CONSTANTS.ParentNotMatching;
		}
		// Incase of empty UL or OL (with empty LI), DW re-creates whole parent we have to update parent ID
		if (idInfoObj.parentSwapped) {
			this.logDebugMsg("getDWIDsForTempIDs from GenericLiveEditBridger Prent Swapped");
			newIdObj.oldParentId = idInfoObj.oldParentId;
			newIdObj.parentSwapped = true;
		}

        idInfoObj.callback(newIdObj);
    } else {
		this.logDebugMsg("getDWIDsForTempIDs no parent- ERROR");
		throw DW_LIVEEDIT_CONSTANTS.ParentNotMatching;
	}
};

GenericLiveEditBridger.prototype.DWLogHeadlightsData = function (argObj) {
    'use strict';
    if (argObj && argObj.subCategory && argObj.eventString) {
        if (argObj.category) {
            dw.logEvent(argObj.category, argObj.subCategory, argObj.eventString);
        } else {
            dw.logEvent(argObj.subCategory, argObj.eventString);
        }
    }
};

GenericLiveEditBridger.prototype.DWLogPartialRefreshFailed = function (argObj) {
    'use strict';
    dw.logPartialRefreshFailed();
};

/*
Checks whether there is any temp ID left out in Titan DOM (due to failure) .
Args:
        node - root node from where to start searching.
*/
GenericLiveEditBridger.prototype.isDWTempIDLeftInDom = function (node) {
	'use strict';

	if (!node) {
        return false;
    }

	if (node[DW_LIVEEDIT_CONSTANTS.DWTempId]) {
		return true;
    }

    var childNodes = node.childNodes,
		i,
		result = false;

	if (childNodes) {
		for (i = 0; i < childNodes.length; i += 1) {
			result = this.isDWTempIDLeftInDom(childNodes[i]);
			if (result) {
				break;
			}
		}
	}

	return result;
};

