/*	
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file has the implementation of for Dreamweaver side utility functions which are exposed to extensions.

AdobePatentID="4273US01"
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global GenericLiveEditBridger,dw, DW_LIVEEDIT_CONSTANTS, DWfile */

function ExtensionsBridgeFunctions(documentDOM) {
    'use strict';
    this.logDebugMsg("creating Extension Bridger");
    this.setDocumentDOM(documentDOM);
}

ExtensionsBridgeFunctions.prototype = new GenericLiveEditBridger();
ExtensionsBridgeFunctions.prototype.constructor = ExtensionsBridgeFunctions;
ExtensionsBridgeFunctions.prototype.baseClass = GenericLiveEditBridger.prototype.constructor;

/*
Function: setResponsiveFeatures - Declares the TitanDoc to contain a supported responsive document and sets the features supported on it

Argument Obj Structure:
    argObj.frameworkName -> Responsive Framework detected
    argObj.supportedFeatures -> List of features supported for the framework

Return Object : None
*/
ExtensionsBridgeFunctions.prototype.setResponsiveFeatures = function (argsObj) {
    'use strict';
    var theDOM = this.getDocumentDOM(),
        i,
        supportedFeature = []; // create a new array since Array passed through JSBridging seems to be rejected.
    for (i = 0; i < argsObj.supportedFeatures.length; i++) {
        supportedFeature.push(argsObj.supportedFeatures[i]);
    }
    if (theDOM) {
        theDOM.setResponsiveFeatures(argsObj.frameworkName, supportedFeature);
    }
};

ExtensionsBridgeFunctions.prototype.readFile = function (argObj) {
    'use strict';
    
    var fileStr;
	if (argObj) {
        var newCallbackObj = {};
        if (DWfile && DWfile.exists(argObj.fileName)) {
            fileStr = DWfile.read(argObj.fileName);
            newCallbackObj.fileStr = fileStr;
        } else {
            this.logDebugMsg("in readFile: file doesn't exist " + argObj.fileName);
        }
        if (argObj.callback) {
            argObj.callback(newCallbackObj);
        }
    }
};

/*
Function: getDWDocumentClassesAndIds - Gets the list of available classes and Ids for use in this document
Argument Obj Structure:
argObj.callback -> callback to be called after executing

Return Object : 
allSelectors - array of available selectors
bAppend - whether to append to existing list or not

Get User Dom Elements Classes and Id's for the passed element
*/
ExtensionsBridgeFunctions.prototype.getDWDocumentClassesAndIds = function (functionCallbackObj) {

    'use strict';
    var allSelectors = null;
    var theDOM = this.getDocumentDOM();
    if (theDOM) {
        allSelectors = theDOM.getSelectorsDefinedInStylesheet('id');
        if (allSelectors) {

            // Remove elements whose ids are being used in the DOM
            var nodeList = theDOM.getElementsByAttributeName('id');
            if (nodeList) {
                var i;
                for (i = 0; i < nodeList.length; i++) {
                    var nodeId = nodeList[i].getAttribute('id');
                    if (nodeId) {
                        var index = allSelectors.indexOf('#' + nodeId);
                        if (index !== -1) {
                            allSelectors.splice(index, 1);
                        }
                    }
                }
            }

            // concat class elements
            allSelectors = allSelectors.concat(theDOM.getSelectorsDefinedInStylesheet('class'));
        } else {
            // get class elements
            allSelectors = theDOM.getSelectorsDefinedInStylesheet('class');
        }
    } else {
        this.logDebugMsg("ERROR: getDWDocumentClassesAndIds: Can not find document DOM");
    }

    if (functionCallbackObj && functionCallbackObj.callback) {
        if (allSelectors.length > DW_LIVEEDIT_CONSTANTS.MaxArraySizeForBridging) {
            //we cannot send a large number of selectors via bridging, 
            //so send in chunks
            var k, tempArray, numSelectors = allSelectors.length, chunkSize = DW_LIVEEDIT_CONSTANTS.MaxArraySizeForBridging;
            for (k = 0; k < numSelectors; k += chunkSize) {
                var endIndex = k + chunkSize;
                if (endIndex > numSelectors) {
                    endIndex = numSelectors;
                }
                tempArray = allSelectors.slice(k, endIndex);
                functionCallbackObj.callback(tempArray, (k === 0 ? false : true));
            }
        } else {
            functionCallbackObj.callback(allSelectors, false);
        }
    } else {
        this.logDebugMsg("ERROR: getDWDocumentClassesAndIds: Invalid callback provided");
    }
};

/*
Function: checkIfFluidElement
Returns true if the selected element is Fluid in an FG page    
*/

ExtensionsBridgeFunctions.prototype.checkIfFluidElement = function (argObj) {

    'use strict';
    var theDOM = this.getDocumentDOM();
    var returnObj = {};
    if (theDOM && theDOM.isFluidGridDoc() && theDOM.getIsElementFluid()) {
        returnObj.isFluid = true;
    } else {
        returnObj.isFluid = false;
    }
    if (argObj.callbackFunc) {
        argObj.callbackFunc(returnObj);
    }
};

/*
Function: isCCLibLinkedImage
Returns: The state of Linked Asset if the selected img element is a linked asset from CC Library    
*/

ExtensionsBridgeFunctions.prototype.isCCLibLinkedImage = function (argObj) {
    'use strict';
    
    var returnObj = {};
    if (dw.ccLib && argObj && argObj.fileName) {
        returnObj.assetState = dw.ccLib.getLinkedAssetState(argObj.fileName);
    }
    if (argObj.callbackFunc) {
        argObj.callbackFunc(returnObj);
    }
};

/*
Function: isAssetOptInProgress
Returns: This method returns if asset optimization is in process   
*/

ExtensionsBridgeFunctions.prototype.isAssetOptInProgress = function (argObj) {
    'use strict';
    
    var returnObj = {};
    if (dw.ccLib && argObj && argObj.fileName) {
        returnObj.assetOptState = dw.ccLib.isAssetOptInProgress(argObj.fileName);
    }
    if (argObj.callbackFunc) {
        argObj.callbackFunc(returnObj);
    }
};

/*
Function: handleCloudIconClick
Argument is the image src of the linked asset from CCLibraries.
           Invoked on clicking the cloud icon from ESH.
*/

ExtensionsBridgeFunctions.prototype.handleCloudIconClick = function (argObj) {
    'use strict';
    
    if (dw.ccLib && argObj && argObj.fileName) {
        dw.ccLib.handleCloudIconClick(argObj.fileName, argObj.linkedState);
    }
};

/*
Function: getUserDOMElementClassAndID
Argument Obj Structure:
argObj.uniqId -> dwid of element
argObj.callback -> callback to be called after executing

Return Object : 
newCallbackOb.idStr -> id of the element 
newCallbackOb.classStr ->  classes of the element

Get User Dom Elements Classes and Id's for the passed element
*/
ExtensionsBridgeFunctions.prototype.getDWDocumentElementClassAndID = function (argObj) {

    'use strict';
    this.logDebugMsg("getDWDocumentElementClassAndID from ExtensionsBridgeFunctions");

	var userDOMElement = null;
	var classStr;
	var idStr;
	if (argObj) {
		var uniqId = argObj.uniqId;
		if (uniqId && uniqId.length > 0) {
			userDOMElement = this.getElementByDWId(uniqId);
        }

		if (userDOMElement) {
			classStr = userDOMElement.getAttribute("class");
			idStr = userDOMElement.getAttribute("id");
		}
		var newCallbackObj = {};
		newCallbackObj.idStr = idStr;
		newCallbackObj.classStr = classStr;

		if (argObj.callback) {
			argObj.callback(newCallbackObj);
        }
	}
};

/*
Function: DWSMSetSelectionOnNode
Utility function to set selection on a node

Arguments : node

Return value : True for success, false for Failure 

Set Selection on a given node
*/
ExtensionsBridgeFunctions.prototype.DWSMSetSelectionOnNode = function (node, forceSyncSelection) {
    'use strict';
    if (!node) {
        this.logDebugMsg("node is not present");
    }
    var theDOM = this.getDocumentDOM();
    var offsets = theDOM.nodeToOffsets(node);
    theDOM.setSelection(offsets[0], offsets[1], true, forceSyncSelection);
    var curSelection = theDOM.getSelection();
    if (curSelection && (offsets[0] === curSelection[0]) && (offsets[1] === curSelection[1])) {
        return true;
    } else {
        var dwSelectNode = theDOM.getSelectedNode();
        if (dwSelectNode && node !== dwSelectNode) {
            theDOM.setSelectedNode(node, false, false, true, false);
            curSelection = theDOM.getSelection();
            if (curSelection && (offsets[0] === curSelection[0]) && (offsets[1] === curSelection[1])) {
                return true;
            }
        }
    }
    return false;
};

/*
Function: DWSMSetSelectionOnNodeWithIdAndInsertHTML
Utility function to set selection on a node and then insertHTML

Arguments :     DWID: dwId,
                HTML: htmlToBeInserted,
                REPLACE_SELECTION: false,
                SWITCH_SELECTION: true

Set Selection on node with given liveedit tag id and the html to be inserted with other options
*/
ExtensionsBridgeFunctions.prototype.DWSMSetSelectionOnNodeWithIdAndInsertHTML = function (argsObj) {
    'use strict';
    
    var node = this.getElementByDWId(argsObj.DWID),
        theDOM = this.getDocumentDOM(),
        replaceSelection = (typeof argsObj.REPLACE_SELECTION !== "undefined") ? argsObj.REPLACE_SELECTION : false,
        switchSelection = (typeof argsObj.SWITCH_SELECTION !== "undefined") ? argsObj.SWITCH_SELECTION : true;
    
    var insertHtmlThroughCodeView = function (args) {
        var html = args.HTML;
        var domSource = theDOM.source;
        if (domSource) {
            if (dw.getFocus(false) !== 'textView') {
                domSource.updateSelectionFromDOM();
            }
            theDOM.beginBatchEdit();
            var selOffsets = domSource.getSelection();
            // it is not going to replace selection.
            domSource.insert(selOffsets[1], html);
            // it is going to select the newly inserted code. Other options are discarded for code view insertion
            domSource.setSelection(selOffsets[1], selOffsets[1] + html.length); // Select the inserted code
            domSource.syncCodeToDOM();
            theDOM.formatSelection();
            theDOM.endBatchEdit();
        }
    };
    
    if (node !== null) {
        var isSetSelectionSuccessful = this.DWSMSetSelectionOnNode(node, false);

        if (argsObj.CODE_INSERT) {
            insertHtmlThroughCodeView(argsObj);
        } else {
            if (isSetSelectionSuccessful) {
                theDOM.insertHTML(argsObj.HTML, replaceSelection, true);
            } else {
                var offsets = theDOM.nodeToOffsets(node);
                theDOM.setSelectionAndInsertHTML(offsets[0], offsets[1], argsObj.HTML, switchSelection, replaceSelection, true);
            }
        }
    }
};
/*
Function: DWSMSetSelectionOnNodeWithId
Utility function to set selection on a node

Arguments : node

Set Selection on node with given liveedit tag id
*/
ExtensionsBridgeFunctions.prototype.DWSMSetSelectionOnNodeWithId = function (argObj) {
    'use strict';
    var node = this.getElementByDWId(argObj.id);
    if (node !== null) {
        this.DWSMSetSelectionOnNode(node, argObj.forceSyncSelection);
    }
};

/*
function:DWSMGetDWIdsForUpdatedTags
(Internal function on spider monkey side)
Arguments:argObj
    
    nodeId      - dwid of node under edit
    parentId    - dwid of ParentNode of that node
    newText     - new text to keep in the place of old content at that offset.
    counter     - how many nodes have been modified(to get corresponding node dwid's for tempid's)
    callback    - callback function to be called

Description: 
    Update the link attribute and fetch dwid's for all the modfied nodes.

*/
ExtensionsBridgeFunctions.prototype.DWSMGetDWIdsForUpdatedTags = function (argObj) {
    'use strict';
    this.logDebugMsg("DWSMGetDWIdsForUpdatedTags from ExtensionsBridgeFunctions");

    if (!argObj) {
        this.logDebugMsg("DWSMGetDWIdsForUpdatedTags :callback Obj not present");
        return;
    }
    var theDOM = this.getDocumentDOM();
    if (!theDOM) {

        this.logDebugMsg("DWSMGetDWIdsForUpdatedTags : DOM is not available");
        return;
    }
    var editNode = this.getElementByDWId(argObj.nodeId);
    var parentNode = this.getElementByDWId(argObj.parentId);
	var actualParent = editNode.parentNode;
    if (!editNode || !parentNode) {
        this.logDebugMsg("DWSMGetDWIdsForUpdatedTags : nodes are invalid" + !editNode + " " + !parentNode + argObj.nodeId + "  " + argObj.parentId);
        return;
    }
    
    if (!argObj.ignoreParentMatching) {
        // Check parents are matching. In case of Editable Region, it won't match
        // because of ER comment node is treated as parent in DW side.
        if (parentNode !== actualParent && !this.isElementERComment(actualParent)) {
            throw DW_LIVEEDIT_CONSTANTS.ParentNotMatching;
        }
    }

    var isSetSelectionSuccessful;
    if (argObj.setSelectionOnParent) {
        isSetSelectionSuccessful = this.DWSMSetSelectionOnNode(editNode.parentNode, false);
    } else {
        isSetSelectionSuccessful = this.DWSMSetSelectionOnNode(editNode, false);
    }
    
    if (isSetSelectionSuccessful) {
        theDOM.insertHTML(argObj.newText, true, true);
    } else {
        var offsets = theDOM.nodeToOffsets(editNode);
		theDOM.setSelectionAndInsertHTML(offsets[0], offsets[1], argObj.newText, false, true, true);
    }
    var newIdObj = {};
    this.populateNewIDObjectInfo(parentNode, newIdObj); // populate new DWID's with repsect to temporary Id's
	
	// Check for Temp Id population error
	// When image/text HUD removes/add links to image/text, it always causes structure changes 
	// and expects atleast one new ID and temp ID always starting from 0.
	// So checking for '0' temp ID would make sure success.
	if (!newIdObj[0]) {
		throw DW_LIVEEDIT_CONSTANTS.ParentNotMatching;
	}
	
    // set selection explicitly to node under edit- image/text tag
    var node_new_id = newIdObj[argObj.node_callback_id]; // new data attribute Id image/text tag has got
    if (node_new_id) {
        editNode = this.getElementByDWId(node_new_id); // image/text node
        if (editNode) {
            this.DWSMSetSelectionOnNode(editNode, false);
        }
    }
    newIdObj.parentId = argObj.parentId;
    newIdObj.callbackId = argObj.counter;
    argObj.callback(newIdObj);
};

/*
function:dwBrowseForFileURL
(Internal function on spider monkey side)
Arguments:argObj
        
        operation: select/open
        subop : name that will be shown at the top
        fileter: filters to be applied while showing the folder 
        callback: on successful/failure of execution callback to js layer.

Description: 
    Browse for file functionality for source and link.

*/
ExtensionsBridgeFunctions.prototype.dwBrowseForFileURL = function (argObj) {
    'use strict';
    if (!argObj) {
        this.logDebugMsg("dwBrowseForFileURL :callback Obj not present");
        return;
    }
    var imageFilter = false;
    if (argObj.filter.indexOf("png") !== -1) {
        imageFilter = true;
    }
    var path = dw.browseForFileURL(argObj.operation, argObj.subOp, imageFilter, false, argObj.filter);
    /* on Mac, the focus is to be explicitly set to the Live view */
	dw.setFocusToLiveView();
    if (argObj.callback) {
        argObj.callback(path);
    }
};

/*
function:rollBackFailedChanges
(Internal function on spider monkey side)
Arguments:none
Description: 
    Undo last change and clean up the redo events.
*/
ExtensionsBridgeFunctions.prototype.rollBackFailedChanges = function () {
	'use strict';
	this.logDebugMsg('ExtensionsBridgeFunctions.prototype.rollBackFailedChanges');

	var dom = this.getDocumentDOM();
	if (!dom) {
		return;
	}

	try {
		if (this.isDWTempIDLeftInDom(dom)) {
			this.logDebugMsg('isDWTempIDLeftInDom is TRUE - doing UNDO');
			var domsource = dom.source;
			if (domsource) {
				domsource.undo();
				dom.clearRedo();
				//domsource.clearRedo();
			}
		}
	} catch (e) {
		this.logDebugMsg('ERROR in rollBackFailedChanges : ' + e.message);
	}
};

/*
function:isInUnhideMode
Arguments:callback function to be called with the result
Description: 
    Figures out if the titan doc is in unhide mode
*/
ExtensionsBridgeFunctions.prototype.isInUnhideMode = function (args) {
    'use strict';
    var theDOM = this.getDocumentDOM();
	if (args.callback) {
        args.callback({ inUnhideMode: theDOM.getIsInUnhideMode()});
    }
};

/*
function:dwHandleLiveViewDuplicate
(Internal function on spider monkey side)

Description: 
    Duplicate the layout and its content of selected element in Responsive framework layout file

*/
ExtensionsBridgeFunctions.prototype.dwHandleLiveViewDuplicate = function () {
    'use strict';
    if (dw.canHandleLiveViewDuplicate()) {
        dw.handleLiveViewDuplicate();
    }
};