/*
 Copyright 2014 Adobe Systems Incorporated.  All rights reserved.

Purpose-
This file has the implementation of Drag Feedback in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global liveViewObject, DW_EXTENSION_EVENT*/

/* Constants used in this file */
var CONSTANTS = {
    ExtensionID: "liveDragFeedback",
    FeedbackElementClass: "feedbackElement",
    ExtensionMinHeight: 100,
    OverlayBorderWidth: 1
};

/* Object definition */

function LiveDragFeedback() {
    'use strict';
}

/*
function:initialize - initialize variabes, add event listeners

Arguments: None

Return : None
*/
LiveDragFeedback.prototype.initialize = function () {
    'use strict';
        
    this.m_currentDragElement = null;
    
    //add listener for messages from Live View
    window.addEventListener("message", this.messageHandler.bind(this), false);
    
    //initialize our container structure
    this.m_feedbackContainer = document.body;
    this.m_feedbackElementDiv = document.createElement("div");
    this.m_feedbackElementDiv.setAttribute('class', CONSTANTS.FeedbackElementClass);
    this.m_feedbackContainer.appendChild(this.m_feedbackElementDiv);
};



/*
function:messageHandler - Handler for messages from Live View

Arguments: None

Return : None
*/
LiveDragFeedback.prototype.messageHandler = function (e) {
    'use strict';
    if (e.data && e.data.type) {
        if (e.data.type === DW_EXTENSION_EVENT.LIVE_SURFACE_DRAG_START) {
            //show the feedback on the current dragged element
            this.m_currentDragElement = liveViewObject.getCurrentDraggedElement();
            this.showDragFeedback(this.m_currentDragElement);
        } else if (e.data.type === DW_EXTENSION_EVENT.LIVE_SURFACE_DRAG_END ||
                    e.data.type === DW_EXTENSION_EVENT.ENABLE_EXTENSIONS) {
            this.hideDragFeedback();
        }
    }
};

/*
function:showDragFeedback - Show the Drag feedback on the dragged element

Arguments: None

Return : None
*/
LiveDragFeedback.prototype.showDragFeedback = function (currentDragElement) {
    'use strict';
	
    //make sure that there is no feedback now
    this.hideDragFeedback();
    
    if (this.m_feedbackElementDiv && currentDragElement) {
        //get the rect for current dragged element
        var elemRect = liveViewObject.getElementRect(currentDragElement);
        if (elemRect.width <= 0 || elemRect.height <= 0) {
            return;
        }
        
        if (!parent.document.documentElement) {
            return;
        }

		//set our extension to use the entire document space
        var documentWidth = parent.document.documentElement.offsetWidth;
        var documentHeight = parent.document.documentElement.offsetHeight;


        //in some cases, element stretches beyond the size provided to us by document element
        //so increase the height and width if needed
        if (elemRect.left + elemRect.width > documentWidth) {
            documentWidth = elemRect.left + elemRect.width;
        }
        
        if (elemRect.top + elemRect.height > documentHeight) {
            documentHeight = elemRect.top + elemRect.height;
        }
        
        if (documentHeight < CONSTANTS.ExtensionMinHeight) {
            documentHeight = CONSTANTS.ExtensionMinHeight;
        }
           
        liveViewObject.positionExtensionById(CONSTANTS.ExtensionID, 0, 0, documentWidth, documentHeight);
        
        //now position our feedback div over the dragged element
        this.m_feedbackElementDiv.style.top = elemRect.top + 'px';
        this.m_feedbackElementDiv.style.left = elemRect.left + 'px';
        //we should reduce the width and height to accommodate the border
        //of the overlay div. so reduce 2*border_width    
        this.m_feedbackElementDiv.style.width = (elemRect.width - 2 * CONSTANTS.OverlayBorderWidth) + 'px';
        this.m_feedbackElementDiv.style.height = (elemRect.height - 2 * CONSTANTS.OverlayBorderWidth) + 'px';
        
        this.m_feedbackElementDiv.style.display = 'block';
    }
};

/*
function:hideDragFeedback - Hide the Drag feedback on the dragged element

Arguments: None

Return : None
*/
LiveDragFeedback.prototype.hideDragFeedback = function () {
    'use strict';
    this.m_feedbackElementDiv.style.display = 'none';
};

/*
function:initLiveDragFeedback - Entry point for Live Drag Feedback. Set on body Onload in HTML

Arguments: none

Return : none
*/
var initLiveDragFeedback = function () {
    'use strict';
    window.liveDragFeedbackObj = new LiveDragFeedback();
    window.liveDragFeedbackObj.initialize();
};
