/*
 Copyright 2014 Adobe Systems Incorporated.  All rights reserved.

Purpose-
This file has the implementation of DwTableHud
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global liveViewObject, DwTableHud, AuxiliaryExtensionObject, dwExtensionController, DW_EXTENSION_EVENT, dwObject, MOUSE */

/* Constants used in this file */
var CONSTANTS = {
    ExtensionID: "dwTableHud",
    OverlayDivID: "overlayDiv",
    CellOverlaySectionID: "cellOverlayHolder",
    BottomHandleDivID: "bottomHandle",
    RightHandleDivID: "rightHandle",
    CornerHandleDivID: "cornerHandle",
    CustomWidthAttribute: "origWidth",
    CustomHeightAttribute: "origHeight",
    CustomCellIDAttribute: "cellID",
    OverlayBorderWidth: 2,
    OverlayBorderOffsetForResize: 2,
    TransformHandleSize: 8,
    DWUniqueId: 'data_liveedit_tagid',
    CellHandleColor: 'black',
    CellHandleHoverColor: '#00BDFE',
    CellHandleSize: '1px',
    CellHandleSelectSize: '2px',
    CellHandleSelectOffset: '-2px',
    CellHandleSelectZeroOffset: '0px',
    CellHighlightColor: 'rgb(255, 0, 0)',
    OverlayBorderOffsetForHighlight: 3,
    HorizontalOverlayDivID: "horizontalDiv",
    VerticalOverlayDivID: "verticalDiv",
    ZIndexForSingleRowColSelect: 20,
    ZIndexForMultiRowColSelect: 30
};

var DRAGTYPES = {
    DRAG_NONE: 0,
    DRAG_TABLE_WIDTH: 1,
    DRAG_TABLE_HEIGHT: 2,
    DRAG_TABLE_CORNER: 3,
    DRAG_CELL_WIDTH: 4,
    DRAG_CELL_HEIGHT: 5,
    DRAG_ROW_OR_COLUMN_SELECT: 6,
    DRAG_ROW_AND_COLUMN_SELECT: 7
};

var VALUETYPES = {
	PIXEL: 1,
	PERCENT: 2
};

var DIMENSIONTYPE = {
	WIDTH: 1,
	HEIGHT: 2
};

var DW_HEADLIGHTS = {
    /* SUB CATEGORY */
    ELV_TABLE_LAYOUT: "ELV TableLayout",
    /* FEATURES */
    ENTER_LAYOUT_MODE: "Layout_Mode_ON",
    RESIZE_OP: "Resize"
};

//Extend from AuxiliaryExtensionObject
DwTableHud.prototype = new AuxiliaryExtensionObject();
DwTableHud.prototype.constructor = DwTableHud;
DwTableHud.prototype.baseClass = AuxiliaryExtensionObject.prototype.constructor;

/* Object definition */

function DwTableHud() {
    'use strict';

    //get the reference of the elements in the HUD
    this.m_overlayDiv = document.getElementById(CONSTANTS.OverlayDivID);
    this.m_cellOverlayHolder = document.getElementById(CONSTANTS.CellOverlaySectionID);
    this.m_bottomHandle = document.getElementById(CONSTANTS.BottomHandleDivID);
    this.m_rightHandle = document.getElementById(CONSTANTS.RightHandleDivID);
    this.m_cornerHandle = document.getElementById(CONSTANTS.CornerHandleDivID);

    //Hold the table's dw id, v'll need it for all our calls to DW
    this.m_selectedTableDWID = null;

    //if the view resizes we need to change
    window.parent.addEventListener('resize', this.positionHud.bind(this));

    //need to readjust if table size changed
    dwExtensionController.addListener(DW_EXTENSION_EVENT.ELEMENT_DIMENSION_CHANGED, this.positionHud, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.PARTIAL_REFRESH_COMPLETE, this.reInitialize, this);
    
    //On going out from tabel layout mode, empty selected cells array
    dwExtensionController.addListener(DW_EXTENSION_EVENT.HIDE_AUX_HUD, this.emptySelectedOverlayCellsArray, this);

    //Tracking variables to hande resize drag
    this.m_dragType = DRAGTYPES.DRAG_NONE;
    this.m_startDragX = -1;
    this.m_startDragY = -1;
    this.m_currentDragX = -1;
    this.m_currentDragY = -1;

    //Variables that will hold the inital dimensions before resize
    this.m_initialTableWidth = -1;
    this.m_initialTableHeight = -1;
    this.m_initialCellWidth = -1;
    this.m_initialCellHeight = -1;

    //Optimisation param to ensure we call don't call requestAnimationFrame multiple times
    this.m_resizePending = false;

    //Variables to track hovered & selected overlay cells
    this.m_currentHoverCellOverlay = null;
    this.m_selectedOverlayCells = [];

    //contains the dwID of the cell whose dimensions we will update
    this.m_currentDraggedCellID = null;

    this.m_highlightedOverlayCells = [];

    //Information from Dreamweaver
    this.m_cellToRowMap = null; //given a table cell, it will give which row does it belong to, considering the rowspan
    this.m_cellToColMap = null; //given a table cell, it will give which column does it belong to, considering the colspan
    this.m_rowIdentifierArray = null; //list of cells whose height needs to be edited when the height changes
    this.m_colIdentifierArray = null; //list of cells whose width needs to be edited when the width changes
    this.m_cellToRowSpanMap = null; //given a table cell, it will give its rowspan
    this.m_cellToColSpanMap = null; //given a table cell, it will give its colspan
    
    //contains the liveedit_tagid of selected cells
    this.m_selectedCellIds = [];
    
    //contains the liveedit_tagid to overlay div mapping
    this.m_mapIdToDiv = {};

    //variable to track which cells dimensions got changed
    this.m_cellWidthEditTracker = {};
    this.m_cellHeightEditTracker = {};
    this.m_resetCellWidths = false;
    this.m_resetCellHeights = false;

	//variable to see if we need to convert any of values to percent
	this.m_isTableWidthInPercent = false;
    this.m_isCellWidthInPercent = false;
    this.m_isCellHeightInPercent = false;

	//the following variable keeps track of the pixel to percent conversion factor
	this.m_tableWidthPercentConvertFactor = 0;

    //array contains all the table cells of the current table
    this.m_tableCellList = [];
    //variable to track of there are any dynamic cells in the table
    this.m_tableContainsDynamicCell = false;

    //contains left offsets and top offsets of all cells in the table
    this.offsetContainer = {
        leftOffsets: [],
        topOffsets: []
    };

    //variables to store previous mouse move range to prevent calling mouse move handler on every move
    this.m_prevLowerLeftPos = -1;
    this.m_prevUpperLeftPos = -1;
    this.m_prevLowerTopPos = -1;
    this.m_prevUpperTopPos = -1;
    
    //variables for drag and select
    this.m_dragStartRegion = null;
    this.m_dragStartPosX = -1;
    this.m_dragStartPosY = -1;
    this.m_mouseDownRange = {
        lower_left: -1,
        upper_left: -1,
        lower_top: -1,
        upper_top: -1
    };
    this.m_startDragCellOverlay = "";
    this.m_endDragCellOverlay = "";
}

DwTableHud.prototype.getHudTag = function () {
    'use strict';
    return "table";
};

DwTableHud.prototype.getExtensionId = function () {
    'use strict';
    return CONSTANTS.ExtensionID;
};

//function to capture when the hud gets turned on or off
DwTableHud.prototype.toggleAuxHud = function (message) {
    'use strict';
    if (AuxiliaryExtensionObject.prototype.toggleAuxHud.call(this, message)) { //let the inital stuff happen already
        if (this.getVisibility() === true) {
            if (!this.reInitialize()) {
				this.setVisibility(false);
			} else {
                dwObject.logHeadlightsData(DW_HEADLIGHTS.ELV_TABLE_LAYOUT, DW_HEADLIGHTS.ENTER_LAYOUT_MODE);
            }
        }
    }
};

//reinitialise Table info and reposition the HUD
DwTableHud.prototype.reInitialize = function () {
    'use strict';
	var ret = false;
    //check for our selected element, it might have got removed from the DOM tree
    if ((!this.m_currentSelectedElement || !window.parent.document.body.contains(this.m_currentSelectedElement))
            && this.getVisibility() === true) {
        this.toggleAuxHud();
    } else if (this.m_currentSelectedElement && this.getVisibility() === true) {
        this.m_selectedTableDWID = this.m_currentSelectedElement.getAttribute(CONSTANTS.DWUniqueId);
        this.getTableCells();
        if (this.m_selectedTableDWID && !this.m_tableContainsDynamicCell) {
            window.parent.DWLECallJsBridgingFunction('TableHudBridger', 'selectTable', {
                dwID : this.m_selectedTableDWID
            }, true);
            this.getTableData();
            this.positionHud();
			ret = true;
        }
    }
	return ret;
};

/**
 * @private positionHud
 * Position the HUD according to element's width, height and position
 * @Param elemRect : contains computed width,height,top,left 
 */
DwTableHud.prototype.positionHud = function () {
    'use strict';
	if (this.m_currentSelectedElement && this.getVisibility() === true) {
        var elemRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
        var index = 0;
        this.clearCellOverlays();
        this.positionExtension(elemRect);
        this.positionOverlayDiv(elemRect);
        this.positionTransformHandles(elemRect);
        this.positionCellOverlays(elemRect);
        this.m_selectedOverlayCells = [];
        while (index < this.m_selectedCellIds.length) {
            var cell = this.m_mapIdToDiv[this.m_selectedCellIds[index]].firstChild;
            cell.style.outlineWidth = CONSTANTS.CellHandleSelectSize;
            cell.style.outlineColor = CONSTANTS.CellHandleHoverColor;
            cell.style.outlineOffset = CONSTANTS.CellHandleSelectOffset;
            this.m_selectedOverlayCells.push(cell);
            index++;
        }
        this.selectCellsInTable();
    }
};

/**
 * @private clearCellOverlays
 * delete all the cell overlays
 * @Param none
 */
DwTableHud.prototype.clearCellOverlays = function () {
    'use strict';
	this.m_cellOverlayHolder.innerHTML = '';
};

/*
function:getTableCells - Get the list of all the table cells in the table
Return : none
*/
DwTableHud.prototype.getTableCells = function () {
    'use strict';
    this.m_tableCellList = [];
    this.m_tableContainsDynamicCell = false;
    if (this.m_currentSelectedElement) {
        var tableCells = this.m_currentSelectedElement.querySelectorAll('td,th');
        tableCells = Array.prototype.slice.call(tableCells); // for convenience of array type
        this.m_tableCellList = tableCells.filter(function (val) { //idea here is incase of nested tables we want only direct children
            var ret = false;
            if (val) {
                var parent = val.parentElement;
                while (parent) {
                    if (parent.tagName.toLowerCase() === "table") {
                        if (parent === this.m_currentSelectedElement) {
                            ret = true;
                            if (!val.getAttribute(CONSTANTS.DWUniqueId)) {
                                this.m_tableContainsDynamicCell = true;
                            }
                        }
                        break;
                    }
                    parent = parent.parentElement;
                }
            }
            return ret;
        }, this);
    }
};

/*
function:positionExtension - Position the extension so that we don't get cut off
Arguments: elemRect - rect to be used
Return : none
*/
DwTableHud.prototype.positionExtension = function (elemRect) {
    'use strict';
    if (elemRect) {
        //set our extension to use the entire document space
        var documentWidth = window.parent.document.documentElement.offsetWidth;
        var documentHeight = window.parent.document.documentElement.offsetHeight;
        //in some cases, element stretches beyond the size provided to us by document element
        //so increase the height and width if needed
        if (elemRect.left + elemRect.width > documentWidth) {
            documentWidth = elemRect.left + elemRect.width;
        }
        if (elemRect.top + elemRect.height > documentHeight) {
            documentHeight = elemRect.top + elemRect.height;
        }

        liveViewObject.positionExtensionById(CONSTANTS.ExtensionID, 0, 0, documentWidth, documentHeight);
    }
};

/*
function:updateSizeOfOverlayDivs - updates the sizes of overlay divs hovering over which highlights single row or column 
Arguments: none
Return : none
*/
DwTableHud.prototype.updateSizeOfOverlayDivs = function () {
    'use strict';
    var cellOuterHorizontalOverlay = document.getElementById(CONSTANTS.HorizontalOverlayDivID);
    var cellOuterVerticalOverlay = document.getElementById(CONSTANTS.VerticalOverlayDivID);
    var tableRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
    
    if (cellOuterHorizontalOverlay && cellOuterVerticalOverlay) {
        cellOuterHorizontalOverlay.style.top = tableRect.top;
        cellOuterHorizontalOverlay.style.left = this.offsetContainer.leftOffsets[0] + CONSTANTS.OverlayBorderOffsetForHighlight;
        cellOuterHorizontalOverlay.style.width = tableRect.left + tableRect.width - this.offsetContainer.leftOffsets[0] - CONSTANTS.OverlayBorderOffsetForHighlight;
        cellOuterHorizontalOverlay.style.height = this.offsetContainer.topOffsets[0] - tableRect.top + CONSTANTS.OverlayBorderOffsetForHighlight;
        cellOuterHorizontalOverlay.style.zIndex = CONSTANTS.ZIndexForSingleRowColSelect;

        cellOuterVerticalOverlay.style.left = tableRect.left;
        cellOuterVerticalOverlay.style.top = this.offsetContainer.topOffsets[0] + CONSTANTS.OverlayBorderOffsetForHighlight;
        cellOuterVerticalOverlay.style.width = this.offsetContainer.leftOffsets[0] - tableRect.left + CONSTANTS.OverlayBorderOffsetForHighlight;
        cellOuterVerticalOverlay.style.height = tableRect.top + tableRect.height - this.offsetContainer.topOffsets[0] - CONSTANTS.OverlayBorderOffsetForHighlight;
        cellOuterVerticalOverlay.style.zIndex = CONSTANTS.ZIndexForSingleRowColSelect;
    }
};

/*
function:positionCellOverlays - Position the cell overlays, this will be used for resizing cells as well as multi selection
Arguments: tableRect - rect to be used
Return : none
*/
DwTableHud.prototype.positionCellOverlays = function (tableRect) {
    'use strict';
    if (this.m_currentSelectedElement && tableRect) {
        this.offsetContainer.leftOffsets = [];
        this.offsetContainer.topOffsets = [];
        var docFrag = document.createDocumentFragment();

        var tableCellsRect = [];
        var tableCells = this.m_tableCellList;
        tableCells.forEach(function (val) {
            if (val) {
                var elemRect = liveViewObject.getElementRect(val);
                if (elemRect) {
                    tableCellsRect.push(elemRect);
                    this.offsetContainer.leftOffsets.push(elemRect.left);
                    this.offsetContainer.topOffsets.push(elemRect.top);
                }
            }
        }, this);

        this.offsetContainer.leftOffsets.sort(function (a, b) {
            return a - b;
        });
        this.offsetContainer.topOffsets.sort(function (a, b) {
            return a - b;
        });

        var cellOuterHorizontalOverlay = document.createElement('div');
        var cellOuterVerticalOverlay = document.createElement('div');

        cellOuterHorizontalOverlay.style.position = 'absolute';
        cellOuterHorizontalOverlay.setAttribute("id", CONSTANTS.HorizontalOverlayDivID);
        cellOuterHorizontalOverlay.addEventListener('mousemove', this.cellOuterMouseMoveHandler.bind(this));
        cellOuterHorizontalOverlay.addEventListener('mouseout', this.cellOuterMouseOutHandler.bind(this));
        cellOuterHorizontalOverlay.addEventListener('mousedown', this.cellOuterMouseDownHandler.bind(this));
    
        cellOuterVerticalOverlay.style.position = 'absolute';
        cellOuterVerticalOverlay.setAttribute("id", CONSTANTS.VerticalOverlayDivID);
        cellOuterVerticalOverlay.addEventListener('mousemove', this.cellOuterMouseMoveHandler.bind(this));
        cellOuterVerticalOverlay.addEventListener('mouseout', this.cellOuterMouseOutHandler.bind(this));
        cellOuterVerticalOverlay.addEventListener('mousedown', this.cellOuterMouseDownHandler.bind(this));
        
        tableCells.forEach(function (val, itemNum) {
            var elemRect = tableCellsRect[itemNum];
            if (elemRect) {
                var width = 0, height = 0, index = 0;
                for (index = 0; index < this.offsetContainer.leftOffsets.length; ++index) {
                    if (this.offsetContainer.leftOffsets[index] > (elemRect.left + elemRect.width)) {
                        width = this.offsetContainer.leftOffsets[index] - elemRect.left - 1;
                        break;
                    }
                }
                if (width === 0) {
                    width = tableRect.left + tableRect.width - elemRect.left;
                }

                for (index = 0; index < this.offsetContainer.topOffsets.length; ++index) {
                    if (this.offsetContainer.topOffsets[index] > (elemRect.top + elemRect.height)) {
                        height = this.offsetContainer.topOffsets[index] - elemRect.top - 1;
                        break;
                    }
                }
                if (height === 0) {
                    height = tableRect.top + tableRect.height - elemRect.top;
                }

                var cellOverlay = document.createElement('div');
                cellOverlay.style.position = 'absolute';
                cellOverlay.style.left = elemRect.left;
                cellOverlay.style.top = elemRect.top;
                cellOverlay.style.width = width;
                cellOverlay.style.height = height;
                cellOverlay.style.zIndex = 10;
                cellOverlay.setAttribute(CONSTANTS.CustomWidthAttribute, elemRect.width);
                cellOverlay.setAttribute(CONSTANTS.CustomHeightAttribute, elemRect.height);
                cellOverlay.setAttribute(CONSTANTS.CustomCellIDAttribute, val.getAttribute(CONSTANTS.DWUniqueId));
                cellOverlay.addEventListener('mousemove', this.cellOverLayMouseMoveHandler.bind(this));
                cellOverlay.addEventListener('mousedown', this.cellOverLayMouseDownHandler.bind(this));
                cellOverlay.addEventListener('mouseout', this.cellOverLayMouseOutHandler.bind(this));
                
                this.m_mapIdToDiv[val.getAttribute(CONSTANTS.DWUniqueId)] = cellOverlay;

                var cellHandle = document.createElement('div');
                cellHandle.style.width = elemRect.width;
                cellHandle.style.height = elemRect.height;
                cellHandle.setAttribute('class', 'cellHandle');
                cellHandle.setAttribute(CONSTANTS.CustomCellIDAttribute, val.getAttribute(CONSTANTS.DWUniqueId));

                cellOverlay.appendChild(cellHandle);
                docFrag.appendChild(cellOverlay);
            }
        }, this);

        docFrag.appendChild(cellOuterVerticalOverlay);
        docFrag.appendChild(cellOuterHorizontalOverlay);
        this.m_cellOverlayHolder.appendChild(docFrag);
        this.updateSizeOfOverlayDivs();
    }
};

/*
function:cellOuterMouseDownHandler - Mouse Down Handler for the horizontal and overlay divs for selecting single row/column or multiple row/column using drag
Arguments: mouse event
Return : none
*/
DwTableHud.prototype.cellOuterMouseDownHandler = function (event) {
    'use strict';
    if (event && event.which === MOUSE.LeftClick && event.currentTarget && this.m_dragType === DRAGTYPES.DRAG_NONE) {
        var tableRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
        this.m_dragType = DRAGTYPES.DRAG_ROW_OR_COLUMN_SELECT;
        this.m_dragStartRegion.style.left = tableRect.left;
        this.m_dragStartRegion.style.top = tableRect.top;
        this.m_dragStartRegion.style.height = tableRect.height;
        this.m_dragStartRegion.style.width = tableRect.width;
        this.m_dragStartRegion.style.zIndex = CONSTANTS.ZIndexForMultiRowColSelect;
        this.calculateRange(event.clientX, event.clientY, this.m_mouseDownRange);
        if (this.m_dragStartRegion.getAttribute("id") === CONSTANTS.HorizontalOverlayDivID) {
            this.m_dragStartPosX = event.clientX;
        } else if (this.m_dragStartRegion.getAttribute("id") === CONSTANTS.VerticalOverlayDivID) {
            this.m_dragStartPosY = event.clientY;
        }
        if (!(event.ctrlKey || event.metaKey)) {
            this.selectCellOverlay(null, false);
        }
        document.addEventListener("mousemove", this.dragSelectRowColMouseMoveHandler.bind(this));
        document.addEventListener("mouseup", this.dragSelectMouseUpHandler.bind(this));
        event.stopPropagation();
        return false;
    }
    return true;
};

/*
function:dragSelectMouseUpHandler - Mouse Up Handler for row/column selection
Arguments: mouse event
Return : none
*/
DwTableHud.prototype.dragSelectMouseUpHandler =  function (event) {
    'use strict';
    if (this.m_dragType >= DRAGTYPES.DRAG_ROW_OR_COLUMN_SELECT && event) {
        var rangeObj = {lower_left: -1, upper_left: -1, lower_top: -1, upper_top: -1};
        var isRowOrColumnSelected = true;
        this.m_highlightedOverlayCells.forEach(function (val) {
            if (this.m_selectedOverlayCells.indexOf(val) === -1) {
                this.selectCellOverlay(val, true);
                isRowOrColumnSelected = false;
            }
        }, this);
        
        if (isRowOrColumnSelected && (event.ctrlKey || event.metaKey)) {
            this.m_highlightedOverlayCells.forEach(function (val) {
                this.selectCellOverlay(val, true);
            }, this);
        }
        
        this.selectCellsInTable();
        this.updateSizeOfOverlayDivs();
        this.updateRange(rangeObj);
        this.m_dragStartPosX = -1;
        this.m_dragStartPosY = -1;

        if (this.m_dragType === DRAGTYPES.DRAG_ROW_OR_COLUMN_SELECT) {
            document.removeEventListener("mousemove", this.dragSelectRowColMouseMoveHandler);
        }
        this.m_highlightedOverlayCells = [];
        this.m_dragType = DRAGTYPES.DRAG_NONE;
        document.removeEventListener("mouseup", this.dragSelectMouseUpHandler);
        window.document.body.style.cursor = "auto";
        event.stopPropagation();
        return false;
    }
    return true;
};

/*
function:dragSelectRowColMouseMoveHandler - Mouse Move Handler for either row or column multi selection
Arguments: mouse event
Return : none
*/
DwTableHud.prototype.dragSelectRowColMouseMoveHandler = function (event) {
    'use strict';
    if (event && this.m_dragType === DRAGTYPES.DRAG_ROW_OR_COLUMN_SELECT) {
        var tableRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
        var rangeObj = {lower_left: -1, upper_left: -1, lower_top: -1, upper_top: -1};
        
        if ((this.m_dragStartPosY === -1 && event.clientX < this.offsetContainer.leftOffsets[0]) || (this.m_dragStartPosX === -1 && event.clientY <                      this.offsetContainer.topOffsets[0]) || event.ctrlKey || event.metaKey) {
            event.stopPropagation();
            return false;
        }
        
        if (!(event.clientX >= tableRect.left && event.clientX < (tableRect.left + tableRect.width) && event.clientY >= tableRect.top && event.clientY < (tableRect.top + tableRect.height))) {
            event.stopPropagation();
            return false;
        }
        
        if (!((event.clientX >= this.m_prevLowerLeftPos && event.clientX < this.m_prevUpperLeftPos && this.m_prevLowerTopPos === -1) ||
              (event.clientY >= this.m_prevLowerTopPos && event.clientY < this.m_prevUpperTopPos && this.m_prevLowerLeftPos === -1))) {
            this.calculateRange(event.clientX, event.clientY, rangeObj);
            this.updateRange(rangeObj);
            if (this.m_mouseDownRange.lower_top >= rangeObj.upper_top && rangeObj.lower_left === -1) {
                rangeObj.upper_top = this.m_mouseDownRange.upper_top;
            } else if (this.m_mouseDownRange.lower_top < rangeObj.upper_top && rangeObj.lower_left === -1) {
                rangeObj.lower_top = this.m_mouseDownRange.lower_top;
            } else if (this.m_mouseDownRange.lower_left >= rangeObj.upper_left && rangeObj.lower_top === -1) {
                rangeObj.upper_left = this.m_mouseDownRange.upper_left;
            } else if (this.m_mouseDownRange.lower_left < rangeObj.upper_left && rangeObj.lower_top === -1) {
                rangeObj.lower_left = this.m_mouseDownRange.lower_left;
            }
            this.selectRowOrColInRange(rangeObj);
        }
        event.stopPropagation();
        return false;
    }
    return true;
};
/*
function:calculateRange - calculates the lower and upper top/left corner in which the mouse down/move lies 
Arguments: event coordinates and range object to populate
Return : none
*/
DwTableHud.prototype.calculateRange = function (X_ordinate, Y_ordinate, rangeObj) {
    'use strict';
    var i = 0;
    var tableRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
    if (this.m_dragStartRegion.getAttribute("id") === CONSTANTS.HorizontalOverlayDivID) {
        rangeObj.lower_left = this.offsetContainer.leftOffsets[this.offsetContainer.leftOffsets.length - 1];
        rangeObj.upper_left = tableRect.left + tableRect.width;
        for (i = 0; i < this.offsetContainer.leftOffsets.length - 1; ++i) {
            if (X_ordinate >= this.offsetContainer.leftOffsets[i] && X_ordinate < this.offsetContainer.leftOffsets[i + 1]) {
                rangeObj.lower_left = this.offsetContainer.leftOffsets[i];
                rangeObj.upper_left = this.offsetContainer.leftOffsets[i + 1];
                break;
            }
        }
    } else if (this.m_dragStartRegion.getAttribute("id") === CONSTANTS.VerticalOverlayDivID) {
        rangeObj.lower_top = this.offsetContainer.topOffsets[this.offsetContainer.topOffsets.length - 1];
        rangeObj.upper_top = tableRect.top + tableRect.height;
        for (i = 0; i < this.offsetContainer.leftOffsets.length - 1; ++i) {
            if (Y_ordinate >= this.offsetContainer.topOffsets[i] && Y_ordinate < this.offsetContainer.topOffsets[i + 1]) {
                rangeObj.lower_top = this.offsetContainer.topOffsets[i];
                rangeObj.upper_top = this.offsetContainer.topOffsets[i + 1];
                break;
            }
        }
    }
};

/*
function:updateRange - stores the current Range of mouse event to prevent handling another mouse event if the range is same 
Arguments: current range object
Return : none
*/
DwTableHud.prototype.updateRange = function (rangeObj) {
    'use strict';
    this.m_prevLowerLeftPos = rangeObj.lower_left;
    this.m_prevUpperLeftPos = rangeObj.upper_left;
    this.m_prevLowerTopPos = rangeObj.lower_top;
    this.m_prevUpperTopPos = rangeObj.upper_top;
};

/*
function:setCellHandleOutlineAttributes
Arguments: cell handle
Return : none
*/
DwTableHud.prototype.setCellHandleOutlineAttributes = function (elem) {
    'use strict';
    if (elem) {
        elem.style.outlineWidth = CONSTANTS.CellHandleSize;
        elem.style.outlineColor = CONSTANTS.CellHandleColor;
        elem.style.outlineOffset = CONSTANTS.CellHandleSelectZeroOffset;
    }
};

/*
function:cellHighlight - to highlight the cell handle, not selecting
Arguments: cell handle
Return : none
*/
DwTableHud.prototype.cellHighlight = function (elem) {
    'use strict';
    if (this.m_selectedOverlayCells.indexOf(elem) === -1) {
        elem.style.outlineColor = CONSTANTS.CellHighlightColor;
        elem.style.outlineOffset = CONSTANTS.CellHandleSelectOffset;
    }
    this.m_highlightedOverlayCells.push(elem);
};

/*
function:selectRowOrColInRange - select rows/columns which are in drag range
Arguments: cell handle
Return : none
*/
DwTableHud.prototype.selectRowOrColInRange = function (rangeObj) {
    'use strict';
    var overlay = this.m_cellOverlayHolder.firstChild;
    while (overlay !== document.getElementById(CONSTANTS.VerticalOverlayDivID)) {
        var elemRect = overlay.getBoundingClientRect();
        var elemIndex = this.m_selectedOverlayCells.indexOf(overlay.firstChild);
        if ((elemRect.left >= rangeObj.lower_left && elemRect.left < rangeObj.upper_left) || (elemRect.top >= rangeObj.lower_top && elemRect.top < rangeObj.upper_top)) {
            if (elemIndex === -1) {
                this.selectCellOverlay(overlay.firstChild, true);
            }
        } else if (elemIndex !== -1) {
            this.setCellHandleOutlineAttributes(overlay.firstChild);
            this.m_selectedOverlayCells.splice(elemIndex, 1);
        }
        overlay = overlay.nextSibling;
    }
};
/*
function:cellOuterMouseMoveHandler - handler when user hovers on the outer divs
Arguments: mouse event
Return : none
*/
DwTableHud.prototype.cellOuterMouseMoveHandler = function (event) {
    'use strict';
    if (event && event.currentTarget && this.m_dragType === DRAGTYPES.DRAG_NONE) {
        //checking this condition to prevent iteration over all cells if user clicks in the previous region
        if (!((event.clientX >= this.m_prevLowerLeftPos && event.clientX < this.m_prevUpperLeftPos && this.m_prevLowerTopPos === -1) ||
               (event.clientY >= this.m_prevLowerTopPos && event.clientY < this.m_prevUpperTopPos && this.m_prevLowerLeftPos === -1))) {
            var rangeObj = {lower_left: -1, upper_left: -1, lower_top: -1, upper_top: -1};
            this.m_dragStartRegion = event.currentTarget;
            this.calculateRange(event.clientX, event.clientY, rangeObj);
            
            this.m_highlightedOverlayCells.forEach(function (val) {
                if (this.m_selectedOverlayCells.indexOf(val) === -1) {
                    this.setCellHandleOutlineAttributes(val);
                }
            }, this);
            this.m_highlightedOverlayCells = [];
            
            var overlay = this.m_cellOverlayHolder.firstChild;
            while (overlay !== document.getElementById(CONSTANTS.VerticalOverlayDivID)) {
                var elemRect = overlay.getBoundingClientRect();
                if ((elemRect.left >= rangeObj.lower_left && elemRect.left < rangeObj.upper_left) || (elemRect.top >= rangeObj.lower_top && elemRect.top < rangeObj.upper_top)) {
                    this.cellHighlight(overlay.firstChild);
                }
                overlay = overlay.nextSibling;
            }
            this.updateRange(rangeObj);
        }
        event.stopPropagation();
        return false;
    }
    return true;
};
/*
function:cellOuterMouseOutHandler - mouse out handler when mouse goes out of the horizontal or vertical div
Arguments: mouse event
Return : none
*/
DwTableHud.prototype.cellOuterMouseOutHandler = function (event) {
    'use strict';
    if (this.m_dragType === DRAGTYPES.DRAG_NONE) {
        var rangeObj = {lower_left: -1, upper_left: -1, lower_top: -1, upper_top: -1};
        this.m_highlightedOverlayCells.forEach(function (val) {
            if (this.m_selectedOverlayCells.indexOf(val) === -1) {
                this.setCellHandleOutlineAttributes(val);
            }
        }, this);
        this.m_highlightedOverlayCells = [];
        this.updateRange(rangeObj);
    }
};

DwTableHud.prototype.cellOverLayMouseOutHandler = function (event) {
    'use strict';
    if (this.m_currentHoverCellOverlay) {
        if (this.m_selectedOverlayCells.indexOf(this.m_currentHoverCellOverlay) === -1) {
            this.setCellHandleOutlineAttributes(this.m_currentHoverCellOverlay);
        }
    }
};
/*
function:selectRowColumn - selecting both rows and columns depending on the drag
Arguments: cell overlay on which mouse event initiates
Return : none
*/
DwTableHud.prototype.selectRowColumn = function (target) {
    'use strict';
    var currentTargetData = {Row: -1, Col: -1, RowSpan: -1, ColSpan: -1};
    var prevTargetData = {Row: -1, Col: -1, RowSpan: -1, ColSpan: -1};
    var currentRange = {rowLower: -1, rowUpper: -1, colLower: -1, colUpper: -1};
    var ID = target.getAttribute(CONSTANTS.CustomCellIDAttribute);
    currentTargetData.Row = this.m_cellToRowMap[ID];
    currentTargetData.Col = this.m_cellToColMap[ID];
    currentTargetData.RowSpan = this.m_cellToRowSpanMap[ID];
    currentTargetData.ColSpan = this.m_cellToColSpanMap[ID];
    
    ID = this.m_startDragCellOverlay.getAttribute(CONSTANTS.CustomCellIDAttribute);
    prevTargetData.Row = this.m_cellToRowMap[ID];
    prevTargetData.Col = this.m_cellToColMap[ID];
    prevTargetData.RowSpan = this.m_cellToRowSpanMap[ID];
    prevTargetData.ColSpan = this.m_cellToColSpanMap[ID];
    
    if (prevTargetData.Row - prevTargetData.RowSpan <= currentTargetData.Row - currentTargetData.RowSpan && prevTargetData.Col - prevTargetData.ColSpan <= currentTargetData.Col - currentTargetData.ColSpan) {
        currentRange.rowLower = prevTargetData.Row - prevTargetData.RowSpan + 1;
        currentRange.rowUpper = currentTargetData.Row > prevTargetData.Row ? currentTargetData.Row : prevTargetData.Row;
        currentRange.colLower = prevTargetData.Col - prevTargetData.ColSpan + 1;
        currentRange.colUpper = currentTargetData.Col > prevTargetData.Col ? currentTargetData.Col : prevTargetData.Col;
    } else if (prevTargetData.Row - prevTargetData.RowSpan <= currentTargetData.Row - currentTargetData.RowSpan && prevTargetData.Col - prevTargetData.ColSpan > currentTargetData.Col - currentTargetData.ColSpan) {
        currentRange.rowLower = prevTargetData.Row - prevTargetData.RowSpan + 1;
        currentRange.rowUpper = currentTargetData.Row > prevTargetData.Row ? currentTargetData.Row : prevTargetData.Row;
        currentRange.colLower = currentTargetData.Col - currentTargetData.ColSpan + 1;
        currentRange.colUpper = prevTargetData.Col;
    } else if (prevTargetData.Row - prevTargetData.RowSpan > currentTargetData.Row - currentTargetData.RowSpan && prevTargetData.Col - prevTargetData.ColSpan <= currentTargetData.Col - currentTargetData.ColSpan) {
        currentRange.rowLower = currentTargetData.Row - currentTargetData.RowSpan + 1;
        currentRange.rowUpper = prevTargetData.Row;
        currentRange.colLower = prevTargetData.Col - prevTargetData.ColSpan + 1;
        currentRange.colUpper = currentTargetData.Col > prevTargetData.Col ? currentTargetData.Col : prevTargetData.Col;
    } else if (prevTargetData.Row - prevTargetData.RowSpan > currentTargetData.Row - currentTargetData.RowSpan && prevTargetData.Col - prevTargetData.ColSpan > currentTargetData.Col - currentTargetData.ColSpan) {
        currentRange.rowLower = currentTargetData.Row - currentTargetData.RowSpan + 1;
        currentRange.rowUpper = prevTargetData.Row;
        currentRange.colLower = currentTargetData.Col - currentTargetData.ColSpan + 1;
        currentRange.colUpper = prevTargetData.Col;
    }
    
    var overlay = this.m_cellOverlayHolder.firstChild;
    var rowCondition = false, colCondition = false;
    while (overlay !== document.getElementById(CONSTANTS.VerticalOverlayDivID)) {
        var DWID = overlay.getAttribute(CONSTANTS.CustomCellIDAttribute);
        var row = this.m_cellToRowMap[DWID] - this.m_cellToRowSpanMap[DWID] + 1;
        var col = this.m_cellToColMap[DWID] - this.m_cellToColSpanMap[DWID] + 1;
        rowCondition = row >= currentRange.rowLower && row <= currentRange.rowUpper;
        colCondition = col >= currentRange.colLower && col <= currentRange.colUpper;
        var elemIndex = this.m_selectedOverlayCells.indexOf(overlay.firstChild);
        if (rowCondition && colCondition) {
            if (elemIndex === -1) {
                this.selectCellOverlay(overlay.firstChild, true);
            }
        } else if (elemIndex !== -1 && overlay !== this.m_startDragCellOverlay) {
            this.setCellHandleOutlineAttributes(overlay.firstChild);
            this.m_selectedOverlayCells.splice(elemIndex, 1);
        }
        overlay = overlay.nextSibling;
    }
};
/*
function:cellOverLayMouseMoveHandler - This is just to ensure we have the correct cursor all the time
Arguments: event - is the mousemove event
Return : none
*/
DwTableHud.prototype.cellOverLayMouseMoveHandler = function (event) {
    'use strict';
    if (event && event.currentTarget && event.currentTarget.parentElement && event.currentTarget.parentElement.id === CONSTANTS.CellOverlaySectionID && this.m_dragType === DRAGTYPES.DRAG_NONE) {
        var target = event.currentTarget;
        var cursor = "";
        var xBorder = parseFloat(target.getAttribute(CONSTANTS.CustomWidthAttribute), 10);
        var yBorder = parseFloat(target.getAttribute(CONSTANTS.CustomHeightAttribute), 10);
		// this is to make sure that we will be able to show hover feedback we are adding 2px offset.
        if (event.offsetX >= xBorder - CONSTANTS.OverlayBorderOffsetForResize) {
            cursor = "col-resize";
        } else if (event.offsetY >= yBorder - CONSTANTS.OverlayBorderOffsetForResize) {
            cursor = "row-resize";
        }

        if (cursor === "") {
            this.setCurrentHoverCellOverlay(target.querySelector('div'));
        } else {
            if (event.ctrlKey || event.metaKey) {
                cursor = "";
            }
            this.setCurrentHoverCellOverlay(null);
        }

        if (target.style.cursor !== cursor) {
            target.style.cursor = cursor;
        }
        event.stopPropagation();
        return false;
    } else if (this.m_dragType === DRAGTYPES.DRAG_ROW_AND_COLUMN_SELECT) {
        var dragTarget = event.currentTarget;
        var xCellBorder = parseFloat(dragTarget.getAttribute(CONSTANTS.CustomWidthAttribute), 10);
        var yCellBorder = parseFloat(dragTarget.getAttribute(CONSTANTS.CustomHeightAttribute), 10);
        if (this.m_endDragCellOverlay !== dragTarget && event.offsetX < xCellBorder && event.offsetY < yCellBorder) {
            this.selectRowColumn(dragTarget);
            this.m_endDragCellOverlay = dragTarget;
        }
        event.stopPropagation();
        return false;
    }
    return true;
};


/*
function:setCurrentHoverCellOverlay - We set the overlay div
Arguments: div element for which we need to add overlay
Return : none
*/
DwTableHud.prototype.setCurrentHoverCellOverlay = function (elem) {
    'use strict';
    if (this.m_currentHoverCellOverlay) {
        if (this.m_selectedOverlayCells.indexOf(this.m_currentHoverCellOverlay) === -1) {
            this.setCellHandleOutlineAttributes(this.m_currentHoverCellOverlay);
        }
        this.m_currentHoverCellOverlay = null;
    }

    if (elem) {
        this.m_currentHoverCellOverlay = elem;
        //this.m_currentHoverCellOverlay.style.outlineColor = CONSTANTS.CellHandleHoverColor;
    }

};

/*
function:selectCellOverlay - We add the list of elements into selected list
Arguments: elem - a single element, append - true if we want to append, false if we want clear 
Return : none
*/
DwTableHud.prototype.selectCellOverlay = function (elem, append) {
    'use strict';
    if (!append) {
        if (this.m_selectedOverlayCells.length > 0) {
            this.m_selectedOverlayCells.forEach(function (val) {
                if (val && this.m_highlightedOverlayCells.indexOf(val) === -1) {
                    this.setCellHandleOutlineAttributes(val);
                }
            }, this);
        }
        this.m_selectedOverlayCells = [];
    }

    if (elem) {
        var elemIndex = this.m_selectedOverlayCells.indexOf(elem);
        if (elemIndex === -1) {
            elem.style.outlineWidth = CONSTANTS.CellHandleSelectSize;
            elem.style.outlineColor = CONSTANTS.CellHandleHoverColor;
            elem.style.outlineOffset = CONSTANTS.CellHandleSelectOffset;
            this.m_selectedOverlayCells.push(elem);
        } else {
            this.setCellHandleOutlineAttributes(elem);
            this.m_selectedOverlayCells.splice(elemIndex, 1);
        }
    }
    this.m_selectedCellIds = [];
};
/*
function:selectCellsInTable - select the table cells which are in m_selectedOverlayCells 
Arguments: none
Return : none
*/
DwTableHud.prototype.selectCellsInTable = function () {
    'use strict';
    var dwIDArray = this.m_selectedOverlayCells.map(function (val) {
        var ret = "";
        if (val) {
            ret = val.getAttribute(CONSTANTS.CustomCellIDAttribute);
        }
        return ret;
    }, this);

    this.setIgnoreSelectionChange();
    
    this.m_selectedOverlayCells.forEach(function (val) {
        this.m_selectedCellIds.push(val.getAttribute(CONSTANTS.CustomCellIDAttribute));
    }, this);
    
    window.parent.DWLECallJsBridgingFunction('TableHudBridger', 'selectTableCells', {
        dwID : this.m_selectedTableDWID,
        cellArray : dwIDArray
    }, true);

};

/*
function:cellOverLayMouseDownHandler - This is starting point for resizing the column and row
Arguments: event - is the mousedown event
Return : none
*/
DwTableHud.prototype.cellOverLayMouseDownHandler = function (event) {
    'use strict';
    if (event && event.which === MOUSE.LeftClick && event.currentTarget && event.currentTarget.parentElement && event.currentTarget.parentElement.id === CONSTANTS.CellOverlaySectionID && this.m_dragType === DRAGTYPES.DRAG_NONE) {
        var target = event.currentTarget;
        var cursor = "";
        var xBorder = parseFloat(target.getAttribute(CONSTANTS.CustomWidthAttribute), 10);
        var yBorder = parseFloat(target.getAttribute(CONSTANTS.CustomHeightAttribute), 10);
        var dwID = target.getAttribute(CONSTANTS.CustomCellIDAttribute);
        if (dwID) {
            if (event.offsetX >= xBorder - CONSTANTS.OverlayBorderOffsetForResize && this.m_cellToColMap) {
                var colNum = this.m_cellToColMap[dwID];
                if (colNum < this.m_colIdentifierArray.length) {
                    var origColCell = this.m_currentSelectedElement.querySelector("[" + CONSTANTS.DWUniqueId + "='" + this.m_colIdentifierArray[colNum] + "']");
                    var cssWidthObj = parent.getComputedStyle(origColCell);
                    if (origColCell) {
                        this.m_currentDraggedCellID = this.m_colIdentifierArray[colNum];
                        this.m_initialCellWidth = parseFloat(cssWidthObj.width, 10);
                        this.m_cellWidthEditTracker = {};
                        this.m_resetCellWidths = true;
						this.m_isCellWidthInPercent = this.detectPercentValues(DIMENSIONTYPE.WIDTH);
                        this.startDrag(DRAGTYPES.DRAG_CELL_WIDTH, event, "col-resize");
                    }
                }
            } else if (event.offsetY >= yBorder - CONSTANTS.OverlayBorderOffsetForResize && this.m_cellToRowMap) {
                var rowNum = this.m_cellToRowMap[dwID];
                if (rowNum < this.m_rowIdentifierArray.length) {
                    var origRowCell = this.m_currentSelectedElement.querySelector("[" + CONSTANTS.DWUniqueId + "='" + this.m_rowIdentifierArray[rowNum] + "']");
                    var cssHeightObj = parent.getComputedStyle(origRowCell);
                    if (origRowCell) {
                        this.m_currentDraggedCellID = this.m_rowIdentifierArray[rowNum];
                        this.m_initialCellHeight = parseFloat(cssHeightObj.height, 10);
                        this.m_cellHeightEditTracker = {};
                        this.m_resetCellHeights = true;
						this.m_isCellHeightInPercent = this.detectPercentValues(DIMENSIONTYPE.HEIGHT);
                        this.startDrag(DRAGTYPES.DRAG_CELL_HEIGHT, event, "row-resize");
                    }
                }
            } else {
                this.m_dragType = DRAGTYPES.DRAG_ROW_AND_COLUMN_SELECT;
                this.selectCellOverlay(target.firstChild, event.ctrlKey || event.metaKey);
                this.m_startDragCellOverlay = target;
                this.m_endDragCellOverlay = target;
                document.addEventListener("mouseup", this.dragSelectMouseUpHandler.bind(this));
            }
        }
        event.stopPropagation();
        return false;
    }
    return true;
};

/*
function:positionOverlayDiv - Position our overlay div at the specified rect
Arguments: elemRect - rect to be used
Return : none
*/
DwTableHud.prototype.positionOverlayDiv = function (elemRect) {
    'use strict';

    if (this.m_overlayDiv && elemRect) {
        if (elemRect.width <= 0 || elemRect.height <= 0) {
            this.m_overlayDiv.style.display = 'none';
            return;
        }

        this.m_overlayDiv.style.top = elemRect.top + 'px';
        this.m_overlayDiv.style.left = elemRect.left + 'px';
        //we should reduce the width and height to accommodate the border
        //of the overlay div. so reduce 2*border_width    
        this.m_overlayDiv.style.width = (elemRect.width - 2 * CONSTANTS.OverlayBorderWidth) + 'px';
        this.m_overlayDiv.style.height = (elemRect.height - 2 * CONSTANTS.OverlayBorderWidth) + 'px';
        this.m_overlayDiv.style.display = 'block';
    }
};

/*
function:positionTransformHandles - Position our Transform Handles div at the specified rect
Arguments: elemRect - rect to be used
Return : none
*/
DwTableHud.prototype.positionTransformHandles = function (elemRect) {
    'use strict';

    if (this.m_bottomHandle && this.m_rightHandle && this.m_cornerHandle && elemRect) {
        if (elemRect.width <= 0 || elemRect.height <= 0) {
            this.m_bottomHandle.style.display = 'none';
            this.m_rightHandle.style.display = 'none';
            this.m_cornerHandle.style.display = 'none';
            return;
        }

        this.m_bottomHandle.style.top = elemRect.top + elemRect.height - CONSTANTS.TransformHandleSize + 'px';
        this.m_bottomHandle.style.left = elemRect.left + (elemRect.width - CONSTANTS.TransformHandleSize) / 2 + 'px';
        this.m_bottomHandle.style.display = 'block';

        this.m_rightHandle.style.top = elemRect.top + (elemRect.height - CONSTANTS.TransformHandleSize) / 2 + 'px';
        this.m_rightHandle.style.left = elemRect.left + elemRect.width - CONSTANTS.TransformHandleSize + 'px';
        this.m_rightHandle.style.display = 'block';

        this.m_cornerHandle.style.top = elemRect.top + elemRect.height - CONSTANTS.TransformHandleSize + 'px';
        this.m_cornerHandle.style.left = elemRect.left + elemRect.width - CONSTANTS.TransformHandleSize + 'px';
        this.m_cornerHandle.style.display = 'block';
    }
};

/*
function:startDrag - Called when someone does a mousedown with a possible intention of drag
Arguments: dragtype - suggests what is the resize for, event - is the mousedown event, cursor - gives the cursor type to show during drag
Return : none
*/

DwTableHud.prototype.startDrag = function (dragType, event, cursor) {
    'use strict';
    var cssObj = parent.getComputedStyle(this.m_currentSelectedElement);
    if (dragType > DRAGTYPES.DRAG_NONE && event && event.button === 0 && this.m_currentSelectedElement && cssObj) {
        //store the necessary values for calculations
        this.m_dragType = dragType;
        this.m_startDragX = event.pageX;
        this.m_startDragY = event.pageY;
        this.m_currentDragX = event.pageX;
        this.m_currentDragY = event.pageY;
        this.m_initialTableHeight = this.m_currentSelectedElement.height ? parseFloat(this.m_currentSelectedElement.height, 10) : parseFloat(cssObj.height, 10);
        this.m_initialTableWidth = this.m_currentSelectedElement.width ? parseFloat(this.m_currentSelectedElement.width, 10) : parseFloat(cssObj.width, 10);
        this.m_isTableWidthInPercent = this.m_currentSelectedElement.width && this.m_currentSelectedElement.width.indexOf('%') !== -1;
		this.m_tableWidthPercentConvertFactor = this.m_initialTableWidth / parseFloat(cssObj.width, 10);

        //kickout cell Selection
        this.selectCellOverlay(null, false);

        //add event listeners to the window to track the drag and also to end it
        window.addEventListener("mousemove", this.mouseMoveHandler.bind(this));
        window.addEventListener("mouseup", this.stopDrag.bind(this));
        window.parent.addEventListener("mouseup", this.stopDrag.bind(this)); //required when you drag beyond browser window
        window.parent.addEventListener("mousemove", this.mouseMoveHandler.bind(this)); //required when you drag beyond browser window

        //prior to this the body was not responding, however for above mousemove and mouseup to work, we need to start pointerevent
        window.document.body.style.pointerEvents = "auto";
        window.document.body.style.cursor = cursor;
        event.stopPropagation();
        return false;
    }
    return true;
};

/*
function:mouseMoveHandler - If the drag was initiated in previous mouse down, this will materialise resize
Arguments: event - is the mousemove event
Return : none
*/
DwTableHud.prototype.mouseMoveHandler = function (event) {
    'use strict';
    if (this.m_dragType > DRAGTYPES.DRAG_NONE && this.m_dragType < DRAGTYPES.DRAG_ROW_OR_COLUMN_SELECT && event) {
        //store the current position of drag
        this.m_currentDragX = event.pageX;
        this.m_currentDragY = event.pageY;

        //if we havent already requested for animation frame, do so
        if (!this.m_resizePending) {
            window.requestAnimationFrame(this.applyResize.bind(this, false));
            this.m_resizePending = true;
        }
        event.stopPropagation();
        return false;
    }

    return true;
};

/*
function:applyResize - This will actually commit the drag
Arguments: commitToDOM - if true, it will commit to the doc, or else it will just modify the DOM in browser only
Return : none
*/

DwTableHud.prototype.applyResize = function (commitToDOM) {
    'use strict';

    if (!this.m_currentSelectedElement) {
        return;
    }

    var heightDiff = 0, widthDiff = 0, changeTableHeight = false, changeTableWidth = false, changeCellWidth = false, changeCellHeight = false, finalTableWidth = "", finalTableHeight = "";
    switch (this.m_dragType) {
    case DRAGTYPES.DRAG_TABLE_HEIGHT:
        changeTableHeight = true;
        break;
    case DRAGTYPES.DRAG_TABLE_WIDTH:
        changeTableWidth = true;
        break;
    case DRAGTYPES.DRAG_TABLE_CORNER:
        changeTableHeight = true;
        changeTableWidth = true;
        break;
    case DRAGTYPES.DRAG_CELL_HEIGHT:
        changeCellHeight = true;
        changeTableHeight = true;
        break;
    case DRAGTYPES.DRAG_CELL_WIDTH:
        changeCellWidth = true;
        changeTableWidth = true;
        break;
    default:
        break;
    }

    if (changeTableHeight || changeCellHeight) {
        heightDiff = this.m_currentDragY - this.m_startDragY;
    }

    if (changeTableWidth || changeCellWidth) {
        widthDiff = this.m_currentDragX - this.m_startDragX;
    }

	if (changeCellHeight && heightDiff !== 0) {
		heightDiff = this.applyCellDimensions(heightDiff, DIMENSIONTYPE.HEIGHT);
	}

	if (changeTableHeight && heightDiff !== 0) {
		finalTableHeight = this.calculateFinalTableDimension(heightDiff, DIMENSIONTYPE.HEIGHT);
	}

	if (changeCellWidth && widthDiff !== 0) {
		widthDiff = this.applyCellDimensions(widthDiff, DIMENSIONTYPE.WIDTH);
	}

	if (changeTableWidth && widthDiff !== 0) {
		finalTableWidth = this.calculateFinalTableDimension(widthDiff, DIMENSIONTYPE.WIDTH);
	}

    if (commitToDOM) {
        if (heightDiff !== 0 && finalTableHeight) {
            window.parent.DWLECallJsBridgingFunction('TableHudBridger', 'setInlineStyle', {
                dwID : this.m_selectedTableDWID,
                prop : 'height',
                value : finalTableHeight
            }, true);
        }

        if (widthDiff !== 0 && finalTableWidth) {

			if (this.m_isTableWidthInPercent) {
				finalTableWidth = this.adjustPercentValues(finalTableWidth);
				finalTableWidth += '%';
				this.m_currentSelectedElement.setAttribute('width', finalTableWidth);
			}

            window.parent.DWLECallJsBridgingFunction('TableHudBridger', 'setInlineStyle', {
                dwID : this.m_selectedTableDWID,
                prop : 'width',
                value : finalTableWidth
            }, true);
        }

        var editList = null, prop = "";
        if (this.m_dragType === DRAGTYPES.DRAG_CELL_HEIGHT && heightDiff !== 0) {
            editList = this.m_cellHeightEditTracker;
            prop = 'height';
			if (this.m_isCellHeightInPercent) {
				this.adjustCellDimensions(DIMENSIONTYPE.HEIGHT);
			}
        } else if (this.m_dragType === DRAGTYPES.DRAG_CELL_WIDTH && widthDiff !== 0) {
            editList = this.m_cellWidthEditTracker;
            prop = 'width';
			if (this.m_isCellWidthInPercent) {
				this.adjustCellDimensions(DIMENSIONTYPE.WIDTH);
			}
        }

        if (editList && prop !== "") {
            var keys = Object.keys(editList);
            var index = 0;
            for (index = 0; index < keys.length; ++index) {
                var key = keys[index];
                var value = editList[key];
                window.parent.DWLECallJsBridgingFunction('TableHudBridger', 'setInlineStyle', {
                    dwID : key,
                    prop : prop,
                    value : value
                }, true);
            }
        }
        
        dwObject.logHeadlightsData(DW_HEADLIGHTS.ELV_TABLE_LAYOUT, DW_HEADLIGHTS.RESIZE_OP);
    }

    this.m_resizePending = false;
    this.elementDimensionChanged();
};

/*
function: applyCellDimensions - this helps to track for which cells the width/height have been put
Arguments: diff
return: returns the actual diff that was applied
*/
DwTableHud.prototype.applyCellDimensions = function (diff, dimension) {
    'use strict';

    var ret = diff;
	var attr = dimension === DIMENSIONTYPE.WIDTH ? 'width' : 'height';
	var identifierArr = dimension === DIMENSIONTYPE.WIDTH ? this.m_colIdentifierArray : this.m_rowIdentifierArray;
	var trackerArr = dimension === DIMENSIONTYPE.WIDTH ? this.m_cellWidthEditTracker : this.m_cellHeightEditTracker;
	var initialDimension = dimension === DIMENSIONTYPE.WIDTH ? this.m_initialCellWidth : this.m_initialCellHeight;
	var resetParam = dimension === DIMENSIONTYPE.WIDTH ? this.m_resetCellWidths : this.m_resetCellHeights;

    //apply width/height to the holders
    if (identifierArr && resetParam) {
        var index = 0;
        for (index = 0; index < identifierArr.length; ++index) {
            var dwID = identifierArr[index];
            var cell = this.m_currentSelectedElement.querySelector("[" + CONSTANTS.DWUniqueId + "='" + dwID + "']");
            var cellCSS = parent.getComputedStyle(cell);
            if (cell && cellCSS && parseFloat(cell[attr], 10) !== parseFloat(cellCSS[attr])) {
                var temp = parseFloat(cellCSS[attr]);
                trackerArr[dwID] = cell[attr] = temp.toString();
            }
        }
		if (dimension === DIMENSIONTYPE.WIDTH) {
			this.m_resetCellWidths = false;
        } else {
			this.m_resetCellHeights = false;
        }
    }

    if (this.m_currentDraggedCellID) {
        var origCell = this.m_currentSelectedElement.querySelector("[" + CONSTANTS.DWUniqueId + "='" + this.m_currentDraggedCellID + "']");
        if (origCell) {
			var initVal = initialDimension + diff;
            origCell[attr] = initVal;
            trackerArr[this.m_currentDraggedCellID] = initVal;
			var cssObj = parent.getComputedStyle(origCell);
			if (cssObj && parseFloat(cssObj[attr], 10) !== initVal) {
				ret = parseFloat(cssObj[attr], 10) - initialDimension;
				origCell[attr] = parseFloat(cssObj[attr], 10).toString();
				trackerArr[this.m_currentDraggedCellID] = origCell[attr];
			}
        }
    }

    return ret;
};

/*
function:stopDrag - This signals the end of drag if ever there was one.
Arguments: event - is the mouseup/mouseout event
Return : none
*/
DwTableHud.prototype.stopDrag = function (event) {
    'use strict';

    if (this.m_dragType > DRAGTYPES.DRAG_NONE && this.m_dragType < DRAGTYPES.DRAG_ROW_OR_COLUMN_SELECT && event && event.button === 0) {
        //commit the values
        if (this.m_startDragX !== this.m_currentDragX || this.m_startDragY !== this.m_currentDragY) {
            this.applyResize(true);
        }
        this.resetDrag();
        event.stopPropagation();
        return false;
    }

    return true;
};

/*
function:resetDrag - This resets the drag details
Arguments: none
Return : none
*/
DwTableHud.prototype.resetDrag = function () {
    'use strict';
    //reset all the tracking variables
    this.m_dragType = DRAGTYPES.DRAG_NONE;
    this.m_startDragX = -1;
    this.m_startDragY = -1;
    this.m_currentDragX = -1;
    this.m_currentDragY = -1;
    this.m_initialTableWidth = -1;
    this.m_initialTableHeight = -1;
    this.m_initialCellHeight = -1;
    this.m_initialCellWidth = -1;
    this.m_currentDraggedCellID = null;
    this.m_cellWidthEditTracker = {};
    this.m_cellHeightEditTracker = {};
    this.m_resetCellWidths = false;
    this.m_resetCellHeights = false;

	this.m_isTableWidthInPercent = false;
    this.m_isCellWidthInPercent = false;
    this.m_isCellHeightInPercent = false;
	this.m_tableWidthPercentConvertFactor = 0;

    //remove all the event listeners as these are overhead
    window.removeEventListener("mousemove", this.mouseMoveHandler);
    window.removeEventListener("mouseup", this.stopDrag);
    window.parent.removeEventListener("mouseup", this.stopDrag);
    window.parent.removeEventListener("mousemove", this.mouseMoveHandler);

    //Not doing this will not cause any selection change by clicking else where in live view
    window.document.body.style.pointerEvents = "none";
    window.document.body.style.cursor = "auto";
};

/*
function:getTableData - This asks the Bridger to fetch the data
Arguments: none
Return : none
*/
DwTableHud.prototype.getTableData = function () {
    'use strict';
    window.parent.DWLECallJsBridgingFunction('TableHudBridger', 'getTableInfo', {
        callback : this.storeTableData.bind(this)
    }, true);
};

/*
function:storeTableData - stores the data recived from the Bridger
Arguments: info about the table
Return : none
*/
DwTableHud.prototype.storeTableData = function (infoObj) {
    'use strict';
    this.m_cellToRowMap = infoObj.rowMap;
    this.m_cellToColMap = infoObj.colMap;
    this.m_rowIdentifierArray = infoObj.rowIdentifier;
    this.m_colIdentifierArray = infoObj.colIdentifier;
    this.m_cellToRowSpanMap = infoObj.rowSpanMap;
    this.m_cellToColSpanMap = infoObj.colSpanMap;
};


/*
function:calculateFinalTableDimension - for a given diff, it calculates the width/height value of table
Arguments: diff
Return : final DimensionValue
*/
DwTableHud.prototype.calculateFinalTableDimension = function (diff, dimension) {
    'use strict';
	var finalVal = "";
	var actualVal = 0;
	var cssObj = parent.getComputedStyle(this.m_currentSelectedElement);
	if (dimension ===  DIMENSIONTYPE.WIDTH) {
		if (this.m_isTableWidthInPercent) {
			var actualDiff = diff * this.m_tableWidthPercentConvertFactor;
			var suffix = '%';
			finalVal = this.m_initialTableWidth + actualDiff;
			this.m_currentSelectedElement.setAttribute('width', finalVal + suffix);
			var parentCssObj = parent.getComputedStyle(this.m_currentSelectedElement.parentNode);
			if (parentCssObj && cssObj) {
				actualVal = ((parseFloat(cssObj.width, 10) / parseFloat(parentCssObj.width, 10)) * 100);
				if (actualVal > finalVal) {
					finalVal = actualVal;
                    this.m_currentSelectedElement.setAttribute('width', finalVal + suffix);
				}
			}
            finalVal += suffix;
		} else {
			finalVal = this.m_initialTableWidth + diff;
			this.m_currentSelectedElement.setAttribute('width', finalVal);
			if (cssObj) {
				actualVal = parseFloat(cssObj.width, 10);
				if (actualVal > finalVal) {
					finalVal = actualVal;
                    this.m_currentSelectedElement.setAttribute('width', finalVal);
				}
			}
		}
	} else if (dimension ===  DIMENSIONTYPE.HEIGHT) {
		finalVal = this.m_initialTableHeight + diff;
		this.m_currentSelectedElement.setAttribute('height', finalVal);
		if (cssObj) {
			actualVal = parseFloat(cssObj.height, 10);
			if (actualVal > finalVal) {
				finalVal = actualVal;
		        this.m_currentSelectedElement.setAttribute('height', finalVal);
			}
		}
	}
    
	return finalVal;
};

/*
function:adjustPercentValues - for a given percent value, adjust it by rounding it off
Arguments: current value
Return : final value
*/
DwTableHud.prototype.adjustPercentValues = function (val) {
    'use strict';
	var numVal = parseFloat(val, 10);
	var intVal = Math.round(numVal);
	var finalVal = intVal < 1 ? 1 : intVal;
	return finalVal;
};

/*
function:adjustCellDimensions - when in percent mode, we need to re adjust the values of cell dimensions
Arguments: width / height;
Return : none
*/
DwTableHud.prototype.adjustCellDimensions = function (dimension) {
    'use strict';
    var cssObj = parent.getComputedStyle(this.m_currentSelectedElement);
	if (!this.m_currentSelectedElement || !cssObj) {
		return;
	}

	var editList = dimension === DIMENSIONTYPE.WIDTH ? this.m_cellWidthEditTracker : this.m_cellHeightEditTracker;
	var attr = dimension === DIMENSIONTYPE.WIDTH ? 'width' : 'height';
	var baseVal = parseFloat(cssObj[attr], 10);

	if (editList) {
		var keys = Object.keys(editList);
		var index = 0;
		var sum = 0;
		var key = "";
		var value = "";
		for (index = 0; index < keys.length; ++index) {
			key = keys[index];
			value = editList[key];
			sum += parseFloat(value, 10);
		}

		var leftover = 0;
		for (index = 0; index < keys.length; ++index) {
			key = keys[index];
			value = editList[key];
			var origVal = (parseFloat(value, 10) * 100) / sum;

			if (Math.abs(leftover) > 0.5) {
				origVal -= leftover;
				leftover = 0;
			}

			var numVal = this.adjustPercentValues(origVal);
			leftover += numVal - origVal;
			numVal += '%';
			editList[key] = numVal;
			var cell = this.m_currentSelectedElement.querySelector("[" + CONSTANTS.DWUniqueId + "='" + key + "']");
			if (cell) {
				cell[attr] = numVal;
			}
		}
	}
};

/*
function: detectPercentValues - check if we are to put cell dimensions in percent
Arguments: dimension;
Return : true/false;
*/
DwTableHud.prototype.detectPercentValues = function (dimension) {
    'use strict';
	var ret = true;
	var identifierArr = dimension === DIMENSIONTYPE.WIDTH ? this.m_colIdentifierArray : this.m_rowIdentifierArray;
	var attr = dimension === DIMENSIONTYPE.WIDTH ? 'width' : 'height';

	if (identifierArr) {
        var index = 0;
        for (index = 0; index < identifierArr.length; ++index) {
            var dwID = identifierArr[index];
            var cell = this.m_currentSelectedElement.querySelector("[" + CONSTANTS.DWUniqueId + "='" + dwID + "']");
            if (cell && cell[attr].indexOf('%') === -1) {
				ret = false;
				break;
			}
        }
    }
	return ret;
};

/*
function:escapeKeyPressed - handler for escape key press

Arguments: None

Return : None
*/
DwTableHud.prototype.escapeKeyPressed = function () {
    'use strict';
    this.emptySelectedOverlayCellsArray();
    //check if a drag is in progress
    if (this.m_dragType > DRAGTYPES.DRAG_NONE) {
        //commit the values
        if (this.m_startDragX !== this.m_currentDragX || this.m_startDragY !== this.m_currentDragY) {
            this.applyResize(true);
        }
        this.resetDrag();
    } else {
        //exit layout mode
        AuxiliaryExtensionObject.prototype.escapeKeyPressed.call(this);
    }
};

/*
function:deleteOrbackspaceKeyPressed - handler for delete or backspace key press
Arguments: None
Return : None
*/
DwTableHud.prototype.deleteOrbackspaceKeyPressed = function () {
    'use strict';
    window.parent.DWLECallJsBridgingFunction('TableHudBridger', 'deleteRowOrColumn', {
        dwID : this.m_selectedTableDWID
    }, false);
};

DwTableHud.prototype.emptySelectedOverlayCellsArray =  function () {
    'use strict';
    this.m_selectedOverlayCells = [];
    this.m_selectedCellIds = [];
};

/*
function:initDwTableHud - Entry point for DwTableHud. Set on body Onload in HTML
Arguments: none
Return : none
*/
var initDwTableHud = function () {
    'use strict';
    window.dwTableHudObj = new DwTableHud();
    window.dwTableHudObj.initialize();
};

