// editor.js is picked up from emmet
_snippet.define('editorProxy', function(require, _) {
	return {
		dom: function() {
			return dw.getDocumentDOM();
		},
        
		/**
		 * Returns character indexes of selected text: object with <code>start</code>
		 * and <code>end</code> properties. If there's no selection, should return 
		 * object with <code>start</code> and <code>end</code> properties referring
		 * to current caret position
		 * @return {Object}
		 * @example
		 * var selection = editor.getSelectionRange();
		 * alert(selection.start + ', ' + selection.end); 
		 */
		getSelectionRange: function() {
		
			var selection = this.dom().source.getSelection();
			
			return {
				start: selection[0],
				end: selection[1]
                
			};
		},
		
		/**
		 * Creates selection from <code>start</code> to <code>end</code> character
		 * indexes. If <code>end</code> is ommited, this method should place caret 
		 * and <code>start</code> index
		 * @param {Number} start
		 * @param {Number} [end]
		 * @example
		 * editor.createSelection(10, 40);
		 * 
		 * //move caret to 15th character
		 * editor.createSelection(15);
		 */
		createSelection: function(start, end) {
			
			this.dom().source.setSelection(start, end);
		},
		
		/**
		 * Returns current line's start and end indexes as object with <code>start</code>
		 * and <code>end</code> properties
		 * @return {Object}
		 * @example
		 * var range = editor.getCurrentLineRange();
		 * alert(range.start + ', ' + range.end);
		 */
		getCurrentLineRange: function() {
			var caretPos = this.getCaretPos();
			if (caretPos === null) return null;
			return require('utils').findNewlineBounds(this.getContent(), caretPos);
		},
		
		/**
		 * Returns current caret position
		 * @return {Number}
		 */
		getCaretPos: function() {
			var caretPos = this.getSelectionRange().start;
			return ~caretPos ? caretPos : null;
		},
		
		/**
		 * Set new caret position
		 * @param {Number} pos Caret position
		 */
		setCaretPos: function(pos) {
			this.createSelection(pos, pos);
		},
		
		/**
		 * Returns content of current line
		 * @return {String}
		 */
		getCurrentLine: function() {
			var range = this.getCurrentLineRange();
            return range.start < range.end ? this.dom().source.getText(range.start, range.end) : '';
		},
        
		/**
		 * Returns editor's content
		 * @return {String}
		 */
		getContent: function() {
			return this.dom().source.getText() || '';
		},		
		
		getSelection: function() {
			var dom = this.dom();
			var selection = dom.source.getSelection();
			return dom.source.getText(selection[0], selection[1]);
		}
	}
});