/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2010 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and may be covered by U.S. and Foreign Patents,
* patents in process, and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
*
**************************************************************************/


/*
The data structures used here are as follows:
-gJsonMap is an array containing all the FontFamilies.
-All the data structures that need to store reference to font, store the font's index value at gJsonMap
-gPrimaryFilterData and gSecondaryFilterData are array of arrays of fonts classified based on their type and recommendations respectively.

gPrimaryFilterData arrayindex
0 = sans-serif
1 = serif
2 = slab-serif
3 = script
4 = blackletter
5 = monospace
6 = handmade
7 = decorative

gSecondaryFilterData arrayindex
0-headings
1-paragraphs
2-headings and paragraphs

-gVisibleFonts has all the indices of fonts currently visible to the user
-gAllVisiblityFontList is a dummy array containing 1 to n where n is the total number of fonts
It is used to display all fonts when no filter is on

-gPreSelectedFonts contains IDs of fonts that have already been imported in dreamweaver
-gIDtoIndexMap contains mapping between fontID and index at jsonMap, helps in retriving the index for gPreSelectedFonts
*/
var gJsonMap = new Array();
var gIDtoIndexMap = new Object();
var gPrimaryFilterData = new Array();
var gSecondaryFilterData = new Array();
var gSelectedFamilies = new Array();
var gPreSelectedFonts = new Array();
var gVisibleFonts = new Array();
var gAllVisiblityFontList = new Array();
var gPathToFonts;
var gPrimaryFiltersLength = 8;
var gSecondaryFiltersLength = 3;
var gTimeOutFunc = null;

//variables to hold the current state of filters
var gCurrentprimaryFilter = null;
var gCurrentRecommendation = null;
var gCurrentSearchString = null;
var gShowSelectedFonts = false;

//variable to store last displayed font, used during lazy scrolling
var gLastFontDisplayed = -1;
//a lock like varibale telling if the current filter is still eing applied
var gDOMUpdateInProgress = false;

//WebkitBoxShadow property and value for selected fontsContainer
var gBoxShadowValue = "0 0 5px 0 #000";
var gBoxShadowProperty = "WebkitBoxShadow";

//Variables with respect to showing of fonts
var gScrollThreshold = 60;
var gInitialFontNumber = 32;
var gSubsequentFontNumber = 52;

//Separators used to construct Font Variation String
var gIDNameSeparator = "~";
var gVariationSeparator = "^";

//KeyCodes for Enter and Escape
var gKEYCODE_ENTER = 13;
var gKEYCODE_ESC = 27;
var gKEYCODE_TAB = 9;
var gKEYCODE_TILDE = 192;
var gKEYCODE_COMMAND_LEFT = 91;
var gKEYCODE_COMMAND_RIGHT = 93;
var gCommandButtonPressed = false;

//--------------------------------------------------------------------
// FUNCTION:
//   initUI
//
// DESCRIPTION:
//	Called from CPP code, when the document is ready
//
// ARGUMENTS:
//  none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------

function initUI(configPath, selectedFonts){
	gPathToFonts = configPath + "FreeFonts/";
	var url= gPathToFonts + "FreeFonts.json";
	if(selectedFonts)
		gPreSelectedFonts = selectedFonts;
		
	$.getJSON(url,handleJSONData).error(function() {
		/*if(gParent)
		{
			gParent.handleCorruptFile();
		}*/
	});
	
	document.body.addEventListener("keyup", handleKeyupEvent );
	//document.body.addEventListener("keydown", handleKeydownEvent );
}


//--------------------------------------------------------------------
// FUNCTION:
//   handleKeyEvent
//
// DESCRIPTION:
//	Handle Key Events
//
// ARGUMENTS:
//  none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
function handleKeyupEvent(event){
	//If Escape or Enter or TAB is pressed pass call the transferKeyMessage function
	//transferKeyMessage function is added through CPP code
	switch(event.keyCode)
	{
		case gKEYCODE_ENTER:
		case gKEYCODE_ESC:
		case gKEYCODE_TAB:
		//case gKEYCODE_TILDE:
			transferKeyMessage(event.keyCode, gCommandButtonPressed, event.ctrlKey);
			break;
			/*
		case gKEYCODE_COMMAND_LEFT:
		case gKEYCODE_COMMAND_RIGHT:
			gCommandButtonPressed = false;
			break;*/
		default:
			break;
	}
}
/*
function handleKeydownEvent(event){
	if(event.keyCode == gKEYCODE_COMMAND_LEFT || event.keyCode == gKEYCODE_COMMAND_RIGHT)
		gCommandButtonPressed = true;
}
*/
//--------------------------------------------------------------------
// FUNCTION:
//   getSelectedFonts
//
// DESCRIPTION:
//	Called when the OK button on dialog is clicked, returns the selected families to parent window
//
// ARGUMENTS:
//   
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------

function getSelectedFonts()
{
	var selectedFontsList = new Array();
	for(var i =0; i < gSelectedFamilies.length; i++)
	{
		selectedFontsList.push(gJsonMap[gSelectedFamilies[i]]);
	}
	return selectedFontsList;
}

//--------------------------------------------------------------------
// FUNCTION:
//   getFilterID
//
// DESCRIPTION:
//	called to check if the given string is a valid filter, if yes return the index in gPrimaryFilterData
//
// ARGUMENTS:
//   string representing the filter
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------

function getFilterID(str) {
	
	if(str == "none")
		return -1;
	else if( str == "sans-serif" || str == "sans")
		return 0;
	else if( str == "serif")
		return 1;
	else if( str == "slab-serif" || str == "slab")
		return 2;
	else if( str == "script")
		return 3;
	else if( str == "blackletter")
		return 4;
	else if( str == "monospaced" || str == "mono")
		return 5;
	else if( str == "handmade" || str == "hand")
		return 6;
	else if( str == "decorative")
		return 7;
	else 
		return -1;
}

//--------------------------------------------------------------------
// FUNCTION:
//   getRecommendationID
//
// DESCRIPTION:
//	called to check if the given string is a valid recommendation, if yes return the index in gSecondaryFilterData
//
// ARGUMENTS:
//   string representing the recommendation
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------

function getRecommendationID(str) {
	if(str == "none")
		return -1;
	else if(str == "headings" || str == "head")
		return 0;
	else if(str == "paragraphs" || str == "para")
		return 1;
	else if(str == "both")
		return 2;
	else
		return -1;
}

//--------------------------------------------------------------------
// FUNCTION:
//   changeFilter
//
// DESCRIPTION:
//	recieves the notification if a filter is applied or removed
//
// ARGUMENTS:
//   string representing the filter
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------


function changeFilter(str) {
	gDOMUpdateInProgress = true;
	var currentFilter = getFilterID(str);
	gCurrentprimaryFilter = ( currentFilter != -1) ? currentFilter:null;
	
	startDisplayFonts(prepareVisibleFontList());
	gDOMUpdateInProgress = false;
}

//--------------------------------------------------------------------
// FUNCTION:
//   changeRecommendation
//
// DESCRIPTION:
//	recieves the notification if a recommendation is applied or removed
//
// ARGUMENTS:
//   string representing the recommendation
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------

function changeRecommendation(str) {
	gDOMUpdateInProgress = true;
	var currentFilter = getRecommendationID(str);
	gCurrentRecommendation = ( currentFilter != -1) ? currentFilter:null;
	
	startDisplayFonts(prepareVisibleFontList());
	gDOMUpdateInProgress = false;
}

//--------------------------------------------------------------------
// FUNCTION:
//   showSelectedFonts
//
// DESCRIPTION:
//	recieves the notification if showSelectedFonts filter is applied or removed
//
// ARGUMENTS:
//   bool representing showSelectedFonts filter state
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------

function showSelectedFonts(bool)
{
	if(bool)
	{
		//small optimization, the filter is not applied on entire list but on the visibleFonts only
		gShowSelectedFonts = true;
		var currentFontList = findIntersectionOfArrays(gVisibleFonts, gSelectedFamilies.sort(numSort));
		startDisplayFonts(currentFontList);
	}
	else
	{
		gShowSelectedFonts = false;
		startDisplayFonts(prepareVisibleFontList());
	}
		
}

//--------------------------------------------------------------------
// FUNCTION:
//   prepareVisibleFontList
//
// DESCRIPTION:
//	central function that computes the visible set based on current filters applied
//
// ARGUMENTS:
//   NONE
//
// RETURNS:
//  array containing indices of visible fonts 
//--------------------------------------------------------------------

function prepareVisibleFontList() {
	var visibleFonts = new Array();
	
	//apply primary filter and recommendations
	if(gCurrentRecommendation == null && gCurrentprimaryFilter == null)
		visibleFonts = gAllVisiblityFontList;
	else if(gCurrentRecommendation == null)
		visibleFonts = gPrimaryFilterData[gCurrentprimaryFilter];
	else if(gCurrentprimaryFilter == null)
		visibleFonts = gSecondaryFilterData[gCurrentRecommendation];
	else
	{
		var pFilter = gPrimaryFilterData[gCurrentprimaryFilter];
		var sFilter = gSecondaryFilterData[gCurrentRecommendation];
		
		visibleFonts = findIntersectionOfArrays(pFilter, sFilter);
	}	
	
	//if showSelectedFonts is set, create the visible set accordingly
	if(gShowSelectedFonts)
		visibleFonts = findIntersectionOfArrays(visibleFonts, gSelectedFamilies.sort(numSort));
		
	//if some text is typed, apply search on the fontset
	visibleFonts = findMatchingFonts(visibleFonts, gCurrentSearchString);
	
	return visibleFonts;
}

//function to sort gSelectedFamilies
function numSort(a,b)
{
	return a-b;
}

//--------------------------------------------------------------------
// FUNCTION:
//   handleJSONData
//
// DESCRIPTION:
//	once the json file is picked and parse, this function is called
//
// ARGUMENTS:
//   json Data
//
// RETURNS:
//  array containing indices of visible fonts 
//--------------------------------------------------------------------
function handleJSONData(json) 
{
	json.families.sort(function(a,b) {
		return a.name.toLowerCase() > b.name.toLowerCase()? 1:-1;
	});	
	classifyFonts(json);
	startDisplayFonts(gAllVisiblityFontList);
}


//--------------------------------------------------------------------
// FUNCTION:
//   applySearch
//
// DESCRIPTION:
//	recieves the notification when the textfield has the focus and it has text written in it
//
// ARGUMENTS:
//  string containing the search string
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
function applySearch(str)
{
	if(	gDOMUpdateInProgress )
		return;
	
	gDOMUpdateInProgress = true;
	str = str.toLowerCase();
	if( str != null  && str.length == 0 ) 
	{
		//if the search string is null show all fonts
		if(gCurrentSearchString != null)
		{
			gCurrentSearchString = null;
			startDisplayFonts(prepareVisibleFontList());
		}
	}
	else if(str != gCurrentSearchString)
	{
		var currentFontList = new Array();
		//slight optimization if the previous search string is part of current string, apply the search on the visible set
		if(gCurrentSearchString == null || str.toLowerCase().substring(0,gCurrentSearchString.length) == gCurrentSearchString)
		{
			currentFontList = findMatchingFonts(gVisibleFonts, str);
			startDisplayFonts(currentFontList);
			gCurrentSearchString = str;
		}
		else
		{
			gCurrentSearchString = str;
			startDisplayFonts(prepareVisibleFontList());
		}
	}
	gDOMUpdateInProgress = false;
}

//utility function to merge 2 sorted array of numbers
function findIntersectionOfArrays(arrayOne, arrayTwo)
{
	var currentFontList = new Array();
	if(arrayOne != null && arrayOne.length != 0 && arrayTwo != null && arrayTwo.length != 0)
	{
		var i = 0;
		var j = 0;
		
		while(i < arrayOne.length && j <arrayTwo.length)
		{
			if(arrayOne[i] == arrayTwo[j])
			{
				currentFontList.push(arrayOne[i]);
				i++; j++;
			}
			else if(arrayOne[i] < arrayTwo[j])
			{
				i++;
			}
			else
			{
				j++;
			}
		}
	}
	return currentFontList;
}

//--------------------------------------------------------------------
// FUNCTION:
//   findMatchingFonts
//
// DESCRIPTION:
//	Takes a array of font indices and applies the search string on it
//
// ARGUMENTS:
//   arrayOfIndices and the searchString
//
// RETURNS:
//  array containing fonts that match with search string
//--------------------------------------------------------------------

function findMatchingFonts(arrayOfIndex, matchingString)
{
	var currentFontList = new Array();
	if(arrayOfIndex != null)
	{
		if(matchingString != null && matchingString.length > 0)
		{
			matchingString = matchingString.toLowerCase();
			for(var i =0 ; i < arrayOfIndex.length; i++)
			{
				var family = gJsonMap[arrayOfIndex[i]];
				if(family.name.toLowerCase().indexOf(matchingString) != -1)
					currentFontList.push(arrayOfIndex[i]);
			}
		}
		else
		{
			currentFontList = arrayOfIndex;
		}
	}
	
	return currentFontList;
}

//--------------------------------------------------------------------
// FUNCTION:
//   startDisplayFonts
//
// DESCRIPTION:
//	Initiates the display of fonts listed in the current array
//
// ARGUMENTS:
//   arrayOfIndices 
//
// RETURNS:
//  
//--------------------------------------------------------------------

function startDisplayFonts(fontList)
{
	document.getElementById('fontGrid').innerHTML = '';
	if(fontList != null && fontList.length == 0)
	{
		gVisibleFonts = [];
		gLastFontDisplayed = 0;
		return;
	}
	
	document.body.scrollTop = 0;
	gLastFontDisplayed = -1;
    gVisibleFonts = fontList;
	displayFonts(gInitialFontNumber);
	gTimeOutFunc = setInterval(loadMoreFonts, 70);
}

function loadMoreFonts()
{
    var diff = $(window).scrollTop() - ($(document).height() - $(window).height());
	if  (diff >= -gScrollThreshold && diff <= gScrollThreshold){
        displayFonts(gSubsequentFontNumber);
    }
}
	
//--------------------------------------------------------------------
// FUNCTION:
//   displayFonts
//
// DESCRIPTION:
//	displays limited number of fonts by adding them to the DOM
//
// ARGUMENTS:
//  number of fonts to be displayed
//
// RETURNS:
//  
//--------------------------------------------------------------------
function displayFonts(displayNumber)
{	
	if(!displayNumber || displayNumber == 0)
		displayNumber = gInitialFontNumber;
	
	var parent = document.getElementById('fontGrid');
	var range = gVisibleFonts.length - gLastFontDisplayed - 1;
	var upperLimit = range > displayNumber ?  displayNumber : range;
	for(var j = 0; j < upperLimit ; j++)
	{
		gLastFontDisplayed++;
		
		if(gVisibleFonts.length - 1 == gLastFontDisplayed)
		{
			clearInterval(gTimeOutFunc);
		}
			
		var fontIndex = gVisibleFonts[gLastFontDisplayed];
		var family = gJsonMap[fontIndex];
		
		var fontItem = document.createElement("div");
		fontItem.className = "gridData";
		var fontImage = document.createElement("div");
		fontImage.className = "fontImage";
		fontImage.style.backgroundImage = "url('" + gPathToFonts + family.id + ".png')";
		
		var fontBanner = document.createElement("div");
		fontBanner.innerHTML = family.name;
		fontBanner.className = "fontName";
		
		var fontContainer = document.createElement("div");
		fontContainer.className = "fontContainer";
		fontContainer.addEventListener('click',toggleSelection,false);
		fontContainer.addEventListener('mouseover',mouseOverFont,false);
		fontContainer.addEventListener('mouseout',mouseOutFont,false);
		fontContainer.title = family.name;
		fontContainer.id = fontIndex;
		
		var selectedIndex = gSelectedFamilies.indexOf(fontIndex);
		if(selectedIndex != -1)
		{
			addCheckMark(fontContainer, "selected");
			fontContainer.style[gBoxShadowProperty] = gBoxShadowValue;
		}
		
		fontContainer.appendChild(fontImage);
		fontContainer.appendChild(fontBanner);
		fontItem.appendChild(fontContainer)
		parent.appendChild(fontItem);
	}
    
}

//Utility function to add a hover or selected image to the font image cell
function addCheckMark(element, type)
{
	//valid values for type are 'hover' and 'selected'
	var children = element.getElementsByClassName('checkMark');
	var child = children.length != 0 ? children[0] : null;
	if(!child)
	{
		var imgItem = document.createElement("div");
		imgItem.style.background = "url('images/checkmark_" + type + ".png') no-repeat";
		imgItem.className = "checkMark";
		element.appendChild(imgItem);
	}
}

//--------------------------------------------------------------------
// FUNCTIONS:
//   mouseOverFont,mouseOutFont,toggleSelection
//
// DESCRIPTION:
//	handles mouseover, mouseout and click events over the font images
//
// ARGUMENTS:
//  
//
// RETURNS:
//  
//--------------------------------------------------------------------


function mouseOverFont()
{
	if(gSelectedFamilies.indexOf(parseInt(this.id)) == -1)
	{
		addCheckMark(this, "hover");
	}
}

function mouseOutFont(event)
{
	var children = this.getElementsByClassName('checkMark');
	var child = children.length != 0 ? children[0] : null;
	if(gSelectedFamilies.indexOf(parseInt(this.id)) == -1 && event.relatedTarget != child)
	{
		this.removeChild(child);
	}
}

function toggleSelection()
{
	//adds or removes the font from gSelectedFamilies
	var selectedIndex = gSelectedFamilies.indexOf(parseInt(this.id));
	
	var children = this.getElementsByClassName('checkMark');
	var child = children.length != 0 ? children[0] : null;
	
	if( selectedIndex != -1)
	{
		gSelectedFamilies.splice(selectedIndex,1);
		//if show selected fonts mode is selected and user deselects a font, remove the deselected font
		if(gShowSelectedFonts)
		{
			var visibleIndex = gVisibleFonts.indexOf(parseInt(this.id));
			gVisibleFonts.splice(visibleIndex,1);
			var parent = this.parentNode;
			var grid = document.getElementById('fontGrid');
			grid.removeChild(parent);
		}
		else
		{
			this.removeChild(child);
			addCheckMark(this, "hover");
			this.style[gBoxShadowProperty] = "";
		}
		
	}
	else
	{
		gSelectedFamilies.push(parseInt(this.id));
		this.removeChild(child);
		addCheckMark(this, "selected");
		this.style[gBoxShadowProperty] = gBoxShadowValue;
	}
	return false;
}


//--------------------------------------------------------------------
// FUNCTION:
//   classifyFonts
//
// DESCRIPTION:
//	classify fonts into gJsonMap, gPrimaryFilterData, gSecondaryFilterData
//
// ARGUMENTS:
//  sorted json Data
//
// RETURNS:
//  
//--------------------------------------------------------------------
function classifyFonts(jsonData){

	for(var i = 0; i < gPrimaryFiltersLength ; i++)
	{
		gPrimaryFilterData[i] = new Array();
	}
	
	for(var i = 0; i < gSecondaryFiltersLength ; i++)
	{
		gSecondaryFilterData[i] = new Array();
	}
	
	for (var i = 0; i < jsonData.families.length; i++) {
		gJsonMap[i] = jsonData.families[i];
		gAllVisiblityFontList[i] = i;
		gIDtoIndexMap[jsonData.families[i].id] = i;
		for (var j = 0; j < jsonData.families[i].classifications.length; j++){
			gPrimaryFilterData[getFilterID(jsonData.families[i].classifications[j])].push(i);
		}
		switch(jsonData.families[i].recommended_for.length)
		{
			case 1:
				gSecondaryFilterData[getRecommendationID(jsonData.families[i].recommended_for[0])].push(i);
				break;
			case 2:
				//if it is recommended for both heading and paragraph, add it to individual list as well as 'both' list
				gSecondaryFilterData[getRecommendationID("both")].push(i);
				gSecondaryFilterData[getRecommendationID("head")].push(i);
				gSecondaryFilterData[getRecommendationID("para")].push(i);
				break;
			default:
				break;
		}
	}
	
	if(gPreSelectedFonts)
	{
		for(var j = 0; j < gPreSelectedFonts.length; j++)
		{
			var index = gIDtoIndexMap[gPreSelectedFonts[j]];
			if(index || index == 0)
				gSelectedFamilies.push(index);
		}
	}
}


//--------------------------------------------------------------------
// FUNCTION:
//   addSelectedFontsToDreamweaver
//
// DESCRIPTION:
//	adds the selected fonts to dreamweaver
//
// ARGUMENTS:
//  none
//
// RETURNS:
//  none
//
// Please Note that the functions clearFreeFontList() and addFreeFont()
// are added to this window object through CPP code in EdgeWebFontsView.cpp
//--------------------------------------------------------------------
function addSelectedFontsToDreamweaver()
{
	clearFreeFontList();
	var selectedFonts = getSelectedFonts();
	if(selectedFonts)
	{
		for(var k = 0; k < selectedFonts.length; k++)	
		{
			var family = selectedFonts[k];
			var fontVariationList = "";
			for (var j = 0; j < family.variations.length; j++)
			{
				var vID = family.variations[j].fvd;
				var vName = family.variations[j].name;
				//The variations are passed in a string with pattern id~name^id~name^
				fontVariationList += vID + gIDNameSeparator + vName + gVariationSeparator;
			}
			var trackUsage = gPreSelectedFonts.indexOf(family.id) != -1 ? false : true;
			addFreeFont(family.id, family.name, family.slug, fontVariationList, family.css_stack, trackUsage);
		}
	}
}
