/*
Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 

Purpose:
Initialization of Preview Feature and Preview Bridging

*/
/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global dw,DWfile,DWLP_FILEPATHS*/

function DevicePreviewBridgingObject(browser) {
    /*
        Member variables : browser, browser object window and dreamweaver configuration path 
    */
    'use strict';
    this.m_browser = browser;
    this.m_context = this;
    this.m_browserWindow = browser.getWindow();
    this.m_devicePreviewConfigPath = dw.getConfigurationPath() + DWLP_FILEPATHS.DevicePreviewPath;
}

DevicePreviewBridgingObject.prototype.applyCallback = function (argsObj, parameter) {
	"use strict";
    if (typeof argsObj.callback === "function") {
        if (parameter !== undefined && parameter !== null) {
            argsObj.callback(parameter);
        } else {
            argsObj.callback();
        }
    }
};

//--------------------------------------------------------------------
// FUNCTION:
//   launchPreviewURL
//
// DESCRIPTION:
//  Launches the URL in default system browser
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.launchPreviewURL = function (argsObj) {
    "use strict";
    if (dw.devicePreview) {
        dw.devicePreview.launchPreviewURL(argsObj.url);
    }
    this.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setbrowserpreviewURL
//
// DESCRIPTION:
//  Launches the URL in default system browser
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.setbrowserpreviewURL = function (argsObj) {
    "use strict";
    if (dw.devicePreview) {
        dw.devicePreview.setBrowserPreviewURL(argsObj);
    }
    this.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setupJSBridging
//
// DESCRIPTION:
//  Set up the JS Bridging Infrastructure between the Device Preview window and Dw
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.setupJSBridging = function () {
	"use strict";
    var browserWindow = this.m_browserWindow,
		self = this;
    // set up the JS Bridging Functions
	browserWindow.DWDevicePreviewCallJsBridgingFunction = function (funcName, argsObj) {
		if (funcName && self[funcName]) {
			dw.logDebugMsg("function called: " + funcName);
			self[funcName](argsObj);
		} else {
			dw.logDebugMsg("invalid function called: " + funcName + "\n" + funcName + " not defined in DevicePreviewBridgingObject");
		}
	};
};

//--------------------------------------------------------------------
// FUNCTION:
//   logPreviewEvent
//
// DESCRIPTION:
//  Call the devicePreview Object to log preview event in Dw
//
// ARGUMENTS:
//   argsObj.eventname :- eventname for log
//	 argsObj.subevent  :- subevent for log
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.logPreviewEvent = function (argsObj) {
    "use strict";
    dw.logDebugMsg(argsObj.eventcategory + argsObj.eventname + argsObj.subevent);
	
    if (argsObj.eventcategory) {
        dw.logEvent(argsObj.eventcategory, argsObj.eventname, argsObj.subevent);
    } else {
        dw.logEvent(argsObj.eventname, argsObj.subevent);
    }
};

//--------------------------------------------------------------------
// FUNCTION:
//   logDataGroupPreviewEvent
//
// DESCRIPTION:
//  Function to log an event as a data group in Dreamweaver
//
// ARGUMENTS:
//   argsObj.eventname :- eventname for log
//	 argsObj.subevent  :- subevent for log
//   argsObj.dataGroup :- array of strings where each string is of form "key"+":"+"value"
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.logDataGroupPreviewEvent = function (argsObj) {
    "use strict";
    
    //while passing between browser and the dw bridging layer, 
    //array has been modified and ahve indexes are changed to string
    var i, valArray = [];
    for (i = 0; i < argsObj.dataGroup.length; ++i) {
        valArray.push(argsObj.dataGroup[String(i)]);
    }
    
    if (argsObj.eventcategory) {
        dw.logDataGroupEvent(argsObj.eventcategory, argsObj.eventname, argsObj.subevent, valArray);
    } else {
        dw.logDataGroupEvent("", argsObj.eventname, argsObj.subevent, valArray);
    }
};

//--------------------------------------------------------------------
// FUNCTION:
//   increaseWindowHeight
//
// DESCRIPTION:
//  increase the pop up window height by the provided margin
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.increaseWindowHeight = function (argsObj) {
    "use strict";
    if (dw.devicePreview) {
        dw.devicePreview.increaseWindowHeight(argsObj.height);
    }
    this.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   updateLiveViewViewport
//
// DESCRIPTION:
//  This function updates the liveview viewport size.
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.updateLiveViewViewport = function (argsObj) {
    "use strict";
    if (dw.devicePreview) {
        dw.devicePreview.updateLiveViewViewport(argsObj.width, argsObj.height);
    }
    this.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setLiveViewInspectMode
//
// DESCRIPTION:
//  This function sets the LiveInspect mode on/off.
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.setLiveViewInspectMode = function (argsObj) {
    "use strict";
    if (dw.devicePreview) {
        dw.devicePreview.setLiveViewInspectMode(argsObj.mode);
    }
    this.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   showOrHidePreviewPopup
//
// DESCRIPTION:
//  This function shows/hides the device preview popup
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.showOrHidePreviewPopup = function (argsObj) {
    "use strict";
    if (dw.devicePreview) {
        dw.devicePreview.showOrHidePreviewPopup(argsObj.show);
    }
    this.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   getPreviewError
//
// DESCRIPTION:
//  Fetch the error for dreamweaver
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.getPreviewError = function (argsObj) {
    "use strict";
    var error = 0;
    if (dw.devicePreview) {
        error = dw.devicePreview.getPreviewError();
    }
    this.applyCallback(argsObj, error);
};


//--------------------------------------------------------------------
// FUNCTION:
//   updateDeviceCount
//
// DESCRIPTION:
//  Fetch the error for dreamweaver
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.updateDeviceCount = function (argsObj) {
    "use strict";
    var error = 0;
    if (dw.devicePreview) {
        error = dw.devicePreview.updateDeviceCount(argsObj.count);
    }
    this.applyCallback(argsObj, error);
};

//--------------------------------------------------------------------
// FUNCTION:
//   getDefaultbrowserList
//
// DESCRIPTION:
//  Fetch the list of installed browsers for dreamweaver
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.getDefaultbrowserList = function (argsObj) {
    "use strict";
    var browsers = [];
    if (dw.devicePreview) {
        browsers = dw.devicePreview.getDefaultBrowserList();
    }
    this.applyCallback(argsObj, browsers);
};

//--------------------------------------------------------------------
// FUNCTION:
//   getPrimaryBrowser
//
// DESCRIPTION:
//  Fetch the primary browser for dreamweaver
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.getPrimaryBrowser = function (argsObj) {
    "use strict";
    var primaryBrowser = dw.getPrimaryBrowser();
    this.applyCallback(argsObj, primaryBrowser);
};

//--------------------------------------------------------------------
// FUNCTION:
//   EditBrowserList
//
// DESCRIPTION:
//  Edit list of installed browsers
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.EditBrowserList = function (argsObj) {
    "use strict";
	if (dw.devicePreview) {
        dw.devicePreview.editBrowserList();
    }
};

//--------------------------------------------------------------------
// FUNCTION:
//   LaunchURLinBrowser
//
// DESCRIPTION:
//  Edit list of installed browsers
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  nothing 
//--------------------------------------------------------------------
DevicePreviewBridgingObject.prototype.LaunchURLinBrowser = function (argsObj) {
    "use strict";
    if (dw.devicePreview) {
        dw.launchPIB(argsObj.theBrowser);
    }
};