/*
    Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 

    Purpose:
        SocketApp is responsible for 
        Handling Socket communication for Preview feature.
        Socket Handlers for Preview Operations.
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global io, $, URI */

function SocketApp() {
    //member variables
    'use strict';
    this._socket = null;
    this._initialized = false;
	this._userid = '';
    /*********************************************************************************************
    //get/set functions
    *********************************************************************************************/
    /*
        This function returns the socket object connected with the preview service.
    */
    this.getSocket = function () {
        return this._socket;
    };
    /*********************************************************************************************
    //member functions    
    *********************************************************************************************/
    
    /*
        Initialization of socket and all the handlers to
        communicate with the preview service.
    */
    this.initWebsocket = function (ip) {
        var userId = window.preview.main.getUserId(),
            sessionId = window.preview.main.getSessionId();
        
		if (this._socket) {
            this._socket.io.reconnect();
        } else {
			var hostURL = window.preview.config.DW_SERVER_PROTOCOL + '://' + ip + ':' + window.preview.config.DW_SERVER_PORT + '/';
			if (userId) {
                if (sessionId) {
					this._socket = io.connect(hostURL, {'sync disconnect on unload': true, query: "userid=" + userId + "&sessionid=" + sessionId + "&controller=true"});
				} else {
				    this._socket = io.connect(hostURL, {'sync disconnect on unload': true, query: "userid=" + userId + "&controller=true"});
				}
                    
			} else {
                this._socket = io.connect(hostURL, {'sync disconnect on unload': true, query: "controller=true"});
			}
        }
    };
    
    /*
        Disconnect the socket.
    */
    this.kickAllDevicesOut = function () {
        var socket = this.getSocket();
        socket.emit('disconnect:yourself', window.preview.config.RECONNECT_ERROR_MARKER);
    };
    
    /*
        Disconnect the socket.
    */
    this.teardown = function () {
        this.removeSocketHandlers();
        this._socket.io.disconnect();
        this._initialized = false;
    };
    
    this.removeSocketHandlers = function () {
        var socket = this.getSocket();
        socket.removeAllListeners('connect');
        socket.removeAllListeners('check:authorization:cb');
        socket.removeAllListeners('join:room:cb');
        socket.removeAllListeners('new:client:connected');
        socket.removeAllListeners('client:disconnected');
		socket.removeAllListeners('dw:log:event');
        socket.removeAllListeners('query:device:viewport:dw:cb');
    };
    
	this.initSocketCallbacks = function () {
        var socket = this.getSocket(),
            userId = window.preview.main.getUserId();

        socket.on('connect', function () {
            socket.emit('check:authorization', userId);
        });
        
		socket.on('check:authorization:cb', function (isauhtorized) {
			if (isauhtorized) {
				socket.emit("join:room", userId);
			}
		});

        socket.on('join:room:cb', function (data) {
            if (data.success === '0') {
				console.log("Dreamweaver is ready for preview");
            }
        });
		
        socket.on("new:client:connected", function (clientData) {
            window.preview.handlers.newDeviceConnected(clientData);
        });
        
        socket.on("client:disconnected", function (clientId) {
            window.preview.handlers.deviceDisconnected(clientId);
        });
        
        socket.on('dw:log:event', function (eventParams) {
            window.preview.metricsController.trackEvent(eventParams.eventName, eventParams.params);
        });
        
        socket.on('query:device:viewport:dw:cb', function (viewportSize) {
            if (viewportSize.width && viewportSize.height) {
                window.DWDevicePreviewCallJsBridgingFunction("updateLiveViewViewport", viewportSize);
            }
        });
	};
	
	this.configureWebsocket = function () {
        var socket = this.getSocket();
        if (socket) {
            this.initSocketCallbacks();
        } else {
            console.log("socket null");
        }
	};
    
    this.purgeUrl = function (url) {
        if (!url) {
            return '';
        }
        
        var urlObj = new URI(url),
            newUrl;
        
        newUrl = (urlObj.protocol() ? urlObj.protocol() + "://" : '')
            + window.preview.config.DW_SERVER_HOST
            + (window.preview.config.DW_SERVER_PORT ? ':' + window.preview.config.DW_SERVER_PORT : '')
            + '/content/files/'
            + urlObj.pathname();

        return newUrl;
    };
    
    this.getRootUrl = function (url) {
        var urlObj = new URI(url);
        return {
            'protocol': urlObj.protocol(),
            'hostname' : urlObj.hostname(),
            'port': urlObj.port()
        };
    };
    
    /*
        TODO: This function receives notification that file has been saved.
        Broadcasts messages to all the connected clients to refresh the URL.
    */
    this.broadcastUrlHandler = function (url) {
        var rootUrl = this.getRootUrl(url),
            purgedUrl = this.purgeUrl(url),
            socket = this.getSocket();
        
        if (socket) {
            var proxyUrlInfo = {
                'protocol': rootUrl.protocol,
                'hostname': rootUrl.hostname,
                'port':   rootUrl.port ? parseInt(rootUrl.port, 10) : 80
            };
            
            if (proxyUrlInfo.hostname && proxyUrlInfo.protocol !== 'file') {
                socket.emit('update:proxy:url', proxyUrlInfo);
            } else {
                socket.emit('update:proxy:url', null);
            }
            
            socket.emit('change:url', purgedUrl);
            
	    } else {
	        console.log("socket null");
        }
    };
    
    this.broadcastRefreshUrlHandler = function (url) {
        var socket = this.getSocket();
        
        if (socket) {
            socket.emit('refresh:url', null);
	    } else {
	        console.log("socket null");
        }
    };
    
    this.sendMouseHoverIn = function (clientData) {
        var socket = this.getSocket();
        socket.emit('mouse:hover:in', clientData);
    };
    
    this.sendMouseHoverOut = function (clientId) {
        var socket = this.getSocket();
        socket.emit('mouse:hover:out', clientId);
    };
    
    this.sendScrollCommand = function (scrollParams) {
        var socket = this.getSocket();
        if (socket) {
            socket.emit('scroll:dw', scrollParams);
        }
    };
    
    this.sendMouseHoverOutIfAny = function () {
        var socketId = window.preview.handlers.getHoveredSocketId();
        if (socketId) {
            this.sendMouseHoverOut(socketId);
        }
    };
    
    this.queryDeviceViewportCommand = function (socketId) {
        var socket = this.getSocket();
        if (socket) {
            socket.emit('query:device:viewport:dw', socketId);
        }
    };
    
    this.sendPreviewUrl = function (previewUrl) {
        var socket = this.getSocket();
        if (socket) {
            socket.emit('update:preview:service:url', previewUrl);
        }
    };
    
    this.sendInspectOffCommand = function (socketId) {
        var socket = this.getSocket();
        if (socket) {
            socketId = socketId || '';
            socket.emit('inspect:off:dw', socketId);
        }
    };
    
    this.sendInspectCommand = function (liveId, selectorString) {
        var socketId = window.preview.handlers.getInspectedSocketId(),
            inspectData = {
                'socketId': socketId,
                'liveId': liveId,
                'selectorString': selectorString
            },
            socket = this.getSocket();
        
        if (socket) {
            socket.emit('inspect:dw', inspectData);
        }
    };
    
    this.initializeSocketApp = function (ip) {
        if (!this._initialized) {
            this.initWebsocket(ip);
            this.configureWebsocket();
            this._initialized = true;
        }
    };
    /* TODO: Implementation of socket handlers.  */
}

window.preview.socketApp = window.preview.socketApp || new SocketApp();