import gdb
from freertos.List import ListManager
import json


def get_task_name_by_handle(handle, tcb_type):
    tcb_ptr = gdb.Value(handle).cast(tcb_type.pointer())
    tcb = tcb_ptr.dereference()
    return tcb['pcTaskName'].string()


def get_curr_task(config, tcb_type):
    curr_tcb_ptr, _ = gdb.lookup_symbol("pxCurrentTCB")
    curr_tcb = curr_tcb_ptr.value().cast(tcb_type.pointer()).dereference()
    result = prepare_task_data(None, curr_tcb, "Running", config)
    return json.dumps(result)


def get_freertos_tasks(config, types):
    running_task_ptr = int(gdb.parse_and_eval("pxCurrentTCB"))
    blocked = ListManager("xSuspendedTaskList", types)
    delayed1 = ListManager("xDelayedTaskList1", types)
    delayed2 = ListManager("xDelayedTaskList2", types)
    ready_lists = [ListManager("xPendingReadyList", types)]

    ready_tasks_lists_name = "pxReadyTasksLists"
    ready_lists_sym, _ = gdb.lookup_symbol(ready_tasks_lists_name)
    if ready_lists_sym is not None:
        _ready_lists = ready_lists_sym.value()  # List_t [configMAX_PRIORITIES]
        min_index, max_index = _ready_lists.type.range()  # 0, configMAX_PRIORITIES - 1
        for i in range(min_index, max_index + 1):
            ready_lists.append(ListManager(_ready_lists[i], types))
    else:
        raise ValueError("Failed to Find Symbol: %s" % ready_tasks_lists_name)

    if config.generate_runtime_stats:
        total_runtime, _ = gdb.lookup_symbol("ulTotalRunTime")
        if total_runtime is None:
            config.generate_runtime_stats = False

    result = []
    for i, rlist in enumerate(ready_lists):
        items = rlist.get_elements(types.tcb_type)
        if len(items) > 0:
            for tcb, val in items:
                result.append(prepare_task_data(running_task_ptr, tcb, "Ready", config))

    items = blocked.get_elements(types.tcb_type)
    for tcb, val in items:
        result.append(prepare_task_data(running_task_ptr, tcb, "Blocked", config))

    items = delayed1.get_elements(types.tcb_type)
    for tcb, val in items:
        result.append(prepare_task_data(running_task_ptr, tcb, "Delayed", config))

    items = delayed2.get_elements(types.tcb_type)
    for tcb, val in items:
        result.append(prepare_task_data(running_task_ptr, tcb, "Delayed", config))

    if not config.use_trace_facility:  # have to take care of the identifiers ourselves
        result.sort(key=lambda tsk: tsk['t_name'])
        task_index = 1
        for task in result:
            task['t_id'] = str(task_index)
            task_index += 1
    else:
        result.sort(key=lambda tsk: tsk['t_id'])

    return json.dumps(result)


def prepare_task_data(running_task_ptr, task, state, config):  # : gdb.Value, : str
    task_addr = int(task.address)

    if task_addr == running_task_ptr:
        state = "Running"

    # if ( configUSE_TRACE_FACILITY == 1 )
    #   UBaseType_t uxTaskNumber; /*< Stores a number specifically for use by third party trace code.* /
    # #endif
    if config.use_trace_facility:
        task_number = int(task['uxTCBNumber'])
    else:
        task_number = -1

    try:
        pc_buff = gdb.selected_inferior().read_memory(int(task['pxTopOfStack']) + 56, 4)
        pc = ''.join('%.2x' % ord(pc_buff[i]) for i in reversed(range(len(pc_buff))))
    except gdb.error:
        pc = 0

    return {
        't_name': task['pcTaskName'].string(),
        't_state': state,
        't_id': str(task_number),
        's_top_addr': str(int(task['pxTopOfStack'])),
        't_pc': '0x%s' % pc
    }
