; MODULE_ID IGESOUT_LSP_
;;;----------------------------------------------------------------------------
;;;
;;;   IGESOUT.LSP
;;;   Copyright (C) 1992-1996 by Autodesk, Inc.
;;;
;;;   Permission to use, copy, modify, and distribute this software
;;;   for any purpose and without fee is hereby granted, provided
;;;   that the above copyright notice appears in all copies and that
;;;   both that copyright notice and this permission notice appear in
;;;   all supporting documentation.
;;;
;;;   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED
;;;   WARRANTY.  ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR
;;;   PURPOSE AND OF MERCHANTABILITY ARE HEREBY DISCLAIMED.
;;;
;;;----------------------------------------------------------------------------
;;; Description:
;;;   This module contains proteus interface programming for IGESOUT GUI.
;;;
;;; Author:  Data Exchange Group
;;;          Autodesk, Inc.
;;;
;;;----------------------------------------------------------------------------
;;; Check to see if AI_UTILS is loaded, If not, try to find it,
;;; and then try to load it.
;;;
;;; If it can't be found or it can't be loaded, then abort the
;;; loading of this file immediately, preserving the (autoload)
;;; stub function.
  (cond
    ((and ai_dcl (listp ai_dcl)) )            ; it's already loaded.
    ((not (findfile ;|MSG0|;"ai_utils.lsp"))  ; find it
      (ait_abort ;|MSG0|;"IGESOUT" 
		 (strcat ;|MSG1|;"Can't locate file ai_utils.lsp."
			 ;|MSG2|;"\n Check support directory.")
		 ;|MSG0|;"igesout"
		 igesout))
    ((eq ;|MSG0|;"failed" (load ;|MSG0|;"ai_utils" ;|MSG0|;"failed")) ; load it
      (ait_abort ;|MSG0|;"IGESOUT" 
		 ;|MSG3|;"Can't load file ai_utils.lsp"
		 ;|MSG0|;"igesout"
		 igesout))
  )
  (if (not (ai_acadapp))     ; defined in ai_utils.lsp
    (ait_abort ;|MSG0|;"IGESOUT" nil 
	       ;|MSG0|;"igesout" igesout)  ; a Nil <msg> supresses
  )                                        ; ait_abort's alert box dialog.

  ;; Initialize AutoSurf dialog support.
  (setq arxlist (arx))
  (if (member "asrf.arx" arxlist)
	(setq asurf T)
	(setq asurf nil) ;; disable AutoSurf option usage
  )

;;; ==================== end load-time operations ===========================

;;;----------------------------------------------------------------------------
;;; The main IGESOUT function.
;;;----------------------------------------------------------------------------
(defun c:igesout ( / old_cmd)

  ;; Set up error function.
  (setq old_cmd (getvar ;|MSG0|;"CMDECHO")    ; save current setting of cmdecho
	old_error  *error*            ; save current error function
	*error* ai_error              ; new error function
  )
  (setvar ;|MSG0|;"cmdecho" 0)
  (cond
    ((not (ait_cmdactive (+ 1 2 8))))         ; transparent/dialog box not OK
    ((not (ai_acadapp)) )                      ; ACADAPP.EXP xloaded?
    ((or (= 0 (getvar ;|MSG0|;"cmddia"))               ; Command line version
	 (= 4 (logand (getvar ;|MSG0|;"cmdactive") 4)) ; also do if script running
     ) 
      (if (ait_rxload ;|MSG0|;"igesout" igesout)
	(igesout)
      )
      (ait_rxunload ;|MSG0|;"igesout" igesout)
    )
    (T                                         ; Proteus version 
      (ddigesout)
    )
  )
  (princ)
)
;;
;;
;;;---------------------------------------------------------------------------
;;; The alternative -IGESOUT function.
;;;----------------------------------------------------------------------------
(defun c:-igesout ( / old_cmd)
  ;; Set up error function.
  (setq old_cmd (getvar ;|MSG0|;"CMDECHO")    ; save current setting of cmdecho
	old_error  *error*                    ; save current error function
	*error* ai_error                      ; new error function
  ) 
  (setvar ;|MSG0|;"CMDECHO" 0)
  (ait_rxunload ;|MSG0|;"igesout" igesout) 
  (cond 
    ((not (ait_cmdactive (+ 1 2 8))))    ; transparent/dialog box not OK
    ((not (ai_acadapp)))                 ; ACADAPP loaded?
    (if (ait_rxload ;|MSG0|;"igesout" igesout)
      (igesout) 
    )
  ) 
  (ait_rxunload ;|MSG0|;"igesout" igesout)
  (princ)
)
;;
;;
(defun ddigesout (/
  a pickf pickigsf dcl_id optname what_next1 dname tmode lfiles canopts
  curoptdesc sfp_list mfp_list dwg_list soptname moptname igesext igs_name
  tmp_opts desctxt canflag tmp_list f f1 sg_opts str_opts geo_opts anno_opts
  demap_opts idp_opts misc_opts allopts modflag fcnt proceed defopts scrpath 
  asurf_opts 
  curr defs invdir posrn saving lnf0 lnf1 lnf2 lnf3 lnf4 lnf5
  u_inch u_mm u_user u_feet u_mile u_m u_km u_mils u_um u_cm u_uin
		 )


  (ait_rxload ;|MSG0|;"igesgui" ISXOGD)

  ;; Setup messages
  (setq
     curr   ;|MSG4|;"*Current*"
     defs   ;|MSG5|;"*Defaults*"
     invdir ;|MSG6|;"Invalid directory."
     posrn  ;|MSG7|;"Must be a positive real number."
     saving ;|MSG8|;"Saving... Please Wait"
     lnf0   ;|MSG9|;"0 None"
     lnf1   ;|MSG10|;"1 Solid"
     lnf2   ;|MSG11|;"2 Dashed"
     lnf3   ;|MSG12|;"3 Phantom"
     lnf4   ;|MSG13|;"4 Centerline"
     lnf5   ;|MSG14|;"5 Dotted"
     u_inch ;|MSG15|;"Inches (1)"
     u_mm   ;|MSG16|;"Millimeters (2)"
     u_user ;|MSG17|;"User Defined (3)"
     u_feet ;|MSG18|;"Feet (4)"
     u_mile ;|MSG19|;"Miles (5)" 
     u_m    ;|MSG20|;"Meters (6)"
     u_km   ;|MSG21|;"Kilometers (7)"
     u_mils ;|MSG22|;"Mils[0.001 in] (8)"
     u_um   ;|MSG23|;"Microns (9)" 
     u_cm   ;|MSG24|;"Centimeters (10)"
     u_uin  ;|MSG25|;"Microinches (11)"
  )


;;
;;
(defun greylists()
  (if (= tmode ;|MSG0|;"1")
    (progn
      (if dwg_list (mode_tile ;|MSG0|;"clr_all" 0))
      (if (read (get_tile ;|MSG0|;"dwg_list"))
	(progn
	  (mode_tile ;|MSG0|;"out_file" 0)
	  (mode_tile ;|MSG0|;"del_cur_out" 0)
	)
	(progn
	  (mode_tile ;|MSG0|;"out_file" 1)
	  (mode_tile ;|MSG0|;"del_cur_out" 1)
	)
      )
    )
    (progn
      (mode_tile ;|MSG0|;"del_cur_out" 1)
      (mode_tile ;|MSG0|;"clr_all" 1)
      (if (read (get_tile ;|MSG0|;"fp_list"))
	(mode_tile ;|MSG0|;"out_file" 0)
	(mode_tile ;|MSG0|;"out_file" 1)
      )
    )
  )
)
;;
;;
(defun greydir (path_on)
  (if (and (= (get_tile ;|MSG0|;"iges_path") ;|MSG0|;"spec_path") path_on)
    (mode_tile ;|MSG0|;"outdir" 0)
    (mode_tile ;|MSG0|;"outdir" 1)
  )
)
;;
;;
(defun grey_modes ()
  (if (= tmode ;|MSG0|;"0")
    (progn
      (mode_tile ;|MSG0|;"add_to_list" 1)
      (mode_tile ;|MSG0|;"iges_path" 1)
      (greydir nil)
      ;; (mode_tile ;|MSG0|;"iges_ext" 1)
    )
    (progn
      (mode_tile ;|MSG0|;"add_to_list" 0)
      (mode_tile ;|MSG0|;"iges_path" 0)
      (greydir T)
      ;; (mode_tile ;|MSG0|;"iges_ext" 0)
    )
  )
  (greylists)
)
;;
;;
(defun rs_err ()
  (set_tile ;|MSG0|;"error" ;|MSG0|;"")
)
;;
;;
(defun read_fls (/ noerr validf)
  (setq noerr
    (cond
      ((not (setq f (open ;|MSG0|;"igesout.cfg" ;|MSG0|;"r"))) nil)
      ((not (or (= ;|MSG0|;"0" (setq tmode (read-line f))) 
		(= ;|MSG0|;"1" tmode)))
	(setq tmode ;|MSG0|;"0") nil)
      ((/= ;|MSG0|;"single" (read-line f))
	nil)
      ((not (or (findfile (setq soptname (read-line f)))
		(= soptname defs)))     ; single opt filename
	(setq soptname defs) nil)
      ((/= ;|MSG0|;"multiple" (read-line f))
	nil)
      ((not (chkdir (setq dname (read-line f))))         ; out directory
	(setq dname nil))
      ((not (check_igsext (setq igesext (read-line f)))) ; file extension
	(setq igesext ;|MSG0|;"igs") nil)
      ((not (or (= defs (setq moptname (read-line f)))
		(findfile moptname)))                    ; multi opt filename
	(setq moptname defs) nil)
      ((not (is_posint (itoa (setq lfiles (atoi (read-line f)))))) ;file count
	(if (= 0 lfiles) T nil))
      ((not (progn
	 (setq fcnt lfiles validf T)
	 (while (and (> fcnt 0) validf)
	   (if (setq validf (setq a (read-line f))) ; if not end of file
	     (if (setq validf (findfile a))         ; if file exists
	       (setq dwg_list (cons a dwg_list))
	     )
	   )
	   (setq fcnt (1- fcnt))
	 )
	 (if validf
	   (setq dwg_list (reverse dwg_list))
	   (setq dwg_list  nil)
	 )
       ))
	nil)
      ((not (progn
	 (setq fcnt lfiles)
	 (while (and (> fcnt 0) validf)
	   (if (setq validf (setq a (read-line f))) ; if not end of file
	     (if (setq validf (chkdir (strpfn a)))  ; if valid directory
	       (setq mfp_list (cons a mfp_list))
	     )
	   )
	   (setq fcnt (1- fcnt))
	 )
	 (if validf
	   (setq mfp_list (reverse mfp_list))
	   (setq mfp_list nil dwg_list  nil)
	 )
       ))
	nil)
      ((not (read-line f)) T)
      (t (setq mfp_list nil dwg_list  nil))
    )
  )
  (if f
    (close f)
    (setq noerr T)
  )
  noerr
)
;;
;;
(defun is_posint (value / tmpval)
  (setq tmpval (distof value))
  (cond
    ((= tmpval nil) nil)  ; not a number
    ((< tmpval 1) nil)    ; less than 1
    ((/= (/ tmpval (fix tmpval)) 1) nil) ; not an integer.
    (t value)
  )
)
;;
;;
(defun checkigslist (/ n)
  (setq canflag 0)
  (foreach n fp_list
    (if (findfile n)
      (alertusr n)
      (set_tile ;|MSG0|;"error" ;|MSG26|;"Please Wait...")
    )
  )
  (rs_err)
)
;;
;;
(defun alertusr (dupname)
  (if (= canflag 0)
    (progn
      (if (not (new_dialog ;|MSG0|;"dupfile" dcl_id))(exit))
      (set_tile ;|MSG0|;"dupfilename" dupname)
      (action_tile ;|MSG0|;"accept" ;|MSG0|;"(setq canflag 0)(done_dialog 40)")
      (action_tile ;|MSG0|;"cancel" ;|MSG0|;"(setq canflag 1)(done_dialog 41)")
      (setq what_next1 (start_dialog))
    )
  )
)
;;
;;
(defun save_env ()
  (setq f (open ;|MSG0|;"igesout.cfg" ;|MSG0|;"w"))
  (write-line tmode f)
  (write-line ;|MSG0|;"single" f)
  (if (= soptname curr)
    (write-line defs f)
    (write-line soptname f)
  )
  (write-line ;|MSG0|;"multiple" f)
  (write-line dname f)
  (write-line igesext f)
  (if (= moptname curr)
    (write-line defs f)
    (write-line moptname f)
  )
  (write-line (itoa lfiles) f)
  (if dwg_list
    (foreach n dwg_list (write-line n f))
  )
  (if mfp_list
    (foreach n mfp_list (write-line n f))
  )
  (close f)
)
;;
;;
(defun addfile (/ fname)
  (setq fname ;|MSG0|;"")
  (if (and 
	(not (null dwg_list))
	(/= ;|MSG0|;"" pickigsf))
    (if (not (wcmatch pickigsf ;|MSG0|;"* *")) ; test for multiple select
      (setq fname (strpfn (nth (read pickigsf) dwg_list)))
    )
  )
  (if (setq fname 
	    (getfiled ;|MSG27|;"Select DRAWING File" fname ;|MSG0|;"dwg" 2))
    (progn
      (add2lists fname)
      (setq lfiles (length dwg_list))
    )
  )
)
;;
; Add a file to the internal lists for display
(defun add2lists (filename / igsfilename)
  (if (not (member filename dwg_list))
    (progn
      (setq igsfilename 
	(strcat (if (= (get_tile ;|MSG0|;"iges_path") ;|MSG0|;"spec_path")
		  (strcat (addendslash dname) (strpdirs filename))
		  (strpext filename)
		)
		;|MSG0|;"." igesext)
      )
      (if (member igsfilename mfp_list)
	(alert ;|MSG28|;"**Output File exists in the list**\n\nChange Output Filename"))
      (setq dwg_list (append dwg_list (list filename)))
      (setlist dwg_list ;|MSG0|;"dwg_list")
      (setq mfp_list (append mfp_list (list igsfilename)))
      (setq fp_list mfp_list)
      (setlist fp_list ;|MSG0|;"fp_list")
      (set_tile ;|MSG0|;"dwg_list" (itoa (what_pos filename dwg_list)))
      (set_tile ;|MSG0|;"fp_list" (itoa (what_pos filename dwg_list)))
      (setq pickf (get_tile ;|MSG0|;"dwg_list"))
      (greylists)
    )
  )
)
;
;;
(defun what_pos (item the_list)
  (- (length the_list) (length (member item the_list)))
)
;;
;;
(defun extlen2 (s1 / dupstr)
  (setq dupstr s1)
  (setq strindex 1)
  (while (wcmatch dupstr ;|MSG0|;"*`.*")
    (progn
      (setq dupstr (substr s1 strindex))
      (setq strindex (1+ strindex))
    )
  )
  (if (> strindex 1)
    (setq lext (1+ (strlen dupstr)))
    (setq lext 0)
  )
)
;;
;; /*+ .doc
;; Strips the extension from a file name and returns
;; the filename.
;; -*/
;;
(defun strpext (s1)
   (substr s1 1 (- (strlen s1) (extlen2 s1)))
)
;;
;;
(defun strpdirs (s1)
  (setq s1 (strpext s1))
  (substr s1 (findfn s1))
)
;;
;;
(defun strpfn (s1)
  (substr s1 1 (- (findfn s1) 1))
)
;;
;;
(defun findfn (s1 / dupstr)
  (setq dupstr s1 strindex 0)
  (while (or (wcmatch dupstr ;|MSG0|;"*/*")(wcmatch dupstr ;|MSG0|;"*\\*"))
    (setq strindex (1+ strindex))
    (setq dupstr (substr s1 strindex))
  )
  strindex
)
;;
;;
(defun copylists ()
  (cond
    ((= tmode ;|MSG0|;"0") (setq sfp_list fp_list))
    ((= tmode ;|MSG0|;"1") (setq mfp_list fp_list))
    (t nil)
  )
)
;;
;;  
(defun remfile1 (/ pickf_list pickf_no dwg_list1 fp_list1)
  (while (setq pickf_no (read pickf))
    (setq pickf_list (cons pickf_no pickf_list))
    (setq pickf (substr pickf (+ 2 (strlen (itoa pickf_no)))))
  )
  (setq n 0)
  (while (< n (length dwg_list))
    (if (not (member n pickf_list))
      (progn
	(setq dwg_list1 (cons (nth n dwg_list) dwg_list1))
	(setq fp_list1 (cons (nth n fp_list) fp_list1))
      )
    )
    (setq n (1+ n))
  )
  (setq dwg_list (reverse dwg_list1))
  (setq fp_list (reverse fp_list1))
  (if (= tmode ;|MSG0|;"1")(setq lfiles (length dwg_list)))
  (copylists)
  (setlist dwg_list ;|MSG0|;"dwg_list")
  (setlist fp_list ;|MSG0|;"fp_list")
)
;;
;;
(defun clraccept ()
  (setq dwg_list nil fp_list nil pickf ;|MSG0|;"" pickigsf ;|MSG0|;"")
  (copylists)
)
;;
;;
(defun clr_all ()
  (if (not (new_dialog ;|MSG0|;"modealert" dcl_id))(exit))
  (action_tile ;|MSG0|;"accept" ;|MSG0|;"(clraccept)(done_dialog 26)")
  (action_tile ;|MSG0|;"cancel" ;|MSG0|;"(done_dialog 25)")
  (setq what_next1 (start_dialog))
  (setlist dwg_list  ;|MSG0|;"dwg_list")
  (setlist fp_list ;|MSG0|;"fp_list")
  (greylists)
)
;;
;; If it is in single mode invoke the dialog box and set only if they
;; made the choice.
;;
(defun mod_igsname (/ pickf_no old_igs new_igs newext lastch)
  (if (= tmode ;|MSG0|;"0") ; Single mode
    (progn
      (setq new_igs (getfiled ;|MSG29|;"Change Output IGES File" 
			      (car fp_list) igesext 5))
      (if new_igs (setq fp_list (setq sfp_list (list new_igs))))
    )
    (progn
      (setq pickf_no (read pickf))
      (setq old_igs (nth pickf_no fp_list))
      (setq new_igs (getfiled ;|MSG30|;"Change Output IGES File" 
			      old_igs igesext 5))
      (if new_igs (checkdup old_igs new_igs))
    )
  )
  (setlist fp_list ;|MSG0|;"fp_list")
  (set_tile ;|MSG0|;"fp_list" pickf)
  ; Update current IGES extension if changed 
  (setq newext Nil)
  (if (not (null new_igs))
    (progn
      (setq lastch (strlen new_igs))
      (setq newext 
	    (substr new_igs (- lastch (- (extlen2 new_igs) 2)) lastch))
      (if (and (/= igesext newext)
	       (check_igsext newext))
	(setq igesext newext)
      )
    )
  )
) 
;;
;;  This function checks the validity of the file extension.  If legitimate,
;;  the file extension is returned, nil otherwise.
;;
(defun check_igsext (extension / plat)
  (setq plat (getvar ;|MSG0|;"PLATFORM"))
  (cond
    ;; Ensure iges extension is valid for DOS and Windows      
    ((and (> (strlen extension) 3)
	  (or (= ;|MSG0|;"Microsoft" (substr plat 1 9))
	      (= ;|MSG0|;"386 DOS" (substr plat 1 7))))
      nil
    )
    ((wcmatch extension ;|MSG0|;"*[] `#`@`.`?`*`~`[`,`'!%^&()+={}|`\\:;\"<>/]*")
      (set_tile ;|MSG0|;"error" ;|MSG31|;"Invalid character(s) in file extension. ")
      ;; (mode_tile ;|MSG0|;"iges_ext" 3)
      nil
    )
    ((< 31 (strlen extension))
      (set_tile ;|MSG0|;"error" ;|MSG32|;"Exceeds 31 character limit.")
      ;; (mode_tile ;|MSG0|;"iges_ext" 3)
      nil
    )
    (t (rs_err) extension)
  )
)
;;
;;
(defun checkdup (/old_igs new_igs)
  (if (member new_igs fp_list)
    (set_tile ;|MSG0|;"error" ;|MSG33|;"File is already in Output File List.")
    (progn
      (setq fp_list (subst new_igs old_igs fp_list))
      (copylists)
    )
  )
)
;;
;; Update list for a list_box
;;
(defun setlist (listname tilename)
  (if listname
   (progn
     (start_list tilename)
     (mapcar (quote add_list) listname)
     (end_list)
   )
   (progn
     (start_list tilename)
     (end_list)
   )
  )
)
;;
;;
(defun delendslash (s1 / lchar)
  (if (/= ;|MSG0|;"" s1)
    (if (and (wcmatch s1 ;|MSG0|;"~*:`\\")(wcmatch s1 ;|MSG0|;"*?[`/`\\]"))
      (substr s1 1 (- (strlen s1) 1))
      s1
    )
    s1
  )
)
;;
;;
(defun revbackslash (s1 / dupstr si testchar)
  (cond
    ((wcmatch s1 ;|MSG0|;"*\\*")
      (setq dupstr s1 si 0)
      (while (wcmatch dupstr ;|MSG0|;"*\\*")
	(setq testchar (substr dupstr 1 1))
	(if (wcmatch testchar ;|MSG0|;"\\")
	  (setq s1 (strcat (substr s1 1 (1- si)) ;|MSG0|;"/" (substr dupstr 2)))
	)
	(setq si (1+ si))
	(setq dupstr (substr s1 si))
      )
      s1
    )   
    (t s1)
  )
)
;;
;;
(defun addendslash (s1 / vers)
  (setq vers (getvar ;|MSG0|;"PLATFORM"))
  (cond
    ((and (or (= vers ;|MSG0|;"386 DOS Extender") 
	      (= (substr vers 1 9) ;|MSG0|;"Microsoft"))
	  (wcmatch s1 ;|MSG0|;"*?*") (wcmatch s1 ;|MSG0|;"~*:`\\")
	  (wcmatch s1 ;|MSG0|;"~`\\") (wcmatch s1 ;|MSG0|;"~*`\\"))
      (setq s1 (strcat s1 ;|MSG0|;"\\"))
    )
    ((and (/= vers ;|MSG0|;"386 DOS Extender")
	  (/= (substr vers 1 9) ;|MSG0|;"Microsoft")
	  (wcmatch s1 ;|MSG0|;"*?*")(wcmatch s1 ;|MSG0|;"~*/"))  ;unix
      (setq s1 (strcat s1 ;|MSG0|;"/"))
    )
    (t s1)
  )
)
;;
;;
(defun chkdir (dirname / vers dret flname)
  (if (/= ;|MSG0|;"" dirname)
    (progn
      (setq vers (getvar ;|MSG0|;"PLATFORM"))
      (cond
	((or (= vers ;|MSG0|;"386 DOS Extender")
	     (= (substr vers 1 9) ;|MSG0|;"Microsoft"))
	  (setq flname (strcat (addendslash dirname) ;|MSG0|;"igesout.tmp"))
	  (setq f1 (open flname ;|MSG0|;"w"))
	  (if f1
	    (progn
	      (close f1)
	      (if (not (member flname tmp_list))
		(setq tmp_list (append tmp_list (list flname)))
	      )
	    )
	  )
	  (setq dret f1)
	)
	(t (setq dret (open dirname ;|MSG0|;"r"))) ; Default platform (unix)
      )
    )
    t
  )
)
;;
;;
(defun del_tmps ()
  (if tmp_list
    (foreach n tmp_list (command ;|MSG0|;"_del" n))
  )
)
;;
;;
(defun chgigsdir ()
  (if (chkdir $value)
    (Progn
      (setq dname (delendslash $value))
      (rs_err)
    )
    (set_tile ;|MSG0|;"error" invdir)
  )
)
;;
;;
(defun chgoption ( / tmpname)
  (if (isoptfile)
    (setq tmpname optname)
    (setq tmpname ;|MSG0|;"")
  )
  (setq tmpname (getfiled ;|MSG34|;"Select OPTION SETTINGS File" 
			  tmpname ;|MSG0|;"opt" 10))
  (if tmpname
    (progn
      (getallopts (setq optname tmpname))
      (setopt nil)
      (if (= tmode ;|MSG0|;"0")
	(setq soptname optname)
	(setq moptname optname)
      )
    )
  )
)
;;
;;
(defun switchmode (mode)
  (if (or (= mode ;|MSG0|;"batch") (= mode ;|MSG0|;"1"))
    (setq tmode ;|MSG0|;"1")
    (setq tmode ;|MSG0|;"0")
  )
  (cond
    ((= tmode ;|MSG0|;"0")
      (setlist nil ;|MSG0|;"dwg_list")
      (setq fp_list sfp_list)
      (setq optname soptname)
    )
    ((= tmode ;|MSG0|;"1")
      (setlist dwg_list ;|MSG0|;"dwg_list")
      (setq fp_list mfp_list)
      (setq optname moptname)
      ;(set_tile ;|MSG0|;"outdir" dname)
      ;(set_tile ;|MSG0|;"iges_ext" igesext)
    )
    (t nil)
  )
  (setlist fp_list ;|MSG0|;"fp_list")
  (set_tile ;|MSG0|;"fp_list" (setq pickf ;|MSG0|;"0"))
  (setpick)

  (if optname (setopt nil))
)
;;
;;
(defun setopt (tmpdesc)
  (set_tile ;|MSG0|;"curopt" optname)
  (if (not tmpdesc)
    (cond
      ((= optname curr) (setq tmpdesc curoptdesc))
      ((= optname defs) (setq tmpdesc ;|MSG0|;""))
      (T (if (not (setq tmpdesc (ISXOGD (findfile optname))))
	   (setq tmpdesc ;|MSG0|;"")))
    )
  )
  (set_tile ;|MSG0|;"opts_desc_text" tmpdesc)
)
;;
;;
(defun isoptfile ()
  (if (or (= optname defs)
	  (= optname curr)) nil T)
)
;;
;;
(defun waitmsg ()
   (set_tile ;|MSG0|;"error" ;|MSG35|;"Working...Please Wait")
)
;;
;;
(defun optmenu ( / canexit tmpopts tmpsubopts)

  (if (not (new_dialog ;|MSG0|;"optdialog" dcl_id))(exit))
  (setq modflag 0
	tmpopts allopts
	tmpoptn optname
	canexit nil)
  (cond ((= optname defs) )
	((= optname curr) )
	(T (getallopts optname))
  )

  (set_tile ;|MSG0|;"opt_file_name" optname)
  (set_tile ;|MSG0|;"desc_text" (car allopts))
  (action_tile ;|MSG0|;"desc_text" ;|MSG0|;"(setq desctxt $value) (setmflag)")
  (action_tile ;|MSG0|;"start_global" ;|MSG0|;"(waitmsg)(startmenu)(rs_err)")
  (action_tile ;|MSG0|;"struct" ;|MSG0|;"(waitmsg)(structmenu)(rs_err)")
  (action_tile ;|MSG0|;"geom" ;|MSG0|;"(waitmsg)(geomenu)(rs_err)")
  (action_tile ;|MSG0|;"anno" ;|MSG0|;"(waitmsg)(annomenu)(rs_err)")
  (action_tile ;|MSG0|;"demap" ;|MSG0|;"(waitmsg)(demapmenu)(rs_err)")
  (action_tile ;|MSG0|;"idpmap" ;|MSG0|;"(waitmsg)(idpmenu)(rs_err)")
  (action_tile ;|MSG0|;"misclog" ;|MSG0|;"(waitmsg)(logmenu)(rs_err)")
  (action_tile ;|MSG0|;"misc" ;|MSG0|;"(waitmsg)(miscmenu)(rs_err)")
  (action_tile ;|MSG0|;"def_options" ;|MSG0|;"(popdefopts)")
  (action_tile ;|MSG0|;"save_as_options" ;|MSG0|;"(saveasopts)")
  (action_tile ;|MSG0|;"save_options" ;|MSG0|;"(saveinopts)")
  (action_tile ;|MSG0|;"accept" 
	       ;|MSG0|;"(exitopts)(if (not canexit) (done_dialog 19))")
  (action_tile ;|MSG0|;"cancel" ;|MSG0|;"(canallopts)")
  (action_tile ;|MSG0|;"help" 
	       ;|MSG0|;"(ait_helpdlg \"opt_outoptsDialogBox\")")
  (setq what_next1 (start_dialog))
  (if (not canopts) (setopt desctxt))
)
;;
;;
(defun canallopts ()
  (setq canopts T
	desctxt (car tmpopts)
	allopts tmpopts
	optname tmpoptn)
  (setq sg_opts (nth 1 allopts))
  (setq str_opts (nth 2 allopts))
  (setq geo_opts (nth 3 allopts))
  (setq anno_opts (nth 4 allopts))
  (setq demap_opts (nth 5 allopts))
  (setq idp_opts (nth 6 allopts))
  (setq misc_opts (nth 7 allopts))
  (if (not (null asurf))
    (setq asurf_opts (nth 8 allopts))
    (setq asurf_opts nil)
  )
  (done_dialog 17)
)
;;
;;
(defun setofiletxt (ofile)
  (setq optname ofile) 
  (set_tile ;|MSG0|;"opt_file_name" ofile)
  (cond ((= tmode ;|MSG0|;"0")(setq soptname optname))
	((= tmode ;|MSG0|;"1")(setq moptname optname))
	(t nil)
  )
)
;;
;;
(defun getolist ()
  (setq allopts (list desctxt sg_opts str_opts geo_opts anno_opts demap_opts 
		      idp_opts misc_opts asurf_opts))
)
;;
;;
(defun saveinopts (/ savefile err)
  (if (isoptfile)
    (progn
      (setq savefile (get_tile ;|MSG0|;"opt_file_name"))
      (set_tile ;|MSG0|;"error" saving)
      (setq err (ISXOPO savefile (getolist)))
      (if (null err)
         (progn
           (setq modflag 0)
           (rs_err)
         )
         (set_tile ;|MSG0|;"error" err)
      )
    )
    (saveasopts)
  )
)
;;
;;
(defun saveasopts (/ savefile err)
  (if (isoptfile)
    (setq savefile optname)
    (setq savefile ;|MSG0|;"")
  )
  (setq savefile (getfiled ;|MSG36|;"Save As..." savefile ;|MSG0|;"opt" 1))
  (if savefile
    (progn
      (set_tile ;|MSG0|;"error" saving)
      (setq err (ISXOPO savefile (getolist)))
      (if (null err)
         (progn 
           (rs_err)
           (setofiletxt savefile)
           (setq modflag 0)
         )
         (set_tile ;|MSG0|;"error" err)
      )
    )
  )
)
;;
;;
(defun exitopts ()
  (if (= modflag 1)
    (progn                          ; If changes were made and not saved.
      (setq canexit nil)
      (if (not (new_dialog ;|MSG0|;"optmod" dcl_id)) (exit))
      (action_tile ;|MSG0|;"accept" ;|MSG0|;"(setup_curopts)")
      (action_tile ;|MSG0|;"cancel" ;|MSG0|;"(setq canexit T)(done_dialog 20)")
      (setq what_next1 (start_dialog))
    )
    (setq canexit nil canopts nil)   ; If default or saved options.
  )
)
;;
;;
(defun setup_curopts ()
  (setofiletxt curr)
  (ISXOPO ;|MSG0|;"tmpiges.opt" (getolist))
  (setq canopts nil curoptdesc (car allopts))
  (done_dialog 19)
)
;;
;;
(defun getallopts (ofilename)

  (if ofilename
    (setq allopts (ISXOGO (findfile ofilename)))
    (setq defopts (setq allopts (ISXOGO)))
  )

  (set_tile ;|MSG0|;"desc_text" (setq desctxt (nth 0 allopts)))
  (setq sg_opts (nth 1 allopts))
  (setq str_opts (nth 2 allopts))
  (setq geo_opts (nth 3 allopts))
  (setq anno_opts (nth 4 allopts))
  (setq demap_opts (nth 5 allopts))
  (setq idp_opts (nth 6 allopts))
  (setq misc_opts (nth 7 allopts))
  (if (not (null asurf))
    (setq asurf_opts (nth 8 allopts))
    (setq asurf_opts nil)
  )
)
;;
;;
(defun popdefopts ()
  (set_tile ;|MSG0|;"desc_text" (setq desctxt (nth 0 defopts)))
  (setq sg_opts (nth 1 defopts)
	str_opts (nth 2 defopts)
	geo_opts (nth 3 defopts)
	anno_opts (nth 4 defopts)
	demap_opts (nth 5 defopts)
	idp_opts (nth 6 defopts)
	misc_opts (nth 7 defopts)
	allopts defopts)
  (if (not (null asurf))
    (setq asurf_opts (nth 8 defopts))
    (setq asurf_opts nil)
  )
  (setofiletxt defs)
  (setq modflag 0)
)
;;
;;
(defun startmenu (/
  unitpick stfile authval orgsval sidval ridval eofval eorval mssval usrunit)
  (if (not (new_dialog ;|MSG0|;"stglobal" dcl_id))(exit))

  (setq eof_err_msg ;|MSG37|;"Invalid Parameter Delimiter Character."
	eor_err_msg ;|MSG38|;"Invalid Record Delimiter Character."
	mss_err_msg posrn)

  ;; Initialize version list
  (setlist (list ;|MSG39|;"Version 1.0 (1)"
		 ;|MSG40|;"ANSI Y14.26M-1981 (2)" 
		 ;|MSG41|;"Version 2.0 (3)" 
		 ;|MSG42|;"Version 3.0 (4)"
		 ;|MSG43|;"ASME/ANSI Y14.26M-1987 (5)"
		 ;|MSG44|;"Version 4.0 (6)"
		 ;|MSG45|;"ANSI Y14.26M-1989 (7)"
		 ;|MSG46|;"Version 5.0 (8)" 
		 ;|MSG47|;"Version 5.1 (9)"
		 ;|MSG48|;"Version 5.2 (10)"
		 ;|MSG48|;"Version 5.3 (11)") 
	    ;|MSG0|;"igs_vers")

  ;; Initialize unit list
  (setlist (list u_inch u_mm u_user u_feet u_mile u_m u_km u_mils u_um u_cm
		 u_uin) ;|MSG0|;"unit_box")

  ;; Initialize draft std list
  (setlist (list ;|MSG49|;"None (0)" 
		 ;|MSG50|;"ISO (1)" 
		 ;|MSG51|;"AFNOR (2)" 
		 ;|MSG52|;"ANSI (3)" 
		 ;|MSG53|;"BSI (4)" 
		 ;|MSG54|;"CSA (5)" 
		 ;|MSG55|;"DIN (6)" 
		 ;|MSG56|;"JIS (7)") 
	   ;|MSG0|;"draft_std")

  (popstartg)

  (setq wasmod modflag tmpsubopts sg_opts)
  (action_tile ;|MSG0|;"inc_startfile" ;|MSG0|;"(incstfile)")
  (action_tile ;|MSG0|;"get_startfile" ;|MSG0|;"(getstfile)")
  (action_tile ;|MSG0|;"inc_levnotes" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"eofchar" 
	       ;|MSG0|;"(setq eofval (setdelimchr eofval eof_err_msg))")
  (action_tile ;|MSG0|;"eorchar" 
	       ;|MSG0|;"(setq eorval (setdelimchr eorval eor_err_msg))")
  (action_tile ;|MSG0|;"author"
	       ;|MSG0|;"(if (test_chg authval $value)(setq authval $value))")
  (action_tile ;|MSG0|;"orgs"
	       ;|MSG0|;"(if (test_chg orgsval $value)(setq orgsval $value))")
  (action_tile ;|MSG0|;"send_id"
	       ;|MSG0|;"(if (test_chg sidval $value)(setq sidval $value))")
  (action_tile ;|MSG0|;"rec_id"
	       ;|MSG0|;"(if (test_chg ridval $value)(setq ridval $value))")
  (action_tile ;|MSG0|;"draft_std"
	       ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"msscale"
	       ;|MSG0|;"(setq mssval (setposlunit mssval mss_err_msg))")
  (action_tile ;|MSG0|;"unit_box"
	       ;|MSG0|;"(setunit 2)")
  (action_tile ;|MSG0|;"usr_units" ;|MSG0|;"(setusrunit)")
  (action_tile ;|MSG0|;"igs_vers" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"accept" ;|MSG0|;"(check_sg)")
  (action_tile ;|MSG0|;"cancel"
	       ;|MSG0|;"(setq sg_opts (canopt))(done_dialog 41)")
  (action_tile ;|MSG0|;"default"
	       ;|MSG0|;"(setq sg_opts (nth 1 defopts))(popstartg)")
  (action_tile ;|MSG0|;"help" ;|MSG0|;"(ait_helpdlg \"opt_startGlobalDialogBox\")")
  (setq what_next1 (start_dialog))
)
;;
;; This function tests for a change in the tile's value.  If there is
;; a change, the modflag is set and T is returned, otherwise nil is returned.
;;   
(defun test_chg (previous new)
  (rs_err)
  (if (/= new previous)
    (progn (setmflag) T)
    nil
  )
)
;;
;;  This function checks the validity of an acad object name.  If legitimate,
;;  the object name is returned in uppercase, nil otherwise.  Setting `wild'
;;  to `T' allows wild card characters.
;;
;;  Note: the '-' has been removed from teststr to allow hypenated names.
(defun check_objname (object err_msg wild / teststr)
  (if wild   ; allows wildcard characters.
    (setq teststr ;|MSG0|;"*[ !%^&()=+\\|{}:\";`,/<>]*")
    (setq teststr ;|MSG0|;"*[] !@#%^&*()=+\\|{}[:\"~;'`,./<>?]*")
  )
  (if (wcmatch object teststr)
    (progn (set_tile ;|MSG0|;"error" err_msg) nil)
    (strcase object)
  )
)
;;
;; When canceling, test to make sure that the modflag was not set prior to 
;; entering this dialog.  If it was, then do not reset.  Also returns unchanged
;; suboptions list.
;;
(defun canopt ()
  (if (= wasmod 0)
    (setq modflag 0)
  )
  tmpsubopts
)
;;
;;
(defun incstfile ()
  (rs_err)
  (if (= ;|MSG0|;"1" $value)
    (progn
      (grey_startf 0)
      (if (= stfile ;|MSG0|;"")
	(if (not (getstfile))
	  (progn (set_tile ;|MSG0|;"inc_startfile" ;|MSG0|;"0") (grey_startf 1))
	)
	(setmflag)
      )
    )
    (progn
      (grey_startf 1)
      (setmflag)
    )
  )
)
;;
;;
(defun grey_startf (gmode)
  (mode_tile ;|MSG0|;"get_startfile" gmode)
  (mode_tile ;|MSG0|;"start_file" gmode)
)
;;
;;
(defun getstfile ( / tmpfile)
  (setq tmpfile (getfiled ;|MSG57|;"Select Start Section File" stfile "" 14))
  (if tmpfile
    (progn
      (set_tile ;|MSG0|;"start_file" (setq stfile tmpfile))
      (setmflag)
      T
    )
    (progn (set_tile ;|MSG0|;"error" ;|MSG58|;"Invalid file name.") nil)
  )
)
;;
;;
(defun check_sg ()
  (cond
    ((not (chkdelim (get_tile ;|MSG0|;"eofchar") eof_err_msg)) 
      (mode_tile ;|MSG0|;"eofchar" 2))
    ((not (chkdelim (get_tile ;|MSG0|;"eorchar") eor_err_msg))
      (mode_tile ;|MSG0|;"eorchar" 2))
    ((not (ai_num (get_tile ;|MSG0|;"msscale") mss_err_msg 6))
      (mode_tile ;|MSG0|;"msscale" 2)
    )
    (t (waitmsg)(getstartg)(rs_err)(done_dialog 40))
  )
)
;;
;; Varify the delimiter character is Valid.  Return T if valid,
;; otherwise return nil.
;;
(defun chkdelim (testchar err_msg)
  (if (wcmatch testchar ;|MSG0|;"[0-9` DEH`.+`-]")
    (progn (if err_msg (set_tile ;|MSG0|;"error" err_msg)) nil)
    (if (= (get_tile ;|MSG0|;"eorchar") (get_tile ;|MSG0|;"eofchar"))
      (progn (set_tile ;|MSG0|;"error" 
		       ;|MSG59|;"Delimiters must be different characters.") nil)
      T
    )        
  )  
)
;;
;;
(defun setdelimchr (currval err_msg)
  (if (chkdelim $value err_msg)
    (if (test_chg currval $value)
      (setq currval $value)
      currval
    )
    currval
  )
)
;;
;; Returns a real number in a string format determined AutoCAD's LUNITS and
;; LUPREC system variables. `real_val' is a string containing the number to
;; convert.
;;
(defun conv_lunits (real_val)
  (rtos (atof real_val))
)
;;
;;
(defun setposlunit (currval err_msg)
  (if (ai_num $value err_msg 6)
    (if (test_chg currval $value)
      (progn
	(rs_err)
	(set_tile $key (setq currval (conv_lunits $value)))
	currval
      )
      currval
    )
    currval
  )
)
;;
;;
(defun popstartg ()
   (set_tile ;|MSG0|;"start_file" (setq stfile (car sg_opts)))
   (set_tile ;|MSG0|;"inc_levnotes" (itoa (cadr sg_opts)))
   (if (= stfile ;|MSG0|;"")
     (progn
       (grey_startf 1)
       (set_tile ;|MSG0|;"inc_startfile" ;|MSG0|;"0")
     )
     (progn
       (grey_startf 0)
       (set_tile ;|MSG0|;"inc_startfile" ;|MSG0|;"1")
     )
   )
   (if (= ;|MSG0|;"" (setq eofval (caddr sg_opts)))
       (setq eofval ;|MSG0|;",")
   )
   (set_tile ;|MSG0|;"eofchar" eofval)
   (if (= ;|MSG0|;"" (setq eorval (nth 3 sg_opts)))
       (setq eorval ;|MSG0|;";")
   )
   (set_tile ;|MSG0|;"eorchar" eorval)
   (set_tile ;|MSG0|;"igs_vers" (itoa (- (nth 4 sg_opts) 1)))
   (set_tile ;|MSG0|;"author" (setq authval (nth 5 sg_opts)))
   (set_tile ;|MSG0|;"orgs" (setq orgsval (nth 6 sg_opts)))
   (set_tile ;|MSG0|;"send_id" (setq sidval (nth 7 sg_opts)))
   (set_tile ;|MSG0|;"rec_id" (setq ridval (nth 8 sg_opts)))
   (set_tile ;|MSG0|;"msscale" (setq mssval (rtos (nth 9 sg_opts) 2 4)))
   (set_tile ;|MSG0|;"usr_units" (setq usrunit (nth 11 sg_opts)))
   (set_tile ;|MSG0|;"unit_box" (setq unitpick (itoa (1- (nth 10 sg_opts)))))
   (if (= unitpick ;|MSG0|;"2")
       (mode_tile ;|MSG0|;"usr_units" 0)
       (mode_tile ;|MSG0|;"usr_units" 1)
   )
   (set_tile ;|MSG0|;"draft_std" (itoa (nth 12 sg_opts)))
)
;;
;;
(defun getstartg ()
  (setq sg_opts 
	(list (if (= ;|MSG0|;"1" (get_tile ;|MSG0|;"inc_startfile")) stfile ;|MSG0|;"")
	      (atoi (get_tile ;|MSG0|;"inc_levnotes")) eofval eorval
	      (+ 1 (atoi (get_tile ;|MSG0|;"igs_vers"))) authval orgsval sidval ridval
	      (atof mssval) (1+ (atoi unitpick)) usrunit
	      (atoi (get_tile ;|MSG0|;"draft_std")))
  )
)
;;
;;
(defun setunit (row)
  (if (test_chg unitpick $value)
    (progn
      (setq unitpick (ai_strltrim $value))
      (if (= unitpick (itoa row))
	(progn
	  (mode_tile ;|MSG0|;"usr_units" 0)
	  (mode_tile ;|MSG0|;"usr_units" 2)
	)
	(mode_tile ;|MSG0|;"usr_units" 1)
      )
    )
  )
)
;;
;;
(defun setusrunit ()
  (rs_err)
  (if (test_chg usrunit $value)
    (set_tile $key (setq usrunit (strcase $value)))
  )
)
;;
;;   
(defun structmenu (/ lexlist unitpick shtval 
  xsizeval ysizeval xrefval grpval lexpick usrunit laynval dwgsizeval)
  (if (not (new_dialog ;|MSG0|;"stropts" dcl_id))(exit))
  (if lexlist
    (progn
      (set_tile ;|MSG0|;"excl_layer" ;|MSG0|;"0")
      (setq lexpick (get_tile ;|MSG0|;"excl_layer"))
    )
    (mode_tile ;|MSG0|;"del_excl_layer" 1)
  )
  (setlist (list ;|MSG61|;"No Mapping" 
		 ;|MSG62|;"Level Property (406:3)" 
		 ;|MSG63|;"Name Property (406:15)"
		 ;|MSG64|;"Level Attributes IDP") 
	   ;|MSG0|;"lyr_name_map")
  (setlist (list ;|MSG65|;"No Mapping" 
		 ;|MSG66|;"Unordered (402:7)" 
		 ;|MSG67|;"Ordered (402:15)")
	   ;|MSG0|;"group_map")
  (setlist (list ;|MSG68|;"No Mapping" 
		 ;|MSG69|;"Bind" 
		 ;|MSG70|;"XREF (416:1)" 
		 ;|MSG71|;"XREF (416:3)")
	   ;|MSG0|;"xref_map")
  (setlist (list ;|MSG72|;"None" 
		 u_inch u_mm u_user u_feet u_mile u_m u_km u_mils u_um 
		 u_cm u_uin) ;|MSG0|;"dwg_units")
  (setlist (list ;|MSG73|;"Extents" 
		 ;|MSG74|;"User Defined"
		 ;|MSG75|;"Limits" 
		 ;|MSG76|;"None") ;|MSG0|;"dwg_size")
  (popstructs)
  (setq wasmod modflag tmpsubopts str_opts)
  (action_tile ;|MSG0|;"lyr_name_map"
	       ;|MSG0|;"(if (test_chg laynval $value) (setq laynval $value))")
  (action_tile ;|MSG0|;"lyr_excls" ;|MSG0|;"(exclyrmenu)")
  (action_tile ;|MSG0|;"dwg_name"
	       ;|MSG0|;"(if (test_chg shtval $value) (setq shtval $value))")
  (action_tile ;|MSG0|;"dwg_units" ;|MSG0|;"(setunit 3)")
  (action_tile ;|MSG0|;"usr_units" ;|MSG0|;"(setusrunit)")
  (action_tile ;|MSG0|;"dwg_size" ;|MSG0|;"(setdwgsize)")
  (action_tile ;|MSG0|;"xsize"
	       ;|MSG0|;"(setq xsizeval (setposlunit xsizeval size_emsg))")
  (action_tile ;|MSG0|;"ysize"
	       ;|MSG0|;"(setq ysizeval (setposlunit ysizeval size_emsg))")
  (action_tile ;|MSG0|;"xref_map"
	       ;|MSG0|;"(if (test_chg xrefval $value)(setq xrefval $value))")
  (action_tile ;|MSG0|;"group_map"
	       ;|MSG0|;"(if (test_chg grpval $value)(setq grpval $value))")
  (action_tile ;|MSG0|;"map_pers" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"named_views" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"map_tiledvp" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"viewp_frame" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"allow_neg" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"rect_array" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"accept" ;|MSG0|;"(check_str)")
  (action_tile ;|MSG0|;"cancel"
	       ;|MSG0|;"(setq str_opts (canopt))(done_dialog 41)")
  (action_tile ;|MSG0|;"default"
	       ;|MSG0|;"(setq str_opts (nth 2 defopts))(popstructs)")
  (action_tile ;|MSG0|;"help"
	       ;|MSG0|;"(ait_helpdlg \"opt_structOptsDialogBox\")")
  (setq what_next1 (start_dialog))
)
;;
;;
(defun check_str ()
  (cond
    ((not (ai_num (get_tile ;|MSG0|;"xsize") size_emsg 6))
      (mode_tile ;|MSG0|;"xsize" 2))
    ((not (ai_num (get_tile ;|MSG0|;"ysize") size_emsg 6))
      (mode_tile ;|MSG0|;"ysize" 2))
    (t (waitmsg)(getstructs)(rs_err)(done_dialog 40))
  )
)
;;
;;
(defun lex_list_action ()
  (rs_err)
  (setq lexpick $value)
  (if (= $value ;|MSG0|;"")
    (mode_tile ;|MSG0|;"del_excl_layer" 1)
    (mode_tile ;|MSG0|;"del_excl_layer" 0)
  )
)
;;
;;
(defun exclyrmenu ( / tmplst)
  (if (not (new_dialog ;|MSG0|;"layer_exclusions" dcl_id))
    (exit)
  )
  (setq size_emsg posrn
	layn_emsg ;|MSG60|;"Invalid character(s) in layer name.")
  ;; Save current list
  (setq tmplst lexlist)
  ;; Update dialog with current list of layer exclusions
  (setlist lexlist ;|MSG0|;"excl_layer")
  (action_tile ;|MSG0|;"excl_layer" ;|MSG0|;"(lex_list_action)")
  (action_tile ;|MSG0|;"enter_layer" ;|MSG0|;"(addexclyr)")
  (action_tile ;|MSG0|;"add_excl_layer" ;|MSG0|;"(addexclyr)")
  (action_tile ;|MSG0|;"del_excl_layer" ;|MSG0|;"(delexclyr)")
  (action_tile ;|MSG0|;"accept" ;|MSG0|;"(done_dialog 40)")
  (action_tile ;|MSG0|;"cancel"
	       ;|MSG0|;"(setq lexlist tmplst)(done_dialog 41)")
  (action_tile ;|MSG0|;"help"
	       ;|MSG0|;"(ait_helpdlg \"opt_opt_structOptsDialogBox_layerExclusions\")")

  (setq what_next1 (start_dialog))
)
;;
(defun delexclyr (/ n pickf_no pickf_list lexlist1)
  (rs_err)
  (while (setq pickf_no (read lexpick))
    (setq pickf_list (cons pickf_no pickf_list))
    (setq lexpick (substr lexpick (+ 2 (strlen (itoa pickf_no)))))
  )
  (setq n 0)
  (while (< n (length lexlist))
    (if (not (member n pickf_list))
      (setq lexlist1 (cons (nth n lexlist) lexlist1))
    )
    (setq n (1+ n))
  )
  (setq lexlist (reverse lexlist1) modflag 1)
  (setlist lexlist ;|MSG0|;"excl_layer")
  (mode_tile ;|MSG0|;"del_excl_layer" 1)
)
;;
;;
(defun addexclyr (/ lexname temp)
  (rs_err)
  (setq temp (get_tile ;|MSG0|;"excl_layer"))
  (setq lexname 
	(check_objname (get_tile ;|MSG0|;"enter_layer") layn_emsg T))
  (if (and lexname (/= lexname ;|MSG0|;""))
    (progn
      (if (not (member lexname lexlist))
	(progn
	  (setq lexlist (append lexlist (list lexname)) modflag 1)
	  (setlist lexlist ;|MSG0|;"excl_layer")
	  (set_tile ;|MSG0|;"excl_layer" temp)
	  (get_tile ;|MSG0|;"excl_layer")
	  (set_tile ;|MSG0|;"enter_layer" ;|MSG0|;"")
	  (mode_tile ;|MSG0|;"enter_layer" 2)
	)
	(set_tile ;|MSG0|;"error" ;|MSG77|;"Duplicate layer name.")
      )
    )
    (setq lexpick (get_tile ;|MSG0|;"excl_layer"))
  )
)
;;
;;
(defun setdwgsize ()
  (if (test_chg dwgsizeval $value)
    (progn
      (setq dwgsizeval $value)
      (greyxy)
    )
  )
)
;;
;;
(defun popstructs (/ temp dinc tfcount)
  (set_tile ;|MSG0|;"lyr_name_map" (setq laynval (itoa (car str_opts))))
  (setq tfcount (cadr str_opts))
  (if (> tfcount 0)
    (setq lexlist (caddr str_opts) dinc 0)
    (setq lexlist nil dinc 1)
  )
  (setlist lexlist ;|MSG0|;"excl_layer")
  (set_tile ;|MSG0|;"xref_map" (setq xrefval (itoa (nth (- 3 dinc) str_opts))))
  (set_tile ;|MSG0|;"group_map" (setq grpval (itoa (nth (- 4 dinc) str_opts))))
  (set_tile ;|MSG0|;"dwg_name" (setq shtval (nth (- 5 dinc) str_opts)))
  (set_tile ;|MSG0|;"usr_units" (setq usrunit (nth (- 7 dinc) str_opts)))
  (set_tile ;|MSG0|;"dwg_units" (setq unitpick (itoa (nth (- 6 dinc) str_opts))))
  (if (= unitpick ;|MSG0|;"3")
    (mode_tile ;|MSG0|;"usr_units" 0)
    (mode_tile ;|MSG0|;"usr_units" 1)
  )
  (set_tile ;|MSG0|;"dwg_size" (setq dwgsizeval (itoa (nth (- 8 dinc) str_opts))))
  (set_tile ;|MSG0|;"xsize" (setq xsizeval (rtos (nth (- 9 dinc) str_opts) 2 4)))
  (set_tile ;|MSG0|;"ysize" (setq ysizeval (rtos (nth (- 10 dinc) str_opts) 2 4)))
  (greyxy)
  (set_tile ;|MSG0|;"allow_neg"  (itoa (nth (- 11 dinc) str_opts)))
  (set_tile ;|MSG0|;"viewp_frame" (itoa (nth (- 12 dinc) str_opts)))
  (set_tile ;|MSG0|;"named_views" (itoa (nth (- 13 dinc) str_opts)))
  (set_tile ;|MSG0|;"map_tiledvp" (itoa (nth (- 14 dinc) str_opts)))
  (set_tile ;|MSG0|;"map_pers" (itoa (nth (- 15 dinc) str_opts)))
  (if (= 0 (nth (- 16 dinc) str_opts))
    (set_tile ;|MSG0|;"rect_array" ;|MSG0|;"412")
    (set_tile ;|MSG0|;"rect_array" ;|MSG0|;"408")
  )
)
;;
;;
(defun getstructs ( / tmplist tfcount)
  (setq tmplist
	(list (atoi xrefval) (atoi grpval)  shtval (atoi unitpick) usrunit
	      (atoi dwgsizeval) (atof xsizeval) (atof ysizeval)
	      (atoi (get_tile ;|MSG0|;"allow_neg"))
	      (atoi (get_tile ;|MSG0|;"viewp_frame")) 
	      (atoi (get_tile ;|MSG0|;"named_views"))
	      (atoi (get_tile ;|MSG0|;"map_tiledvp")) 
	      (atoi (get_tile ;|MSG0|;"map_pers"))
	      (if (= ;|MSG0|;"412" (get_tile ;|MSG0|;"rect_array")) 0 1)))
  (setq tfcount (length lexlist))
  (setq str_opts
	(if (> tfcount 0)
	  (append (list (atoi laynval) tfcount lexlist) tmplist)
	  (append (list (atoi laynval) tfcount) tmplist)
	))
)
;;
;;   
(defun greyxy ()
  (if (/= ;|MSG0|;"1" (get_tile ;|MSG0|;"dwg_size"))
    (progn
      (mode_tile ;|MSG0|;"xsize" 1)
      (mode_tile ;|MSG0|;"ysize" 1)
      (mode_tile ;|MSG0|;"usr_size" 1)
    )
    (progn
      (mode_tile ;|MSG0|;"xsize" 0)
      (mode_tile ;|MSG0|;"ysize" 0)
      (mode_tile ;|MSG0|;"usr_size" 0)
    )
  )
)
;;
;;
(defun geomenu (/ pntval mk110 mk186 mk230 splval mk144 mk106_13)
  (if (not (new_dialog ;|MSG0|;"geopts" dcl_id))
    (exit))
  (popgeo)
  (setq wasmod modflag tmpsubopts geo_opts)
;  (action_tile ;|MSG0|;"splines_iges"
;               ;|MSG0|;"(if (test_chg splval $value) (setq splval $value))")
  (action_tile ;|MSG0|;"brepout_mapping"
	       ;|MSG0|;"(if (test_chg mk186 $value) (setq mk186 $value))")
;  (action_tile ;|MSG0|;"region_mapping"
;	       ;|MSG0|;"(if (test_chg mk230 $value) (setq mk230 $value))")
  (action_tile ;|MSG0|;"map_110"
	       ;|MSG0|;"(if (test_chg mk110 $value) (setq mk110 $value))")
  (action_tile ;|MSG0|;"trimmed_surface_mapping"
	       ;|MSG0|;"(if (test_chg mk144 $value) (setq mk144 $value))")
  
  (if (not (null asurf))
    (progn
      (action_tile ;|MSG0|;"asurf_create_106_13"
	;|MSG0|;"(if (test_chg mk106_13 $value) (setq mk106_13 $value))")
    )
    (mode_tile ;|MSG0|;"asurf_create_106_13" 1) ;; else - disable option
  )

  (action_tile ;|MSG0|;"map_points"
	       ;|MSG0|;"(if (test_chg pntval $value) (setq pntval $value))")
  (action_tile ;|MSG0|;"accept"
	       ;|MSG0|;"(waitmsg)(getgeo)(rs_err)(done_dialog 40)")
  (action_tile ;|MSG0|;"cancel" 
	       ;|MSG0|;"(setq geo_opts (canopt))(done_dialog 41)")
  (action_tile ;|MSG0|;"default" 
	       ;|MSG0|;"(defgeo)")
  (action_tile ;|MSG0|;"help" 
	       ;|MSG0|;"(ait_helpdlg \"opt_geometryDialogBox\")")
	       (setq what_next1 (start_dialog))
)
;;
(defun defgeo (/ oldlst)
  ; set the geometry options to their defaults
  (setq geo_opts (nth 3 defopts))
  ; set the AutoSurf geometry options to their defaults
  (if (and (not (null asurf)) (not (null asurf_opts)))
    (progn
      (setq oldlst (nth 8 defopts))
      (setq mk106_13 (nth 2 oldlst))
      (if (/= mk106_13 (nth 2 asurf_opts))
	(progn
	  (if (= mk106_13 0)
	    (set_tile ;|MSG0|;"asurf_create_106_13" 
	       (setq mk106_13 ;|MSG0|;"e106_12"))
	    (set_tile ;|MSG0|;"asurf_create_106_13" 
	       (setq mk106_13 ;|MSG0|;"e106_13"))
	  )
	  (setq asurf_opts (list (car oldlst) (nth 1 oldlst) mk106_13))
	)
      )
    )
  )
  (popgeo)
)
;;
(defun popgeo (/ temp)
  (setq temp (car geo_opts))
; The splines_crv_out option has been dropped for AIT 13 GUI
;  (cond
;    ((= temp 0) 
;        (set_tile ;|MSG0|;"splines_iges" (setq splval ;|MSG0|;"b_splines")))
;    ((= temp 1) 
;        (set_tile ;|MSG0|;"splines_iges" (setq splval ;|MSG0|;"pc_splines")))
;    ((= temp 2) 
;        (set_tile ;|MSG0|;"splines_iges" (setq splval ;|MSG0|;"cop_data")))
;    (t nil)
;  )
  (if (not (null asurf_opts))
     (progn
	(setq mk106_13 (nth 2 asurf_opts))
	(if (= mk106_13 0)
	   (set_tile ;|MSG0|;"asurf_create_106_13" (setq mk106_13 ;|MSG0|;"e106_12"))
	   (set_tile ;|MSG0|;"asurf_create_106_13" (setq mk106_13 ;|MSG0|;"e106_13"))
	)
     )
  )
  (if (= 0 (nth 1 geo_opts))
    (setq pntval ;|MSG0|;"1")
    (setq pntval ;|MSG0|;"0")
  )
  (set_tile ;|MSG0|;"map_points" pntval)
  (if (= 2 (nth 2 geo_opts))
    (setq mk186 ;|MSG0|;"eSurf")
    (if (= 1 (nth 2 geo_opts))
      (setq mk186 ;|MSG0|;"e186")
      (setq mk186 ;|MSG0|;"e186a")
    )
  )
  (set_tile ;|MSG0|;"brepout_mapping" mk186)
  ;(if (= 0 (nth 3 geo_opts))
  ;   (setq mk230 ;|MSG0|;"e230")
  ;   (setq mk230 ;|MSG0|;"eTsurf")
  ;)
  ;(set_tile ;|MSG0|;"region_mapping" mk230) 
  ; map_110 option
  (if (= 0 (nth 3 geo_opts))
    (setq mk110 ;|MSG0|;"0")
    (setq mk110 ;|MSG0|;"1")
  )
  (set_tile ;|MSG0|;"map_110" mk110)
  ;; Trimmed Surface Mapping option
  (setq mk144 (nth 4 geo_opts))
  (if (= mk144 0)
     (setq mk144 ;|MSG0|;"e143")
     (setq mk144 ;|MSG0|;"e144")
  )
  (set_tile ;|MSG0|;"trimmed_surface_mapping" mk144)
)
;;
;;
(defun getgeo (/ oldlst)
; Removed spline_crv_out option for R13 AIT 5.2.1
;  (cond
;    ((= splval ;|MSG0|;"pc_splines") (setq splval 1))
;    ((= splval ;|MSG0|;"cop_data") (setq splval 2))
;    (t (setq splval 0))  ; Default
;  )
  (if (= mk144 ;|MSG0|;"e143") 
     (setq mk144 0)
     (setq mk144 1)
  ) 
  (if (not (null asurf_opts))
    (progn
      (if (= mk106_13 ;|MSG0|;"e106_12") 
	(setq mk106_13 0)
	(setq mk106_13 1)
      ) 
    )
  )
  (if (= mk186 ;|MSG0|;"e186")
     (setq mk186 1)
     (if (= mk186 ;|MSG0|;"e186a")
	(setq mk186 0)
	(setq mk186 2)
     )
  )
  ;(if (= mk230 ;|MSG0|;"e230")
  ;   (setq mk230 0)
  ;   (setq mk230 1)
  ;)
  (if (= mk110 ;|MSG0|;"0")
     (setq mk110 0)
     (setq mk110 1)
  )
  (setq geo_opts 
	(list (car (nth 3 defopts)) 
	      (if (= ;|MSG0|;"0" pntval) 1 0) 
	      mk186 mk110 mk144)
  )
  ;; Update asurf_opts with mk106_13.
  (if (not (null asurf_opts))
    (progn 
      (if (/= mk106_13 (nth 2 asurf_opts))
	(progn
	  (setq oldlst asurf_opts)
	  (setq asurf_opts 
		(list (car oldlst) (nth 1 oldlst) mk106_13))
	  (setmflag)
	)
      )
    )
  )
)
;;
;;
(defun annomenu (/ dfont_list defntval opickfnt
   pickf iflist aslist arrval ahdyval ahdxval fcode ascode)
  (if (not (new_dialog ;|MSG0|;"annopts" dcl_id))(exit))
  (setq opickfnt ;|MSG0|;"0")

  ; Initialize default font code list
  (setq dfont_list (ait_fontlist))
  (setlist dfont_list ;|MSG0|;"def_font")
  (popanno)
  (setq wasmod modflag tmpsubopts anno_opts)

  (mode_tile ;|MSG0|;"acad_name" 2)
  (set_tile ;|MSG0|;"map_list" (setq pickf ;|MSG0|;""))
  (settfaction pickf)

  (action_tile ;|MSG0|;"def_font" ;|MSG0|;"(actdeffont)") 
  (action_tile ;|MSG0|;"map_list" ;|MSG0|;"(settfaction $value)")
  (action_tile ;|MSG0|;"add_font" ;|MSG0|;"(addfontmap dfont_list nil)")
  (action_tile ;|MSG0|;"mod_font" ;|MSG0|;"(modfontmap dfont_list nil)")
  (action_tile ;|MSG0|;"del_font" ;|MSG0|;"(delfontmap)(settfaction pickf)")
  (action_tile ;|MSG0|;"clines" ;|MSG0|;"(grey_cl (atoi $value))(setmflag)")
  (action_tile ;|MSG0|;"cl_wdims" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"rad_dims" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"de_dims" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"text_lines" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"map_406_31" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"accept" 
	       ;|MSG0|;"(waitmsg)(getanno)(rs_err)(done_dialog 40)")
  (action_tile ;|MSG0|;"cancel" 
	       ;|MSG0|;"(setq anno_opts (canopt))(done_dialog 41)")
  (action_tile ;|MSG0|;"default" 
	       ;|MSG0|;"(setq anno_opts (nth 4 defopts))(popanno)")
  (action_tile ;|MSG0|;"help" 
	       ;|MSG0|;"(ait_helpdlg \"opt_annotationDialogBox\")")
  (setq what_next1 (start_dialog))
)
;;
;;
(defun settfaction (value)
  (listaction 34 16 dfont_list value)
)
;;
;;
(defun actdeffont ()
  (rs_err)
  (if (test_chg defntval $value)
    (setq defntval $value)
  )
)
;;
;;
(defun delfontmap ( / n pickf_no pickf_list aslist1 delitem)
  (rs_err)
  (while (setq pickf_no (read pickf))
    (setq pickf_list (cons pickf_no pickf_list))
    (setq pickf (substr pickf (+ 2 (strlen (itoa pickf_no)))))
  )
  (setq n 0)
  (while (< n (length fontmap_list))
    (if (not (member n pickf_list))
      (setq aslist1 (cons (nth n fontmap_list) aslist1))
      (setq delitem (nth n fontmap_list))
    )
    (setq n (1+ n))
  )
  (setq fontmap_list (reverse aslist1))
  (setlist fontmap_list ;|MSG0|;"map_list")
  (setmflag)
  delitem
)
;;
;;
(defun addfontmap (iges_list list304 / n tmpstr tmpacad tmpiges)
  (rs_err)
  (setq tmpiges (nth (atoi (get_tile ;|MSG0|;"iges_list")) iges_list))
  (setq tmpacad (ai_strrtrim (strcase (get_tile ;|MSG0|;"acad_name"))))
  (if (and (/= tmpacad ;|MSG0|;"") (check_objname tmpacad ;|MSG0|;"" T))
    (progn
      (setq tmpstr
	(if list304
	  (if (= list304 T)
	    (strcat tmpacad ;|MSG0|;"\t" ;|MSG0|;"....." ;|MSG0|;"\t" tmpiges)
	    (strcat tmpacad ;|MSG0|;"\t" list304 ;|MSG0|;"\t" tmpiges)
	  )
	  (strcat tmpacad ;|MSG0|;"\t" tmpiges)
	)
      )
      (if (not (member tmpacad (subcolumn 1 fontmap_list)))
	(progn
	  (setq fontmap_list (append fontmap_list (list tmpstr)))
	  (setq fontmap_list (sortmax fontmap_list))
	  (setlist fontmap_list ;|MSG0|;"map_list")
	  ; set focus to new entry
	  (setq n 0)
	  (while (/= (nth n fontmap_list) tmpstr)
	    (setq n (1+ n))
	  )
	  (set_tile ;|MSG0|;"map_list" (setq pickf (itoa n)))
	  (if list304
	    (setltaction pickf)
	    (settfaction pickf)
	  )
	  (setmflag)
	  T
	)
	(progn
	  (set_tile ;|MSG0|;"error" (strcat tmpacad ;|MSG78|;" mapping already exists."))
	  nil
	)
      )
    )
    (if list304
      (progn (set_tile ;|MSG0|;"error" ;|MSG79|;"Invalid line type name.") nil)
      (progn (set_tile ;|MSG0|;"error" ;|MSG80|;"Invalid text style name.") nil)
    )
  )
)
;;
;;
(defun modfontmap (alist list304 / tmplist tmpick tmpfnt)
  (setq tmplist fontmap_list tmpick pickf)
  (if list304
    (progn
      (setq tmpfnt (delfontmap))
      (setq list304 (substr tmpfnt (col_start 2 tmpfnt) 5))
    )
    (delfontmap)
  )
  (if (not (addfontmap alist list304))
    (progn
      (setlist (setq fontmap_list tmplist) ;|MSG0|;"map_list")
      (set_tile ;|MSG0|;"map_list" (setq pickf tmpick))
    )
  )
)
;;
;;
(defun popanno (/ tfcount tmp dinc tlval)
  (setq arrval (car anno_opts))
  (setq ahdyval (cadr anno_opts))
  (setq ahdxval (caddr anno_opts))
  (setq defntval (nth 3 anno_opts))
  (setq defntval 
    (cond
      ((= defntval 0) ;|MSG0|;"0")
      ((= defntval 1) ;|MSG0|;"1")
      ((= defntval 2) ;|MSG0|;"2")
      ((= defntval 3) ;|MSG0|;"3")
      ((= defntval 6) ;|MSG0|;"4")
      ((= defntval 12) ;|MSG0|;"5")
      ((= defntval 13) ;|MSG0|;"6")
      ((= defntval 14) ;|MSG0|;"7")
      ((= defntval 17) ;|MSG0|;"8")
      ((= defntval 18) ;|MSG0|;"9")
      ((= defntval 19) ;|MSG0|;"10")
      ((= defntval 1001) ;|MSG0|;"11")
      ((= defntval 1002) ;|MSG0|;"12")
      ((= defntval 1003) ;|MSG0|;"13")
      ((= defntval 2001) ;|MSG0|;"14")
      ((= defntval 3001) ;|MSG0|;"15")
      (t  ;|MSG0|;"1")
    )
  )
  (set_tile ;|MSG0|;"def_font" defntval)
  (setq aslist nil iflist nil)
  (setq tfcount (nth 4 anno_opts))
  (if (> tfcount 0)
    (progn
      (setq aslist (nth 5 anno_opts) dinc 0)
      (setq iflist (mapcar (quote itoa) (nth 6 anno_opts)))
      (foreach n dfont_list
	(setq iflist (subst n (itoa (read n)) iflist))
      )
      (setq fontmap_list (sortmax (catlists aslist 33 iflist)))
    )
    (setq dinc 2)
  )
  (setlist fontmap_list ;|MSG0|;"map_list")
  (setlist dfont_list ;|MSG0|;"iges_list")
  (setq tmp (nth (- 7 dinc) anno_opts))
  (cond
    ((= tmp 2)(set_tile ;|MSG0|;"clines" ;|MSG0|;"1")
	      (set_tile ;|MSG0|;"cl_wdims" ;|MSG0|;"1")(grey_cl 1))
    ((= tmp 0)(set_tile ;|MSG0|;"clines" ;|MSG0|;"0")
	      (set_tile ;|MSG0|;"cl_wdims" ;|MSG0|;"0")(grey_cl 0))
    (t (set_tile ;|MSG0|;"clines" ;|MSG0|;"1")
	      (set_tile ;|MSG0|;"cl_wdims" ;|MSG0|;"0")(grey_cl 1)) ; default=1
  )
  (set_tile ;|MSG0|;"de_dims" (itoa (nth (- 8 dinc) anno_opts)))
  (set_tile ;|MSG0|;"rad_dims" (itoa (nth (- 9 dinc) anno_opts)))
  (if (= 0 (nth (- 10 dinc) anno_opts))
    (setq tlval ;|MSG0|;"1")
    (setq tlval ;|MSG0|;"0")
  )
  (set_tile ;|MSG0|;"text_lines" tlval)
  (set_tile ;|MSG0|;"map_406_31" (itoa (nth (- 11 dinc) anno_opts)))
  (setq tol_mapping (nth (- 12 dinc) anno_opts))
)
;;
;;
(defun grey_cl (mode)
  (if (= mode 1)
    (setq mode 0)
    (setq mode 1)
  )
  (mode_tile ;|MSG0|;"cl_wdims" mode)
)
;;
;;  This function gets the start column for a substring in a list
;;
(defun subcolumn (col the_list / n tmplist)
  (setq tmplist nil)
  (foreach tmpstr the_list
    (setq tmplist
	  (append tmplist
		   (list (substr tmpstr 
				 (setq n (col_start col tmpstr))
				 (strwth (substr tmpstr n))
				 )
		   )
	  )
    )
  )
)
;;
;;  This function gets the width of the substring in a list.
;;
(defun strwth (str / n)
  (setq lng (strlen str))
  (setq n 1)
  (while (and (not (= lng (1- n))) (not (= ;|MSG0|;"\t" (substr str n 1))))
    (setq n (1+ n))
  )    
  (1- n)
)
;;
;;
(defun subfield (col recno the_list / tmpstr)
  (setq tmpstr (nth recno the_list))
  (substr tmpstr (setq cnt (col_start col tmpstr)) (strwth (substr tmpstr cnt)))
)
;;
;;
(defun catlists (firstlist colwidth secondlist)
  (mapcar (quote (lambda (first second)
	     (strcat first ;|MSG0|;"\t" second)))
	     firstlist
	     secondlist)
)
;;
;;
(defun col_start (col str / n)
  (setq n 0)
  (if (= col 1)
    col
    (+ 
      (repeat (- col 1)
	(setq n (+ n 1))
	(while (not (equal ;|MSG0|;"\t" (substr str n 1)))
	    (setq n (+ n 1))
	)
      )  
      1
    )  
  )  
)
;;
;;
(defun getanno (/ clval flist slist tlist tlval)
  (setq defntval
    (cond
      ((= defntval ;|MSG0|;"0") 0)
      ((= defntval ;|MSG0|;"2") 2)
      ((= defntval ;|MSG0|;"3") 3)
      ((= defntval ;|MSG0|;"4") 6)
      ((= defntval ;|MSG0|;"5") 12)
      ((= defntval ;|MSG0|;"6") 13)
      ((= defntval ;|MSG0|;"7") 14)
      ((= defntval ;|MSG0|;"8") 17)
      ((= defntval ;|MSG0|;"9") 18)
      ((= defntval ;|MSG0|;"10") 19)
      ((= defntval ;|MSG0|;"11") 1001)
      ((= defntval ;|MSG0|;"12") 1002)
      ((= defntval ;|MSG0|;"13") 1003)
      ((= defntval ;|MSG0|;"14") 2001)
      ((= defntval ;|MSG0|;"15") 3001)
      (t  1)  ; Default=1
    )
  )
  (if (= ;|MSG0|;"1" (get_tile ;|MSG0|;"clines"))
    (if (= ;|MSG0|;"1" (get_tile ;|MSG0|;"cl_wdims"))
      (setq clval 2)
      (setq clval 1)
    )
    (setq clval 0)
  )
  (setq tfcount (length fontmap_list))
  (setq flist (list arrval ahdyval ahdxval defntval tfcount))
  (if (= "0" (get_tile ;|MSG0|;"text_lines"))
    (setq tlval 1)
    (setq tlval 0) 
  )
  (setq slist 
	(list clval (atoi (get_tile ;|MSG0|;"de_dims")) 
		    (atoi (get_tile ;|MSG0|;"rad_dims")) 
		    tlval
		    (atoi (get_tile ;|MSG0|;"map_406_31")) 
		    tol_mapping 
	))
  (if (> tfcount 0)
    (progn
      (setq aslist (list (subcolumn 1 fontmap_list)))
      (setq iflist (list (mapcar (quote read) (subcolumn 2 fontmap_list))))
      (setq anno_opts (append flist aslist iflist slist))
    )
    (setq anno_opts (append flist slist))
  )
)
;;
;; 
(defun demapmenu (/ pickf altlist ilflist list_304
    altcode lfcode code304 fontmap_list)
  (if (not (new_dialog ;|MSG0|;"demapopts" dcl_id))(exit))
  (setq lineflist (list lnf0 lnf1 lnf2 lnf3 lnf4 lnf5))
  (setlist lineflist ;|MSG0|;"iges_lfname")

  (popdemap)
  (setq wasmod modflag tmpsubopts demap_opts)

  (mode_tile ;|MSG0|;"acad_name" 2)
  (set_tile ;|MSG0|;"map_list" (setq pickf ;|MSG0|;""))
  (setltaction pickf)

  (action_tile ;|MSG0|;"out_314" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"out_handle" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"map_list" ;|MSG0|;"(setltaction $value)")
  (action_tile ;|MSG0|;"add_font" ;|MSG0|;"(addfontmap lineflist T)")
  (action_tile ;|MSG0|;"del_font" ;|MSG0|;"(delfontmap)(setltaction pickf)")
  (action_tile ;|MSG0|;"mod_font" ;|MSG0|;"(modfontmap lineflist T)")
  (action_tile ;|MSG0|;"m304_on" ;|MSG0|;"(modm304 \"304:2\")")
  (action_tile ;|MSG0|;"m304_off" ;|MSG0|;"(modm304 \".....\")")
  (action_tile ;|MSG0|;"accept" 
	       ;|MSG0|;"(waitmsg)(getdemap)(rs_err)(done_dialog 40)")
  (action_tile ;|MSG0|;"cancel"
	       ;|MSG0|;"(setq demap_opts (canopt))(done_dialog 41)")
  (action_tile ;|MSG0|;"default"
	       ;|MSG0|;"(setq demap_opts (nth 5 defopts))(popdemap)")
  (action_tile ;|MSG0|;"help"
	       ;|MSG0|;"(ait_helpdlg \"opt_annotationDialogBox_deMapping\")")
  (setq what_next1 (start_dialog))
)
;;
;;
(defun modm304 (newstr / pickf_no tmpstr tmppck 304cl)
  (rs_err)
  (setq tmppck pickf)
  (while (setq pickf_no (read tmppck))
    (setq tmpstr (nth pickf_no fontmap_list))
    (setq 304cl (+ 1 (strwth tmpstr)))
    (setq fontmap_list
      (subst
	(strcat
	  (substr tmpstr 1 304cl)
	  newstr ;|MSG0|;"\t"
	  (substr tmpstr (col_start 3 tmpstr))
	)
      tmpstr fontmap_list
      )
    )
    (setq tmppck (substr tmppck (+ 2 (strlen (itoa pickf_no)))))
  )
  (setlist fontmap_list ;|MSG0|;"map_list")
  (set_tile ;|MSG0|;"map_list" pickf)
)
;;
;;
(defun setltaction (value)
  (listaction 42 6 lineflist value)
)
;;
;;
(defun listaction (igsbegin igesmax iges_list value / cnt field tmpstr)
  (rs_err)
  (setq pickf value)

  (if (read value)
    (progn
      (if (= igsbegin 42) ; kludge test for a call by demapmenu
	(progn 
           (setq field 3) ;; field within fontmap_list for iges line font
           (mode_tile ;|MSG0|;"m304_on" 0)
           (mode_tile ;|MSG0|;"m304_off" 0)
        )
        (setq field 2)    ;; field within fontmap_list for IGES font code 
      )
      (mode_tile ;|MSG0|;"del_font" 0)
      (if (wcmatch value ;|MSG0|;"* *") ; test for multiple select
	(progn
	  (set_tile ;|MSG0|;"acad_name" ;|MSG0|;"")
	  (mode_tile ;|MSG0|;"mod_font" 1)
	  (set_tile ;|MSG0|;"iges_list" (itoa igesmax))
	)
	(progn
          (set_tile ;|MSG0|;"acad_name" (subfield 1 (atoi pickf) fontmap_list))
          (setq tmpstr (subfield field (atoi pickf) fontmap_list))
          (setq cnt 0)
          (while (and (/= (nth cnt iges_list) tmpstr) (< cnt igesmax))
            (setq cnt (1+ cnt))
          )
          (set_tile ;|MSG0|;"iges_list" (itoa cnt))
	  (mode_tile ;|MSG0|;"mod_font" 0)
	)
      )
    )
    (progn
      (if (= igsbegin 42) ; kludge test for a call by demapmenu
	(progn (mode_tile ;|MSG0|;"m304_on" 1) (mode_tile ;|MSG0|;"m304_off" 1))
      )
      (mode_tile ;|MSG0|;"del_font" 1)
      (mode_tile ;|MSG0|;"mod_font" 1)
      (set_tile ;|MSG0|;"acad_name" ;|MSG0|;"")
      (set_tile ;|MSG0|;"iges_list" ;|MSG0|;"") ;(itoa igesmax))
    )
  )
)
;;
;; 
(defun popdemap (/ tfcount)
  (if (= (car demap_opts) 1)
    (set_tile ;|MSG0|;"out_314" ;|MSG0|;"0")
    (set_tile ;|MSG0|;"out_314" ;|MSG0|;"1")
  )
  (set_tile ;|MSG0|;"out_handle" (itoa (cadr demap_opts)))
  (setq tfcount (caddr demap_opts))
  (if (> tfcount 0)
    (progn
      (setq altlist (cadddr demap_opts))
      (setq ilflist (mapcar (quote convlfont) (nth 4 demap_opts)))
      (setq list_304 (mapcar (quote conv304) (nth 5 demap_opts)))
      (setq fontmap_list
	    (sortmax (catlists altlist 33 (catlists list_304 8 ilflist)))
      )
    )
  )
  (setlist fontmap_list ;|MSG0|;"map_list")
  (setlist lineflist ;|MSG0|;"iges_list")
  (setlist list_304 ;|MSG0|;"make_304_list")
)
;;
;;
(defun getdemap (/ elem1 tfcount flist)
  (setq elem1 (get_tile ;|MSG0|;"out_314"))
  (cond ((= elem1 ;|MSG0|;"1")(setq elem1 0))
	(t (setq elem1 1)) ; default=0
  )
  (setq tfcount (length fontmap_list))
  (setq flist (list elem1 (atoi (get_tile ;|MSG0|;"out_handle")) tfcount))
  (if (> tfcount 0)
    (progn
      (setq aslist (list (subcolumn 1 fontmap_list)))
      (setq ilflist (list (mapcar (quote lftonum) (subcolumn 3 fontmap_list))))
      (setq list_304 (list (mapcar (quote convto304) (subcolumn 2 fontmap_list))))
      (setq demap_opts (append flist aslist ilflist list_304))
    )
    (setq demap_opts flist)
  )
)
;;
;;
(defun lftonum (s / num)
  (cond
    ((= s lnf0) (setq num 0))
    ((= s lnf1) (setq num 1))
    ((= s lnf2) (setq num 2))
    ((= s lnf3) (setq num 3))
    ((= s lnf4) (setq num 4))
    ((= s lnf5) (setq num 5))
    (t (setq num 0))
  )
)
;;
;;
(defun convto304 (s / num)
  (cond
    ((= s ;|MSG0|;".....") (setq num 0))
    ((= s ;|MSG0|;"304:2") (setq num 1))
    (t (setq num 0))
  )
)
;;
;;
(defun setmflag ()
  (setq modflag 1)
)
;;
;;
(defun conv304 (s / str304)
  (cond
    ((= s 0) (setq str304 ;|MSG0|;"....."))
    ((= s 1) (setq str304 ;|MSG0|;"304:2"))
    (t (setq str304 ;|MSG81|;"Map to an IGES Line Font"))
  )
)
;;
;;
(defun convlfont (s / strlf)
  (cond
    ((= s 0) (setq strlf lnf0))
    ((= s 1) (setq strlf lnf1))
    ((= s 2) (setq strlf lnf2))
    ((= s 3) (setq strlf lnf3))
    ((= s 4) (setq strlf lnf4))
    ((= s 5) (setq strlf lnf5))
    (t (setq strlf ;|MSG82|;"0 - No Pattern Specified"))
  )
)
;;
;;
(defun idpmenu ()
  (if (not (new_dialog ;|MSG0|;"idpopts" dcl_id))
     (exit)
  )
  (popidp)
  (setq wasmod modflag tmpsubopts idp_opts)
  (action_tile ;|MSG0|;"create_idp" ;|MSG0|;"(greyidp)")
  (action_tile ;|MSG0|;"accept"
	       ;|MSG0|;"(waitmsg)(getidp)(rs_err)(done_dialog 40)")
  (action_tile ;|MSG0|;"cancel" 
	       ;|MSG0|;"(setq idp_opts (canopt))(done_dialog 41)")
  (action_tile ;|MSG0|;"default" ;|MSG0|;"(defidp)")
  (action_tile ;|MSG0|;"help" 
	       ;|MSG0|;"(ait_helpdlg \"opt_idpMapDialogBox\")")
  (setq what_next1 (start_dialog))
)
;;
;;
(defun defidp ()
  (setq idp_opts (nth 6 defopts))
  (popidp)
)
;;
;;
(defun popidp ()
  (set_tile ;|MSG0|;"create_idp" (itoa (nth 0 idp_opts)))
  (set_tile ;|MSG0|;"non_idp" (itoa (nth 1 idp_opts)))  
  (set_tile ;|MSG0|;"txt_idp" (itoa (nth 2 idp_opts)))  
  (set_tile ;|MSG0|;"txtstyle_idp" (itoa (nth 3 idp_opts)))  
  (set_tile ;|MSG0|;"attdef_idp" (itoa (nth 4 idp_opts)))  
  (set_tile ;|MSG0|;"attrib_idp" (itoa (nth 5 idp_opts)))  
  (set_tile ;|MSG0|;"dim_idp" (itoa (nth 6 idp_opts)))  
  (set_tile ;|MSG0|;"dimstyle_idp" (itoa (nth 7 idp_opts)))  
  (setq viewp_idp (nth 8 idp_opts))
  (setq view_idp (nth 9 idp_opts))
  (set_tile ;|MSG0|;"ucs_idp" (itoa (nth 10 idp_opts)))  
  (setq shape_idp (nth 11 idp_opts))
  (setq sysvar_idp (nth 12 idp_opts))
  (set_tile ;|MSG0|;"ltype_idp" (itoa (nth 13 idp_opts)))  
  (set_tile ;|MSG0|;"pface_idp" (itoa (nth 14 idp_opts)))  
  (setq xed_idp (nth 15 idp_opts))
  (set_tile ;|MSG0|;"mesh_idp" (itoa (nth 16 idp_opts)))
  (greyidp)
)
;;
;;
(defun getidp (/ idp_topts)
  (setq idp_topts 
	(list (atoi (get_tile ;|MSG0|;"create_idp"))
	      (atoi (get_tile ;|MSG0|;"non_idp"))
	      (atoi (get_tile ;|MSG0|;"txt_idp"))
	      (atoi (get_tile ;|MSG0|;"txtstyle_idp"))  
	      (atoi (get_tile ;|MSG0|;"attdef_idp"))  
	      (atoi (get_tile ;|MSG0|;"attrib_idp"))  
	      (atoi (get_tile ;|MSG0|;"dim_idp"))  
	      (atoi (get_tile ;|MSG0|;"dimstyle_idp"))  
	      viewp_idp
	      view_idp
	      (atoi (get_tile ;|MSG0|;"ucs_idp"))  
	      shape_idp
	      sysvar_idp
	      (atoi (get_tile ;|MSG0|;"ltype_idp"))  
	      (atoi (get_tile ;|MSG0|;"pface_idp")) 
	      xed_idp 
	      (atoi (get_tile ;|MSG0|;"mesh_idp"))
	))
  (mapcar (quote (lambda (first second)
	     (if (/= first second) (setmflag))))
	  idp_topts
	  idp_opts)
  (setq idp_opts idp_topts)
)
;;
;;
(defun greyidp (/ gmode)
  (if (= (get_tile ;|MSG0|;"create_idp") ;|MSG0|;"0")
    (setq gmode 1)
    (setq gmode 0)
  )
  (mode_tile ;|MSG0|;"non_idp" gmode)
  (mode_tile ;|MSG0|;"txt_idp" gmode)
  (mode_tile ;|MSG0|;"txtstyle_idp" gmode)
  (mode_tile ;|MSG0|;"attdef_idp" gmode)
  (mode_tile ;|MSG0|;"attrib_idp" gmode)
  (mode_tile ;|MSG0|;"dim_idp" gmode)
  (mode_tile ;|MSG0|;"dimstyle_idp" gmode)
  (mode_tile ;|MSG0|;"ucs_idp" gmode)
  (mode_tile ;|MSG0|;"ltype_idp" gmode)
  (mode_tile ;|MSG0|;"pface_idp" gmode)
  (mode_tile ;|MSG0|;"mesh_idp" gmode) 
)
;;
;;
(defun miscmenu (/ logpathval trace acadent anno_angtol anno_pttol
		   tolpt xed msg_quota)
  (if (not (new_dialog ;|MSG0|;"miscopts" dcl_id))(exit))
  (popmisc)

  (setq wasmod modflag tmpsubopts misc_opts)
  (action_tile ;|MSG0|;"yes_log" ;|MSG0|;"(grey_log)(setmflag)")
  (action_tile ;|MSG0|;"log_loc" ;|MSG0|;"(setlogloc)")
  (action_tile ;|MSG0|;"log_dir" ;|MSG0|;"(chglogdir)(setmflag)")
  (action_tile ;|MSG0|;"map_ucs" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"map_unref" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"map_proxy" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"cals1_view" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"jamais" ;|MSG0|;"(setmflag)")
  (action_tile ;|MSG0|;"accept" ;|MSG0|;"(check_misc)")
  (action_tile ;|MSG0|;"cancel"
	       ;|MSG0|;"(setq misc_opts (canopt))(done_dialog 41)")
  (action_tile ;|MSG0|;"default" 
	       ;|MSG0|;"(setq misc_opts (nth 7 defopts))(popmisc)")
  (action_tile ;|MSG0|;"help"
	       ;|MSG0|;"(ait_helpdlg \"opt_outMiscDialogBox\")")
  (setq what_next1 (start_dialog))
)
;;
;;
(defun check_misc ()

  (cond
    ((not (chkdir (get_tile ;|MSG0|;"log_dir")))
      (mode_tile ;|MSG0|;"log_dir" 2))
    (t (waitmsg)(getmisc)(rs_err)(done_dialog 40))
  )
)
;;
;;
(defun popmisc (/ temp cals_list vdais_list)

  (if (= ;|MSG0|;"" (setq logpathval (cadr misc_opts)))
    (set_tile ;|MSG0|;"log_loc" ;|MSG0|;"iges_file")
    (progn
      (set_tile ;|MSG0|;"log_loc" ;|MSG0|;"spec_loc")
      (set_tile ;|MSG0|;"log_dir" logpathval)
    )
  )
  
  (set_tile ;|MSG0|;"yes_log" (itoa (car misc_opts)))
  (grey_log)
  
  (set_tile ;|MSG0|;"map_ucs" (itoa (caddr misc_opts)))
  
  (if (= 0 (nth 3 misc_opts))
    (setq temp ;|MSG0|;"1")
    (setq temp ;|MSG0|;"0")
  )
  (set_tile ;|MSG0|;"map_unref" temp)

  (if (= 0 (nth 4 misc_opts))
    (setq temp ;|MSG0|;"0")
    (setq temp ;|MSG0|;"1")
  )
  (set_tile ;|MSG0|;"map_proxy" temp)
  (setq cals_subset (nth 5 misc_opts))
  (setq vdais_subset (nth 6 misc_opts))
  (setq vdais_anno (nth 7 misc_opts))
  (setq vdais_free (nth 8 misc_opts))
  (set_tile ;|MSG0|;"cals1_view" (itoa (nth 9 misc_opts)))
  (setq trace (nth 10 misc_opts))
  (setq acadent (nth 11 misc_opts))
  (setq anno_angtol (nth 12 misc_opts))
  (setq anno_pttol (nth 13 misc_opts))
  (setq tolpt (nth 14 misc_opts))
  (setq xed (nth 15 misc_opts))
  (setq msg_quota (nth 16 misc_opts))
  (set_tile ;|MSG0|;"jamais" (itoa (nth 17 misc_opts)))
)
;;
;;
(defun setlogloc ()
  (if (chkdir (get_tile ;|MSG0|;"log_dir"))
    (progn
      (grey_logdir 0)
      (if (= (get_tile ;|MSG0|;"log_loc") ;|MSG0|;"spec_loc")
	(mode_tile ;|MSG0|;"log_dir" 2)
      )
      (setmflag)
    )
    (progn
      (set_tile ;|MSG0|;"log_loc" ;|MSG0|;"spec_loc")
      (mode_tile ;|MSG0|;"log_dir" 2)
    )
  )
)
;;
;;
(defun chglogdir ()
  (if (chkdir $value)
    (Progn
      (setq logpathval (delendslash $value))
      (rs_err)
    )
    (set_tile ;|MSG0|;"error" invdir)
  )
)
;;
;;
(defun getmisc (/ mapunref)

  (if (= ;|MSG0|;"iges_file" (get_tile ;|MSG0|;"log_loc"))
    (setq logpathval ;|MSG0|;"")
  )
  (if (= ;|MSG0|;"0" (get_tile ;|MSG0|;"map_unref"))
    (setq mapunref 1)
    (setq mapunref 0)
  )

  (setq misc_opts
	(list (atoi (get_tile ;|MSG0|;"yes_log")) logpathval
	      (atoi (get_tile ;|MSG0|;"map_ucs")) mapunref
	      (atoi (get_tile ;|MSG0|;"map_proxy"))
	      cals_subset vdais_subset vdais_anno vdais_free
	      (atoi (get_tile ;|MSG0|;"cals1_view")) trace acadent anno_angtol
	      anno_pttol tolpt xed msg_quota
	      (atoi (get_tile ;|MSG0|;"jamais"))
	      ))
)
;;
;;
(defun grey_log (/ mode)
  (if (= (get_tile ;|MSG0|;"yes_log") ;|MSG0|;"1")
    (setq mode 0)
    (setq mode 1)
  )
  (mode_tile ;|MSG0|;"log_loc" mode)
  (grey_logdir mode)
)
;;
;;
(defun grey_logdir (mode)
  (if (and (= mode 0) (= (get_tile ;|MSG0|;"log_loc") ;|MSG0|;"iges_file"))
    (setq mode 1)
  )
  (mode_tile ;|MSG0|;"log_dir" mode)
)
;;
;; sort in accordance with maxsort.
;;
(defun sortmax (sort_list)
  (if (>= (getvar ;|MSG0|;"MAXSORT") (length sort_list))       
    (acad_strlsort sort_list)
    sort_list
  )
)
;;
;;
(defun setpick ()
  (if dwg_list
    (set_tile ;|MSG0|;"dwg_list" (setq pickf ;|MSG0|;"0" pickigsf ;|MSG0|;"0"))
    (setq pickf ;|MSG0|;"" pickigsf ;|MSG0|;"")
  )
  (grey_modes)
)
;;
;;
(defun setigespath ()
  (if (chkdir (get_tile ;|MSG0|;"outdir"))
    (progn
      (greydir T)
      (if (= (get_tile ;|MSG0|;"iges_path") ;|MSG0|;"spec_path")
	(mode_tile ;|MSG0|;"outdir" 2)
      )
      (setmflag)
    )
    (progn
      (set_tile ;|MSG0|;"iges_path" ;|MSG0|;"spec_path")
      (mode_tile ;|MSG0|;"outdir" 2)
    )
  )
)
;;
;;
(defun check_mode ()
  (cond
    ((not (chkdir (get_tile ;|MSG0|;"outdir")))
      (set_tile ;|MSG0|;"error" invdir)
      (set_tile ;|MSG0|;"input_mode" ;|MSG0|;"batch")
      (mode_tile ;|MSG0|;"outdir" 2)
    )
    (t 
      (switchmode $value)
    )
  )
)
;;
;;
(defun check_out ()
  (cond
    ((not (chkdir (get_tile ;|MSG0|;"outdir")))
      (mode_tile ;|MSG0|;"outdir" 2)
      (set_tile ;|MSG0|;"error" invdir)
    )
    (t 
      (checkigslist)
      (if (= canflag 0) 
	(progn
	  (if fp_list
	    (progn
	      (save_env)
	      (setq proceed 1)
	    )
	  )
	  (done_dialog 1)
	)
      )
    )
  )
)
;; About IGESOUT
(defun ao_about ( / about )
  (if (not (new_dialog ;|MSG0|;"about" dcl_id))(exit))
  (ait_rxload ;|MSG0|;"igesout" igesout)
  (setq about (about_igesout))
  (if (/= nil about)
    (ait_about about ;|MSG0|;"IGESOUT")
  )
  (ait_rxunload ;|MSG0|;"igesout" igesout)
  (setq what_next1 (start_dialog))
)
;;
;; Initialization
;;
  (setq sfp_list nil  igs_name nil
	mfp_list nil  tmode    ;|MSG0|;"0"
	fp_list  nil  dwg_list nil
	tmp_list nil  proceed  nil
	dname    nil  canflag  0
	lfiles   0    igesext ;|MSG0|;"igs")

  (setq soptname (setq moptname defs))

  ;;; Load DCL files and exit if not available. 
  (if (not (setq dcl_id (ai_dcl ;|MSG0|;"igesout"))) (exit))
  (if (not (new_dialog ;|MSG0|;"igesout" dcl_id)) (exit))

  (if (not (read_fls))
    (set_tile ;|MSG0|;"error" ;|MSG83|;"Error reading igesout.cfg file.")
  )

  (if dname
    (progn
      (set_tile ;|MSG0|;"outdir" (delendslash dname))
      (set_tile ;|MSG0|;"iges_path" ;|MSG0|;"input_path")
    )
    (progn
      (set_tile ;|MSG0|;"outdir" 
		(setq dname (delendslash (getvar ;|MSG0|;"DWGPREFIX"))))
      (set_tile ;|MSG0|;"iges_path" ;|MSG0|;"spec_path")
    )
  )

  ;; (set_tile ;|MSG0|;"iges_ext" igesext)
  ;; Note: DWGNAME now has a .dwg suffix and no path 
  (setq igs_name (strcat (getvar ;|MSG0|;"DWGPREFIX") 
                         (getvar ;|MSG0|;"DWGNAME")))
  (setq igs_name (strcat (substr igs_name 1 (- (strlen igs_name) 4)) 
                         ;|MSG0|;"." igesext))
  (setq sfp_list (list igs_name))

  (getallopts nil)  ; get default options

  (if (= tmode ;|MSG0|;"0")
    (set_tile ;|MSG0|;"input_mode" ;|MSG0|;"input1")
    (set_tile ;|MSG0|;"input_mode" ;|MSG0|;"batch")
  )
  (switchmode tmode)

  (action_tile ;|MSG0|;"input_mode" ;|MSG0|;"(check_mode)")
  (action_tile ;|MSG0|;"dwg_list" 
	       ;|MSG0|;"(rs_err)(setq pickf $value)(setq pickigsf pickf)(if (= tmode \"1\")(set_tile \"fp_list\" pickf))(greylists)")
  (action_tile ;|MSG0|;"fp_list" ;|MSG0|;"(rs_err)(setq pickigsf $value)(set_tile \"dwg_list\" pickigsf)(setq pickf pickigsf)(greylists)")
  (action_tile ;|MSG0|;"add_to_list" ;|MSG0|;"(rs_err)(addfile)(mode_tile \"add_to_list\" 2)")
  (action_tile ;|MSG0|;"out_file" 
	       ;|MSG0|;"(mod_igsname)(setlist fp_list \"fp_list\")(set_tile \"fp_list\" pickf)")
  (action_tile ;|MSG0|;"del_cur_out" ;|MSG0|;"(rs_err)(remfile1)(greylists)")
  (action_tile ;|MSG0|;"clr_all" ;|MSG0|;"(clr_all)")
  (action_tile ;|MSG0|;"iges_path" ;|MSG0|;"(setigespath)")
  (action_tile ;|MSG0|;"outdir" ;|MSG0|;"(chgigsdir)")
  ;; (action_tile ;|MSG0|;"iges_ext" ;|MSG0|;"(check_igsext (setq igesext $value))")
  (action_tile ;|MSG0|;"cur_opt" ;|MSG0|;"(chgoption)")
  (action_tile ;|MSG0|;"menu_options" ;|MSG0|;"(optmenu)")
  (action_tile ;|MSG0|;"cancel" 
	       ;|MSG0|;"(ait_rxunload \"igesgui\" ISXOGD)(done_dialog 0)")
  (action_tile ;|MSG0|;"accept" ;|MSG0|;"(check_out)")
  (action_tile ;|MSG0|;"help" ;|MSG0|;"(ait_helpdlg \"maindiag_igesout\")")
  (action_tile ;|MSG0|;"about" ;|MSG0|;"(waitmsg)(ao_about)(rs_err)")
  (setq what_next (start_dialog))
  ;(del_tmps)
  (ait_rxunload ;|MSG0|;"igesgui" ISXOGD)
  (if (= optname curr) (setq optname ;|MSG0|;"tmpiges.opt"))
  (if proceed
    (cond
      ((= tmode ;|MSG0|;"0")
	(ait_rxload ;|MSG0|;"igesout" igesout)
	(igesout (car fp_list) optname)
	(ait_rxunload ;|MSG0|;"igesout" igesout)
      )
      ((= tmode ;|MSG0|;"1")
	(setq scrpath (strcat (strpfn (car dwg_list)) ;|MSG0|;"igesout.scr"))
	(setq f1 (open scrpath ;|MSG0|;"w"))
	(setq fcnt 0)
	(foreach n dwg_list 
	  (write-line ;|MSG0|;"_open" f1)
	  (write-line ;|MSG0|;"(if (> (getvar \"dbmod\") 0) \"_yes\" (princ))" f1)
	  (write-line (strcat "\"" n "\"") f1)
	  (write-line ;|MSG0|;"(arxload \"igesout\")" f1)
	  (write-line (strcat ;|MSG0|;"(igesout \"" (revbackslash (nth fcnt fp_list)) "\" \"" (revbackslash optname) "\")") f1)
	  (write-line ;|MSG0|;"(arxunload \"igesout\")" f1)
	  (setq fcnt (1+ fcnt))
	)
	(close f1)
      )
    )
  )
  (setvar ;|MSG0|;"cmdecho" old_cmd)
  (setq *error* old_error)
  (if (and proceed (= tmode ;|MSG0|;"1")) (command ;|MSG0|;"_script" scrpath))
)
(princ ;|MSG84|;"  IGESOUT loaded.\n")
(princ)

;;;-----BEGIN-SIGNATURE-----
;;; 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
;;; CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEE+U0vr78xSzq0vXnmuWytEw
;;; DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
;;; YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
;;; MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
;;; IC0gRzIwHhcNMjAwNjE1MDAwMDAwWhcNMjEwODEyMjM1OTU5WjCBijELMAkGA1UE
;;; BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
;;; FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
;;; b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
;;; AQEBBQADggEPADCCAQoCggEBAMt3bX7WxfbqKZpcgiX4jvJBSg6MCucqh4aDRHA8
;;; Hny4Fc2vKP+Xgn3mY4qtHoeNZ/j4d669d5+guDfOpD7jZX/JyR81wLE22GwswGLN
;;; M6n3N6Vudo0PcdCNXRxccGdPmJMIiEZNp84RjWN7Gy0JfgChxNiNEWS+Dezr5cYK
;;; B7UlGjGivbUdK6/HP0JlO3t3QxXYERASgPWj6YXmHGdvYHPBNit1MJTLRUj7uXyk
;;; bWzpaQEnXdey73yxg+FV2us0BicMIChoDXdTY/uAWbhkThET8GffQd08miZDNWMv
;;; IUIfFEkKjGZX6VkfkK/ShlVtjcdYlknpatvAtkFuYnQm6mECAwEAAaOCAT4wggE6
;;; MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
;;; MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
;;; eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
;;; cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
;;; IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
;;; STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
;;; aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBACiY
;;; IcU6uh5Lz5MGiUDT3w+daDMfYW44/ak/wMK1SgkL+iHzsgHaY8plDqZ4oaxT3U45
;;; /L5BdedSZWIvR738xpjcJNKZVILKTMg3mzl0bjO/t91dj5TFlTSpCbQgBh+jbBpJ
;;; R0il59VeZ9LQTvK/2E41TK89VHXZa+8MaTrsVOyTsBuAnNdYdzzExmAqp1BZdirH
;;; JH35aOAnZz3nkcGS4knKOAc7EX8pLkhAl4UDoS2V+2MKw5IXpYO3tU5TwM7y8uGZ
;;; gj/XpMlzWXtVQvOVpwgmT0XCymP5174BFyTP1SLNhlUWUfwFCypOY6tRHd6U9uqc
;;; BATajxbYoRpvjWl9Nu8xggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
;;; BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
;;; cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
;;; b2RlIFNpZ25pbmcgQ0EgLSBHMgIQT5TS+vvzFLOrS9eea5bK0TANBgkqhkiG9w0B
;;; AQsFADANBgkqhkiG9w0BAQEFAASCAQAykvICbrYLHJElD5If933vJBYp1K0Aryxf
;;; i8KTO7wF4vxvaBbnKCczDCH9HLIe/2o7fzwd2CY+jkyLDSwoCEs1V59r1qR9125d
;;; bTEWgEquu3efXPDiB7T4euZEopxdJ1bNg64N7bkMrQ8L5b8Snvk58KMxSbXMI6LJ
;;; CUhx7dheC6xqyZA+AluLZgm7sb8WYK1YfWWonw8Eb22Ht74Nx/0+SWpHEZ0NgmW3
;;; fZK0KJZazqkQ7NLjv9ZjBfUl4cX27pSNWHcJ9FCpddwqjGEEAq53OzwfnW45OcKX
;;; XDqu2lMMN+6Anqp7SHzHKVhL6miEKfwRZhqIiBZpPCnqgMB6YSXj
;;; -----END-SIGNATURE-----