;;;---------------------------------------------------------------------------;
;;;
;;;   BLANK.LSP   Version 1.0
;;;
;;;   Copyright (C) 1995 by Autodesk, Inc.
;;;
;;;   Permission to use, copy, modify, and distribute this software and its
;;;   documentation for any purpose and without fee is hereby granted.
;;;
;;;   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
;;;   ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR PURPOSE AND OF
;;;   MERCHANTABILITY ARE HEREBY DISCLAIMED.
;;;
;;;---------------------------------------------------------------------------;
;;;   BLANK, UNBLANK, and UNBLANKALL
;;;
;;;   This module provides functions to manipulate the visibility field of
;;;   AutoCAD objects.  BLANK will make a selection set of objects invisible.
;;;   UNBLANK will make a specified object (given its handle) visible.
;;;   UNBLANKALL will make all blanked entities visible.
;;;
;;;---------------------------------------------------------------------------;


;;;---------------------------------------------------------------------------;
;;; Internal error handling.
;;;---------------------------------------------------------------------------;
(defun blank_error(s)
  ;; The strings in the following statements can be translated.
  (if (/= s ;|MSG1|;"Function cancelled")
    (princ (strcat ;|MSG2|;"\nBLANK Error: " s))
  )
  (setq *error* olderr)
  (princ)
)

(defun unblank_error(s)
  ;; The strings in the following statements can be translated.
  (if (/= s ;|MSG3|;"Function cancelled")
    (princ (strcat ;|MSG3|;"\nUNBLANK Error: " s))
  )
  (setq *error* olderr)
  (princ)
)

(defun blank60 (e / e2)
  (if (not (null (assoc '60 e)))
    (setq e2 (subst '(60 . 1) '(60 . 0) e))
    (setq e2 (append e '((60 . 1))))
  )
)   

(defun show60 (e / e2)
  (if (not (null (assoc '60 e)))
     (setq e2 (subst '(60 . 0) '(60 . 1) e))
     (setq e2 (append e '((60 . 0))))
  )
)

(defun setvis ( vis ename / e)
  (setq e (entget ename))
  (if (eq vis 0)
     (entmod (show60 e))
     (entmod (blank60 e))
  )
  (entupd ename)
  ;; Blank vertices of polyline, if necessary
  (if (eq (cdr (nth 1 e)) "POLYLINE")
    (progn
      (setq ename (entnext ename))
      (setq e (entget ename))
      (while (eq (cdr (nth 1 e)) "VERTEX")
        (if (eq vis 0)
           (entmod (show60 e))
           (entmod (blank60 e))
        )
        (entupd ename)
        (setq ename (entnext ename))
        (setq e (entget ename))
      ) ; while
    ) ; progn
  ) ; if polyline
  (if (and (eq (cdr (nth 1 e)) "INSERT")
           (assoc '66 e))
    (progn
      (setq ename (entnext ename))
      (setq e (entget ename))
      (while (eq (cdr (nth 1 e)) "ATTRIB")
        (if (eq vis 0)
           (entmod (show60 e))
           (entmod (blank60 e))
        )
        (entupd ename)
        (setq ename (entnext ename))
        (setq e (entget ename))
      ) ; while
    ) ; progn
  )
)

(defun c:blank ( ) ;;; / olderr echo ss i ename )
  (setq olderr *error*                ; Redefine error handler.
        echo (getvar ;|MSG0|;"cmdecho")
        *error* blank_error)
  (setvar ;|MSG0|;"cmdecho" 0)                ; Turn off cmdecho sysvar
  (command ;|MSG0|;"_.undo" ;|MSG0|;"_group")

  (setq ss (ssget))
  (setq i 0)
  (while (< i (sslength ss)) (progn
     (setq ename (ssname ss i))
     (setvis 1 ename)
     (setq i (1+ i))
  ))

  (setq *error* old_error)            ; restore error function
  (command ;|MSG0|;"_.undo" ;|MSG0|;"_end")
  (setvar ;|MSG0|;"cmdecho" echo)             ; restore cmdecho sysvar
  (princ)                             ; Quiet exit.
)

(defun c:unblankall ( ) ;;; / olderr echo ss i ename )
  (setq olderr *error*                ; Redefine error handler.
        echo (getvar ;|MSG0|;"cmdecho")
        *error* unblank_error)
  (setvar ;|MSG0|;"cmdecho" 0)                ; Turn off cmdecho sysvar
  (command ;|MSG0|;"_.undo" ;|MSG0|;"_group")

  ;; Select all blanked entities
  (setq ss (ssget ;|MSG0|;"_x" '((60 . 1))))
  (if (not (null ss))
    (progn
      (setq i 0)
      (princ (sslength ss))
      (princ " blanked entities found.\n");
      ;; Unblank each entity in the set
      (while (< i (sslength ss)) (progn
         (setq ename (ssname ss i))
         (setvis 0 ename)
         (setq i (1+ i))
      )) 
    )   
    (princ "\n0 blanked entities found.\n");
  )

  (setq *error* old_error)            ; restore error function
  (command ;|MSG0|;"_.undo" ;|MSG0|;"_end")
  (setvar ;|MSG0|;"cmdecho" echo)             ; restore cmdecho sysvar
  (princ)                             ; Quiet exit.
)

(defun c:unblank ( ) ;;; / olderr echo ss i ename hand )
  (setq olderr *error*                ; Redefine error handler.
        echo (getvar ;|MSG0|;"cmdecho")
        *error* unblank_error)
  (setvar ;|MSG0|;"cmdecho" 0)                ; Turn off cmdecho sysvar
  (command ;|MSG0|;"_.undo" ;|MSG0|;"_group")

  (setq hand (getstring ;|MSG5|;"\nEnter handle of entity to be unblanked: "))
  ;; Unblank the entity if handle is not an empty string
  (if (> (strlen hand) 0)
    (progn
      (setq ename (handent hand))
      (if (/= nil ename)
        (setvis 0 ename)
        (princ ;|MSG6|;"Invalid handle.")
      )
    )
  )

  (setq *error* old_error)            ; restore error function
  (command ;|MSG0|;"_.undo" ;|MSG0|;"_end")
  (setvar ;|MSG0|;"cmdecho" echo)             ; restore cmdecho sysvar
  (princ)                             ; Quiet exit.
)
(princ)

;;;-----BEGIN-SIGNATURE-----
;;; 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
;;; CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEE+U0vr78xSzq0vXnmuWytEw
;;; DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
;;; YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
;;; MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
;;; IC0gRzIwHhcNMjAwNjE1MDAwMDAwWhcNMjEwODEyMjM1OTU5WjCBijELMAkGA1UE
;;; BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
;;; FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
;;; b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
;;; AQEBBQADggEPADCCAQoCggEBAMt3bX7WxfbqKZpcgiX4jvJBSg6MCucqh4aDRHA8
;;; Hny4Fc2vKP+Xgn3mY4qtHoeNZ/j4d669d5+guDfOpD7jZX/JyR81wLE22GwswGLN
;;; M6n3N6Vudo0PcdCNXRxccGdPmJMIiEZNp84RjWN7Gy0JfgChxNiNEWS+Dezr5cYK
;;; B7UlGjGivbUdK6/HP0JlO3t3QxXYERASgPWj6YXmHGdvYHPBNit1MJTLRUj7uXyk
;;; bWzpaQEnXdey73yxg+FV2us0BicMIChoDXdTY/uAWbhkThET8GffQd08miZDNWMv
;;; IUIfFEkKjGZX6VkfkK/ShlVtjcdYlknpatvAtkFuYnQm6mECAwEAAaOCAT4wggE6
;;; MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
;;; MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
;;; eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
;;; cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
;;; IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
;;; STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
;;; aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBACiY
;;; IcU6uh5Lz5MGiUDT3w+daDMfYW44/ak/wMK1SgkL+iHzsgHaY8plDqZ4oaxT3U45
;;; /L5BdedSZWIvR738xpjcJNKZVILKTMg3mzl0bjO/t91dj5TFlTSpCbQgBh+jbBpJ
;;; R0il59VeZ9LQTvK/2E41TK89VHXZa+8MaTrsVOyTsBuAnNdYdzzExmAqp1BZdirH
;;; JH35aOAnZz3nkcGS4knKOAc7EX8pLkhAl4UDoS2V+2MKw5IXpYO3tU5TwM7y8uGZ
;;; gj/XpMlzWXtVQvOVpwgmT0XCymP5174BFyTP1SLNhlUWUfwFCypOY6tRHd6U9uqc
;;; BATajxbYoRpvjWl9Nu8xggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
;;; BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
;;; cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
;;; b2RlIFNpZ25pbmcgQ0EgLSBHMgIQT5TS+vvzFLOrS9eea5bK0TANBgkqhkiG9w0B
;;; AQsFADANBgkqhkiG9w0BAQEFAASCAQAsLVJWfCJkDfRyer3aBPL8KRo+nVOiv5vY
;;; K0IdvHm9zX4Hhn+2NSpW/ZF0jHGaowwRaKIK5wm+x5UEROlovirsGnSA+9MeKN/3
;;; PJBIdLgkLv/4YH+hc3punuDFRoyIG0c4g1UvY10gDmY+kR6Lw/I9NvBbusVbOGNi
;;; RiA8YLDdY6hHAtoFMWVnjxL1CgqTnc+9kmGcXTafLsO2hFy7AVCjYCERTjCacSf0
;;; oOKY9imtc/ZnUVf3IswOh0J6ourAWbeUUXeg9QvMZu6Y962z6AFNQ7VWa7Mvxv/L
;;; pL+CqSSCfQP6EhEYCynn0jFNO3KsqiATYvgmWrCi+e3R4mLCu+Su
;;; -----END-SIGNATURE-----