//**************************************************************************/
// Copyright 2009 Autodesk, Inc.
// All rights reserved.
//
// This computer source code and related instructions and comments are the
// unpublished confidential and proprietary information of Autodesk, Inc.
// and are protected under Federal copyright and state trade secret law.
// They may not be disclosed to, copied or used by any third party without
// the prior written consent of Autodesk, Inc.
//**************************************************************************/
// DESCRIPTION: Weighted Product OIT effect.
// AUTHOR: Liming Zhang
// CREATED: February 2010
//**************************************************************************/

// World-view-projection transformation.
float4x4 gWVPXf : WorldViewProjection < string UIWidget = "None"; >;

// Color and alpha accumulation texture.
texture2D gColorTex : InputTexture
<
    string UIName = "Input Texture";
> = NULL;

// Color and alpha accumulation texture sampler.
sampler gColorSamp = sampler_state
{
    Texture   = <gColorTex>;
};

// Depth complexity texture.
texture2D gColorTex2 : InputTexture
<
    string UIName = "Input Texture";
> = NULL;

// Depth complexity texture sampler.
sampler gColorSamp2 = sampler_state
{
    Texture   = <gColorTex2>;
};

// Vertex shader input structure.
struct VS_INPUT
{
    float4 Pos : POSITION;
    float2 UV : TEXCOORD0;
};

// Vertex shader output structure.
struct VS_TO_PS
{
    float4 HPos : SV_Position;
    float2 UV : TEXCOORD0;
};

// Vertex shader.
VS_TO_PS VS_WeightedProductFinal(VS_INPUT In)
{
    VS_TO_PS Out;

    // Transform the position from object space to clip space for output.
    Out.HPos = mul(In.Pos, gWVPXf);

    // Pass the texture coordinates unchanged.
    Out.UV = In.UV;

    return Out;
}


// Pixel shader.
float4 PS_WeightedProductFinal(VS_TO_PS In) : SV_Target
{
    // accumulated weighted color and multiply accumulated no-weight alpha
    float4 accum = gColorTex.Sample(gColorSamp, In.UV);

    // revealage (alpha)
    float reveal = accum.a;

    if (reveal == 1.0)
    { // Save the blending and color texture fetch cost
        discard;
    }

    // get weighted alpha
    accum.a = gColorTex2.Sample(gColorSamp2, In.UV).r;

    // return "normal" color with a reasonable alpha, not the 1-alpha shown in the paper
    return float4(accum.rgb / max(accum.a, 1e-5), 1.0-reveal);
}

// The main technique.
technique10 Main
{
    pass p0
    {
        SetVertexShader(CompileShader(vs_4_0, VS_WeightedProductFinal()));
        SetPixelShader(CompileShader(ps_4_0, PS_WeightedProductFinal()));
    }
}
