//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.page - Implements animations and functionality of new tab page elements
 */

 // Namespace
var ntp = ntp || {};
ntp.page = ntp.page || {};

ntp.page.animationSettings = {
    TAB_SWITCHING_ANIMATION_DURATION: 600,
    ELEMENT_TOGGLING_ANIMATION_DURATION: 200
};

/**
 * Handles page layout that cannot be done via CSS
 */
ntp.page.layout = (function () {
    var $allTabs = null;
    var $columns = null;
    var $getStartedVideo = null;
    var $featuredVideo = null;
    var $onlineResource = null;
    var $continueWorking = null;
    var $connectPanel = null;
    var columnMinHeight = 504;
    //This is video item height. thumb height 90px, thumb border is 2px and thumb bottom-margin is 8px.
    var videoItemHeight = 100;
    //This is video item margin. video item margin-bottom is 16px.
    var videoItemGap = 16;
    
   
    var getColumnHeight = function() {
        var newHeight = $allTabs.height() * 0.7;
        if (newHeight < columnMinHeight){
            newHeight = columnMinHeight;
        }
        return newHeight;
    };
    var adjustContentSpace = function () {
        var allTabsHeight = $allTabs.height();
        var newColumnHeight = allTabsHeight * 0.7;
        var space = (allTabsHeight - newColumnHeight) * 0.375;
        if (space < 0) {
            space = 0;
        }
        // adjust top space
        $allTabs.children('.content').css('margin-top', space);
        // adjust content height
        $allTabs.children('.content').css('height', allTabsHeight - space);
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $allTabs = $('#canvas_slider > .tab');
            $columns = $('section.column');
            $getStartedVideo = $('#getting_started ol');
            $featuredVideo = $('#feature_videos ol');
            $continueWorking = $('#continue_working');
            $onlineResource = $('#online_resources_panel');
            $connectPanel = $('#connect_scroll_panel');
            this.adjustColumnHeight();
        },
        
        adjustColumnHeight: function () {
            var newHeight = getColumnHeight();
            $columns.height(newHeight);
            $continueWorking.height(newHeight);
            this.updateVideoSectionHeight();
            this.updateOnlineResourceHeight();
            this.updateConnectHeight();
        },
        
        update: function () {
            // Update tab width
            var tabWidth = $canvas.width();
            $allTabs.each(function () {
                $(this).width(tabWidth);
            });
            
            this.adjustColumnHeight();
            adjustContentSpace();
        },
        
        updateVideoSectionHeight: function() {
            var height = getColumnHeight();
            if(ntp.content.learnTab.hasGetStartedVideos() && ntp.content.learnTab.hasFeaturedVideos()) {
                var getStartedItemsHeight = $getStartedVideo.children().length * videoItemHeight
                                                + ($getStartedVideo.children().length - 1) * videoItemGap;
                var featuredItemsHeight = $featuredVideo.children().length * videoItemHeight
                                                + ($featuredVideo.children().length - 1) * videoItemGap;
                var getStartedHeight = height * 0.5;
                if (getStartedHeight > getStartedItemsHeight)
                    getStartedHeight = getStartedItemsHeight;
                var featuredHeight = height - getStartedHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                if (featuredHeight > featuredItemsHeight){
                    featuredHeight = featuredItemsHeight;
                    getStartedHeight = height - featuredHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                }
                $getStartedVideo.css({'height': 'auto', 'max-height': getStartedHeight});
                $featuredVideo.css({'height': 'auto','max-height': featuredHeight});
            }
            else if(ntp.content.learnTab.hasGetStartedVideos()) {
                var getStartedHeight = height - $('#getting_started_title').outerHeight(true);
                $getStartedVideo.css({'height': 'auto','max-height': getStartedHeight});
            }
            else if(ntp.content.learnTab.hasFeaturedVideos()) {
                var featuredHeight = height - $('#featured_title').outerHeight(true);
                $featuredVideo.css({'height': 'auto', 'max-height': featuredHeight});
            }
        },
        
        updateOnlineResourceHeight: function() {
            if(ntp.content.learnTab.hasOnlineResource()) {
                var height = getColumnHeight();
                var tipHeight = 0;
                if(ntp.content.learnTab.hasTips()) {
                    tipHeight = $('#Section_tip').outerHeight(true);
                }
                var onlineHeight = height - $('#online_resource_title').outerHeight(true) - tipHeight;
                $onlineResource.css('max-height', onlineHeight);
            }
        },
        
        updateConnectHeight: function() {
            if(ntp.widget.connect.hasConnect()){
                var height = getColumnHeight();
                var panelHeight;
                if(ntp.widget.notifications.hasNotifications()){
                    panelHeight = height - $('#connect_title').outerHeight(true) - $('#section_notifications').outerHeight(true);
                }
                else {
                    panelHeight = height - $('#connect_title').outerHeight(true);
                }
                $connectPanel.css('max-height', panelHeight);
            }
        }
    };
})();

/**
 * Handles tabs and tab switching
 */
ntp.page.tabs = (function () {
    
    var $canvas = null;
    var $tabs = null;
    var tipContentsVisited = false;
    var $visableTabs = null;
    var selectedIndex = 0;
    
    return {
        current: function () {
            return selectedIndex;
        },

        isTipContentsVisited: function () {
            return tipContentsVisited;
        },
        
        count: function () {
            if ($tabs) {
                return $tabs.length;
            } else {
                return 0;
            }
        },
        
        visibleCount: function () {
            if ($visibleTabs)
                return $visibleTabs.length;
            else
                return 1;
        },
        
        init: function () {
            $canvas = $('#canvas');
            $tabs = $('#canvas_slider > .tab');
            $visibleTabs = $('#canvas_slider > .tab.show');
        },
        
        addVisibleTab: function (id, select) {
            var index = 1;
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            var overCurrent = false;
            for(var i = 0; i < $tabs.length; i++){
                if($tabs.eq(i).attr('id') === id){
                    $tabs.eq(i).addClass('show');
                    $('#footer_nav li').eq(i).addClass('show');
                    $visibleTabs = $('#canvas_slider > .tab.show');
                    if(!overCurrent){
                        selectedIndex++;
                        if(!select)
                            this.update();
                    }
                    ntp.page.footerNavs.visibleTabsChange();
                    ntp.page.sideNavs.visibleTabsChange();
                    break;
                }
                else if($tabs.eq(i).attr('id') === $tab.attr('id')){
                    overCurrent = true;
                }
            }
            if(select)
                this.selectById(id, true, true);
        },
        
        removeVisibleTab: function (id, defaultId) {
            $visibleTabs = $('#canvas_slider > .tab.show');
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            if ($tab.attr('id') === id) {
                $tab.removeClass('show');
                $('#footer_nav li').eq(selectedIndex - 1).removeClass('show');
                $visibleTabs = $('#canvas_slider > .tab.show');
                this.selectById(defaultId, true, true);
                ntp.page.footerNavs.visibleTabsChange();
                ntp.page.sideNavs.visibleTabsChange();
            }
            else {
                var overCurrent = false;
                for (var i = 0; i < $visibleTabs.length; i++) {
                    if ($visibleTabs.eq(i).attr('id') === id) {
                        $visibleTabs.eq(i).removeClass('show');
                        $('#footer_nav li').eq(i).removeClass('show');
                        $visibleTabs = $('#canvas_slider > .tab.show');
                        if (!overCurrent) {
                            selectedIndex--;
                            this.update();
                        }
                        ntp.page.footerNavs.visibleTabsChange();
                        ntp.page.sideNavs.visibleTabsChange();
                        break;
                    } else if ($visibleTabs.eq(i).attr('id') === $tab.attr('id')) {
                        overCurrent = true;
                    }
                }
            }
        },
        
        // Selects a tab by its index that is only count for visible tabs.
        selectByIndex: function(index, immediate, forced) {
            var indexChanged = true;
            if (selectedIndex === index) {
                if (!forced) { return; }
                indexChanged = false;
            }
            
            selectedIndex = index;

            // Updates footer navigator class
            $('#footer_nav li.show').each(function (index) {
                if (index + 1 === selectedIndex) {
                    $(this).addClass('highlight');
                } else {
                    $(this).removeClass('highlight');
                }
            });

            // If 'immediate' is auto, then an animation will be used only if canvas is already running an animation.
            if (immediate === 'auto') {
                immediate = !$canvas.is(':animated');
            }

            // Updates tab position
            var $tab = $('.tab.show').eq(index - 1);

            if (tipContentsVisited === true && ntp.content.learnTab.urlIsHilightedContent('tip')) {
                //Leaving learn tab, so unhilight tip
                ntp.content.learnTab.unhilightViewedLearnConents({ tip: $('div#tip_resource_box').data('upload_time') });
                tipContentsVisited = false;

            }else if ($tab.attr('id') === 'tab_learn' && ntp.content.learnTab.urlIsHilightedContent('tip')) {

                //First time Visiting, so notify all others except self.
                ntp.acad.updateLearnContentState('tip', $('div#tip_resource_box').data('upload_time'));
                ntp.acad.updateLearnContentState('tipID', $('div#tip_resource_box').data('tip_ID'));
                ntp.acad.updateLearnContentState('tipExpireDate', $('div#tip_resource_box').data('tip_Expire_Date'));
                tipContentsVisited = true;
            }

            var newScrollLeft = $tab.position().left;
            if ($canvas.scrollLeft() !== newScrollLeft) {
                if (immediate) {
                    $canvas.stop().scrollLeft(newScrollLeft);
                } else {
                    var durationMultiplier = Math.abs(newScrollLeft - $canvas.scrollLeft()) / $canvas.width();
                    if (durationMultiplier > 1) { durationMultiplier = 1; }
                    $canvas.stop().animate(
                        { scrollLeft: newScrollLeft },
                        durationMultiplier * ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                        'linear');
                }
            }
            ntp.page.footerNavs.updateArrow(immediate);
            ntp.page.sideNavs.update(immediate, true, true);
        },
        
        update: function () {
            this.selectByIndex(selectedIndex, 'auto', true);
        },

        // Selects a tab by its id
        selectById: function(id, immediate, forced) {
            var tabIndex = $('.tab.show').index($('#' + id));
            if (tabIndex >= 0 && tabIndex < this.visibleCount()) {
                this.selectByIndex(tabIndex + 1, immediate, forced);
            }
        },
        
        // Auto snaps to the closest tab after scrolling
        snap: function (immediate) {
            var tabWidth = $canvas.width();
            var currentScrollLeft = $canvas.scrollLeft();
            
            // Return if scrolling is caused by resizing
            if (currentScrollLeft === (tabWidth * (selectedIndex - 1))) {
                return;
            }
            
            var closestTabIndex = Math.floor((currentScrollLeft + tabWidth / 2) / tabWidth) + 1;
            this.selectByIndex(closestTabIndex, immediate, true);
        }
    };
})();

/**
 * Handles side navigation arrows
 */
ntp.page.sideNavs = (function () {
    
    var $canvas = null;
    var $tabContents = null;
    var $sideNavs = null;
    
    var tabContentWidth = null;
    var leftSideNavWidth = null;
    var rightSideNavWidth = null;
    
    var overlapped = null;
    
    // Updates the global variable storing left and right of the content of currently selected tab        
    var updateCurrentTabContentWidth = function () {
        if ($tabContents === null) { return null; }
        var selectedIndex = ntp.page.tabs.current();
        var $content = $tabContents.eq(selectedIndex - 1);
        var $columns = $content.children('.column');
        if ($columns.length === 0) {
            tabContentWidth = $content.outerWidth(true);
        } else {
            var $firstColumn = $columns.eq(0);
            var $lastColumn = $columns.eq($columns.length - 1);
            tabContentWidth = $lastColumn.offset().left + $lastColumn.outerWidth(true) - $firstColumn.offset().left;
        }
    };
    
    // Updates side navigator overlap status. If side navigators overlap with page content, they would be hidden.
    var updateOverlap = function (immediate) {
        // Update stored tab content width
        updateCurrentTabContentWidth();

        var tabWidth = $canvas.width();
        var sideNavSpace = (tabWidth - tabContentWidth) / 2;
    
        var newOverlapped = ((sideNavSpace - leftSideNavWidth) < 10 || 
                (sideNavSpace -rightSideNavWidth) < 10);

        if (newOverlapped !== overlapped) {
            if (overlapped === null || immediate) {
                if (newOverlapped) {
                    $sideNavs.addClass('overlapped');
                } else {
                    $sideNavs.removeClass('overlapped');
                }
            } else {
                // Use a fadein or fadeout animation when showing or hiding the side navigators
                $sideNavs.stop(true);
                if (newOverlapped) {
                    $sideNavs.css('opacity', 1);
                    $sideNavs.animate({ opacity: 0 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                        $sideNavs.addClass('overlapped');
                    });
                } else {
                    $sideNavs.css('opacity', 0);
                    $sideNavs.removeClass('overlapped');
                    $sideNavs.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                    });
                }
            }

            overlapped = newOverlapped;
        }
    };
    
    // Updates total width of side navigation tools in global variable.
    var updateWidth = function () {
        leftSideNavWidth = 33;
        leftSideNavWidth += $sideNavs.eq(0).outerWidth(true);
        rightSideNavWidth = 33;
        rightSideNavWidth += $sideNavs.eq(1).outerWidth(true);    
    };
    
    // Updates individual side navigator according to currently selected tab.
    var updateNavigator = function (navigatorSelector, navigatorTabIndex, immediate, handleOverlap) {
        var $sideNav = $(navigatorSelector);

        $sideNav.off('click'); // Clear existing event handler

        if (immediate) {
            var sideNavShouldShow = (navigatorTabIndex > 0 && navigatorTabIndex <= ntp.page.tabs.visibleCount()
                                    && ntp.page.tabs.visibleCount() > 1);
            if (sideNavShouldShow) {
                // Target tab should show. 
                var $footerNav = $('#footer_nav li.show').eq(navigatorTabIndex - 1);
                var label = $footerNav.children('.label').text();
                $sideNav.removeClass('disabled')
                        .children('.label')
                        .text(label);

                $sideNav.one('click', function (event) {
                    ntp.page.tabs.selectByIndex(navigatorTabIndex);
                    event.preventDefault();
                });
            } else {
                //Side navigator need to hide.
                $sideNav.addClass('disabled');
            }

            // Update width of side navigators after their size has changed
            updateWidth();

            if (handleOverlap) {
                updateOverlap(true);
            }
        } else {
            // Change navigators with fade in/fade out effect. The implementation is a little bit complicated,
            // as the simple jQuery fadeIn() fadeOut() do not work so well here.
            var fadeIn = function () {
                $sideNav.stop().css('opacity', '');
                updateNavigator(navigatorSelector, navigatorTabIndex, true, handleOverlap);
                if ($sideNav.css('display') !== 'none') {
                    $sideNav.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNav.css('opacity', '');
                    });
                }
            };

            $sideNav.stop();
            if ($sideNav.css('display') === 'none') {
                setTimeout(fadeIn, ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2);
            } else {
                $sideNav.animate({ opacity: 0 }, 
                    ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                    'linear', fadeIn);
            }
        }
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');         
            $tabContents = $('#canvas_slider > .tab.show > .content');
            $sideNavs = $('#side_nav li');
        },
        
        visibleTabsChange: function () {
            $tabContents = $('#canvas_slider > .tab.show > .content');
            this.update(true, true, true);
        },
        
        // Updates both side navigators according to currently selected tab.
        update: function (immediate, handleOverlap, forced) {
            var currentTab = ntp.page.tabs.current();
            
            if (forced) {
                // Update side navigators
                updateNavigator('#left_side_nav', currentTab - 1, immediate, handleOverlap);
                updateNavigator('#right_side_nav', currentTab + 1, immediate, handleOverlap);
            } else {
                updateOverlap();
            }
        }
    };
})();

/**
 * Handles footer navigation buttons
 */
ntp.page.footerNavs = (function () {
    var $canvas = null;
    var $footerNavs = null;
    var $navArrow = null;
    var $navRoot = null;
    var footerNavWidth = null;
    var isInit = false;
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $navArrow = $('#nav_arrow');
            $footerNavs = $('#footer_nav li.show');
            $navRoot = $('#footer_nav');
            if(!isInit){
                $('#footer_nav li .button').each(
                        function(index) {
                            var tabId = $('#canvas_slider > .tab').eq(index).attr('id');
                            $(this).on("click", function() {
                                ntp.page.tabs.selectById(tabId);
                                event.preventDefault();
                            });
                        });
            }
            isInit = true;
            if($footerNavs.length > 1)
                this.visibleTabsChange();
        },
        
        visibleTabsChange: function () {
            $footerNavs = $('#footer_nav li.show');
            // Makes the footer navs equal width, and store footerNavWidth
            footerNavWidth = 0;

            if ($footerNavs.length > 1) {
                var innerWidth = 0;
                $navRoot.removeClass('hidden');
                $footerNavs.each(function() {
                    $(this).width('auto');
                    var currentNavWidth = parseFloat(window.getComputedStyle(this).width);
                    currentNavWidth = Math.ceil(currentNavWidth);
                    if (currentNavWidth > innerWidth) {
                        innerWidth = currentNavWidth;
                    }
                });

                $footerNavs.each(function(index) {
                    if (index + 1 === ntp.page.tabs.current()) {
                        $(this).addClass('highlight');
                    } else {
                        $(this).removeClass('highlight');
                    }
                    $(this).width(innerWidth);
                });

                // Calculate width from offset of two elements. This is needed to count in a mysterious space between the elements in the width
                // (elements already have margin = 0 )
                var firstTabLeft = $footerNavs.eq(0).offset().left;
                var secondTabLeft = $footerNavs.eq(1).offset().left;
                if (firstTabLeft === secondTabLeft && firstTabLeft > 0)
                    footerNavWidth = $footerNavs.eq(0).innerWidth();
                else
                    footerNavWidth = $footerNavs.eq(1).offset().left - $footerNavs.eq(0).offset().left;
                $navArrow.removeClass("hidden");
                this.updateArrow(true);
            } else {
                $navRoot.addClass('hidden');
                $navArrow.addClass("hidden");
            }
        },
        
        updateArrow: function(immediate){
            if(!$('#nav_arrow').hasClass("hidden")){
                var index = ntp.page.tabs.current() - 1;
                var footerFirstTab = $footerNavs.eq(0);
                var footerLeft = footerFirstTab.offset().left;
                var newOffset = footerLeft + footerNavWidth * (ntp.page.tabs.current() - 0.5) - $navArrow.width() / 2;
                if(immediate){
                    $navArrow.stop().css('left', newOffset);
                } else {
                    var oldLeft = $navArrow.css('left');
                    if(oldLeft !== newOffset){
                        $navArrow.stop().animate(
                                {left: newOffset},
                                ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                                'linear');
                    }
                }
            }
        },
        
        update: function () {
            if (!footerNavWidth) { // footerNavWidth might be 0 if the window size is too small
                this.visibleTabsChange();
            }
            this.updateArrow(true);
        }
    };
})();


/**
 * Page initialization on customization complete
 */
ntp.deferred.customization.done(function () {
    ntp.page.layout.init();
    ntp.page.tabs.init();
    ntp.widget.templatebar.init();
    
    if (ntp.page.tabs.count() > 1) {
        ntp.page.sideNavs.init();
    }

    $(window).resize(function() {
        ntp.widget.templatebar.update();
        ntp.widget.projectbar.update();
        ntp.page.layout.update();

        if (ntp.page.tabs.count() > 1) {
            ntp.page.footerNavs.update();
            // Fix the problem of window resizing when switch tab transition is not finished.
            ntp.page.tabs.update();
        }
    });

    $(document).on('keydown', function(e) {
        if (e.ctrlKey || e.altKey) {
            ntp.acad.notifyShortcutKeyPressed(e);
        }
    });
    
    $(document).mousedown(function(e) {
        if(e.button == 1) {
            e.preventDefault();
        }
    });

    // Initialize layouts
    ntp.page.layout.update();
    
    // Show create tab on start
    if (ntp.page.tabs.count() > 1) {
        ntp.page.tabs.selectById('tab_create', true, true);
    }
});

/**
 * Page initialization on customization and localization complete.
 */
$.when(ntp.deferred.localization, ntp.deferred.customization).done(function () {
    // Updates layout after text is loaded. Some elements are resized automatically.
    if (ntp.page.tabs.count() > 1) {
        ntp.page.footerNavs.init();
        ntp.page.sideNavs.update(true, true, true); // Update side navigator names
    }
});

/**
 * Allows only dropping dwg files
 */
(function () {
    function isDrawingFile(path) {
        path = path || "";
        path = path.substr(path.length - 4).toLowerCase();
        return (path == ".dxf") || (path == ".dwg") || (path == ".dws") || (path == ".dwt");
    }

    function queryDropHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var data = e.originalEvent.dataTransfer;
        data.dropEffect = "none";
        if (data && data.files) {
            for (var i = 0; i < data.files.length; ++i) {
                if (isDrawingFile(data.files[i].name)) {
                    data.dropEffect = "copy";
                    break;
                }
            }
        }
    }

    $(window).on('dragenter', queryDropHandler);
    $(window).on('dragover', queryDropHandler);
    $(window).on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        ntp.acad.openDroppedFiles();
    });
})();

// SIG // Begin signature block
// SIG // MIIZjAYJKoZIhvcNAQcCoIIZfTCCGXkCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // Rns7C6EHCOFHEjMNlXTMG+QmJVWIaTimEl1+hmW5m1qg
// SIG // ggopMIIE2jCCA8KgAwIBAgIQfG5XIF7pXpl44AXkMQgG
// SIG // EDANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xODA3MDUwMDAw
// SIG // MDBaFw0xOTA4MDUyMzU5NTlaMIGKMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEXMBUGA1UECgwOQXV0b2Rlc2ssIElu
// SIG // Yy4xHzAdBgNVBAsMFkRlc2lnbiBTb2x1dGlvbnMgR3Jv
// SIG // dXAxFzAVBgNVBAMMDkF1dG9kZXNrLCBJbmMuMIIBIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA7mRyNXG+
// SIG // QgtKVZfb37ZZm2wDgkFnx9kZy5clwGxvrSFmo0RQb5gR
// SIG // 6Ez0lhlQRNsfBVcYZrp0YyNxKg25oCdLDgPnaqilCia0
// SIG // UkRTpMYIUdaD0acP8C7Cb7r4oeKl2c3qVOGpvyCSM7Ep
// SIG // x16v7HtO0xeiOtCGU+cQ2kussGyDxiuvg0nsCMR4zV7b
// SIG // cg0bjKS3q22JzkzNrJpxSJJPnemEif5NQVQExg8Dg8YO
// SIG // UVAEhPc5+TRCtWH0fT+c4asKEUo9Urz5TBRwqqg2h/Ti
// SIG // k2qUPok3YlHrI8JIOeKfItfgU4PYtwr6yt4PPW7Jie1g
// SIG // cYi6JNButwAUIwRfCI1brIyhwQIDAQABo4IBPjCCATow
// SIG // CQYDVR0TBAIwADAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0l
// SIG // BAwwCgYIKwYBBQUHAwMwYQYDVR0gBFowWDBWBgZngQwB
// SIG // BAEwTDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNi
// SIG // LmNvbS9jcHMwJQYIKwYBBQUHAgIwGQwXaHR0cHM6Ly9k
// SIG // LnN5bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAU1MAGIknr
// SIG // OUvdk+JcobhHdglyA1gwKwYDVR0fBCQwIjAgoB6gHIYa
// SIG // aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcmwwVwYIKwYB
// SIG // BQUHAQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vcmIu
// SIG // c3ltY2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vcmIu
// SIG // c3ltY2IuY29tL3JiLmNydDANBgkqhkiG9w0BAQsFAAOC
// SIG // AQEArEMBlWJ6bc5AKEqwyO4TWNdnsYX9FreWUauXniGK
// SIG // lAwGhAkH7N4Ma1sE3yKtQi6pE7n+1t0bmlPPLiAWfRgI
// SIG // oDZ/ZwiWkOoP7A+31Q7skKjdUN1M96QjcbNUQjDkFqxy
// SIG // bHUraVr0gPlGUpnMR1npXAFHgXCrw45F1NT+LZNHWt/l
// SIG // AmfQsquTmDNvs7o0f/AhQr45Ylo8/Gsr8neojm2Io8iF
// SIG // pcMHqY7jcNnVL4ethYXVVdY0JSBN+Vg/2G0j/Lq8ic+I
// SIG // f8LiWdBllb3DUrVrgdjmBIHYCxPPJTXgsTGv4hwlRifS
// SIG // QkMw83nBdgWsObnssKnDJEOrhm9Da8FRHL4/fzCCBUcw
// SIG // ggQvoAMCAQICEHwbNTVK59t050FfEWnKa6gwDQYJKoZI
// SIG // hvcNAQELBQAwgb0xCzAJBgNVBAYTAlVTMRcwFQYDVQQK
// SIG // Ew5WZXJpU2lnbiwgSW5jLjEfMB0GA1UECxMWVmVyaVNp
// SIG // Z24gVHJ1c3QgTmV0d29yazE6MDgGA1UECxMxKGMpIDIw
// SIG // MDggVmVyaVNpZ24sIEluYy4gLSBGb3IgYXV0aG9yaXpl
// SIG // ZCB1c2Ugb25seTE4MDYGA1UEAxMvVmVyaVNpZ24gVW5p
// SIG // dmVyc2FsIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3Jp
// SIG // dHkwHhcNMTQwNzIyMDAwMDAwWhcNMjQwNzIxMjM1OTU5
// SIG // WjCBhDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFu
// SIG // dGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRl
// SIG // YyBUcnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRl
// SIG // YyBDbGFzcyAzIFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0Eg
// SIG // LSBHMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBANeVQ9Tc32euOftSpLYmMQRw6beOWyq6N2k1lY+7
// SIG // wDDnhthzu9/r0XY/ilaO6y1L8FcYTrGNpTPTC3Uj1Wp5
// SIG // J92j0/cOh2W13q0c8fU1tCJRryKhwV1LkH/AWU6rnXmp
// SIG // AtceSbE7TYf+wnirv+9SrpyvCNk55ZpRPmlfMBBOcWNs
// SIG // WOHwIDMbD3S+W8sS4duMxICUcrv2RZqewSUL+6Mcntim
// SIG // CXBx7MBHTI99w94Zzj7uBHKOF9P/8LIFMhlM07Acn/6l
// SIG // eCBCcEGwJoxvAMg6ABFBekGwp4qRBKCZePR3tPNgKuZs
// SIG // UAS3FGD/DVH0qIuE/iHaXF599Sl5T7BEdG9tcv8CAwEA
// SIG // AaOCAXgwggF0MC4GCCsGAQUFBwEBBCIwIDAeBggrBgEF
// SIG // BQcwAYYSaHR0cDovL3Muc3ltY2QuY29tMBIGA1UdEwEB
// SIG // /wQIMAYBAf8CAQAwZgYDVR0gBF8wXTBbBgtghkgBhvhF
// SIG // AQcXAzBMMCMGCCsGAQUFBwIBFhdodHRwczovL2Quc3lt
// SIG // Y2IuY29tL2NwczAlBggrBgEFBQcCAjAZGhdodHRwczov
// SIG // L2Quc3ltY2IuY29tL3JwYTA2BgNVHR8ELzAtMCugKaAn
// SIG // hiVodHRwOi8vcy5zeW1jYi5jb20vdW5pdmVyc2FsLXJv
// SIG // b3QuY3JsMBMGA1UdJQQMMAoGCCsGAQUFBwMDMA4GA1Ud
// SIG // DwEB/wQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UE
// SIG // AxMRU3ltYW50ZWNQS0ktMS03MjQwHQYDVR0OBBYEFNTA
// SIG // BiJJ6zlL3ZPiXKG4R3YJcgNYMB8GA1UdIwQYMBaAFLZ3
// SIG // +mlIR59TEtXC6gcydgfRlwcZMA0GCSqGSIb3DQEBCwUA
// SIG // A4IBAQB/68qn6ot2Qus+jiBUMOO3udz6SD4Wxw9FlRDN
// SIG // J4ajZvMC7XH4qsJVl5Fwg/lSflJpPMnx4JRGgBi7odSk
// SIG // VqbzHQCR1YbzSIfgy8Q0aCBetMv5Be2cr3BTJ7noPn5R
// SIG // oGlxi9xR7YA6JTKfRK9uQyjTIXW7l9iLi4z+qQRGBIX3
// SIG // FZxLEY3ELBf+1W5/muJWkvGWs60t+fTf2omZzrI4RMD3
// SIG // R3vKJbn6Kmgzm1By3qif1M0sCzS9izB4QOCNjicbkG8a
// SIG // vggVgV3rL+JR51EeyXgp5x5lvzjvAUoBCSQOFsQUecFB
// SIG // NzTQPZFSlJ3haO8I8OJpnGdukAsak3HUJgLDwFojMYIO
// SIG // uzCCDrcCAQEwgZkwgYQxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UE
// SIG // CxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1MDMGA1UE
// SIG // AxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBT
// SIG // aWduaW5nIENBIC0gRzICEHxuVyBe6V6ZeOAF5DEIBhAw
// SIG // DQYJYIZIAWUDBAIBBQCgfDAQBgorBgEEAYI3AgEMMQIw
// SIG // ADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
// SIG // BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG
// SIG // 9w0BCQQxIgQgG73p0emizmLrPZz9eW97ULaPLOG+CFQ6
// SIG // NHXCCrQYC3AwDQYJKoZIhvcNAQEBBQAEggEA2GWVLXfk
// SIG // plL9tQmpT6cZtZ9jMiyNpRixyS1tYB7ly4hXn0Yy17qn
// SIG // I/JyITlOWaZgl8lNMj3yJdA0DUF9uCpvQityERz7TMXW
// SIG // /9aTCdxRRwqWfbRK/bRr2MZcQi9JmzBucW7zWRbHBVY9
// SIG // 0wHQbYMRMDthD/EF9yeDby+9n29sU8IN6c9SFbrKoUcd
// SIG // ZDqU6f4ZMXLTSuMlHxNon8kRED86UXLmyhiVj1mC3d0M
// SIG // GpVGgjlsIwShqhGQg+v4/ZmHLjB4JUDTkfQeWjDCQGGA
// SIG // +jEbHX4XHJCugTx3FDMucyikogbZdbNdPslVV5SPLAZU
// SIG // zk3AOhHUxov28lMrDFCTEDBsgKGCDHQwggxwBgorBgEE
// SIG // AYI3AwMBMYIMYDCCDFwGCSqGSIb3DQEHAqCCDE0wggxJ
// SIG // AgEDMQ8wDQYJYIZIAWUDBAIBBQAwga8GCyqGSIb3DQEJ
// SIG // EAEEoIGfBIGcMIGZAgEBBgkrBgEEAaAyAgMwMTANBglg
// SIG // hkgBZQMEAgEFAAQgPgLL+X+E391XB/pzwP/BO9lqXnEg
// SIG // sts90wutLdcUX5oCFAvOfux2BOPqbukMX3OJgA3uAc5D
// SIG // GA8yMDE5MDEzMDA5NDc0NVqgL6QtMCsxKTAnBgNVBAMM
// SIG // IEdsb2JhbFNpZ24gVFNBIGZvciBBZHZhbmNlZCAtIEcy
// SIG // oIII0zCCBLYwggOeoAMCAQICDAynz10HBySsieeaOjAN
// SIG // BgkqhkiG9w0BAQsFADBbMQswCQYDVQQGEwJCRTEZMBcG
// SIG // A1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMo
// SIG // R2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEy
// SIG // NTYgLSBHMjAeFw0xODAyMTkwMDAwMDBaFw0yOTAzMTgx
// SIG // MDAwMDBaMCsxKTAnBgNVBAMMIEdsb2JhbFNpZ24gVFNB
// SIG // IGZvciBBZHZhbmNlZCAtIEcyMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEAt8eSiijY85HLUA1d6ylE
// SIG // wYJZpcpkzr2DtVHllMxugJoXbNz/YuY2Q14Bwgxkjcp3
// SIG // nvZdd0W5Xsgk60E3EWY3rMjNdJvMs+9M4n8bznO+yZad
// SIG // NR5891T3bOygEtaumZ1jxHhfWakuIvXDJ9iyW0VjJ5D/
// SIG // jLE9qbTzDsDxIGUGiZ0DHKOwUkGFNOu8y85U9hL+swGJ
// SIG // Sqb6VMUf+9McwOO/l+M/0Bv5QMcb1esmfEMwx3DyeBGq
// SIG // hU6lDg0ASjoRd6PKIwGAnsiTVl1GzIrY0qoKhIWyiVvV
// SIG // Ms5BexhOVMqr0VJeEtMknlbzDI/dvZKOedVBicOjoOWR
// SIG // /MdnnYczznEycwIDAQABo4IBqDCCAaQwDgYDVR0PAQH/
// SIG // BAQDAgeAMEwGA1UdIARFMEMwQQYJKwYBBAGgMgEeMDQw
// SIG // MgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNp
// SIG // Z24uY29tL3JlcG9zaXRvcnkvMAkGA1UdEwQCMAAwFgYD
// SIG // VR0lAQH/BAwwCgYIKwYBBQUHAwgwRgYDVR0fBD8wPTA7
// SIG // oDmgN4Y1aHR0cDovL2NybC5nbG9iYWxzaWduLmNvbS9n
// SIG // cy9nc3RpbWVzdGFtcGluZ3NoYTJnMi5jcmwwgZgGCCsG
// SIG // AQUFBwEBBIGLMIGIMEgGCCsGAQUFBzAChjxodHRwOi8v
// SIG // c2VjdXJlLmdsb2JhbHNpZ24uY29tL2NhY2VydC9nc3Rp
// SIG // bWVzdGFtcGluZ3NoYTJnMi5jcnQwPAYIKwYBBQUHMAGG
// SIG // MGh0dHA6Ly9vY3NwMi5nbG9iYWxzaWduLmNvbS9nc3Rp
// SIG // bWVzdGFtcGluZ3NoYTJnMjAdBgNVHQ4EFgQULW5u0Y3l
// SIG // A4Du52Ppp9naCzxXIj0wHwYDVR0jBBgwFoAUkiGnSpVd
// SIG // ZLCbtB7mADdH5p1BK0wwDQYJKoZIhvcNAQELBQADggEB
// SIG // AI39HR+PiNNl+LVcRumn07Y/SstDBcIHDcNeA4QjN6h6
// SIG // jJcj/yechH1h61xLVz6b9ETbgFj3tmXejxIKX6eHdLfa
// SIG // FECHUeyfSkdV8BfoGD5xIDHmKCKrTvaunGvC50T7aA0g
// SIG // f1M/lBuhhoJOMhYQWj+JWMHQtNu2kPuNMumhS0ZQN7bD
// SIG // OukpJBKSWk4rubYKejaMl8DviXwz3jHhoZfk50WtYVL9
// SIG // WonTVH6KTVuJew6Wd9HyPDgb2vmlYSBlArzFrdcMYhAo
// SIG // LN2246hhAKwIJqONhdBkyTaQHopSyYeNNaVdSJH8k6al
// SIG // ZfHs+TrcpSyeBG1iM/7e+p/Fzq+V81Go8s4wggQVMIIC
// SIG // /aADAgECAgsEAAAAAAExicZQBDANBgkqhkiG9w0BAQsF
// SIG // ADBMMSAwHgYDVQQLExdHbG9iYWxTaWduIFJvb3QgQ0Eg
// SIG // LSBSMzETMBEGA1UEChMKR2xvYmFsU2lnbjETMBEGA1UE
// SIG // AxMKR2xvYmFsU2lnbjAeFw0xMTA4MDIxMDAwMDBaFw0y
// SIG // OTAzMjkxMDAwMDBaMFsxCzAJBgNVBAYTAkJFMRkwFwYD
// SIG // VQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYDVQQDEyhH
// SIG // bG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAtIFNIQTI1
// SIG // NiAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
// SIG // CgKCAQEAqpuOw6sRUSUBtpaU4k/YwQj2RiPZRcWVl1ur
// SIG // Gr/SbFfJMwYfoA/GPH5TSHq/nYeer+7DjEfhQuzj46FK
// SIG // bAwXxKbBuc1b8R5EiY7+C94hWBPuTcjFZwscsrPxNHaR
// SIG // ossHbTfFoEcmAhWkkJGpeZ7X61edK3wi2BTX8QceeCI2
// SIG // a3d5r6/5f45O4bUIMf3q7UtxYowj8QM5j0R5tnYDV56t
// SIG // LwhG3NKMvPSOdM7IaGlRdhGLD10kWxlUPSbMQI2CJxtZ
// SIG // IH1Z9pOAjvgqOP1roEBlH1d2zFuOBE8sqNuEUBNPxtyL
// SIG // ufjdaUyI65x7MCb8eli7WbwUcpKBV7d2ydiACoBuCQID
// SIG // AQABo4HoMIHlMA4GA1UdDwEB/wQEAwIBBjASBgNVHRMB
// SIG // Af8ECDAGAQH/AgEAMB0GA1UdDgQWBBSSIadKlV1ksJu0
// SIG // HuYAN0fmnUErTDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAy
// SIG // BggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2ln
// SIG // bi5jb20vcmVwb3NpdG9yeS8wNgYDVR0fBC8wLTAroCmg
// SIG // J4YlaHR0cDovL2NybC5nbG9iYWxzaWduLm5ldC9yb290
// SIG // LXIzLmNybDAfBgNVHSMEGDAWgBSP8Et/qC5FJK5NUPpj
// SIG // move4t0bvDANBgkqhkiG9w0BAQsFAAOCAQEABFaCSnzQ
// SIG // zsm/NmbRvjWek2yX6AbOMRhZ+WxBX4AuwEIluBjH/NSx
// SIG // N8RooM8oagN0S2OXhXdhO9cv4/W9M6KSfREfnops7yyw
// SIG // 9GKNNnPRFjbxvF7stICYePzSdnno4SGU4B/EouGqZ9uz
// SIG // nHPlQCLPOc7b5neVp7uyy/YZhp2fyNSYBbJxb051rvE9
// SIG // ZGo7Xk5GpipdCJLxo/MddL9iDSOMXCo4ldLA1c3PiNof
// SIG // KLW6gWlkKrWmotVzr9xG2wSukdduxZi61EfEVnSAR3hY
// SIG // jL7vK/3sbL/RlPe/UOB74JD9IBh4GCJdCC6MHKCX8x2Z
// SIG // faOdkdMGRE4EbnocIOM28LZQuTGCAqgwggKkAgEBMGsw
// SIG // WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNp
// SIG // Z24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGlt
// SIG // ZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzICDAynz10H
// SIG // BySsieeaOjANBglghkgBZQMEAgEFAKCCAQ4wGgYJKoZI
// SIG // hvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJ
// SIG // BTEPFw0xOTAxMzAwOTQ3NDVaMC8GCSqGSIb3DQEJBDEi
// SIG // BCBKJdk+xV/GpfKkEqerJN+1MnNqVeXoUqi7/YxjbX87
// SIG // mjCBoAYLKoZIhvcNAQkQAgwxgZAwgY0wgYowgYcEFJsS
// SIG // BXrnKq/21jdytJ9qI28mSc2pMG8wX6RdMFsxCzAJBgNV
// SIG // BAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNh
// SIG // MTEwLwYDVQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGlu
// SIG // ZyBDQSAtIFNIQTI1NiAtIEcyAgwMp89dBwckrInnmjow
// SIG // DQYJKoZIhvcNAQEBBQAEggEAkbcBf3EkIfQxLicXJp/y
// SIG // 6HuBTswjszFa8nK2H7XMPgi7POC4QCf/byvqFM9cRwUR
// SIG // qk4igJjkxNg0khloxycjDJ/uYpsVzIHqHmYKShcTm/Yi
// SIG // WsSQJYtBQMiqJBTgYA0VbeL8Ds70fGLj1Lq5ItKL6ukd
// SIG // WSOYeZFg+A3OzSi8TM4w6XqqL1uBOY9QEEnzP59K75Pu
// SIG // I1IDxYOuHqHcBRR2no+xkon47lJztYljsrpkrp4yrVfN
// SIG // BBOM7x1wSIZDOZSx647PK6zJzQcnAZ3MMebrkSZZicgo
// SIG // tS/HBQYFDgd848O6inT6MbyNoo6vCUcbqBLgnzr1eBxM
// SIG // M0iGOaALQZPyUg==
// SIG // End signature block
