// Main app view.
(function($, UTILS) {

    $.widget('adsk.appview', {
        options: {
            title: 'Book Title',

            minNavigationWidth: 320,

            ui: {
                tabs: {
                    order: ['C', 'I', 'S', 'F'],
                    tocEnabled: true,
                    indexEnabled: true,
                    favoritesEnabled: true,
                    searchEnabled: true
                },

                tools: {
                    order: ['home', 'previous', 'parent', 'following', 'separator',
                            'favorites', 'share', 'separator', 'print'],
                    home: true,
                    previous: true,
                    parent: true,
                    following: true,
                    favorites: true,
                    share: true,
                    print: true
                },

                responsive: true,

				textDirection: 'rtl'
            },

            features: {
                tabsbarParent: 'banner',
                banner: true,
                header: true,
                navigation: true,
                splitter: true,
                breadcrums: true,
                splitter: true
            },

            labels: {
                tabs: {
                }
            }
        },

        /* Private members. */

        _$banner:       undefined,
        _$main:         undefined,
        _$breadcrumbs:  undefined,
        _$splitter:     undefined,
        _$navigation:   undefined,
        _$topic:        undefined,

        _splitterTracking:                  false,
        _splitterMoved:                     false,
        _splitterOriginalX:                 undefined,
        _navigationOriginalWidth:           undefined,
        _topicOriginalWidth:                undefined,
        _navigationOpenedWidth:             undefined,
        _topicOriginalMargin: 	            undefined,
		_dirSign: 							1,
		_dirTopicMarginProperty: 			'margin-left',

        _create: function() {
            console.log('appview: _create');

            var self = this;

            // Show "print" button only when functionality supported.
            if (!window.print) {
                this.options.ui.tools.print = false;
            }

			// Determine text direction.
			if ($('html').attr('dir') == 'rtl') {
				this.options.ui.textDirection = 'rtl';
				this._dirSign = -1;
				this._dirTopicMarginProperty = 'margin-right';
			} else {
				this.options.ui.textDirection = 'ltr';
				this._dirSign = 1;
				this._dirTopicMarginProperty = 'margin-left';
			}

            // Create layout.
            this._render();

            // Initial resize.
            this._resizeHandler();
            this._resizeTabButtons();

            // Perform transition.
            this._$main.transition();

            // Add event handlers.
            this._addEventHandlers(self);

            // Initial app resize.
            this._resizeHandler();

            // Show UI.
            this.element.removeClass('hidden');
        },

        _addEventHandlers: function(self) {
            // Add app resize handler.
            $(window).bind('resize', function() {
                self._resizeHandler();
            });

            self._addViewEventHandlers(self);
        },

        _addViewEventHandlers: function(self) {
            console.log('appview: _addEventHandlers');

            // IE7 hover/active fallback.
            if ($('html').hasClass('ie7') || $('html').hasClass('ie8')) {
                $('a')
                .mousedown(function() {
                    $(this).addClass('active');
                })
                .mouseup(function() {
                    $(this).removeClass('active');
                })
                .mouseenter(function() {
                    $(this).addClass('hover');
                })
                .mouseleave(function() {
                    $(this).removeClass('hover');
                });
            }

            // Add toolbar buttons tab bar links handlers.
            self._on('.button', {
                click: "_handleToolbarButtonClick"
            });

            // Add tab bar links handlers.
            self._on('.tab-button', {
                click: "_handleTabButtonClick"
            });

            // Splitter event handlers.
            self._on(this._$splitter, {
                mousedown: "_splitterMouseDownHandler",
                mouseup: function() {
                    // Toggle pane.
                    if (this._splitterMoved == false) {
                        this._toggleNavigationPane();
                    }
                }
            });

            self._on({
                mousemove: "_splitterMouseMoveHandler",
                mouseup: function() {
                    this._splitterTracking = false;
                    this.element.removeClass('no-user-select');
                }
            });
        },

        _toggleNavigationPane: function() {
            var navigationPaneWidth = this._$navigation.width();

            if (navigationPaneWidth > 0) {
                // Remember current width.
                this._navigationOpenedWidth = navigationPaneWidth;

                // Collapse navigation pane.
                var newTopicMargin = this._topicOriginalMargin - navigationPaneWidth;

                // Resize panes.
                this._resizeSplitPanes(0, 0, 0, 0, newTopicMargin);

                // Hide navigation pane.
                this._$navigation.css('visibility', 'hidden');
            } else {
                // Show navigation pane.
                this._$navigation.css('visibility', 'visible');

                // Open navigation pane.
                var newNavigationPaneWidth = this._navigationOpenedWidth,
                    newTopicMargin = this._topicOriginalMargin + newNavigationPaneWidth;

                this._resizeSplitPanes(0, 0, newNavigationPaneWidth, 0, newTopicMargin);
            }
        },

        _splitterMouseDownHandler: function(event) {
            this._splitterMoved = false;

            // Set up splitter.
            this._splitterOriginalX = event.clientX;
            this._navigationOriginalWidth = this._$navigation.width();
            this._topicOriginalWidth = this._$topic.width();
            this._topicOriginalMargin = parseInt(this._$topic.css(this._dirTopicMarginProperty), 10);

            // Prevent user select.
            this.element.addClass('no-user-select');

            // Acgtivate splitter tracking if pane is not collapsed.
            if (this._navigationOriginalWidth > 0) {
                this._splitterTracking = true;
            }
        },

        _splitterMouseMoveHandler: function(event) {
            this._splitterMoved = true;

            if (this._splitterTracking) {
                // Track splitter.
                var topicWidth = this._$topic.width(),
                    navigationWidth = this._$navigation.width(),
                    deltaX = this._dirSign * (event.clientX - this._splitterOriginalX),
                    minNavigationWidth = this.options.minNavigationWidth;

                if ((deltaX < 0 && navigationWidth > this.options.minNavigationWidth) ||
                    (deltaX > 0 && topicWidth > 0)) {
                    // Resize panes.
                    var newNavigationWidth = this._navigationOriginalWidth + deltaX,
                        newTopicMargin = this._topicOriginalMargin + deltaX,
                        newTopicWidth = this._topicOriginalWidth - deltaX;

                    this._resizeSplitPanes(minNavigationWidth, 0, newNavigationWidth, newTopicWidth, newTopicMargin);

                }
            }
        },

        _resizeSplitPanes: function(minNavigationWidth, minTopicWidth, newNavigationWidth, newTopicWidth, newTopicMargin) {
            if (newNavigationWidth < minNavigationWidth) {
                var delta = minNavigationWidth - newNavigationWidth;
                newNavigationWidth += delta;
                newTopicMargin += delta;
            } else if (newTopicWidth < minTopicWidth) {
                var delta = -(minTopicWidth - newTopicWidth);
                newNavigationWidth += delta;
                newTopicMargin += delta;
            }

            this._$navigation.width(newNavigationWidth);
            this._$topic.css(this._dirTopicMarginProperty, newTopicMargin + 'px');

            this._resizeTabButtons();
        },

        _resizeTabButtons: function() {
            // Stretch tab buttons for non-responsive UI.
            var tabButtons = this._$navigation.find('.tab-button'),
                tabButtonsLen = tabButtons.length;

            // Resize tab buttons if all of them are within navigation pane (non-responsive UI).
            if (tabButtonsLen) {
                var navWidth = this._$navigation.width();
                tabButtons.css('width', Math.floor(navWidth / tabButtons.length)  + 'px');

                // Show tab icons navigation pane is wide enough.
                if (navWidth >= 600) {
                    this._$navigation.addClass('wide');
                } else {
                    this._$navigation.removeClass('wide');
                }
            }
        },

        _render: function() {
            console.log('appview: _render');

            // Create app view markup.
            this.element
            .html(this._renderBanner() +
                  '<div id="main">' +
                    this._renderNavigation() +
                    this._renderSplitter() +
                    '<div id="topic"></div>' +
                  '</div>' +
                  this._renderBreadcrumbs() +
                  '<div id="share" class="hidden"></div>');

            var e = this.element;
            this._$banner       = e.find('#banner');
            this._$main         = e.find('#main');
            this._$splitter     = e.find('#splitter');
            this._$navigation   = e.find('#navigation');
            this._$tabsbar   = this._$navigation.find('.tabsbar');
            this._$topic        = e.find('#topic');
            this._$breadcrumbs  = e.find('#breadcrumbs');
        },

        _renderBanner: function() {
            console.log('appview: _renderBanner');

            var html = ''

            if (this.options.features.banner) {
                html +=  '<div id="banner">' +
                            this._renderHeader() +
                            this._renderToolbar();

                if (this.options.features.tabsbarParent == 'banner') {
                    html += this._renderTabsBar();
                }

                html += '</div>';
            }

            return html;
        },

        _renderToolbar: function() {
            var html = '<div class="toolbar"><div class="wrapper">',
                tabs = this.options.ui.tabs,
                tools = this.options.ui.tools;

            $.each(tools.order, function(index, tool) {
                switch(tool) {
                    case 'separator':
                        html += '<span class="separator"></span>';
                        break;
                    case 'home':
                        if (tools.home) {
                            html += '<a href="#" id="main-home" class="button icon home"></a>';
                        }
                        break;
                    case 'previous':
                        if (tools.previous) {
                            html += '<a href="#" id="main-previous" class="button icon previous"></a>';
                        }
                        break;
                    case 'parent':
                        if (tools.parent) {
                            html += '<a href="#" id="main-parent" class="button icon parent"></a>';
                        }
                        break;
                    case 'following':
                        if (tools.following) {
                            html += '<a href="#" id="main-following" class="button icon following"></a>';
                        }
                        break;
                    case 'favorites':
                        if (tabs.favoritesEnabled && tools.favorites) {
                            html += '<a href="#" id="main-favorites" class="button icon favorites"></a>';
                        }
                        break;
                    case 'share':
                        if (tools.share) {
                            html += '<a href="#" id="main-share" class="button icon share"></a>';
                        }
                        break;
                    case 'print':
                        if (tools.print) {
                            html += '<a href="#" id="main-print" class="button icon print"></a>';
                        }
                        break;
                }
            });

            html += '</div></div>';

            return html;
        },

        _renderTabsBar: function() {
            var that = this,
                html = '<div class="tabsbar"><div class="wrapper">',
                tabs = this.options.ui.tabs,
                caps = this.options.labels.tabs,
                pos = 0;

            $.each(tabs.order, function(index, tab) {
                switch(tab) {
                    case 'C':
                        if (tabs.tocEnabled) {
                            html += that._renderTabsBarButton('tab-button-toc', caps.contents, pos);
                            ++pos;
                        }
                        break;
                    case 'I':
                        if (tabs.indexEnabled) {
                            html += that._renderTabsBarButton('tab-button-index', caps.index, pos);
                            ++pos;
                        }
                        break;
                    case 'S':
                        if (tabs.searchEnabled) {
                            html += that._renderTabsBarButton('tab-button-search', caps.search, pos);
                            ++pos;
                        }
                        break;
                    case 'F':
                        if (tabs.favoritesEnabled) {
                            html += that._renderTabsBarButton('tab-button-favorites', caps.favorites, pos);
                            ++pos;
                        }
                        break;
                }
            });

            html += '</div></div>';

            return html;
        },

        _renderTabsBarButton: function (id, label, pos) {
            var html = '',
                classes = 'tab-button';

            if (pos == 0) {
                classes += ' opened';
            }

            html += '<a id="' + id + '" class="' + classes + '" href="#">' +
                        '<span>' + label + '</span>' +
                    '</a>';

            return html;
        },

        _renderNavigation: function() {
            console.log('appview: _renderNavigation');

            var html = '',
                position = 0,
                that = this,
                features = this.options.features,
                tabs = this.options.ui.tabs;

            if (features.navigation) {
                html += '<div id="navigation">';

                if (this.options.features.tabsbarParent == 'navigation') {
                    html += this._renderTabsBar();
                }

                $.each(tabs.order, function(index, tab) {
                    switch(tab) {
                        case 'C':
                            if (tabs.tocEnabled) {
                                html += that._renderNavigationTab('tab-toc', position);
                                ++position;
                            }
                            break;
                        case 'I':
                            if (tabs.indexEnabled) {
                                html += that._renderNavigationTab('tab-index', position);
                                ++position;
                            }
                            break;
                        case 'S':
                            if (tabs.searchEnabled) {
                                html += that._renderNavigationTab('tab-search', position);
                                ++position;
                            }
                            break;
                        case 'F':
                            if (tabs.favoritesEnabled) {
                                html += that._renderNavigationTab('tab-favorites', position);
                                ++position;
                            }
                            break;
                    }
                });

                html += '</div>';

            } else {
                this.element.addClass('no-navigation');
            }
            return html;
        },

        _renderNavigationTab: function(id, position) {
            var html = '<div id="' + id + '" class = "tab';

            // Only first tab should be visible.
            if (position > 0) {
                html += ' hidden'
            }

            html += '"></div>';

            return html;
        },

        _renderSplitter: function() {
            console.log('appview: _renderSplitter');

            if (this.options.features.navigation) {
                return '<div id="splitter"></div>';
            } else {
                return '';
            }
        },

        _renderBreadcrumbs: function() {
            console.log('appview: _renderBreadcrumbs');

            if (this.options.features.breadcrumbs) {
                return '<div id="breadcrumbs"></div>';
            } else {
                return '';
            }
        },

        _renderHeader: function() {
            console.log('appview: _renderHeader');

            if (this.options.features.header) {
                return  '<div class="header">' +
                            '<div class="wrapper">' +
                                '<a class="title" href="#">' +
                                    '<img class="product-title" src="images/product-title.png"/>' +
                                    '<span>' + this.options.title + '</span>' +
                                '</a>' +
                            '</div>' +
                        '</div>';
            } else {
                return '';
            }
        },

        _handleToolbarButtonClick: function(event) {
            console.log('appview: _handleToolbarButtonClick');

            event.preventDefault();
            event.stopPropagation();

            var $target = $(event.target);

            if (!$target.hasClass('disabled') && $target.attr('id').indexOf('main') != -1) {
                this._trigger('buttonclick', event, $target.attr('id'));
            }
        },

        _handleTabButtonClick: function(event) {
            console.log('appview: _handleTabButtonClick');

            event.preventDefault();
            event.stopPropagation();

            var $target = $(event.currentTarget);

            if (!$target.hasClass('disabled')) {
                this._trigger('tabbuttonclick', event, $target.attr('id'));
            }
        },

        _showTabButtonById: function(id) {
            var tabButtons = this._$navigation.find('.tab-button'),
                targetButton = tabButtons.filter(id);

            if (tabButtons.length) {
                tabButtons.removeClass('opened');
                targetButton.addClass('opened');
            }
        },

        _getShareButtonOffset: function() {
            console.log('appview: _getShareButtonOffset');

        },

        _resizeHandler: function() {
            console.log('appview: _resizeHandler');

            var $window = $(window);

            // Resize according to a new app height.
            var height = $window.height() - this._$banner.height() - this._$breadcrumbs.height();
            this._$main.css('height', height + 'px');

            // Resize according to a new app width.
            if (this.options.ui.responsive) {
                var width = $window.width();
                if (width > 768) {
                    // Full viewport view.
                    $(document.body).removeClass('snap');
                } else {
                    // Snap view.
                    $(document.body).addClass('snap');
                }
            } else {
                this._$navigation.find('.tab').css('height', height - this._$tabsbar.height() + 'px');
            }
        },

        _constructBreadcrumbHtml: function(breadcrumb) {
            console.log('appview: _constructBreadcrumbHtml');

            return '<li>' +
                      '<a href="#!/url=' + breadcrumb.href + '">' +
                        breadcrumb.title +
                      '</a>' +
                   '</li>' +
                   '&rarr;';
        },

        /* Public members. */

        showPreloader: function() {
            console.log('appview: showPreloader');

            $(document.body).addClass('loading');
        },

        hidePreloader: function() {
            console.log('appview: hidePreloader');

            $(document.body).removeClass('loading');
        },

        initGoToPreviousButton: function(enabled) {
            console.log('appview: initGoToPreviousButton');

            if (enabled) {
                this._$banner.find('#main-previous').removeClass('disabled');
            } else {
                this._$banner.find('#main-previous').addClass('disabled');
            }
        },

        initGoToParentButton: function(enabled) {
            console.log('appview: initGoToParentButton');

            if (enabled) {
                this._$banner.find('#main-parent').removeClass('disabled');
            } else {
                this._$banner.find('#main-parent').addClass('disabled');
            }
        },

        initGoToNextButton: function(enabled) {
            console.log('appview: initGoToNextButton');

            if (enabled) {
                this._$banner.find('#main-following').removeClass('disabled');
            } else {
                this._$banner.find('#main-following').addClass('disabled');
            }
        },

        initFavoritesButton: function(enabled) {
            console.log('appview: initFavoritesButton');

            if (enabled) {
                this._$banner.find('#main-favorites').removeClass('disabled');
            } else {
                this._$banner.find('#main-favorites').addClass('disabled');
            }
        },

        getShareButtonOffset: function() {
            console.log('appview: getShareButtonOffset');

            var offset = this.element.find('#main-share').offset();

            return offset;
        },

        showTab: function(id) {
            console.log('appview: showTab');

            var tabSelector = '#tab-' + id,
                tabButtonSelector = '#tab-button-' + id,
                tabToShow = this._$navigation.find(tabSelector),
                tabs = this._$navigation.children('.tab');

            this._showTabButtonById(tabButtonSelector);

            if (tabToShow.length > 0 && tabToShow.hasClass('hidden')) {
                // Hide all tabs.
                tabs.addClass('hidden').attr('style', '');

                // Show selected tab.
                tabToShow.transition().removeClass('hidden');

                // Recalculate tab height.
                this._resizeHandler();

                // Go to current topic in TOC.
                if (tabToShow.attr('id') == 'tab-toc') {
                    //this.expandToc(tabToShow);
                    //this.scrollTocToSelectedTopic(tabToShow);
                }
            } else {
                console.log('appview: error: tab with id ' + id + ' not found.');
            }
        },

        // Flag if topic is in Favorites or not.
        flagFavorites: function(addedToFavorites) {
            console.log('mainview: flagFavorites, added = ', addedToFavorites);

            if (addedToFavorites) {
                this._$banner.find('#main-favorites').addClass('flagged');
            } else {
                this._$banner.find('#main-favorites').removeClass('flagged');
            }
        },

        updateBreadcrumbs: function(breadcrumbs, rootTopicData) {
            var len = breadcrumbs.length,
                i,
                html = '<ol class="breadcrumbs">';

                 + this._constructBreadcrumbHtml({
                    href: rootTopicData.attr.href,
                    title: rootTopicData.data || 'Home'
                });

            for (i = 0; i < len; ++i) {
                html = html + this._constructBreadcrumbHtml(breadcrumbs[i]);
            }

            html += '</ol>';

            this._$breadcrumbs.empty();

            $(html).appendTo(this._$breadcrumbs);
        }
    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIXrwYJKoZIhvcNAQcCoIIXoDCCF5wCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFHwhbstdk9iP
// SIG // KeijvLCSGoJrF81AoIIS3zCCA+4wggNXoAMCAQICEH6T
// SIG // 6/t8xk5Z6kuad9QG/DswDQYJKoZIhvcNAQEFBQAwgYsx
// SIG // CzAJBgNVBAYTAlpBMRUwEwYDVQQIEwxXZXN0ZXJuIENh
// SIG // cGUxFDASBgNVBAcTC0R1cmJhbnZpbGxlMQ8wDQYDVQQK
// SIG // EwZUaGF3dGUxHTAbBgNVBAsTFFRoYXd0ZSBDZXJ0aWZp
// SIG // Y2F0aW9uMR8wHQYDVQQDExZUaGF3dGUgVGltZXN0YW1w
// SIG // aW5nIENBMB4XDTEyMTIyMTAwMDAwMFoXDTIwMTIzMDIz
// SIG // NTk1OVowXjELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5
// SIG // bWFudGVjIENvcnBvcmF0aW9uMTAwLgYDVQQDEydTeW1h
// SIG // bnRlYyBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBIC0g
// SIG // RzIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
// SIG // AQCxrLNJVEuXHBIK2CV5kSJXKm/cuCbEQ3Nrwr8uUFr7
// SIG // FMJ2jkMBJUO0oeJF9Oi3e8N0zCLXtJQAAvdN7b+0t0Qk
// SIG // a81fRTvRRM5DEnMXgotptCvLmR6schsmTXEfsTHd+1Fh
// SIG // AlOmqvVJLAV4RaUvic7nmef+jOJXPz3GktxK+Hsz5HkK
// SIG // +/B1iEGc/8UDUZmq12yfk2mHZSmDhcJgFMTIyTsU2sCB
// SIG // 8B8NdN6SIqvK9/t0fCfm90obf6fDni2uiuqm5qonFn1h
// SIG // 95hxEbziUKFL5V365Q6nLJ+qZSDT2JboyHylTkhE/xni
// SIG // RAeSC9dohIBdanhkRc1gRn5UwRN8xXnxycFxAgMBAAGj
// SIG // gfowgfcwHQYDVR0OBBYEFF+a9W5czMx0mtTdfe8/2+xM
// SIG // gC7dMDIGCCsGAQUFBwEBBCYwJDAiBggrBgEFBQcwAYYW
// SIG // aHR0cDovL29jc3AudGhhd3RlLmNvbTASBgNVHRMBAf8E
// SIG // CDAGAQH/AgEAMD8GA1UdHwQ4MDYwNKAyoDCGLmh0dHA6
// SIG // Ly9jcmwudGhhd3RlLmNvbS9UaGF3dGVUaW1lc3RhbXBp
// SIG // bmdDQS5jcmwwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDgYD
// SIG // VR0PAQH/BAQDAgEGMCgGA1UdEQQhMB+kHTAbMRkwFwYD
// SIG // VQQDExBUaW1lU3RhbXAtMjA0OC0xMA0GCSqGSIb3DQEB
// SIG // BQUAA4GBAAMJm495739ZMKrvaLX64wkdu0+CBl03X6ZS
// SIG // nxaN6hySCURu9W3rWHww6PlpjSNzCxJvR6muORH4KrGb
// SIG // sBrDjutZlgCtzgxNstAxpghcKnr84nodV0yoZRjpeUBi
// SIG // JZZux8c3aoMhCI5B6t3ZVz8dd0mHKhYGXqY4aiISo1EZ
// SIG // g362MIIEozCCA4ugAwIBAgIQDs/0OMj+vzVuBNhqmBsa
// SIG // UDANBgkqhkiG9w0BAQUFADBeMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2Vy
// SIG // dmljZXMgQ0EgLSBHMjAeFw0xMjEwMTgwMDAwMDBaFw0y
// SIG // MDEyMjkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjE0MDIGA1UE
// SIG // AxMrU3ltYW50ZWMgVGltZSBTdGFtcGluZyBTZXJ2aWNl
// SIG // cyBTaWduZXIgLSBHNDCCASIwDQYJKoZIhvcNAQEBBQAD
// SIG // ggEPADCCAQoCggEBAKJjCzlEuLsjp0RJuw7/ofBhClOT
// SIG // sJjbrSwPSsVu/4Y8U1UPFc4EPyv9qZaW2b5heQtbyUyG
// SIG // duXgQ0sile7CK0PBn9hotI5AT+6FOLkRxSPyZFjwFTJv
// SIG // TlehroikAtcqHs1L4d1j1ReJMluwXplaqJ0oUA4X7pbb
// SIG // YTtFUR3PElYLkkf8q672Zj1HrHBy55LnX80QucSDZJQZ
// SIG // vSWA4ejSIqXQugJ6oXeTW2XD7hd0vEGGKtwITIySjJEt
// SIG // nndEH2jWqHR32w5bMotWizO92WPISZ06xcXqMwvS8aMb
// SIG // 9Iu+2bNXizveBKd6IrIkri7HcMW+ToMmCPsLvalPmQjh
// SIG // EChyqs0CAwEAAaOCAVcwggFTMAwGA1UdEwEB/wQCMAAw
// SIG // FgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/
// SIG // BAQDAgeAMHMGCCsGAQUFBwEBBGcwZTAqBggrBgEFBQcw
// SIG // AYYeaHR0cDovL3RzLW9jc3Aud3Muc3ltYW50ZWMuY29t
// SIG // MDcGCCsGAQUFBzAChitodHRwOi8vdHMtYWlhLndzLnN5
// SIG // bWFudGVjLmNvbS90c3MtY2EtZzIuY2VyMDwGA1UdHwQ1
// SIG // MDMwMaAvoC2GK2h0dHA6Ly90cy1jcmwud3Muc3ltYW50
// SIG // ZWMuY29tL3Rzcy1jYS1nMi5jcmwwKAYDVR0RBCEwH6Qd
// SIG // MBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTIwHQYD
// SIG // VR0OBBYEFEbGaaMOShQe1UzaUmMXP142vA3mMB8GA1Ud
// SIG // IwQYMBaAFF+a9W5czMx0mtTdfe8/2+xMgC7dMA0GCSqG
// SIG // SIb3DQEBBQUAA4IBAQB4O7SRKgBM8I9iMDd4o4QnB28Y
// SIG // st4l3KDUlAOqhk4ln5pAAxzdzuN5yyFoBtq2MrRtv/Qs
// SIG // JmMz5ElkbQ3mw2cO9wWkNWx8iRbG6bLfsundIMZxD82V
// SIG // dNy2XN69Nx9DeOZ4tc0oBCCjqvFLxIgpkQ6A0RH83Vx2
// SIG // bk9eDkVGQW4NsOo4mrE62glxEPwcebSAe6xp9P2ctgwW
// SIG // K/F/Wwk9m1viFsoTgW0ALjgNqCmPLOGy9FqpAa8VnCwv
// SIG // SRvbIrvD/niUUcOGsYKIXfA9tFGheTMrLnu53CAJE3Hr
// SIG // ahlbz+ilMFcsiUk/uc9/yb8+ImhjU5q9aXSsxR08f5Lg
// SIG // w7wc2AR1MIIE5TCCA82gAwIBAgIQGGKWU9W9DKB4gLDR
// SIG // /Nm99TANBgkqhkiG9w0BAQsFADB/MQswCQYDVQQGEwJV
// SIG // UzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24x
// SIG // HzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsx
// SIG // MDAuBgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2
// SIG // IENvZGUgU2lnbmluZyBDQTAeFw0xNjAxMjgwMDAwMDBa
// SIG // Fw0xODA0MTAyMzU5NTlaMH0xCzAJBgNVBAYTAkNIMRIw
// SIG // EAYDVQQIEwlOZXVjaGF0ZWwxEjAQBgNVBAcTCU5ldWNo
// SIG // YXRlbDEiMCAGA1UEChQZQXV0b2Rlc2sgRGV2ZWxvcG1l
// SIG // bnQgU2FybDEiMCAGA1UEAxQZQXV0b2Rlc2sgRGV2ZWxv
// SIG // cG1lbnQgU2FybDCCASIwDQYJKoZIhvcNAQEBBQADggEP
// SIG // ADCCAQoCggEBALOBje/8PFtBHV0+YQz+EJZzXU+JAy6j
// SIG // OhMiK0n4pWQrgLdfLmoNZUvUL5AVuWoHkzK4f6WLWEjQ
// SIG // eGR4SEZcvDCJjol0n6vdsYAfRurwhekestab635tKUEy
// SIG // JRNKXhC4zuDSgryc+2zUsZa08HSHZaTsFGZzEuQAdxWf
// SIG // iyvD6G5oTu72nj3kDWWn3vqh+zyGkOIhtsIDKZyHpg71
// SIG // 7hPiqEe96bX5LpZgkSOJdAdwpl40NEUFfpNdelT2sOCh
// SIG // MfG63Dg2PR6EjOK5zHgMZm09mPgZBfyNoBV03C1ySSUi
// SIG // UsAHM2ND4wnGfl/GbYEgIVQXWlspRbtpRHTaNavebLJY
// SIG // tWECAwEAAaOCAV0wggFZMAkGA1UdEwQCMAAwDgYDVR0P
// SIG // AQH/BAQDAgeAMCsGA1UdHwQkMCIwIKAeoByGGmh0dHA6
// SIG // Ly9zdi5zeW1jYi5jb20vc3YuY3JsMGEGA1UdIARaMFgw
// SIG // VgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8v
// SIG // ZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0
// SIG // dHBzOi8vZC5zeW1jYi5jb20vcnBhMBMGA1UdJQQMMAoG
// SIG // CCsGAQUFBwMDMFcGCCsGAQUFBwEBBEswSTAfBggrBgEF
// SIG // BQcwAYYTaHR0cDovL3N2LnN5bWNkLmNvbTAmBggrBgEF
// SIG // BQcwAoYaaHR0cDovL3N2LnN5bWNiLmNvbS9zdi5jcnQw
// SIG // HwYDVR0jBBgwFoAUljtT8Hkzl699g+8uK8zKt4YecmYw
// SIG // HQYDVR0OBBYEFHqJl6/oWFvaZKJVCptH8DQRl8JIMA0G
// SIG // CSqGSIb3DQEBCwUAA4IBAQAXhA6ZwymxqqV+k4uKThLd
// SIG // PgPsvZBRcCJJZi3eYMl7HjIDbxSdx03ueVc/Sx3n9OUK
// SIG // U+DupPpLFUnKUJa/X8HPWhcvJftXom9NvGWbYPJM7CBc
// SIG // eTKuXgP5ffKwf1r9pJhok0pEhSA8b6N97pmKQwJDE6XI
// SIG // Z4M5FYw9YELKPRCHhinZCfPHLo+0GxaH+W4AbGd1N2BZ
// SIG // dX2egsgNTGBnC6xGxGMC8ckaj5wyIm4LgklE0pKbxuG9
// SIG // 075426gJ6ASAtCevNViBj9hH3TcNy4aU0ruH6By5gZaR
// SIG // 5lc7ZdaTuBWz0+4wGIK1IiviMSJv19v+h0vpEyn1b8sE
// SIG // 6L7D8JD4szHnMIIFWTCCBEGgAwIBAgIQPXjX+XZJYLJh
// SIG // ffTwHsqGKjANBgkqhkiG9w0BAQsFADCByjELMAkGA1UE
// SIG // BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8w
// SIG // HQYDVQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTow
// SIG // OAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAt
// SIG // IEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQD
// SIG // EzxWZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5
// SIG // IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwHhcN
// SIG // MTMxMjEwMDAwMDAwWhcNMjMxMjA5MjM1OTU5WjB/MQsw
// SIG // CQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29y
// SIG // cG9yYXRpb24xHzAdBgNVBAsTFlN5bWFudGVjIFRydXN0
// SIG // IE5ldHdvcmsxMDAuBgNVBAMTJ1N5bWFudGVjIENsYXNz
// SIG // IDMgU0hBMjU2IENvZGUgU2lnbmluZyBDQTCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAJeDHgAWryyx
// SIG // 0gjE12iTUWAecfbiR7TbWE0jYmq0v1obUfejDRh3aLvY
// SIG // NqsvIVDanvPnXydOC8KXyAlwk6naXA1OpA2RoLTsFM6R
// SIG // clQuzqPbROlSGz9BPMpK5KrA6DmrU8wh0MzPf5vmwsxY
// SIG // aoIV7j02zxzFlwckjvF7vjEtPW7ctZlCn0thlV8ccO4X
// SIG // fduL5WGJeMdoG68ReBqYrsRVR1PZszLWoQ5GQMWXkorR
// SIG // U6eZW4U1V9Pqk2JhIArHMHckEU1ig7a6e2iCMe5lyt/5
// SIG // 1Y2yNdyMK29qclxghJzyDJRewFZSAEjM0/ilfd4v1xPk
// SIG // OKiE1Ua4E4bCG53qWjjdm9sCAwEAAaOCAYMwggF/MC8G
// SIG // CCsGAQUFBwEBBCMwITAfBggrBgEFBQcwAYYTaHR0cDov
// SIG // L3MyLnN5bWNiLmNvbTASBgNVHRMBAf8ECDAGAQH/AgEA
// SIG // MGwGA1UdIARlMGMwYQYLYIZIAYb4RQEHFwMwUjAmBggr
// SIG // BgEFBQcCARYaaHR0cDovL3d3dy5zeW1hdXRoLmNvbS9j
// SIG // cHMwKAYIKwYBBQUHAgIwHBoaaHR0cDovL3d3dy5zeW1h
// SIG // dXRoLmNvbS9ycGEwMAYDVR0fBCkwJzAloCOgIYYfaHR0
// SIG // cDovL3MxLnN5bWNiLmNvbS9wY2EzLWc1LmNybDAdBgNV
// SIG // HSUEFjAUBggrBgEFBQcDAgYIKwYBBQUHAwMwDgYDVR0P
// SIG // AQH/BAQDAgEGMCkGA1UdEQQiMCCkHjAcMRowGAYDVQQD
// SIG // ExFTeW1hbnRlY1BLSS0xLTU2NzAdBgNVHQ4EFgQUljtT
// SIG // 8Hkzl699g+8uK8zKt4YecmYwHwYDVR0jBBgwFoAUf9Nl
// SIG // p8Ld7LvwMAnzQzn6Aq8zMTMwDQYJKoZIhvcNAQELBQAD
// SIG // ggEBABOFGh5pqTf3oL2kr34dYVP+nYxeDKZ1HngXI939
// SIG // 7BoDVTn7cZXHZVqnjjDSRFph23Bv2iEFwi5zuknx0ZP+
// SIG // XcnNXgPgiZ4/dB7X9ziLqdbPuzUvM1ioklbRyE07guZ5
// SIG // hBb8KLCxR/Mdoj7uh9mmf6RWpT+thC4p3ny8qKqjPQQB
// SIG // 6rqTog5QIikXTIfkOhFf1qQliZsFay+0yQFMJ3sLrBkF
// SIG // IqBgFT/ayftNTI/7cmd3/SeUx7o1DohJ/o39KK9KEr0N
// SIG // s5cF3kQMFfo2KwPcwVAB8aERXRTl4r0nS1S+K4ReD6bD
// SIG // dAUK75fDiSKxH3fzvc1D1PFMqT+1i4SvZPLQFCExggQ8
// SIG // MIIEOAIBATCBkzB/MQswCQYDVQQGEwJVUzEdMBsGA1UE
// SIG // ChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsT
// SIG // FlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMDAuBgNVBAMT
// SIG // J1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2ln
// SIG // bmluZyBDQQIQGGKWU9W9DKB4gLDR/Nm99TAJBgUrDgMC
// SIG // GgUAoHAwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcN
// SIG // AQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
// SIG // MAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFG8W
// SIG // huu7B9D4zQbbs75TUUE3R2l6MA0GCSqGSIb3DQEBAQUA
// SIG // BIIBAAXBq01x9PLo1Ce8OJe/zlCMFyzBX1uhzvnsEQWf
// SIG // 4yle1NK8tRU867DHeERHK7kCE1ane4mQ1YWcSfMw1cb0
// SIG // k8VJ27EMouTwhA3wmBR/HamDOAgfX7velyDqCG/YBE/O
// SIG // xnOGd6DVuKoNRrKfoYvyFS4Bp8SwCWm9qQW5LUfMxsOP
// SIG // bnoaH+HgqA7oCMH4xuxjFXbLT/zNtGk1S/3xNZyEemv4
// SIG // sfPPNdjBGYB8v1PYK7HsU/Y2N2eD+k4J4iokxBCyZ842
// SIG // O5swlvIv1z92SGFGKPoTayERCwCox09GgrkEh2LMn/Df
// SIG // sW7SXUfWdK/USMsaKl3/juZ0DVC5tzIGuOmb3EuhggIL
// SIG // MIICBwYJKoZIhvcNAQkGMYIB+DCCAfQCAQEwcjBeMQsw
// SIG // CQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29y
// SIG // cG9yYXRpb24xMDAuBgNVBAMTJ1N5bWFudGVjIFRpbWUg
// SIG // U3RhbXBpbmcgU2VydmljZXMgQ0EgLSBHMgIQDs/0OMj+
// SIG // vzVuBNhqmBsaUDAJBgUrDgMCGgUAoF0wGAYJKoZIhvcN
// SIG // AQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUxDxcN
// SIG // MTgwMjEyMDc0MTU3WjAjBgkqhkiG9w0BCQQxFgQUXP0F
// SIG // KzCTiHci0BAS4MGoT7MvBqAwDQYJKoZIhvcNAQEBBQAE
// SIG // ggEAoE+6SqK3bpJxlXdc06bhk1TkzX89tRs+zLeIAvum
// SIG // isrYzGL5OfXCOTJkPli5OsT98OjBJsMsxx60o4bOWZGp
// SIG // 07TD+souRMaOT+K0RTS6SJuhBj6fPDo7XAY+ReqDX1Ko
// SIG // rBbG38AGsC69ItdUwtKyaePJoD+rDqCwD4covdCKys8F
// SIG // caDIGXBcbvIoQxJQTlG+gdKwxctQHr0VXVB1VtaUzgKc
// SIG // rswet+9nSjSa6QhCJ4yHSL/d0oOzSpyRXmX0xrVcTHHn
// SIG // ynasg7hNNwE3O2gDFcScEi/+yzIBMHQbnyiSYCmmBRy5
// SIG // aINojRrZh/d0JBwnkX1KGaxZJDt+iSUGroTggw==
// SIG // End signature block
