//////////////////////////////////////////////////////////////////////////////
//  Copyright 2016 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.widget - Defines widgets objects
 */

var ntp = ntp || {};
ntp.widget = ntp.widget || {};

var showTooltip = function () {
    $(this).attr('title', $(this).attr('data-title'));
}

var hideTooltip = function () {
    $(this).attr('title', '');
}

/**
 * Get Start widget
 */
ntp.widget.getStarted = {
    options : {
        title : 'column_get_started_title'
    },

    _create : function() {
        // title
        ntp.widget.showWidget($('<div>'), 'Title').appendTo(this.element);

        // get started box
        ntp.widget.showWidget($('<div>'), 'GetStartedBox').appendTo(this.element);

        // template bar
        ntp.widget.showWidget($('<div>'), 'Templatebar').appendTo(this.element);

        // command groups
        ntp.widget.showWidget($('<div>'), 'CommandsGroup').appendTo(this.element);
    }
};

/**
 * Title Widget
 */

ntp.widget.title = {
    options : {
        title : 'column_get_started_title'
    },

    _create : function() {
        $('<h1>').attr('data-text-id', this.options.title).appendTo(this.element);
    }
};

/**
 * Tile widget (Get Started Box)
 */
ntp.widget.tile = {
    options : {
        caption : '',
        background : '',
        action : '',
        startDrawing: 'enabled'
    },

    _create : function() {
        if (this.options.startDrawing === 'enabled') {
            var $getStartedBox = $('<div>').attr({ 'id' : 'get_started_box'}).addClass('shadow_box');
            var $span = $('<span>').attr({ 'id' : 'start_drawing_span' }).appendTo($getStartedBox);

            // Caption
            if ( typeof this.options.caption == 'string' && this.options.caption !== '') {
                ntp.utils.localize($span, this.options.caption);
            } else {
                ntp.utils.localize($span, 'get_started_box_background');
            }

            // Background
            if ( typeof this.options.background === 'string' && this.options.background !== '') {
                // Mark $getStartedBox as custom
                $getStartedBox.addClass('custom');

                var backgroundUrl = "url('" + ntp.utils.resolvePath(this.options.background) + "')";

                ntp.utils.addRule('#get_started_box.custom {' + '   background-image: ' + backgroundUrl + ',' + '                      -webkit-linear-gradient(top, rgba(255,255,255,.1) 0%, rgba(240,240,240,.1) 30%, rgba(193,193,193,1) 100%);' + '}');
                ntp.utils.addRule('#get_started_box.custom:hover {' + '   background-image: ' + backgroundUrl + ',' + '                     -webkit-linear-gradient(top, rgba(255,255,255,0),rgba(0,156,248,.2));' + '}');
                ntp.utils.addRule('#get_started_box.custom:active {' + '   background-image: ' + backgroundUrl + ',' + '                     -webkit-linear-gradient(top, rgba(255,255,255,0),rgba(255,255,255,.2));' + '}');
            }

            // Action
            if (this.options.action === '')
                $getStartedBox.attr('onclick', 'ntp.acad.createDrawingFile()');
            else if ( typeof this.options.action === 'string')
                $getStartedBox.attr('onclick', '').off('click').click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', this.options.action));

            this.element.append($getStartedBox);
        } 
    }
};

/**
 * External widget which can load any widget which is defined in some file like *.html etc.
 *
 */
ntp.widget.external = {
    options : {
        url : ''    // required
    },
    _create : function() {
        var that = this.element;
        if (this.options.url !== '') {
            $.ajax({
                url : this.options.url,
                success : function(data) {
                    that.html(data);
                },
                async : false
            });
        }
    }
};

/**
 * template bar widget
 */
ntp.widget.templatebar = (function() {
    var $templatePanel = null;
    var $templateFooter = null;
    var toggling = false;
    var $listItemForlastUsedTemplate = null;

    // Defines the event handler that closes the panel when user click anywhere outside the panel
    var handleClickedOutsideTemplatePanel = function() {
        // Check if the mouse if hovering anywhere outside the panel
        if ((!$templatePanel.is(':hover') || $('#create_new_sheet_set').is(':hover')) && !toggling) {
            $templatePanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).off('click', handleClickedOutsideTemplatePanel);
        }
        toggling = false;
    };

    // Adjust the height of template panel
    var updateFloatingTemplatePanelHeight = function() {
        // Calculate the maximum height of the panel
        var $templateList = $('#template_list');
        var $templateBar = $('#template_bar');
        var $tabCreate = $('#tab_create');
        var templatePanelHeight = $tabCreate.offset().top + $tabCreate.height() - $templateBar.offset().top - $templateBar.outerHeight() - 24;
        var templateFooterHeight = 0;
        if ($templateFooter !== null) {
            templateFooterHeight = $templateFooter.outerHeight(true);
            if ($templatePanel.css('display') === 'none') {
                $templatePanel.css({ 'display' : 'block', 'height' : 0 });
                templateFooterHeight = $templateFooter.outerHeight(true);
                $templatePanel.css({ 'display' : 'none', 'height' : 'auto'});
            }
        }
        $templateList.css('max-height', templatePanelHeight - templateFooterHeight);
    };

    var processMenuActions = function(menuActions, parent) {
        if (!( menuActions instanceof Array)) {
            return;
        }

        // the id selector has some problem right now, let's use this way
        var $templateFooterGroups = parent.children().eq(1).children(1).eq(2).children().eq(0);

        $.each(menuActions, function(i, menuAction) {
            if (!( typeof menuAction.category === 'string' && typeof menuAction.text === 'string' && typeof menuAction.action === 'string')) {
                return;
            }

            var $group = $templateFooterGroups.children('[data-group="' + menuAction.category + '"]');
            if ($group.size() == 0) {
                // Create group
                $group = $('<li>').attr('data-group', menuAction.category);
                var $groupCaption = $('<span>');
                $groupCaption.appendTo($group);

                $group.append($('<ol>'));
                $group.appendTo($templateFooterGroups);

                ntp.utils.localize($groupCaption, menuAction.category);
            }

            // Create menu item
            var $menuItem = $('<li>').addClass('button');
            $group.children('ol').append($menuItem);

            ntp.utils.localize($menuItem, menuAction.text);

            if ( typeof menuAction.icon === 'string') {
                $menuItem.css('background-image', 'url("' + ntp.utils.resolvePath(menuAction.icon) + '")');
            }

            $menuItem.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', menuAction.action));
        });
    };

    var addNoTemplateToList = function() {
        var $noTempalteList = $('<li>');
        var $ol = $('<ol>');
        $ol.append($('<hr>'));
        // imperial
        $('<li>').attr('id', 'template_imperial').addClass('dwg button').text(ntp.utils.localize('no_template_imperial')).click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'createDrawingFileWithNoTemplate', true)).appendTo($ol);
        // metric
        $('<li>').attr('id', 'template_metric').addClass('dwg button').text(ntp.utils.localize('no_template_metric')).click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'createDrawingFileWithNoTemplate', false)).appendTo($ol);
        $noTempalteList.append($ol).appendTo($('#template_list'));
    };
    
    var loadTemplates = function(categories) {
        var i, j;
        for ( i = 0; i < categories.length; i++) {
            var catName = categories[i].name;
            var catFullPath = categories[i].fullPath;

            var $templateList = $('<li>');
            var $categoryNameSpan = null;
            var $arrowSpan = null;
            var $rightArrowSVG = null;
            var $downArrowSVG = null;
            var $itemDiv = null;

            if (catName !== '.') {
                $rightArrowSVG = $('<svg><polygon points="1,8 9,12 1,16" style="fill:#999"/></svg>').hide();
                $downArrowSVG = $('<svg><polygon points="1,8 9,8 5,16" style="fill:#999"/></svg>');
                $rightArrowSVG.attr('width', '10px').attr('height', '20px');
                $downArrowSVG.attr('width', '10px').attr('height', '20px');
                $arrowSpan = $('<span>').append($downArrowSVG).append($rightArrowSVG).css('display', 'block').css('margin-right', '3px');

                // Title attribute is used to show the tooltip info.
                $categoryNameSpan = $('<span>').text(catName).attr('data-title', catFullPath).css({ 'display' : 'block', 'width' : '232px' });
                $categoryNameSpan.hover(showTooltip, hideTooltip);

                $itemDiv = $('<div>').append($arrowSpan).append($categoryNameSpan);
                $itemDiv.css({ 'display' : '-webkit-box', '-webkit-box-align' : 'center', '-webkit-box-orient' : 'horizontal' });

                $templateList.append($itemDiv);
            }

            var $ol = $('<ol>').appendTo($templateList);

            var eventData = { 'ol' : $ol, 'rightArrowSVG' : $rightArrowSVG, 'downArrowSVG' : $downArrowSVG };

            if ($categoryNameSpan !== null)
                $categoryNameSpan.click(eventData, ntp.widget.templatebar.handleCategoryClicked);
            if ($arrowSpan != null)
                $arrowSpan.click(eventData, ntp.widget.templatebar.handleCategoryClicked);

            var files = categories[i].files;

            for (j = 0; j < files.length; j++) {

                var templateToolTipText ='';

                if (typeof files[j].description !== 'undefined' && files[j].description.length > 0)
                    templateToolTipText = files[j].description + '\r\n\n';

                templateToolTipText += files[j].fullPath;

                var $item = $('<li>').addClass('dwg button').text(files[j].name).attr('data-title', templateToolTipText);
                $item.hover(showTooltip, hideTooltip);

                $item.data({
                fullPath : files[j].fullPath
                });

                $ol.append($item);
                $item.click(ntp.widget.templatebar.handleItemClicked);
            }
            $('#template_list').append($templateList);
            
            if ( i === 0 )
                addNoTemplateToList();
        }
    };

    return {
        options : {
            createSheetSet : 'enabled',
            customMenuActions : [],
            templateBar: ''
        },

        _create : function() {
            if (this.options.templateBar !== "disabled") {
                var $div = $('<div>');
                $div.css('position', 'relative');
                $div = ntp.widget.showWidgetWithOptions($div, 'External', {
                    url : 'js/widget_templates/templatebar.html'
                }).appendTo(this.element);

                if (this.options.createSheetSet !== 'enabled')
                    $div.children().eq(1).children(1).eq(2).children().eq(0).children().eq(0).remove();

                processMenuActions(this.options.customMenuActions, $div);

                this.beginRefresh();
            }
        },

        isEnabled : function() {
            return ntp.settings.templateBar !== 'disabled';
        },

        refresh : function(data) {
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            $('#template_list').empty();

            if ( typeof (data.templates) !== "undefined")
                loadTemplates(data.templates);

            // Toggle template empty placeholder.
            if ($('#template_list').children().length > 1) {
                $('#template_empty').addClass('hidden');
            } else {
                $('#template_empty').removeClass('hidden');
            }
        },

        beginRefresh : function() {
            if (!this.isEnabled()) {
                return null;
            }
            return ntp.acad.loadTemplateFilesAsJSON().done(ntp.utils.makeCallback(this, 'refreshWithJSON'));
        },
        
        hilightLastUsedTemplateListItem: function (data) {
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (!this.isEnabled() || typeof data === 'undefined' || typeof (data.templatename) === 'undefined')
                return;
            
            //Remove the previous hilight if any
            if ($listItemForlastUsedTemplate != null) {
                $listItemForlastUsedTemplate.css({ 'background': 'transparent', 'border-radius': '0' });
            }
            //Hilight the Item matching lastused template
            $('#template_list li.button').each(function () {
                if (typeof $(this).data('fullPath') !== 'undefined' && $(this).data('fullPath').toLocaleLowerCase() === data.templatename.toLocaleLowerCase()) {
                    $listItemForlastUsedTemplate = $(this);
                    $(this).css({ 'background': '-webkit-linear-gradient(top,#e5e5e5,#c8c8c8)', 'border-radius': '2px' });
                }
            });
        },

        refreshLastUsedTemplateWithJSON : function(lastTemplateFile){
            if (!this.isEnabled()) {
                return;
            }
            this.hilightLastUsedTemplateListItem($.parseJSON(lastTemplateFile));
        },

        refreshWithJSON : function(templatesJSON) {
            if (!this.isEnabled()) {
                return;
            }
            
            this.refresh($.parseJSON(templatesJSON));
            ntp.acad.getLastUsedTemplateAsJSON().done(ntp.utils.makeCallback(this, 'refreshLastUsedTemplateWithJSON'));
        },

        init : function() {
            if (ntp.settings.templateBar !== "disabled") {
                $templatePanel = $('#template_panel');
                var $footer = $('#template_footer');
                if ($footer.length > 0) {
                    $templateFooter = $footer;
                }
            }
        },

        update : function() {
            // Adjust the size of floating template panel
            if ($templatePanel !== null && $templatePanel.css('display') !== 'none') {
                updateFloatingTemplatePanelHeight();
            }
        },

        handleTemplateBarClicked : function(event) {
            if ($templatePanel.css('display') === 'none') {
                toggling = true;

                updateFloatingTemplatePanelHeight();

                // Show the template panel
                $templatePanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
                $(document).on('click', handleClickedOutsideTemplatePanel);
            }
        },

        // The browser button is only visible when there is not templates in the templates menu.
        handleTemplateBrowseButtonClicked : function(event) {
            ntp.acad.loadTemplateFilesFromSelectedFolder().done(function(templateFilesJSON) {
                ntp.widget.templatebar.refreshWithJSON(templateFilesJSON);
            });
        },

        handleItemClicked : function() {
            $templatePanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).off('click', handleClickedOutsideTemplatePanel);

            if(typeof $(this).data('fullPath') !== 'undefined')
                ntp.acad.createDrawingFileWithTemplate($(this).data('fullPath'));

            // Setting the title of the item to empty string to enforce turning off tooltip.
            $(this).attr('title', '');
        },

        handleCategoryClicked : function(event) {
            event.data.ol.children().toggle();
            event.data.downArrowSVG.toggle();
            event.data.rightArrowSVG.toggle();
        }
    };
})();

/**
 * the command group : openFile, openSheetSet etc.
 */
ntp.widget.commandsGroup = {
    options : {
        openFile : 'enabled',
        openSheetSet : 'enabled',
        exploreSamples : 'enabled',
        getOnlineTemplates : "",
        customLinks : []
    },

    _create : function() {
        var that = this;
        var $ul = $('<ul>').attr('id', "get_started_commands");
        // predefined commands
        if (this.options.openFile === 'enabled')
            ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_open', text : 'get_started_open_label', action : 'ntp.acad.showOpenFileDialog()', custom : false }).appendTo($ul);
        if (this.options.openSheetSet === 'enabled')
            ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_open_sheet_set', text : 'get_started_open_sheet_set_label', action : 'ntp.acad.showOpenSheetSetDialog()', custom : false }).appendTo($ul);
        if (this.options.getOnlineTemplates !== '' && this.options.getOnlineTemplates !== 'disabled') {
            var $getTemplates = ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_templates', text : 'get_started_templates_label', custom : false });
            ntp.deferred.localization.done(function() {
                $getTemplates.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'openUrlWithBrowser', ntp.utils.localize(that.options.getOnlineTemplates)));
            });
            $ul.append($getTemplates);
        }
        if (this.options.exploreSamples === 'enabled')
            ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_samples', text : 'get_started_samples_label', action : 'ntp.acad.showOpenLocalSampleFileDialog()', custom : false }).appendTo($ul);
        // custom links
        this.processLinks(this.options.customLinks, $ul);
        this.element.append($ul);
    },

    processLinks : function(links, parent) {
        if (!( links instanceof Array)) {
            return;
        }

        $.each(links, function(i, link) {
            if (!( typeof link.text === 'string' && typeof link.action === 'string')) {
                return;
            }

            var $link = $('<li>').addClass('button');

            ntp.utils.localize($link, link.text);

            if ( typeof link.icon === 'string') {
                $link.css('background-image', 'url("' + ntp.utils.resolvePath(link.icon) + '")');
            }

            $link.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', link.action));
            parent.append($link);
        });
    }
};

/**
 * widget for command, e.g. openFile, openSheetSet, getSamples etc.
 */
ntp.widget.command = {
    options : {
        id : '', // required
        text : '', // required
        class_ : '', // optional
        icon : '', // optional
        action : '', // optional
        content : '', // optional
        custom : true
    },

    _create : function() {
        var $item = $('<li>').attr({
            'id' : this.options.id,
            'data-text-id' : this.options.text
        }).addClass('button').text('');
        if (this.options.action !== '' && !this.options.custom)
            $item.attr('onclick', this.options.action);
        else
            $item.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', this.options.action));

        this.element.append($item);
    },

    _setOption : function(key, value) {
        switch( key ) {
            case "action":
                break;
        }
        $.Widget.prototype._setOption.apply(this, arguments);
        this._super("_setOption", key, value);
    },

    destroy : function() {
        $.Widget.prototype.destroy.call(this);
    }
};

/*
/* Define widget for Notification box that includes hardware acceleration, license information and live update.
*/
ntp.widget.notifications = (function() {
    var $notificationBox = null;
    var $updateBox = null;
    var $licenseBox = null;
    var $hwaccBox = null;
    var $containerBox = null;
    var $licenseImage = null;
    var $notificationSection = null;
    var $offlineHelpBox = null;
    var $notificationBadge = null;
    var $footerBadge = null;
    var notificationNum = 0;
    
    var getDayImage = function(day) {
        return 'images/day' + day.toString() + '.png';
    };

    return {
        //options
        options : {
            title : 'general_notifications',
            id : 'section_notifications',
            hwaccAction : 'javascript:ntp.acad.showGsConfigDialog();', //button click action.
            hwaccText : 'hardware_acceleration_message',
            hwaccBText : 'hardware_acceleration_title',
            updateText : 'general_update',
            trialText : 'trial_description_long',
            remainingText : 'trial_days_remaining',
            activeText : 'general_activate',
            activateAction : 'trial_days_remaining',
            updateAction: 'javascript:ntp.acad.showProductUpdateDialog();',
            Update : 'enabled',
            Hwacc : 'enabled',
            License : 'enabled',
            OfflineHelp : 'enabled',
            offlineHelpText : 'offline_help_message'
        },
  
        //the constructor
        _create : function() {
            this.element.attr({ 'id' : this.options.id }).addClass('hidden');
            $notificationSection = this.element;

            $('<h1>').attr({ 'id' : 'notification_title', 'data-text-id' : this.options.title }).appendTo(this.element);
            $notificationBadge = $('<div>').attr({'id' : 'notification_badge'}).html(notificationNum.toString()).addClass('badge hidden').appendTo(this.element);
            $footerBadge = $('#footer_create_badge');
            $containerBox = $('<div>').attr({ 'id' : 'notification_container'});
            $notificationBox = $('<div>').attr({ 'id' : 'notification_box'});
     
            this._initUpdateBox();
            this._initHwaccBox();
            this._initLicenseBox();
            this._initOfflineHelpBox();
            $notificationBox.appendTo($containerBox);
            $containerBox.appendTo(this.element);
        },
        
        _initHwaccBox : function() {
            if (this.options.Hwacc === 'enabled') {
                $hwaccBox = $('<div>').attr({ 'id' : "notification_hwacc" }).addClass('hidden shadow_box');

                $('<span>').attr({ 'data-text-id' : this.options.hwaccText }).addClass("description").appendTo($hwaccBox);
                $('<img>').attr({ 'alt' : "Hardware Acceleration", 'src' : "images/hardware_acc.png" }).appendTo($hwaccBox);
                $('<button>').attr({ 'id' : 'button_hardware_acceleration', 'data-text-id' : this.options.hwaccBText, 'onclick' : this.options.hwaccAction, 'tabindex' : '-1' }).addClass("blue").appendTo($hwaccBox);
                $hwaccBox.appendTo($notificationBox);

                //Set show/hide hardware acceleration notification based on acad information.
                this.beginGetHwaccStatus();
            }
        },
        
        _initUpdateBox : function() {
            if (this.options.Update === 'enabled') {
                $updateBox = $('<div>').attr({'id' : 'notification_update'}).addClass('hidden shadow_box');
                
                $('<span>').attr({'data-text-id' : 'update_description_critical'}).addClass("description").appendTo($updateBox);
                $('<img>').attr({ 'alt' : "Update", 'src' : "images/update.png" }).appendTo($updateBox);
                $('<button>').attr({'data-text-id' : this.options.updateText, 'id' : 'button_update', 'onclick' : this.options.updateAction, 'tabindex' : '-1'}).addClass("blue").appendTo($updateBox);
                
                $updateBox.appendTo($notificationBox);
                this.beginGetUpdateStatus();
            }
        },
        
        _initLicenseBox : function() {
            if (this.options.License === 'enabled') {
                $licenseBox = $('<div>').attr({'id' : 'notification_trial'}).addClass('hidden shadow_box');
                
                $('<span>').attr({'data-text-id' : this.options.trialText}).addClass("description").appendTo($licenseBox);
                $licenseImage = $('<img>').attr({ 'alt' : "Remaining Days", 'src' : "images/day30.png" });
                $licenseImage.appendTo($licenseBox);
                
                $licenseBox.appendTo($notificationBox);
                this.beginGetLicenseStatus();
            }
        },
        
        _initOfflineHelpBox : function() {
            if (this.options.OfflineHelp === 'enabled') {
                $offlineHelpBox = $('<div>').attr({'id' : 'offline_help'}).addClass('hidden shadow_box');
                
                $('<span>').attr({'data-text-id' : this.options.offlineHelpText}).addClass("description").appendTo($offlineHelpBox);
                $('<img>').attr({ 'alt' : "Offline Help", 'src' : "images/offline_help.png" }).appendTo($offlineHelpBox);
                
                $offlineHelpBox.appendTo($notificationBox);
            }
        },
        
        loadAlerts : function(url) {
            $.ajax({
                url : url,
                dataType : 'jsonp',
                jsonpCallback : 'alertCallback',
                crossDomain : true,
                timeout : 5000,
                success : function(data) {
                    ntp.acad.getViewedAlerts().done(function (obj) {
                        var viwedAlertData = $.parseJSON(obj);

                        if (typeof viwedAlertData.retValue !== 'undefined')
                            viwedAlertData = viwedAlertData.retValue; // hoist retValue if present

                        if (!(viwedAlertData.viewedAlerts instanceof Array))
                            viwedAlertData.viewedAlerts = []; // load all alerts

                        ntp.widget.notifications.getAlerts(data, viwedAlertData.viewedAlerts);
                    });
                },
                error : function() {
                }
            });
        },

        getAlerts: function (data, viewedAlerts) {

            if (data.alerts instanceof Array) {
                var alertsList = [];
                for (var i = 0; i < data.alerts.length; i++) {
                    var activeTime = new Date(data.alerts[i].active_time);
                    var id = Number(data.alerts[i].id);
                    var found = false;
                    for (var j = 0 ; j < viewedAlerts.length; j++) {
                        if (viewedAlerts[j].id === id) {
                            found = true; break;
                        }
                    }
                    if (found)
                        continue;
                    var expiryDays = data.alerts[i].expiry;
                    var today = new Date();
                    var signalerturl = false;
                    if (typeof data.alerts[i].signurl !== 'undefined')
                        signalerturl = data.alerts[i].signurl;

                    var expireDate = new Date(activeTime.getTime() + expiryDays * 24 * 60 * 60 * 1000);
                    if (activeTime < today && expireDate > today) {
                        var $alertBox = $('<div>').addClass('shadow_box');
                        var $description = $('<span>').addClass('description').text(data.alerts[i].description).appendTo($alertBox);
                        if (typeof data.alerts[i].image !== 'undefined' && data.alerts[i].image !== "") {
                            $('<img>').attr({ 'src': data.alerts[i].image }).appendTo($alertBox);
                        }
                        else {
                            $description.css('width', '100%');
                        }
                        $('<a>').text(data.alerts[i].link_text).on("click", { link: data.alerts[i].link, id: data.alerts[i].id, signurl: signalerturl, alertbox: $alertBox }, function (e) {

                            if (typeof e.data.link !== 'undefined' && e.data.link !== "" && e.data.id !== 'undefined' && e.data.id !== "") {
                                ntp.acad.openAlertLink(e.data.id, e.data.link, e.data.signurl);
                                // hide the clicked alert box
                                e.data.alertbox.addClass('hidden');
                            };
                        }).appendTo($alertBox);
                        // based on alert priority, it will added into different place.
                        if (data.alerts[i].priority == 1) {
                            $alertBox.addClass('high_priority');
                            if (i == 0) {
                                $alertBox.prependTo($notificationBox);
                            }
                            else {
                                $notificationBox.children().eq(i - 1).after($alertBox);
                            }
                        }
                        else if (data.alerts[i].priority == 3) {
                            $alertBox.addClass('low_priority');
                            $alertBox.appendTo($notificationBox);
                        }
                        else {
                            continue;
                        }
                        alertsList.push(data.alerts[i].id);
                    };
                };
                this.updateNotificationSection();
                if (alertsList.length > 0)
                    ntp.acad.getAlertViewState(alertsList).done(ntp.utils.makeCallback(this, 'refreshAlertsNumber'));
            };
        },

        refreshAlertsNumber : function(numberJson) {
            var data = $.parseJSON(numberJson);
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (data.number != 0) {
                this.updateBadgeNumber(data.number);
            }
        },

        hasNotifications : function() {
            var numbers = $notificationBox.children(':not(.hidden)').length;
            var ret;
            if (numbers === 0)
                ret = false;
            else
                ret = true;
            return ret;
        },
        
        adjustContainerHeight : function() {
            var $firstChild = $notificationBox.children(':not(.hidden)');
            if ($firstChild.length > 0) {
                $containerBox.css("max-height", $firstChild.eq(0).outerHeight(true));
            }
        },
        
        updateNotificationSection : function() {
            var number = $notificationBox.children(':not(.hidden)').length;
            if (number === 0) {
                $notificationSection.addClass('hidden');
            }
            else {
                $notificationSection.removeClass('hidden');
            }
            ntp.widget.notifications.adjustContainerHeight();
            ntp.page.layout.updateConnectHeight();
        },

        updateBadgeNumber : function(updateNumber) {
            notificationNum += updateNumber;
            if (notificationNum > 0) {
                $notificationBadge.html(notificationNum.toString());
                $footerBadge.html(notificationNum.toString());
                $notificationBadge.removeClass('hidden');
                $footerBadge.removeClass('hidden');
            } else{
                $notificationBadge.addClass('hidden');
                $footerBadge.addClass('hidden');
            };

        },
        
        beginGetUpdateStatus : function() {
            return ntp.acad.getUpdateInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshUpdateWithJSON'));
        },
        
        beginGetHwaccStatus : function() {
            return ntp.acad.getHardwareAccelerationInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshHwaccWithJSON'));
        },
        
        beginGetLicenseStatus : function() {
            return ntp.acad.getLicenseInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshLicenseWithJSON'));
        },
        
        beginGetOfflineHelpStatus : function() {
            return ntp.acad.getOfflineHelpInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshOfflineHelpWithJSON'));
        },
        
        refreshUpdateWithJSON : function(updateInfo) {
            var data = $.parseJSON(updateInfo); 
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.update === 0) {
                if (!$updateBox.hasClass('hidden')) {
                    $updateBox.addClass('hidden');
                    num = -1;
                }
            }
            else {
                if(data.update === 1)
                    var $updateTitle = $('span', $updateBox).text(ntp.utils.string.format(ntp.utils.localize('update_info_title'), data.update));
                else
                    var $updateTitle = $('span', $updateBox).text(ntp.utils.string.format(ntp.utils.localize('updates_info_title'), data.update));
                if ($updateBox.hasClass('hidden')) {
                    $updateBox.removeClass('hidden');
                    num = 1;
                }
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        },
        
        refreshHwaccWithJSON : function(hardwareAccelerationInfo) {
            var data = $.parseJSON(hardwareAccelerationInfo);
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.isHardwareAccelerationOn === true) {
                if (!$hwaccBox.hasClass('hidden')) {
                    $hwaccBox.addClass('hidden');
                    num = -1;
                }
            } else {
                if ($hwaccBox.hasClass('hidden')) {
                    $hwaccBox.removeClass('hidden');
                    num = 1;
                }
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        },
        
        refreshLicenseWithJSON : function(licenseInfo) {
            var data = $.parseJSON(licenseInfo);
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.type !== 2) {
                if (!$licenseBox.hasClass('hidden')) {
                    $licenseBox.addClass('hidden');
                    num = -1;
                }
            }
            else {
                if ($licenseBox.hasClass('hidden')) {
                    $licenseBox.removeClass('hidden');
                    num = 1;
                }
                if (data.daysleft >= 0)
                    $licenseImage.attr({'src' : getDayImage(data.daysleft)});
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        },
        
        refreshOfflineHelpWithJSON : function(offlineHelpInfo) {
            var data = $.parseJSON(offlineHelpInfo);
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.isOfflineHelpInstalled === true) {
                if (!$offlineHelpBox.hasClass('hidden')) {
                    $offlineHelpBox.addClass('hidden');
                    num = -1;
                }
            } else {
                if ($offlineHelpBox.hasClass('hidden')) {
                    $offlineHelpBox.removeClass('hidden');
                    num = 1;
                }
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        }
    };
})();

ntp.widget.connect = (function(){
    var $connectPanel = null; 
    var $feedbackPanel = null;
    var $userPanel = null;
    var $loginPanel = null;
    var $widgetPanel = null;
    var $scrollPanel = null;
    var embededWidgetURL = null;
    var embededWidgetEnable = null;
    var widgetIsOnline = false;
     
    return {
        options : {
            id : 'section_connect',
            title : 'general_connect',
            loginTitle : 'connect_login_title',
            loginSummary : 'connect_login_summary',
            loginAction : 'javascript:ntp.acad.showLoginDialog();',
            feedbackTitle : 'feedback_title',
            feedbackSummary : 'feedback_summary',
            feedbackAction : 'javascript:ntp.acad.sendFeedbackWithBrowser();',
            connectWelcome : 'connect_login_welcome',
            connectAction : 'javascript:ntp.acad.openUserHomePageWithBrowser()',
            feedbackPanel : 'enabled',
            userInfoPanel : 'enabled',
            contentWidgetPanel: 'enabled'
        },
        
        _create : function() {
            this.element.attr({ 'id' : this.options.id });
            if(!this.hasConnect())
                this.element.addClass('hidden');

            $('<h1>').attr({ 'data-text-id' : this.options.title, 'id' : 'connect_title'}).appendTo(this.element);

            $scrollPanel = $('<div>').attr({ 'id' : 'connect_scroll_panel'});
            $connectBox = $('<div>').attr({ 'id' : 'connect_box' }).addClass("login shadow_box white_box");
            if (this.options.userInfoPanel === 'enabled') {
                //connect panel.
                $connectPanel = $('<div>').attr({ 'id' : 'connect_user_panel' }).addClass('hidden');

                //login in panel.
                $loginPanel = $('<div>').attr({ 'id' : 'login_panel' }).addClass('hidden');

                $('<h2>').attr({ 'data-text-id' : this.options.loginTitle }).appendTo($loginPanel);
                $('<p>').attr({ 'data-text-id' : this.options.loginSummary }).appendTo($loginPanel);
                $('<button>').attr({ 'data-text-id' : 'connect_login_button', 'class' : 'blue login', 'onclick' : this.options.loginAction, 'tabindex' : '-1' }).appendTo($loginPanel);

                $loginPanel.appendTo($connectPanel);

                //user information panel.
                $userPanel = $('<div>').attr({ 'id' : 'user_info_panel' }).addClass('hidden');
            
                $('<h2>').attr({ 'id' : 'welcome_message', 'data-text-id' : this.options.connectWelcome }).appendTo($userPanel);
                $('<h2>').attr({ 'id' : 'user_name' }).appendTo($userPanel);
                $('<p>').attr({ 'id' : 'user_email' }).appendTo($userPanel);
                $('<p>').attr({ 'id' : 'closing_brace' }).html(')').appendTo($userPanel);

                $userPanel.appendTo($connectPanel);
                $connectPanel.appendTo($connectBox);
   
            }
            
            embededWidgetEnable = this.options.contentWidgetPanel;
            
            if (this.options.contentWidgetPanel === 'enabled') {
                $widgetPanel = $('<iframe>').attr({ 'id': 'content_widget_box' }).addClass("hidden");
                $widgetPanel.appendTo($connectBox);
            }

            if (this.options.feedbackPanel === 'enabled') {
                $feedbackPanel = $('<div>').attr({ 'id' : 'feedback_panel' });

                $('<h2>').attr({ 'data-text-id' : this.options.feedbackTitle }).appendTo($feedbackPanel);
                $('<p>').attr({ 'data-text-id' : this.options.feedbackSummary }).appendTo($feedbackPanel);
                $('<button>').attr({ 'data-text-id' : this.options.feedbackTitle, 'onclick' : this.options.feedbackAction, 'tabindex' : '-1' }).addClass('blue feedback').appendTo($feedbackPanel);

                $feedbackPanel.appendTo($connectBox);
            }
            $connectBox.appendTo($scrollPanel);
            $scrollPanel.appendTo(this.element);
        },
        
        hasConnect : function() {
            return this.options.feedbackPanel === 'enabled' || this.options.userInfoPanel === 'enabled' || this.options.contentWidgetPanel === 'enabled';
        },
        
        isEnabled : function() {
            return this.options.userInfoPanel !== 'disabled';
        },
        
        setWidgetURL : function(url) {
            embededWidgetURL = url;
        },
        
        setIsOnline : function(isOnline) {
            widgetIsOnline = isOnline;
        },

        showWidgetPanel : function() {
            if ($connectPanel !== null)
                $connectPanel.addClass('hidden');
            if ($widgetPanel !== null)
                $widgetPanel.removeClass('hidden');
        },
        
        showLoginPanel : function() {
            if ($userPanel !== null)
                $userPanel.addClass('hidden');
            if ($widgetPanel !== null)
                $widgetPanel.addClass('hidden');
            if ($connectPanel !== null)
                $connectPanel.removeClass('hidden');
            if ($loginPanel !== null)
                $loginPanel.removeClass('hidden');
        },
        
        showUserInfoPanel : function(data) {
            if ($connectPanel === null)
                return;
            $("#closing_brace").addClass('hidden');
            $loginPanel.addClass('hidden');
            $connectPanel.removeClass('hidden');
            $userPanel.removeClass('hidden');
            if ($widgetPanel !== null)
                $widgetPanel.addClass('hidden');
            var $welcomeMsg = $('#welcome_message');
            var $userName = $("#user_name");
            var $userEmail = $("#user_email");

            if (data.loggedInCurrentSession) {
                var userName = data.userName;
                var userEmail = data.userEmail;
                var userFirstName = data.userFirstName;

                // Show the welcome message part
                $welcomeMsg.removeClass('hidden');
                $userEmail.removeClass('hidden');
                $userName.removeClass('hidden');

                $userName.text(" " + userFirstName + "!");
                if ($userName[0].scrollWidth > $userName.width()) {
                    $userName.attr('data-title', $userName.text());
                } else {
                    $userName.attr('data-title', '');
                }
                $userName.hover(showTooltip, hideTooltip);

                $userEmail.text("(" + userEmail + ")");
                if ($userEmail[0].scrollWidth > $userEmail.width()) {
                    $userEmail.attr('data-title', $userEmail.text());
                    $("#closing_brace").removeClass('hidden');
                } else {
                    $userEmail.attr('data-title', '');
                }
                $userEmail.hover(showTooltip, hideTooltip);
            } else {
                // Hide the welcome message part
                $welcomeMsg.addClass('hidden');
                $userEmail.addClass('hidden');
                $userName.addClass('hidden');
            }
        },
        
        refresh : function(data) {
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (data.isLoggedIn === 'undefined')
                return;
            if ( data.isLoggedIn === true) {
                $connectPanel.addClass('hidden');
                if (embededWidgetEnable === 'enabled' && embededWidgetURL !== null){
                    $widgetPanel.attr("src", data.signedUrl);
                    ntp.widget.connect.showWidgetPanel();
                }
                else{
                    ntp.widget.connect.showUserInfoPanel(data);
                }
            }
            else {
                ntp.widget.connect.showLoginPanel();
            }
            ntp.page.layout.updateConnectHeight();
        },
        
        beginRefresh : function() {
            if (!this.isEnabled()) {
                return null;
            }
            return ntp.acad.getSignedRequestDataAsJSON(embededWidgetURL).done(ntp.utils.makeCallback(this, 'refreshWithJSON'));
        },
        
        refreshWithJSON : function(userInfoJSON) {
            if (!this.isEnabled()) {
                return;
            }
            this.refresh($.parseJSON(userInfoJSON));
        },
        
        refreshLoginWithJSON : function(data) {
            ntp.asyncLoadings.push(this.beginRefresh());
        }
    };
})();

ntp.widget.continueWorking = (function() {
    var $recentFiles = null;
    
    var setViewName = function(viewName) {
        $('#recent_file_view_options li').each(function() {
            var $viewOptionButton = $(this);
            var viewOptionName = $viewOptionButton.data('view-name');
            if (viewOptionName !== viewName) {
                $viewOptionButton.removeClass('selected');
                $recentFiles.removeClass(viewOptionName);
            } else {
                $viewOptionButton.addClass('selected');
            }
        });
        $recentFiles.addClass(viewName);
    };
    
    var handleRecentFileClicked = function() {
        ntp.acad.openDocumentWithPath($(this).data('full-path'));
    };
    
    var recentFileCategories = {
        0 : 'pinned',
        1 : 'timing_today',
        2 : 'timing_yesterday',
        3 : 'timing_this_week',
        4 : 'timing_last_week',
        5 : 'timing_this_month',
        6 : 'timing_last_month',
        7 : 'timing_older'
    };
    
    var resetToMidnight = function(d) {
        d.setHours(0, 0, 0, 0);
    };

    var listCategories = function() {
        return recentFileCategories;
    };

    //Implements pin animation of recent files.
    var pinningInProgress = false;
    
    var createCategory = function(categoryId) {
        var categoryName = recentFileCategories[categoryId];
        var categoryText = ntp.utils.localize(categoryName);
        if (categoryText === undefined) {
            // Ensure <h2> element is created to reduce number of edge cases in pin/unpin animation
            categoryText = '';
        }
        return $('<li>').data('category-id', categoryId).append($('<h2>').text(categoryText)).append($('<ol>'));
    };
    
    var handlePinButtonClicked = function(event) {
        event.stopPropagation();

        if (pinningInProgress) {
            return;
        }
        pinningInProgress = true;

        var $currentDoc = $(this).parent().parent();
        var fullPath = $currentDoc.data('full-path');
        var $currentCategory = $currentDoc.parent().parent();
        var $currentCategoryHeader = $currentCategory.children('h2');

        // Update CSS class
        var isPinning = !($(this).hasClass('pinned'));
        if (isPinning) {
            $(this).addClass('pinned');
            $(this).parent().parent().addClass('pinned');
        } else {
            $(this).removeClass('pinned');
            $(this).parent().parent().removeClass('pinned');
        }

        //
        // Show pin/unpin animation
        //

        // Step 1, determine the elements that needs to be moved.
        var $targetCategory = null;

        var shouldRemoveCurrentCategory = $currentDoc.siblings().length === 0;

        var $currentCategoryDocsInBetween = isPinning ? $currentDoc.prevAll() : $currentDoc.nextAll();
        var $currentCategoryDocsBelow = isPinning ? $currentDoc.nextAll() : $([]);
        var targetCategoryId = ntp.widget.continueWorking.getCategory($currentDoc.data('last-opened'), isPinning);
        var currentCategoryId = $currentCategory.data('category-id');

        var $categoriesInBetween = $([]);
        var $categoriesBelow = $([]);

        var $prevCategory = null;
        $recentFiles.children().each(function() {
            var $thisCategory = $(this);
            var categoryId = $thisCategory.data('category-id');
            if (categoryId === targetCategoryId) {
                $targetCategory = $thisCategory;
            } else if (categoryId < targetCategoryId) {
                $prevCategory = $thisCategory;
                if (categoryId > 0) {
                    $categoriesInBetween = $categoriesInBetween.add($thisCategory);
                }
            } else {// categoryId > targetCategoryId
                if (isPinning && categoryId > 0 && categoryId < currentCategoryId) {
                    $categoriesInBetween = $categoriesInBetween.add($thisCategory);
                } else if (categoryId > currentCategoryId) {
                    $categoriesBelow = $categoriesBelow.add($thisCategory);
                }
            }

            if (isPinning && categoryId > 0 && categoryId < currentCategoryId) {
                $categoriesInBetween = $categoriesInBetween.add($thisCategory);
            }
        });

        var $targetCategoryDocsInBetween = $([]);
        var $targetCategoryDocsBelow = $([]);
        var shouldCreateTargetCategory = ($targetCategory === null);
        var $targetCategoryHeader = null;
        if ($targetCategory !== null) {
            $targetCategoryHeader = $targetCategory.children('h2');
        }

        var $prevDoc = null;
        var $targetList = null;
        if (!shouldCreateTargetCategory) {
            $targetList = $targetCategory.children('ol');

            if ($targetList.children().length > 0) {
                var lastOpened = $currentDoc.data('last-opened');
                $targetList.children().each(function() {
                    var $thisDoc = $(this);
                    if ($thisDoc.data('last-opened') > lastOpened) {
                        $prevDoc = $thisDoc;
                        if (!isPinning) {
                            $targetCategoryDocsInBetween = $targetCategoryDocsInBetween.add($thisDoc);
                        }
                    } else {
                        if (isPinning) {
                            $targetCategoryDocsInBetween = $targetCategoryDocsInBetween.add($thisDoc);
                        } else {
                            $targetCategoryDocsBelow = $targetCategoryDocsBelow.add($thisDoc);
                        }
                    }
                });
            }
        }

        // Step 2, calculate element movement
        var $elementsInBetween = $currentCategoryDocsInBetween.add($categoriesInBetween).add($targetCategoryDocsInBetween);

        if (isPinning && $currentCategoryHeader.length > 0 && $currentCategoryHeader.is(':visible')) {
            $elementsInBetween = $elementsInBetween.add($currentCategoryHeader);
        } else if (!isPinning && !shouldCreateTargetCategory) {
            $elementsInBetween = $elementsInBetween.add($targetCategoryHeader);
        }

        var elementsInBetweenTopChange = isPinning ? $currentDoc.outerHeight(true) : -$currentDoc.outerHeight(true);

        var currentDocTopChange = 0;
        $elementsInBetween.each(function() {
            currentDocTopChange += $(this).outerHeight(true);
        });
        if (isPinning) {
            currentDocTopChange = -currentDocTopChange;
        }

        var $elementsBelow = $categoriesBelow.add($targetCategoryDocsBelow).add($currentCategoryDocsBelow);
        var elementsBelowTopChange = 0;

        var headerHeight = $recentFiles.find('li:last-child > h2').outerHeight(true);

        if (isPinning) {
            if (shouldRemoveCurrentCategory) {
                elementsBelowTopChange += -headerHeight;
            }
            if (shouldCreateTargetCategory) {
                elementsInBetweenTopChange += headerHeight;
                elementsBelowTopChange += headerHeight;
            }
        } else {
            if (shouldRemoveCurrentCategory) {
                elementsBelowTopChange += -headerHeight;
                elementsInBetweenTopChange += -headerHeight;
                currentDocTopChange += -headerHeight;
            }
            if (shouldCreateTargetCategory) {
                elementsBelowTopChange += headerHeight;
                currentDocTopChange += headerHeight;
            }
        }

        // Step 3, animate elements
        var animationCount = 0;
        var updateDomTree = function() {
            animationCount--;
            if (animationCount > 0) {
                return;
            }

            if (shouldCreateTargetCategory) {
                $targetCategory = createCategory(targetCategoryId);
                $targetCategoryHeader = $targetCategory.children('h2');
                if ($prevCategory !== null) {
                    $prevCategory.after($targetCategory);
                } else {
                    $recentFiles.prepend($targetCategory);
                }

                $targetList = $targetCategory.children('ol');
            }

            if ($prevDoc !== null) {
                $prevDoc.after($currentDoc);
            } else {
                $targetList.prepend($currentDoc);
            }

            if (shouldRemoveCurrentCategory) {
                $currentCategory.remove();
            }

            var $secondHeader = $recentFiles.find('li:nth-child(2) > h2');
            if ($secondHeader.length > 0) {
                if (shouldCreateTargetCategory && isPinning && elementsBelowTopChange !== 0) {
                    $secondHeader.css('opacity', 0);
                    $secondHeader.animate({
                        'opacity' : 1
                    });
                } else {
                    // Force refresh the second header
                    $secondHeader.css('display', 'none');
                    $secondHeader.css('display', '');
                }
            }
            if (shouldCreateTargetCategory && !isPinning) {
                // Simple fadeIn have bugs
                $targetCategoryHeader.css('opacity', 0);
                $targetCategoryHeader.animate({
                    'opacity' : 1
                });
            }

            pinningInProgress = false;

            // Only updates pinned data in AutoCAD after the animation is done.
            // Updating AutoCAD will result in refreshing all the currently opened NTPs
            // which slows down the animation and makes it laggy.
            //
            // Update pinned data in AutoCAD. Made it async so that problems with this
            // call does not impact the pin animation
            setTimeout(function() {
                ntp.acad.pinDocumentWithPath(fullPath, isPinning);
            }, 0);

        };

        if (shouldRemoveCurrentCategory) {
            var $secondHeader = $recentFiles.find('li:nth-child(2) > h2');
            if (isPinning && $currentCategoryHeader.is(':visible')) {
                $elementsInBetween = $elementsInBetween.not($currentCategoryHeader);
                $currentCategoryHeader.css('opacity', 1);
                $currentCategoryHeader.animate({
                    'opacity' : 0
                });
            }

            if (!isPinning && $secondHeader.length > 0) {
                $elementsInBetween = $elementsInBetween.not($secondHeader);

                if (elementsBelowTopChange !== 0) {
                    $secondHeader.css('opacity', 1);
                    $secondHeader.animate({
                        'opacity' : 0
                    });
                }
            }
        }

        var $lastAnimation = null;
        if ($elementsInBetween.length > 0 && elementsInBetweenTopChange !== 0) {
            animationCount += $elementsInBetween.length;
            $elementsInBetween.css('position', 'relative');
            $lastAnimation = $elementsInBetween.animate({
                top : elementsInBetweenTopChange
            }, function() {
                $(this).css({
                    top : '',
                    position : ''
                });

                updateDomTree();
            });
        }

        if ($elementsBelow.length > 0 && elementsBelowTopChange !== 0) {
            animationCount += $elementsBelow.length;
            $elementsBelow.css('position', 'relative');
            $lastAnimation = $elementsBelow.animate({
                top : elementsBelowTopChange
            }, function() {
                $(this).css({
                    top : '',
                    position : ''
                });

                updateDomTree();
            });
        }

        if (currentDocTopChange !== 0) {
            animationCount += $currentDoc.length;
            $currentDoc.css('position', 'relative');
            $currentDoc.css('top', 0);
            $currentDoc.animate({
                top : currentDocTopChange
            }, function() {
                $(this).css({
                    top : '',
                    position : ''
                });

                updateDomTree();
            });
        }

        if (animationCount === 0) {
            updateDomTree();
        }
    };
    
    var categorizeRecentFiles = function(data) {
        var recentFiles = data;
        var categorizedRecentFiles = [];

        var i;
        var categories = listCategories();
        var ids = Object.keys(categories);
        for ( i = 0; i < ids.length; i++) {
            categorizedRecentFiles[ids[i]] = [];
        }

        for ( i = 0; i < recentFiles.length; i++) {
            var lastOpened = new Date(recentFiles[i].lastOpened);
            recentFiles[i].lastOpened = lastOpened;
            recentFiles[i].lastOpenedText = recentFiles[i].lastOpenedText;
        }

        recentFiles.sort(function(a, b) {
            return b.lastOpened.getTime() - a.lastOpened.getTime();
        });

        for ( i = 0; i < recentFiles.length; i++) {
            var id = ntp.widget.continueWorking.getCategory(recentFiles[i].lastOpened, recentFiles[i].pinned);
            categorizedRecentFiles[id].push(recentFiles[i]);
        }

        return categorizedRecentFiles;
    };
    
    var loadCategorizedRecentFiles = function(categorizedRecentFiles) {
        // Add recent files into page
        $.each(categorizedRecentFiles, function(categoryId, recentFiles) {
            if (recentFiles.length == 0) {
                return;
            }

            var $category = createCategory(categoryId);
            var $ol = $category.children('ol');

            $.each(recentFiles, function(i, recentFile) {
                if ( typeof recentFile.type === 'undefined') {
                    recentFile.type = 'dwg';
                }

                var $thumb = $('<img>').addClass('thumb').attr('alt', 'thumbnail');
                if ( typeof recentFile.preview === 'string' && recentFile.preview !== '') {
                    $thumb.attr('src', 'data:image/png;base64,' + recentFile.preview);
                } else {
                    $thumb.attr('src', ntp.content.thumbnail(recentFile.type));
                }

                var $pin = $('<div class="pin">');
                if (recentFile.pinned === true) {
                    $pin.addClass('pinned');
                }
                //Event handler for pin/unpin a document
                $pin.click(handlePinButtonClicked);

                var $icon = $('<div>').addClass(recentFile.type).addClass('icon');
                var $title = $('<div>').addClass('title').text(recentFile.name);
                var lastOpenedBy = ntp.utils.localize('last_opened') + ' ' + recentFile.lastOpenedText;
                var $date = $('<div>').addClass('date').text(lastOpenedBy);
                var $imageContainer = $('<div>').addClass('image_container').append($thumb, $icon, $pin);
                
                if (recentFile.local !== true) {
                    $icon.addClass('cloudIcon');
                    $imageContainer.append($('<div>').addClass('cloud'));
                }
                // version is not added yet
                $li = $('<li>').data('full-path', recentFile.fullPath).data('last-opened', recentFile.lastOpened).append($imageContainer, $('<div>').addClass('details').append($title, $date), $('<div>').css('clear', 'both'));
                $li.attr('data-title', recentFile.fullPath);
                $li.hover(showTooltip, hideTooltip);

                if (recentFile.pinned === true) {
                    $li.addClass('pinned');
                }

                //Event handler for open a recent file
                $li.click(handleRecentFileClicked);

                $ol.append($li);
                // To make the <li> element fully contain the floating image
            });

            $recentFiles.append($category);
        });
    };
    
    var startLoadingRecentFile = function() {
        $recentFiles.addClass('disabled loading');
    };

    var endLoadingRecentFile = function() {
        $recentFiles.removeClass('disabled loading');
    };

    var processRecentFileImages = function(recentFileImages) {
        if (!( recentFileImages instanceof Array)) {
            return;
        }

        $.each(recentFileImages, function(i, recentFileImage) {
            if (!( typeof recentFileImage.filetype === 'string' && typeof recentFileImage.icon === 'string')) {
                return;
            }

            // Add rule for file type icon
            ntp.utils.addRule('#recent_files .' + recentFileImage.filetype + '{' + ' background-image: url("' + ntp.utils.resolvePath(recentFileImage.icon) + '");' + '}');

            // Store preview image
            if ( typeof recentFileImage.thumbnail === 'string') {
                ntp.content.thumbnail(recentFileImage.filetype, recentFileImage.thumbnail);
            }
        });
    };
    
    return {
        options : {
            id : 'continue_working',
            viewOptionId : 'recent_file_view_options',
            titleId : 'column_continue_working_title',
            minHeight : '504px',
           recentFileImages : []
        },
    
        _create : function() {
            this.element.attr({'id' : this.options.id});

            $('<h1>').attr({ 'data-text-id' : this.options.titleId }).appendTo(this.element);
            $recentFiles = $('<ol>').attr({ 'id' : 'recent_files' }).addClass("flex image_text");
            $recentFiles.appendTo(this.element);
            $viewOptions = $('<ul>').attr({ 'id' : this.options.viewOptionId }).addClass("view_options");

            var view = ['image', 'image_text', 'text'];
            for (var i = 0; i < view.length; i++) {
                var classtype = "button";
                $('<li>').attr({
                    'id' : 'recent_file_' + view[i] + '_view',
                    'data-view-name' : view[i]
                }).addClass(classtype).bind('click', function() {
                    ntp.widget.continueWorking.selectView($(this).data('view-name'));
                }).appendTo($viewOptions);
            }

            $viewOptions.appendTo(this.element);
            
            if (this.options.recentFileImages) {
                processRecentFileImages(this.options.recentFileImages);
            }
            
            this.beginRefresh();
        },
        
        getCategory : function(date, pinned) {
            if (pinned === true) {
                return 0;
            }
            var today = new Date();
            //today
            resetToMidnight(today);
            if (date >= today) {
                return 1;
            }

            var yesterday = new Date();
            //yesterday
            resetToMidnight(yesterday);
            yesterday.setDate(today.getDate() - 1);
            if (date >= yesterday) {
                return 2;
            }

            var thisWeek = new Date();
            //this week
            resetToMidnight(thisWeek);
            thisWeek.setDate(today.getDate() - today.getDay());
            if (date >= thisWeek) {
                return 3;
            }

            var lastWeek = new Date();
            //last week
            resetToMidnight(lastWeek);
            lastWeek.setDate(thisWeek.getDate() - 7);
            if (date >= lastWeek) {
                return 4;
            }

            var thisMonth = new Date();
            // this month
            resetToMidnight(thisMonth);
            thisMonth.setDate(1);
            // first day of this month
            if (date >= thisMonth) {
                return 5;
            }

            var lastMonth = new Date();
            //last month
            resetToMidnight(lastMonth);
            lastMonth.setMonth(thisMonth.getMonth() - 1, 1);
            //first day of last month
            if (date >= lastMonth) {
                return 6;
            }
            return 7;
        },

        selectView : function(viewName) {
            // Fade out recent files list
            // DID 148501, due to some display overlapping, we moved animation away from callback function.
            $recentFiles.stop().animate(
                {opacity : 0}, 
                ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION /2 , 
                'linear', function () {
                    // Close all opened version menus.
                    $('#recent_files .versions_menu:visible').each(function () {
                        $(this).hide();
                    });
            });

            // Change the class of recent files and track the change of height to make a smooth transition.
            $recentFiles.height('auto');
            // Clear previous fixed height setting
            var oldHeight = $recentFiles.height();
            setViewName(viewName);
            // Update recent file class
            var newHeight = $recentFiles.height();
            $recentFiles.height(oldHeight);

            // Fade in the recent file list with smooth transition of height.
            $recentFiles.stop().transition({
                opacity: 1,
                height: newHeight
            }, ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);

            ntp.acad.updateUserSetting("NewTabPageViewStyle", viewName);
        },

        refresh : function(data) {
            if ( typeof data === 'undefined' )
                return;
            if ( typeof data.retValue !== 'undefined' )
                data = data.retValue; // hoist retValue if present
            if ( typeof data.docs === 'undefined' )
                return;
            // Cleans old recent files
            $recentFiles.empty();

            // Categorizes recent file data by date
            var categorizedRecentFiles = categorizeRecentFiles(data.docs);

            // Load categorized data
            loadCategorizedRecentFiles(categorizedRecentFiles);
        },

        beginRefresh : function() {
            startLoadingRecentFile();

            ntp.acad.getUserSettingAsJSON("NewTabPageViewStyle").done(ntp.utils.makeCallback(this, 'refreshViewStyleWithJSON'));
            
            // Update recent file list
            var promise = ntp.acad.loadRecentFilesAsJSON().done(ntp.utils.makeCallback(this, 'refreshWithJSON')).always(endLoadingRecentFile);

            // If aync load of recent file list timed out, close
            // the busy indication.
            ntp.utils.timeout(endLoadingRecentFile, 10000);

            return promise;
        },

        refreshWithJSON : function(recentFilesJSON) {
            try {
                startLoadingRecentFile();
                this.refresh($.parseJSON(recentFilesJSON));
            } finally {
                endLoadingRecentFile();
            }
        },
        
        refreshViewStyleWithJSON : function(viewStyleJSON) {
            var data = $.parseJSON(viewStyleJSON);
            if (typeof data === 'undefined')
                return;
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if ( typeof data.NewTabPageViewStyle === 'undefined')
                return;
            var viewName = data.NewTabPageViewStyle;
            setViewName(viewName);
        }
    };
})();

ntp.widget.projectbar = (function() {
    var $projectBar = null;
    var $projectPanel = null;
    var $projectList = null;
    var $projectFooter = null;
    var $currentProject = null;
    var $openProject = null;
    var $newProject = null;
    var toggling = false;
    
    var handleClickedOutsideProjectPanel = function() {
        // Check if the mouse if hovering anywhere outside the panel
        if (!$projectList.is(':hover') && !toggling) {
            $projectPanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).off('click', handleClickedOutsideProjectPanel);
        }
        toggling = false;
    };
    
    var handleProjectBarClicked = function(event) {
        if ($projectPanel.css('display') === 'none') {
            toggling = true;
            updateFloatingProjectPanelHeight();
            
            $projectPanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).on('click', handleClickedOutsideProjectPanel);
        }
    };
    
    var handleProjectClicked = function(event) {
        $projectPanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
        $(document).off('click', handleClickedOutsideProjectPanel);
        
        ntp.acad.handleProjectClicked($(this).attr('data-title'));
        $(this).attr('title', '');
    };
    
    var createProjectItem = function (project) {
        if (project !== null) {
            var $li = $('<li>').addClass('button').attr('data-title', project.fullPath).text(project.name).click(handleProjectClicked);
            $li.hover(showTooltip, hideTooltip);
            return $li;
        }
        return null;
    };
    
    var addSepratorIfNeeded = function (projects) {
        if (projects.length >= 1 && $projectFooter !== null) {
             $projectFooter.before($('<hr>').css('margin-top', '3px'));
        }
    };
    
    var setCurrentProject = function (projectName) {
        if (projectName !== '') 
            $currentProject.text(projectName);
        else
            $currentProject.text(ntp.utils.localize('no_active_project'));
    };
    
    var loadProjects = function (projects) {
        $projectList.empty();
        $('hr', $projectPanel).remove();
        $.each(projects, function(i, project) {
            var $li = createProjectItem(project);
            $li.appendTo($projectList);
        });        
        addSepratorIfNeeded(projects);
    };
    
    var updateFloatingProjectPanelHeight = function() {
        // Calculate the maximum height of the panel
        var $tabCreate = $('#tab_create');
        var projectPanelHeight = $tabCreate.offset().top + $tabCreate.height() - $projectBar.offset().top - $projectBar.outerHeight() - 31;
        
        var projectFooterHeight = 0;
        if ($projectFooter !== null) {
            projectFooterHeight = $projectFooter.outerHeight(true);
            if ($projectPanel.css('display') === 'none') {
                $projectPanel.css({ 'display' : 'block', 'height' : 0 });
                projectFooterHeight = $projectFooter.outerHeight(true);
                $projectPanel.css({ 'display' : 'none', 'height' : 'auto'});
            }
        }
        $projectList.css('max-height', Math.min(projectPanelHeight - projectFooterHeight, 200));
    };
    
    return {
        options : {
            "projectBar" : "disabled",
            "newProject" : "enabled",
            "openProject" : "enabled",
            "projectIcon" : ""
        },
        
        _create : function() {
            if (this.options.projectBar === 'enabled') {
                $projectBar = $('<div>').attr('id', 'project_dropdown').addClass('button shadow_box').click(handleProjectBarClicked);
                if (this.options.projectIcon !== '') {
                    $projectBar.css('background-image', 'url("' + ntp.utils.resolvePath(this.options.projectIcon) + '")' + ',-webkit-linear-gradient(#e5e5e5,#c8c8c8)');
                    $projectBar.css('background-position','7px center, top left');
                }
                
                $currentProject = $('<span>').attr({'id': 'current_project', 'data-text-id': 'no_active_project'}).addClass('label').appendTo($projectBar);
                $('<span>').addClass('arrow').html('&#x25BC;').appendTo($projectBar);
                
                $projectPanel = $('<div>').attr('id', 'project_panel').addClass('shadow_box');
                $projectList = $('<ol>').attr('id', 'project_list').appendTo($projectPanel);
                var $ol = $('<ol>');
                if (this.options.openProject === 'enabled')
                    $openProject = $('<li>').attr('data-text-id', 'open_project').addClass('button').click(ntp.acad.showProjectNavigator).appendTo($ol);
                    
                if (this.options.newProject === 'enabled')
                    $newProject = $('<li>').attr('data-text-id', 'create_new_project').addClass('button').click(ntp.acad.newProject).appendTo($ol);;
                
                if ($openProject !==  null || $newProject !== null)
                    $projectFooter = $('<footer>').attr('id', 'project_footer').append($ol).appendTo($projectPanel);
                $projectPanel.appendTo($projectBar);
                
                this.element.append($projectBar);
                var projectBarRefresh = this.beginRefresh();
                ntp.asyncLoadings.push(projectBarRefresh);
            }
        },
        
        isEnabled : function () {
            return $projectBar !== null;
        },
        
        update: function() {
            if (this.isEnabled())
                updateFloatingProjectPanelHeight();
        },
        
        beginRefresh : function() {
            if (!this.isEnabled()) {
                return null;
            }
            ntp.api.loadProjectsAsJSON().done(ntp.utils.makeCallback(this, 'refreshWithJSON'));
            return ntp.acad.loadCurrentProjectAsJSON().done(ntp.utils.makeCallback(this, 'refreshCurrentProjectWithJSON'));
        },
        
        refreshWithJSON : function(projectsJSON) {
            this.refresh($.parseJSON(projectsJSON));
        },
        
        refresh: function (data) {
            if (!this.isEnabled())
                return;
            if (typeof data === 'undefined')
                return;
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (typeof (data.projects) === "undefined")
                return;
            loadProjects(data.projects);
        },
        
        setCurrentProject : function(data)
        {
            if (!this.isEnabled())
                return;
            if (typeof data === 'undefined')
                return;
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (typeof (data.name) === "undefined")
                return;
            
            setCurrentProject(data.name);
        },
        
        refreshCurrentProjectWithJSON : function(currentProjectJSON) {
            this.setCurrentProject($.parseJSON(currentProjectJSON));
        }
    };
})();

/**
 * setup all the widgets
 */
( function($) {
        $.widget("ntp.Title", ntp.widget.title);
        $.widget("ntp.Command", ntp.widget.command);
        $.widget("ntp.Templatebar", ntp.widget.templatebar);
        $.widget("ntp.GetStartedBox", ntp.widget.tile);
        $.widget("ntp.GetStarted", ntp.widget.getStarted);
        $.widget("ntp.CommandsGroup", ntp.widget.commandsGroup);
        $.widget("ntp.External", ntp.widget.external);
        $.widget("ntp.Notifications", ntp.widget.notifications);
        $.widget("ntp.Connect", ntp.widget.connect);
        $.widget("ntp.ContinueWorking", ntp.widget.continueWorking);
        $.widget("ntp.ProjectBar", ntp.widget.projectbar);
    }(jQuery));

/**
 * extend ntp.widget for loading widgets
 */
(function() {
    $.extend(ntp.widget, {
        learnPage : [],
        createPage : ["#column_get_started", "#column_continue_working", "#column_online"],

        loadWidgets : function(widgets) {
            var that = this;
            if ( typeof widgets.create === 'object' && widgets.create instanceof Array) {
                $.each(widgets.create, function(index, column) {
                    if ( typeof column === 'object' && column instanceof Array) {
                        $.each(column, function(i, w) {
                            if ( typeof w === 'object') {
                                $.each(w, function(widget, options) {
                                    if ( typeof options === 'string' && options === '') {
                                        ntp.widget.showWidget($('<div>'), widget).appendTo($(that.createPage[index]));
                                    } else {
                                        ntp.widget.showWidgetWithOptions($('<div>'), widget, options).appendTo($(that.createPage[index]));
                                    }
                                });
                            }
                        });
                    }
                });
            }
        },

        merge : function(common, spec) {
            // 1. adjust the order if it is specified in spec
            if ( typeof spec.placement !== 'undefined') {
                $.each(spec.placement.create, function(i, val) {
                    var tmp = [];
                    if ( val instanceof Array) {
                        $.each(val, function(j, v) {
                            if ( typeof v === 'string') {
                                var w = {};
                                w[v] = find(common.widgets.create, v);
                                tmp.push(w);
                            } else
                                tmp.push(v);
                        });
                    }
                    common.widgets.create[i] = tmp;
                });
            }
            // 2. update the property value in common with spec.
            $.each(spec, function(k, v) {
                update(common, k, v);
            });

            function find(obj, key) {
                var res = {};
                (function find_(obj, key) {
                    $.each(obj, function(k, v) {
                        if ( typeof k === 'string' && k === key) {
                            res = v;
                        }
                        if ( typeof v === 'object')
                            find_(v, key);
                    });
                })(obj, key);

                return res;
            }

            function update(obj, key, value) {
                $.each(obj, function(k, v) {
                    if ( typeof k === 'string' && k === key) {
                        obj[k] = value;
                        return false;
                    }
                    if ( typeof v === 'object')
                        update(v, key, value);
                });
            }

        },

        showWidget : function(container, widget) {
            return this.showWidgetWithOptions(container, widget, {});
        },

        showWidgetWithOptions : function(container, widget, options) {
            if ( typeof container == "string") {
                ntp.utils.makeCallbackWithArgs($(container), widget, options)();
                return $(container);
            } else if ( typeof container == "object") {
                ntp.utils.makeCallbackWithArgs(container, widget, options)();
                return container;
            }
        },

        applySettings : function(generic, spec) {
            if ( typeof generic.widgets !== 'undefined') {
                if (spec !== null) {
                    ntp.widget.merge(generic, spec);
                }
                this.loadWidgets(generic.widgets);
                if (spec !== null) {
                    if (typeof spec.alert !== 'undefined' && typeof spec.alert.enabled === 'string' && spec.alert.enabled.toLowerCase() === 'yes') {
                        //Load alerts from server JSON file.
                        if (typeof spec.alert.alertUrl === 'string') {
                            ntp.deferred.localization.done(function() {
                                ntp.widget.notifications.loadAlerts(ntp.utils.localize(spec.alert.alertUrl));
                            });
                        };
                    }
                }
            }
        },

        bindToolTips: function () {
            $("[data-title]").hover(showTooltip, hideTooltip);
        },

        unbindToolTips: function () {
            //Done on document switch out of new tabpage
            $("[data-title]").mouseleave().unbind('mouseenter mouseleave');
        }
    });
})();

// SIG // Begin signature block
// SIG // MIIZuQYJKoZIhvcNAQcCoIIZqjCCGaYCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // p4q7MEjb/WzlG2dBRYeYnAQmeGzUImBZjR6cGRJpYEqg
// SIG // ggopMIIE2jCCA8KgAwIBAgIQDl0r5acblsN6Y+ZZVIFE
// SIG // GDANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNzA4MDQwMDAw
// SIG // MDBaFw0xODA4MDQyMzU5NTlaMIGKMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEXMBUGA1UECgwOQXV0b2Rlc2ssIElu
// SIG // Yy4xHzAdBgNVBAsMFkRlc2lnbiBTb2x1dGlvbnMgR3Jv
// SIG // dXAxFzAVBgNVBAMMDkF1dG9kZXNrLCBJbmMuMIIBIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAs9HnSHLU
// SIG // WStY4GY/6wZd8pYVSkUAoH04sOU3dSP3hYEt0eN77CwF
// SIG // q/8A4vUF6fio5rvg9akkdMl3yOmHgxIaEpkTegU0/IBV
// SIG // cJWFQCOqUtphBeQcNYx7LP7LsVZfigrUHUf5I1Vydzb6
// SIG // QAHRsQhPhUAi9BzK9rjVOGlUbM9dkz8koMOkDq2hwxDf
// SIG // 3flQCMdi6+g4AMSQW3k7+URbM9PPRpCRUsaIPXyuAXG5
// SIG // LD5TPaPK4NlnmXI0FBEX4rhy7x+V724vZapto7uSgfPv
// SIG // p2Qms1b3ACaR0YJ1+PcxCj44Pg7kmNIxgcRyaNVsfKj5
// SIG // kr8EhXKQCnFzS183YdElK4eUbQIDAQABo4IBPjCCATow
// SIG // CQYDVR0TBAIwADAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0l
// SIG // BAwwCgYIKwYBBQUHAwMwYQYDVR0gBFowWDBWBgZngQwB
// SIG // BAEwTDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNi
// SIG // LmNvbS9jcHMwJQYIKwYBBQUHAgIwGQwXaHR0cHM6Ly9k
// SIG // LnN5bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAU1MAGIknr
// SIG // OUvdk+JcobhHdglyA1gwKwYDVR0fBCQwIjAgoB6gHIYa
// SIG // aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcmwwVwYIKwYB
// SIG // BQUHAQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vcmIu
// SIG // c3ltY2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vcmIu
// SIG // c3ltY2IuY29tL3JiLmNydDANBgkqhkiG9w0BAQsFAAOC
// SIG // AQEAt+AVE1Tf82ntInlNGzyzy0+c5N3gPlZ76y0sK5DQ
// SIG // jWSNUjKIhj5ndxDPYAH3HxVe5gUtxedPpFzVwxgm2VPI
// SIG // HtnsRmR/FCYWBzhyFV+eC5Ok+pS1+jdAxP9LX5Ah1h8h
// SIG // uTV1K8UGZnBEZq2YZufaHscaIiA2pBo7pMPsXH1MQk/A
// SIG // iKIIrwXdvgQ2lhAeWAfc4B0JEtrUhY994DTfIXc8irz4
// SIG // 2EuyeDDUU2b34LpQ4zeZ/eMfDFOtu3iighmy2M8Dg9Qb
// SIG // XoWSEmDuXZEINHqwT2JPsUwKw7q0AdfelLxp8XI3IRV+
// SIG // AlUp/OiF0M6P5dAuTOlz2/deiHWBMg8K17tU+zCCBUcw
// SIG // ggQvoAMCAQICEHwbNTVK59t050FfEWnKa6gwDQYJKoZI
// SIG // hvcNAQELBQAwgb0xCzAJBgNVBAYTAlVTMRcwFQYDVQQK
// SIG // Ew5WZXJpU2lnbiwgSW5jLjEfMB0GA1UECxMWVmVyaVNp
// SIG // Z24gVHJ1c3QgTmV0d29yazE6MDgGA1UECxMxKGMpIDIw
// SIG // MDggVmVyaVNpZ24sIEluYy4gLSBGb3IgYXV0aG9yaXpl
// SIG // ZCB1c2Ugb25seTE4MDYGA1UEAxMvVmVyaVNpZ24gVW5p
// SIG // dmVyc2FsIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3Jp
// SIG // dHkwHhcNMTQwNzIyMDAwMDAwWhcNMjQwNzIxMjM1OTU5
// SIG // WjCBhDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFu
// SIG // dGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRl
// SIG // YyBUcnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRl
// SIG // YyBDbGFzcyAzIFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0Eg
// SIG // LSBHMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBANeVQ9Tc32euOftSpLYmMQRw6beOWyq6N2k1lY+7
// SIG // wDDnhthzu9/r0XY/ilaO6y1L8FcYTrGNpTPTC3Uj1Wp5
// SIG // J92j0/cOh2W13q0c8fU1tCJRryKhwV1LkH/AWU6rnXmp
// SIG // AtceSbE7TYf+wnirv+9SrpyvCNk55ZpRPmlfMBBOcWNs
// SIG // WOHwIDMbD3S+W8sS4duMxICUcrv2RZqewSUL+6Mcntim
// SIG // CXBx7MBHTI99w94Zzj7uBHKOF9P/8LIFMhlM07Acn/6l
// SIG // eCBCcEGwJoxvAMg6ABFBekGwp4qRBKCZePR3tPNgKuZs
// SIG // UAS3FGD/DVH0qIuE/iHaXF599Sl5T7BEdG9tcv8CAwEA
// SIG // AaOCAXgwggF0MC4GCCsGAQUFBwEBBCIwIDAeBggrBgEF
// SIG // BQcwAYYSaHR0cDovL3Muc3ltY2QuY29tMBIGA1UdEwEB
// SIG // /wQIMAYBAf8CAQAwZgYDVR0gBF8wXTBbBgtghkgBhvhF
// SIG // AQcXAzBMMCMGCCsGAQUFBwIBFhdodHRwczovL2Quc3lt
// SIG // Y2IuY29tL2NwczAlBggrBgEFBQcCAjAZGhdodHRwczov
// SIG // L2Quc3ltY2IuY29tL3JwYTA2BgNVHR8ELzAtMCugKaAn
// SIG // hiVodHRwOi8vcy5zeW1jYi5jb20vdW5pdmVyc2FsLXJv
// SIG // b3QuY3JsMBMGA1UdJQQMMAoGCCsGAQUFBwMDMA4GA1Ud
// SIG // DwEB/wQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UE
// SIG // AxMRU3ltYW50ZWNQS0ktMS03MjQwHQYDVR0OBBYEFNTA
// SIG // BiJJ6zlL3ZPiXKG4R3YJcgNYMB8GA1UdIwQYMBaAFLZ3
// SIG // +mlIR59TEtXC6gcydgfRlwcZMA0GCSqGSIb3DQEBCwUA
// SIG // A4IBAQB/68qn6ot2Qus+jiBUMOO3udz6SD4Wxw9FlRDN
// SIG // J4ajZvMC7XH4qsJVl5Fwg/lSflJpPMnx4JRGgBi7odSk
// SIG // VqbzHQCR1YbzSIfgy8Q0aCBetMv5Be2cr3BTJ7noPn5R
// SIG // oGlxi9xR7YA6JTKfRK9uQyjTIXW7l9iLi4z+qQRGBIX3
// SIG // FZxLEY3ELBf+1W5/muJWkvGWs60t+fTf2omZzrI4RMD3
// SIG // R3vKJbn6Kmgzm1By3qif1M0sCzS9izB4QOCNjicbkG8a
// SIG // vggVgV3rL+JR51EeyXgp5x5lvzjvAUoBCSQOFsQUecFB
// SIG // NzTQPZFSlJ3haO8I8OJpnGdukAsak3HUJgLDwFojMYIO
// SIG // 6DCCDuQCAQEwgZkwgYQxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UE
// SIG // CxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1MDMGA1UE
// SIG // AxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBT
// SIG // aWduaW5nIENBIC0gRzICEA5dK+WnG5bDemPmWVSBRBgw
// SIG // DQYJYIZIAWUDBAIBBQCgfDAQBgorBgEEAYI3AgEMMQIw
// SIG // ADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
// SIG // BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG
// SIG // 9w0BCQQxIgQgMsdFHINbFiJtLqGnWOKsVyeA7d9ejlPZ
// SIG // smSQZG1iBnkwDQYJKoZIhvcNAQEBBQAEggEAnGJ6aBVb
// SIG // ztWLd+CanERuWcolCjcwReS/T2ilkUHWQXi9Dp2DXWYL
// SIG // l3SJb5VYiqsp4OsTtN4IVMBE2pVyCGTogvfMk/RMpOT7
// SIG // V+9znHAvp95TbtR2LGkUC/Z+TmkO6fNzDzVdMNrBTRpA
// SIG // QAHbKZ2/BpK6todASIxvUkT3GaMGBdLsNNj4gRXLW4ze
// SIG // zkuuSU60CeQ6JSaE7xl/31uLHiz3mqU/LwgUDJRIbwJ0
// SIG // k8uNiOeAhZhULBxGyvrjds3jxNc4jfP30UnkVPV5mfMI
// SIG // 3P5qV2irQUPRna0IJRE6ZW0/7I+2wbpabdtIzdElELhF
// SIG // tKyRHBBgCktx/VRy7oXpUDk28qGCDKEwggydBgorBgEE
// SIG // AYI3AwMBMYIMjTCCDIkGCSqGSIb3DQEHAqCCDHowggx2
// SIG // AgEDMQ8wDQYJYIZIAWUDBAIBBQAwgd0GCyqGSIb3DQEJ
// SIG // EAEEoIHNBIHKMIHHAgEBBgkrBgEEAaAyAgMwMTANBglg
// SIG // hkgBZQMEAgEFAAQgD/T2SqXkYJzkhcJDHRwQaPIJ0RNA
// SIG // E5ASz26UnIJF8t4CFAOxwisc+oGcTDvrNs6exOdbLbMq
// SIG // GA8yMDE4MDEzMDA0NTEyMVqgXaRbMFkxCzAJBgNVBAYT
// SIG // AlNHMR8wHQYDVQQKExZHTU8gR2xvYmFsU2lnbiBQdGUg
// SIG // THRkMSkwJwYDVQQDEyBHbG9iYWxTaWduIFRTQSBmb3Ig
// SIG // QWR2YW5jZWQgLSBHMqCCCMYwggSpMIIDkaADAgECAhIR
// SIG // IQbxD85o8Jv65VsYzY8gAXcwDQYJKoZIhvcNAQELBQAw
// SIG // WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNp
// SIG // Z24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGlt
// SIG // ZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzIwHhcNMTYw
// SIG // NTI0MDAwMDAwWhcNMjcwNjI0MDAwMDAwWjBZMQswCQYD
// SIG // VQQGEwJTRzEfMB0GA1UEChMWR01PIEdsb2JhbFNpZ24g
// SIG // UHRlIEx0ZDEpMCcGA1UEAxMgR2xvYmFsU2lnbiBUU0Eg
// SIG // Zm9yIEFkdmFuY2VkIC0gRzIwggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQC3x5KKKNjzkctQDV3rKUTB
// SIG // glmlymTOvYO1UeWUzG6Amhds3P9i5jZDXgHCDGSNynee
// SIG // 9l13RbleyCTrQTcRZjesyM10m8yz70zifxvOc77Jlp01
// SIG // Hnz3VPds7KAS1q6ZnWPEeF9ZqS4i9cMn2LJbRWMnkP+M
// SIG // sT2ptPMOwPEgZQaJnQMco7BSQYU067zLzlT2Ev6zAYlK
// SIG // pvpUxR/70xzA47+X4z/QG/lAxxvV6yZ8QzDHcPJ4EaqF
// SIG // TqUODQBKOhF3o8ojAYCeyJNWXUbMitjSqgqEhbKJW9Uy
// SIG // zkF7GE5UyqvRUl4S0ySeVvMMj929ko551UGJw6Og5ZH8
// SIG // x2edhzPOcTJzAgMBAAGjggFnMIIBYzAOBgNVHQ8BAf8E
// SIG // BAMCB4AwTAYDVR0gBEUwQzBBBgkrBgEEAaAyAR4wNDAy
// SIG // BggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2ln
// SIG // bi5jb20vcmVwb3NpdG9yeS8wCQYDVR0TBAIwADAWBgNV
// SIG // HSUBAf8EDDAKBggrBgEFBQcDCDBGBgNVHR8EPzA9MDug
// SIG // OaA3hjVodHRwOi8vY3JsLmdsb2JhbHNpZ24uY29tL2dz
// SIG // L2dzdGltZXN0YW1waW5nc2hhMmcyLmNybDBYBggrBgEF
// SIG // BQcBAQRMMEowSAYIKwYBBQUHMAKGPGh0dHA6Ly9zZWN1
// SIG // cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2dzdGltZXN0
// SIG // YW1waW5nc2hhMmcyLmNydDAdBgNVHQ4EFgQULW5u0Y3l
// SIG // A4Du52Ppp9naCzxXIj0wHwYDVR0jBBgwFoAUkiGnSpVd
// SIG // ZLCbtB7mADdH5p1BK0wwDQYJKoZIhvcNAQELBQADggEB
// SIG // AFedU+Td7qUs3uS/YoCiYYZKn5udJCMwm48nhy/6BFnO
// SIG // cIg9RlJ3xJtF0o+9tyEYSdguh/+SLnf5Pwr5oCFcE0/k
// SIG // hCsoh8xcFbkTq+ISVcP4RR5JGc6qdQ8h6O9R/DQsCzIF
// SIG // CbLkFmmt2S5+4fwtSL8a03Q1ATrfMENwZ3o417oRwlb7
// SIG // MUi6W/EhEmLgdomRH0k41FhNeQmKqdpvAtat5NqJfUzZ
// SIG // sTBwQGrbRbpTdpDcS0IWIc/k0liU1FVlm3YDrK4YCLaD
// SIG // EoxBPl+ARWTIidQzF6GhTL0/xJL+f4XyvrGs1/aey84T
// SIG // Ku4CQ0/8/3A1NKMzfM/iDTIkIHwAPqMyEd4wggQVMIIC
// SIG // /aADAgECAgsEAAAAAAExicZQBDANBgkqhkiG9w0BAQsF
// SIG // ADBMMSAwHgYDVQQLExdHbG9iYWxTaWduIFJvb3QgQ0Eg
// SIG // LSBSMzETMBEGA1UEChMKR2xvYmFsU2lnbjETMBEGA1UE
// SIG // AxMKR2xvYmFsU2lnbjAeFw0xMTA4MDIxMDAwMDBaFw0y
// SIG // OTAzMjkxMDAwMDBaMFsxCzAJBgNVBAYTAkJFMRkwFwYD
// SIG // VQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYDVQQDEyhH
// SIG // bG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAtIFNIQTI1
// SIG // NiAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
// SIG // CgKCAQEAqpuOw6sRUSUBtpaU4k/YwQj2RiPZRcWVl1ur
// SIG // Gr/SbFfJMwYfoA/GPH5TSHq/nYeer+7DjEfhQuzj46FK
// SIG // bAwXxKbBuc1b8R5EiY7+C94hWBPuTcjFZwscsrPxNHaR
// SIG // ossHbTfFoEcmAhWkkJGpeZ7X61edK3wi2BTX8QceeCI2
// SIG // a3d5r6/5f45O4bUIMf3q7UtxYowj8QM5j0R5tnYDV56t
// SIG // LwhG3NKMvPSOdM7IaGlRdhGLD10kWxlUPSbMQI2CJxtZ
// SIG // IH1Z9pOAjvgqOP1roEBlH1d2zFuOBE8sqNuEUBNPxtyL
// SIG // ufjdaUyI65x7MCb8eli7WbwUcpKBV7d2ydiACoBuCQID
// SIG // AQABo4HoMIHlMA4GA1UdDwEB/wQEAwIBBjASBgNVHRMB
// SIG // Af8ECDAGAQH/AgEAMB0GA1UdDgQWBBSSIadKlV1ksJu0
// SIG // HuYAN0fmnUErTDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAy
// SIG // BggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2ln
// SIG // bi5jb20vcmVwb3NpdG9yeS8wNgYDVR0fBC8wLTAroCmg
// SIG // J4YlaHR0cDovL2NybC5nbG9iYWxzaWduLm5ldC9yb290
// SIG // LXIzLmNybDAfBgNVHSMEGDAWgBSP8Et/qC5FJK5NUPpj
// SIG // move4t0bvDANBgkqhkiG9w0BAQsFAAOCAQEABFaCSnzQ
// SIG // zsm/NmbRvjWek2yX6AbOMRhZ+WxBX4AuwEIluBjH/NSx
// SIG // N8RooM8oagN0S2OXhXdhO9cv4/W9M6KSfREfnops7yyw
// SIG // 9GKNNnPRFjbxvF7stICYePzSdnno4SGU4B/EouGqZ9uz
// SIG // nHPlQCLPOc7b5neVp7uyy/YZhp2fyNSYBbJxb051rvE9
// SIG // ZGo7Xk5GpipdCJLxo/MddL9iDSOMXCo4ldLA1c3PiNof
// SIG // KLW6gWlkKrWmotVzr9xG2wSukdduxZi61EfEVnSAR3hY
// SIG // jL7vK/3sbL/RlPe/UOB74JD9IBh4GCJdCC6MHKCX8x2Z
// SIG // faOdkdMGRE4EbnocIOM28LZQuTGCArQwggKwAgEBMHEw
// SIG // WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNp
// SIG // Z24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGlt
// SIG // ZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzICEhEhBvEP
// SIG // zmjwm/rlWxjNjyABdzANBglghkgBZQMEAgEFAKCCARQw
// SIG // GgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqG
// SIG // SIb3DQEJBTEPFw0xODAxMzAwNDUxMjFaMC8GCSqGSIb3
// SIG // DQEJBDEiBCDJNDeLJ5OaR7tShA3teGnR7NhRC0pnUw2q
// SIG // U11TuKkfQDCBpgYLKoZIhvcNAQkQAgwxgZYwgZMwgZAw
// SIG // gY0EFH1V2OdaVqL8c4JD97hUh1xctSoNMHUwX6RdMFsx
// SIG // CzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWdu
// SIG // IG52LXNhMTEwLwYDVQQDEyhHbG9iYWxTaWduIFRpbWVz
// SIG // dGFtcGluZyBDQSAtIFNIQTI1NiAtIEcyAhIRIQbxD85o
// SIG // 8Jv65VsYzY8gAXcwDQYJKoZIhvcNAQEBBQAEggEAqrgk
// SIG // EhflXTnNhsAOe3rhSIjbuS0WGbN5g3IFaSg98u80oeoZ
// SIG // yOqsL8IU2znuTONh6r/Vt7CZNiSP4PHBwW/vsbtQP6/2
// SIG // tJdSIeEtncUXlVFAmqzxOOLP9MuEVovIjlvSr5Y//Mbj
// SIG // QHX8ihlzAB9VCLy+sjZLv/rE3cQE1vvvnaF0n0iMeh4L
// SIG // pcKt9+Vj562TbLfgARuFaRqPQOow458R80YoLgKQ4662
// SIG // dNaatPFtO3SxFcx7tp9S0Y0/mor08yXc4z6NF9wvai9B
// SIG // HXX8FOYfE97GYFHLdh6LyJwPADsRyFqIKr6Cl5MxHJRA
// SIG // llB0MxL5K84Hy90g0fr93G20w7FQdA==
// SIG // End signature block
