//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.page - Implements animations and functionality of new tab page elements
 */

 // Namespace
var ntp = ntp || {};
ntp.page = ntp.page || {};

ntp.page.animationSettings = {
    TAB_SWITCHING_ANIMATION_DURATION: 600,
    ELEMENT_TOGGLING_ANIMATION_DURATION: 200
};

/**
 * Handles page layout that cannot be done via CSS
 */
ntp.page.layout = (function () {
    var $allTabs = null;
    var $columns = null;
    var $getStartedVideo = null;
    var $featuredVideo = null;
    var $onlineResource = null;
    var $continueWorking = null;
    var $connectPanel = null;
    var columnMinHeight = 504;
    //This is video item height. thumb height 90px, thumb border is 2px and thumb bottom-margin is 8px.
    var videoItemHeight = 100;
    //This is video item margin. video item margin-bottom is 16px.
    var videoItemGap = 16;
    
   
    var getColumnHeight = function() {
        var newHeight = $allTabs.height() * 0.7;
        if (newHeight < columnMinHeight){
            newHeight = columnMinHeight;
        }
        return newHeight;
    };
    var adjustContentSpace = function () {
        var allTabsHeight = $allTabs.height();
        var newColumnHeight = allTabsHeight * 0.7;
        var space = (allTabsHeight - newColumnHeight) * 0.375;
        if (space < 0) {
            space = 0;
        }
        // adjust top space
        $allTabs.children('.content').css('margin-top', space);
        // adjust content height
        $allTabs.children('.content').css('height', allTabsHeight - space);
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $allTabs = $('#canvas_slider > .tab');
            $columns = $('section.column');
            $getStartedVideo = $('#getting_started ol');
            $featuredVideo = $('#feature_videos ol');
            $continueWorking = $('#continue_working');
            $onlineResource = $('#online_resources_panel');
            $connectPanel = $('#connect_scroll_panel');
            this.adjustColumnHeight();
        },
        
        adjustColumnHeight: function () {
            var newHeight = getColumnHeight();
            $columns.height(newHeight);
            $continueWorking.height(newHeight);
            this.updateVideoSectionHeight();
            this.updateOnlineResourceHeight();
            this.updateConnectHeight();
        },
        
        update: function () {
            // Update tab width
            var tabWidth = $canvas.width();
            $allTabs.each(function () {
                $(this).width(tabWidth);
            });
            
            this.adjustColumnHeight();
            adjustContentSpace();
        },
        
        updateVideoSectionHeight: function() {
            var height = getColumnHeight();
            if(ntp.content.learnTab.hasGetStartedVideos() && ntp.content.learnTab.hasFeaturedVideos()) {
                var getStartedItemsHeight = $getStartedVideo.children().length * videoItemHeight
                                                + ($getStartedVideo.children().length - 1) * videoItemGap;
                var featuredItemsHeight = $featuredVideo.children().length * videoItemHeight
                                                + ($featuredVideo.children().length - 1) * videoItemGap;
                var getStartedHeight = height * 0.5;
                if (getStartedHeight > getStartedItemsHeight)
                    getStartedHeight = getStartedItemsHeight;
                var featuredHeight = height - getStartedHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                if (featuredHeight > featuredItemsHeight){
                    featuredHeight = featuredItemsHeight;
                    getStartedHeight = height - featuredHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                }
                $getStartedVideo.css({'height': 'auto', 'max-height': getStartedHeight});
                $featuredVideo.css({'height': 'auto','max-height': featuredHeight});
            }
            else if(ntp.content.learnTab.hasGetStartedVideos()) {
                var getStartedHeight = height - $('#getting_started_title').outerHeight(true);
                $getStartedVideo.css({'height': 'auto','max-height': getStartedHeight});
            }
            else if(ntp.content.learnTab.hasFeaturedVideos()) {
                var featuredHeight = height - $('#featured_title').outerHeight(true);
                $featuredVideo.css({'height': 'auto', 'max-height': featuredHeight});
            }
        },
        
        updateOnlineResourceHeight: function() {
            if(ntp.content.learnTab.hasOnlineResource()) {
                var height = getColumnHeight();
                var tipHeight = 0;
                if(ntp.content.learnTab.hasTips()) {
                    tipHeight = $('#Section_tip').outerHeight(true);
                }
                var onlineHeight = height - $('#online_resource_title').outerHeight(true) - tipHeight;
                $onlineResource.css('max-height', onlineHeight);
            }
        },
        
        updateConnectHeight: function() {
            if(ntp.widget.connect.hasConnect()){
                var height = getColumnHeight();
                var panelHeight;
                if(ntp.widget.notifications.hasNotifications()){
                    panelHeight = height - $('#connect_title').outerHeight(true) - $('#section_notifications').outerHeight(true);
                }
                else {
                    panelHeight = height - $('#connect_title').outerHeight(true);
                }
                $connectPanel.css('max-height', panelHeight);
            }
        }
    };
})();

/**
 * Handles tabs and tab switching
 */
ntp.page.tabs = (function () {
    
    var $canvas = null;
    var $tabs = null;
    var tipContentsVisited = false;
    var $visableTabs = null;
    var selectedIndex = 0;
    
    return {
        current: function () {
            return selectedIndex;
        },

        isTipContentsVisited: function () {
            return tipContentsVisited;
        },
        
        count: function () {
            if ($tabs) {
                return $tabs.length;
            } else {
                return 0;
            }
        },
        
        visibleCount: function () {
            if ($visibleTabs)
                return $visibleTabs.length;
            else
                return 1;
        },
        
        init: function () {
            $canvas = $('#canvas');
            $tabs = $('#canvas_slider > .tab');
            $visibleTabs = $('#canvas_slider > .tab.show');
        },
        
        addVisibleTab: function (id, select) {
            var index = 1;
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            var overCurrent = false;
            for(var i = 0; i < $tabs.length; i++){
                if($tabs.eq(i).attr('id') === id){
                    $tabs.eq(i).addClass('show');
                    $('#footer_nav li').eq(i).addClass('show');
                    $visibleTabs = $('#canvas_slider > .tab.show');
                    if(!overCurrent){
                        selectedIndex++;
                        if(!select)
                            this.update();
                    }
                    ntp.page.footerNavs.visibleTabsChange();
                    ntp.page.sideNavs.visibleTabsChange();
                    break;
                }
                else if($tabs.eq(i).attr('id') === $tab.attr('id')){
                    overCurrent = true;
                }
            }
            if(select)
                this.selectById(id, true, true);
        },
        
        removeVisibleTab: function (id, defaultId) {
            $visibleTabs = $('#canvas_slider > .tab.show');
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            if ($tab.attr('id') === id) {
                $tab.removeClass('show');
                $('#footer_nav li').eq(selectedIndex - 1).removeClass('show');
                $visibleTabs = $('#canvas_slider > .tab.show');
                this.selectById(defaultId, true, true);
                ntp.page.footerNavs.visibleTabsChange();
                ntp.page.sideNavs.visibleTabsChange();
            }
            else {
                var overCurrent = false;
                for (var i = 0; i < $visibleTabs.length; i++) {
                    if ($visibleTabs.eq(i).attr('id') === id) {
                        $visibleTabs.eq(i).removeClass('show');
                        $('#footer_nav li').eq(i).removeClass('show');
                        $visibleTabs = $('#canvas_slider > .tab.show');
                        if (!overCurrent) {
                            selectedIndex--;
                            this.update();
                        }
                        ntp.page.footerNavs.visibleTabsChange();
                        ntp.page.sideNavs.visibleTabsChange();
                        break;
                    } else if ($visibleTabs.eq(i).attr('id') === $tab.attr('id')) {
                        overCurrent = true;
                    }
                }
            }
        },
        
        // Selects a tab by its index that is only count for visible tabs.
        selectByIndex: function(index, immediate, forced) {
            var indexChanged = true;
            if (selectedIndex === index) {
                if (!forced) { return; }
                indexChanged = false;
            }
            
            selectedIndex = index;

            // Updates footer navigator class
            $('#footer_nav li.show').each(function (index) {
                if (index + 1 === selectedIndex) {
                    $(this).addClass('highlight');
                } else {
                    $(this).removeClass('highlight');
                }
            });

            // If 'immediate' is auto, then an animation will be used only if canvas is already running an animation.
            if (immediate === 'auto') {
                immediate = !$canvas.is(':animated');
            }

            // Updates tab position
            var $tab = $('.tab.show').eq(index - 1);

            if (tipContentsVisited === true && ntp.content.learnTab.urlIsHilightedContent('tip')) {
                //Leaving learn tab, so unhilight tip
                ntp.content.learnTab.unhilightViewedLearnConents({ tip: $('div#tip_resource_box').data('upload_time') });
                tipContentsVisited = false;

            }else if ($tab.attr('id') === 'tab_learn' && ntp.content.learnTab.urlIsHilightedContent('tip')) {

                //First time Visiting, so notify all others except self.
                ntp.acad.updateLearnContentState('tip', $('div#tip_resource_box').data('upload_time'));
                ntp.acad.updateLearnContentState('tipID', $('div#tip_resource_box').data('tip_ID'));
                ntp.acad.updateLearnContentState('tipExpireDate', $('div#tip_resource_box').data('tip_Expire_Date'));
                tipContentsVisited = true;
            }

            var newScrollLeft = $tab.position().left;
            if ($canvas.scrollLeft() !== newScrollLeft) {
                if (immediate) {
                    $canvas.stop().scrollLeft(newScrollLeft);
                } else {
                    var durationMultiplier = Math.abs(newScrollLeft - $canvas.scrollLeft()) / $canvas.width();
                    if (durationMultiplier > 1) { durationMultiplier = 1; }
                    $canvas.stop().animate(
                        { scrollLeft: newScrollLeft },
                        durationMultiplier * ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                        'linear');
                }
            }
            ntp.page.footerNavs.updateArrow(immediate);
            ntp.page.sideNavs.update(immediate, true, true);
        },
        
        update: function () {
            this.selectByIndex(selectedIndex, 'auto', true);
        },

        // Selects a tab by its id
        selectById: function(id, immediate, forced) {
            var tabIndex = $('.tab.show').index($('#' + id));
            if (tabIndex >= 0 && tabIndex < this.visibleCount()) {
                this.selectByIndex(tabIndex + 1, immediate, forced);
            }
        },
        
        // Auto snaps to the closest tab after scrolling
        snap: function (immediate) {
            var tabWidth = $canvas.width();
            var currentScrollLeft = $canvas.scrollLeft();
            
            // Return if scrolling is caused by resizing
            if (currentScrollLeft === (tabWidth * (selectedIndex - 1))) {
                return;
            }
            
            var closestTabIndex = Math.floor((currentScrollLeft + tabWidth / 2) / tabWidth) + 1;
            this.selectByIndex(closestTabIndex, immediate, true);
        }
    };
})();

/**
 * Handles side navigation arrows
 */
ntp.page.sideNavs = (function () {
    
    var $canvas = null;
    var $tabContents = null;
    var $sideNavs = null;
    
    var tabContentWidth = null;
    var leftSideNavWidth = null;
    var rightSideNavWidth = null;
    
    var overlapped = null;
    
    // Updates the global variable storing left and right of the content of currently selected tab        
    var updateCurrentTabContentWidth = function () {
        if ($tabContents === null) { return null; }
        var selectedIndex = ntp.page.tabs.current();
        var $content = $tabContents.eq(selectedIndex - 1);
        var $columns = $content.children('.column');
        if ($columns.length === 0) {
            tabContentWidth = $content.outerWidth(true);
        } else {
            var $firstColumn = $columns.eq(0);
            var $lastColumn = $columns.eq($columns.length - 1);
            tabContentWidth = $lastColumn.offset().left + $lastColumn.outerWidth(true) - $firstColumn.offset().left;
        }
    };
    
    // Updates side navigator overlap status. If side navigators overlap with page content, they would be hidden.
    var updateOverlap = function (immediate) {
        // Update stored tab content width
        updateCurrentTabContentWidth();

        var tabWidth = $canvas.width();
        var sideNavSpace = (tabWidth - tabContentWidth) / 2;
    
        var newOverlapped = ((sideNavSpace - leftSideNavWidth) < 10 || 
                (sideNavSpace -rightSideNavWidth) < 10);

        if (newOverlapped !== overlapped) {
            if (overlapped === null || immediate) {
                if (newOverlapped) {
                    $sideNavs.addClass('overlapped');
                } else {
                    $sideNavs.removeClass('overlapped');
                }
            } else {
                // Use a fadein or fadeout animation when showing or hiding the side navigators
                $sideNavs.stop(true);
                if (newOverlapped) {
                    $sideNavs.css('opacity', 1);
                    $sideNavs.animate({ opacity: 0 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                        $sideNavs.addClass('overlapped');
                    });
                } else {
                    $sideNavs.css('opacity', 0);
                    $sideNavs.removeClass('overlapped');
                    $sideNavs.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                    });
                }
            }

            overlapped = newOverlapped;
        }
    };
    
    // Updates total width of side navigation tools in global variable.
    var updateWidth = function () {
        leftSideNavWidth = 33;
        leftSideNavWidth += $sideNavs.eq(0).outerWidth(true);
        rightSideNavWidth = 33;
        rightSideNavWidth += $sideNavs.eq(1).outerWidth(true);    
    };
    
    // Updates individual side navigator according to currently selected tab.
    var updateNavigator = function (navigatorSelector, navigatorTabIndex, immediate, handleOverlap) {
        var $sideNav = $(navigatorSelector);

        $sideNav.off('click'); // Clear existing event handler

        if (immediate) {
            var sideNavShouldShow = (navigatorTabIndex > 0 && navigatorTabIndex <= ntp.page.tabs.visibleCount()
                                    && ntp.page.tabs.visibleCount() > 1);
            if (sideNavShouldShow) {
                // Target tab should show. 
                var $footerNav = $('#footer_nav li.show').eq(navigatorTabIndex - 1);
                var label = $footerNav.children('.label').text();
                $sideNav.removeClass('disabled')
                        .children('.label')
                        .text(label);

                $sideNav.one('click', function (event) {
                    ntp.page.tabs.selectByIndex(navigatorTabIndex);
                    event.preventDefault();
                });
            } else {
                //Side navigator need to hide.
                $sideNav.addClass('disabled');
            }

            // Update width of side navigators after their size has changed
            updateWidth();

            if (handleOverlap) {
                updateOverlap(true);
            }
        } else {
            // Change navigators with fade in/fade out effect. The implementation is a little bit complicated,
            // as the simple jQuery fadeIn() fadeOut() do not work so well here.
            var fadeIn = function () {
                $sideNav.stop().css('opacity', '');
                updateNavigator(navigatorSelector, navigatorTabIndex, true, handleOverlap);
                if ($sideNav.css('display') !== 'none') {
                    $sideNav.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNav.css('opacity', '');
                    });
                }
            };

            $sideNav.stop();
            if ($sideNav.css('display') === 'none') {
                setTimeout(fadeIn, ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2);
            } else {
                $sideNav.animate({ opacity: 0 }, 
                    ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                    'linear', fadeIn);
            }
        }
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');         
            $tabContents = $('#canvas_slider > .tab.show > .content');
            $sideNavs = $('#side_nav li');
        },
        
        visibleTabsChange: function () {
            $tabContents = $('#canvas_slider > .tab.show > .content');
            this.update(true, true, true);
        },
        
        // Updates both side navigators according to currently selected tab.
        update: function (immediate, handleOverlap, forced) {
            var currentTab = ntp.page.tabs.current();
            
            if (forced) {
                // Update side navigators
                updateNavigator('#left_side_nav', currentTab - 1, immediate, handleOverlap);
                updateNavigator('#right_side_nav', currentTab + 1, immediate, handleOverlap);
            } else {
                updateOverlap();
            }
        }
    };
})();

/**
 * Handles footer navigation buttons
 */
ntp.page.footerNavs = (function () {
    var $canvas = null;
    var $footerNavs = null;
    var $navArrow = null;
    var $navRoot = null;
    var footerNavWidth = null;
    var isInit = false;
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $navArrow = $('#nav_arrow');
            $footerNavs = $('#footer_nav li.show');
            $navRoot = $('#footer_nav');
            if(!isInit){
                $('#footer_nav li .button').each(
                        function(index) {
                            var tabId = $('#canvas_slider > .tab').eq(index).attr('id');
                            $(this).on("click", function() {
                                ntp.page.tabs.selectById(tabId);
                                event.preventDefault();
                            });
                        });
            }
            isInit = true;
            if($footerNavs.length > 1)
                this.visibleTabsChange();
        },
        
        visibleTabsChange: function () {
            $footerNavs = $('#footer_nav li.show');
            // Makes the footer navs equal width, and store footerNavWidth
            footerNavWidth = 0;

            if ($footerNavs.length > 1) {
                var innerWidth = 0;
                $navRoot.removeClass('hidden');
                $footerNavs.each(function() {
                    $(this).width('auto');
                    var currentNavWidth = parseFloat(window.getComputedStyle(this).width);
                    currentNavWidth = Math.ceil(currentNavWidth);
                    if (currentNavWidth > innerWidth) {
                        innerWidth = currentNavWidth;
                    }
                });

                $footerNavs.each(function(index) {
                    if (index + 1 === ntp.page.tabs.current()) {
                        $(this).addClass('highlight');
                    } else {
                        $(this).removeClass('highlight');
                    }
                    $(this).width(innerWidth);
                });

                // Calculate width from offset of two elements. This is needed to count in a mysterious space between the elements in the width
                // (elements already have margin = 0 )
                var firstTabLeft = $footerNavs.eq(0).offset().left;
                var secondTabLeft = $footerNavs.eq(1).offset().left;
                if (firstTabLeft === secondTabLeft && firstTabLeft > 0)
                    footerNavWidth = $footerNavs.eq(0).innerWidth();
                else
                    footerNavWidth = $footerNavs.eq(1).offset().left - $footerNavs.eq(0).offset().left;
                $navArrow.removeClass("hidden");
                this.updateArrow(true);
            } else {
                $navRoot.addClass('hidden');
                $navArrow.addClass("hidden");
            }
        },
        
        updateArrow: function(immediate){
            if(!$('#nav_arrow').hasClass("hidden")){
                var index = ntp.page.tabs.current() - 1;
                var footerFirstTab = $footerNavs.eq(0);
                var footerLeft = footerFirstTab.offset().left;
                var newOffset = footerLeft + footerNavWidth * (ntp.page.tabs.current() - 0.5) - $navArrow.width() / 2;
                if(immediate){
                    $navArrow.stop().css('left', newOffset);
                } else {
                    var oldLeft = $navArrow.css('left');
                    if(oldLeft !== newOffset){
                        $navArrow.stop().animate(
                                {left: newOffset},
                                ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                                'linear');
                    }
                }
            }
        },
        
        update: function () {
            if (!footerNavWidth) { // footerNavWidth might be 0 if the window size is too small
                this.visibleTabsChange();
            }
            this.updateArrow(true);
        }
    };
})();


/**
 * Page initialization on customization complete
 */
ntp.deferred.customization.done(function () {
    ntp.page.layout.init();
    ntp.page.tabs.init();
    ntp.widget.templatebar.init();
    
    if (ntp.page.tabs.count() > 1) {
        ntp.page.sideNavs.init();
    }

    $(window).resize(function() {
        ntp.widget.templatebar.update();
        ntp.widget.projectbar.update();
        ntp.page.layout.update();

        if (ntp.page.tabs.count() > 1) {
            ntp.page.footerNavs.update();
            // Fix the problem of window resizing when switch tab transition is not finished.
            ntp.page.tabs.update();
        }
    });

    $(document).on('keydown', function(e) {
        if (e.ctrlKey || e.altKey) {
            ntp.acad.notifyShortcutKeyPressed(e);
        }
    });
    
    $(document).mousedown(function(e) {
        if(e.button == 1) {
            e.preventDefault();
        }
    });

    // Initialize layouts
    ntp.page.layout.update();
    
    // Show create tab on start
    if (ntp.page.tabs.count() > 1) {
        ntp.page.tabs.selectById('tab_create', true, true);
    }
});

/**
 * Page initialization on customization and localization complete.
 */
$.when(ntp.deferred.localization, ntp.deferred.customization).done(function () {
    // Updates layout after text is loaded. Some elements are resized automatically.
    if (ntp.page.tabs.count() > 1) {
        ntp.page.footerNavs.init();
        ntp.page.sideNavs.update(true, true, true); // Update side navigator names
    }
});

/**
 * Allows only dropping dwg files
 */
(function () {
    function isDrawingFile(path) {
        path = path || "";
        path = path.substr(path.length - 4).toLowerCase();
        return (path == ".dxf") || (path == ".dwg") || (path == ".dws") || (path == ".dwt");
    }

    function queryDropHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var data = e.originalEvent.dataTransfer;
        data.dropEffect = "none";
        if (data && data.files) {
            for (var i = 0; i < data.files.length; ++i) {
                if (isDrawingFile(data.files[i].name)) {
                    data.dropEffect = "copy";
                    break;
                }
            }
        }
    }

    $(window).on('dragenter', queryDropHandler);
    $(window).on('dragover', queryDropHandler);
    $(window).on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        ntp.acad.openDroppedFiles();
    });
})();

// SIG // Begin signature block
// SIG // MIIZuQYJKoZIhvcNAQcCoIIZqjCCGaYCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // Rns7C6EHCOFHEjMNlXTMG+QmJVWIaTimEl1+hmW5m1qg
// SIG // ggopMIIE2jCCA8KgAwIBAgIQDl0r5acblsN6Y+ZZVIFE
// SIG // GDANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNzA4MDQwMDAw
// SIG // MDBaFw0xODA4MDQyMzU5NTlaMIGKMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEXMBUGA1UECgwOQXV0b2Rlc2ssIElu
// SIG // Yy4xHzAdBgNVBAsMFkRlc2lnbiBTb2x1dGlvbnMgR3Jv
// SIG // dXAxFzAVBgNVBAMMDkF1dG9kZXNrLCBJbmMuMIIBIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAs9HnSHLU
// SIG // WStY4GY/6wZd8pYVSkUAoH04sOU3dSP3hYEt0eN77CwF
// SIG // q/8A4vUF6fio5rvg9akkdMl3yOmHgxIaEpkTegU0/IBV
// SIG // cJWFQCOqUtphBeQcNYx7LP7LsVZfigrUHUf5I1Vydzb6
// SIG // QAHRsQhPhUAi9BzK9rjVOGlUbM9dkz8koMOkDq2hwxDf
// SIG // 3flQCMdi6+g4AMSQW3k7+URbM9PPRpCRUsaIPXyuAXG5
// SIG // LD5TPaPK4NlnmXI0FBEX4rhy7x+V724vZapto7uSgfPv
// SIG // p2Qms1b3ACaR0YJ1+PcxCj44Pg7kmNIxgcRyaNVsfKj5
// SIG // kr8EhXKQCnFzS183YdElK4eUbQIDAQABo4IBPjCCATow
// SIG // CQYDVR0TBAIwADAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0l
// SIG // BAwwCgYIKwYBBQUHAwMwYQYDVR0gBFowWDBWBgZngQwB
// SIG // BAEwTDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNi
// SIG // LmNvbS9jcHMwJQYIKwYBBQUHAgIwGQwXaHR0cHM6Ly9k
// SIG // LnN5bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAU1MAGIknr
// SIG // OUvdk+JcobhHdglyA1gwKwYDVR0fBCQwIjAgoB6gHIYa
// SIG // aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcmwwVwYIKwYB
// SIG // BQUHAQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vcmIu
// SIG // c3ltY2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vcmIu
// SIG // c3ltY2IuY29tL3JiLmNydDANBgkqhkiG9w0BAQsFAAOC
// SIG // AQEAt+AVE1Tf82ntInlNGzyzy0+c5N3gPlZ76y0sK5DQ
// SIG // jWSNUjKIhj5ndxDPYAH3HxVe5gUtxedPpFzVwxgm2VPI
// SIG // HtnsRmR/FCYWBzhyFV+eC5Ok+pS1+jdAxP9LX5Ah1h8h
// SIG // uTV1K8UGZnBEZq2YZufaHscaIiA2pBo7pMPsXH1MQk/A
// SIG // iKIIrwXdvgQ2lhAeWAfc4B0JEtrUhY994DTfIXc8irz4
// SIG // 2EuyeDDUU2b34LpQ4zeZ/eMfDFOtu3iighmy2M8Dg9Qb
// SIG // XoWSEmDuXZEINHqwT2JPsUwKw7q0AdfelLxp8XI3IRV+
// SIG // AlUp/OiF0M6P5dAuTOlz2/deiHWBMg8K17tU+zCCBUcw
// SIG // ggQvoAMCAQICEHwbNTVK59t050FfEWnKa6gwDQYJKoZI
// SIG // hvcNAQELBQAwgb0xCzAJBgNVBAYTAlVTMRcwFQYDVQQK
// SIG // Ew5WZXJpU2lnbiwgSW5jLjEfMB0GA1UECxMWVmVyaVNp
// SIG // Z24gVHJ1c3QgTmV0d29yazE6MDgGA1UECxMxKGMpIDIw
// SIG // MDggVmVyaVNpZ24sIEluYy4gLSBGb3IgYXV0aG9yaXpl
// SIG // ZCB1c2Ugb25seTE4MDYGA1UEAxMvVmVyaVNpZ24gVW5p
// SIG // dmVyc2FsIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3Jp
// SIG // dHkwHhcNMTQwNzIyMDAwMDAwWhcNMjQwNzIxMjM1OTU5
// SIG // WjCBhDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFu
// SIG // dGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRl
// SIG // YyBUcnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRl
// SIG // YyBDbGFzcyAzIFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0Eg
// SIG // LSBHMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBANeVQ9Tc32euOftSpLYmMQRw6beOWyq6N2k1lY+7
// SIG // wDDnhthzu9/r0XY/ilaO6y1L8FcYTrGNpTPTC3Uj1Wp5
// SIG // J92j0/cOh2W13q0c8fU1tCJRryKhwV1LkH/AWU6rnXmp
// SIG // AtceSbE7TYf+wnirv+9SrpyvCNk55ZpRPmlfMBBOcWNs
// SIG // WOHwIDMbD3S+W8sS4duMxICUcrv2RZqewSUL+6Mcntim
// SIG // CXBx7MBHTI99w94Zzj7uBHKOF9P/8LIFMhlM07Acn/6l
// SIG // eCBCcEGwJoxvAMg6ABFBekGwp4qRBKCZePR3tPNgKuZs
// SIG // UAS3FGD/DVH0qIuE/iHaXF599Sl5T7BEdG9tcv8CAwEA
// SIG // AaOCAXgwggF0MC4GCCsGAQUFBwEBBCIwIDAeBggrBgEF
// SIG // BQcwAYYSaHR0cDovL3Muc3ltY2QuY29tMBIGA1UdEwEB
// SIG // /wQIMAYBAf8CAQAwZgYDVR0gBF8wXTBbBgtghkgBhvhF
// SIG // AQcXAzBMMCMGCCsGAQUFBwIBFhdodHRwczovL2Quc3lt
// SIG // Y2IuY29tL2NwczAlBggrBgEFBQcCAjAZGhdodHRwczov
// SIG // L2Quc3ltY2IuY29tL3JwYTA2BgNVHR8ELzAtMCugKaAn
// SIG // hiVodHRwOi8vcy5zeW1jYi5jb20vdW5pdmVyc2FsLXJv
// SIG // b3QuY3JsMBMGA1UdJQQMMAoGCCsGAQUFBwMDMA4GA1Ud
// SIG // DwEB/wQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UE
// SIG // AxMRU3ltYW50ZWNQS0ktMS03MjQwHQYDVR0OBBYEFNTA
// SIG // BiJJ6zlL3ZPiXKG4R3YJcgNYMB8GA1UdIwQYMBaAFLZ3
// SIG // +mlIR59TEtXC6gcydgfRlwcZMA0GCSqGSIb3DQEBCwUA
// SIG // A4IBAQB/68qn6ot2Qus+jiBUMOO3udz6SD4Wxw9FlRDN
// SIG // J4ajZvMC7XH4qsJVl5Fwg/lSflJpPMnx4JRGgBi7odSk
// SIG // VqbzHQCR1YbzSIfgy8Q0aCBetMv5Be2cr3BTJ7noPn5R
// SIG // oGlxi9xR7YA6JTKfRK9uQyjTIXW7l9iLi4z+qQRGBIX3
// SIG // FZxLEY3ELBf+1W5/muJWkvGWs60t+fTf2omZzrI4RMD3
// SIG // R3vKJbn6Kmgzm1By3qif1M0sCzS9izB4QOCNjicbkG8a
// SIG // vggVgV3rL+JR51EeyXgp5x5lvzjvAUoBCSQOFsQUecFB
// SIG // NzTQPZFSlJ3haO8I8OJpnGdukAsak3HUJgLDwFojMYIO
// SIG // 6DCCDuQCAQEwgZkwgYQxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UE
// SIG // CxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1MDMGA1UE
// SIG // AxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBT
// SIG // aWduaW5nIENBIC0gRzICEA5dK+WnG5bDemPmWVSBRBgw
// SIG // DQYJYIZIAWUDBAIBBQCgfDAQBgorBgEEAYI3AgEMMQIw
// SIG // ADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
// SIG // BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG
// SIG // 9w0BCQQxIgQgG73p0emizmLrPZz9eW97ULaPLOG+CFQ6
// SIG // NHXCCrQYC3AwDQYJKoZIhvcNAQEBBQAEggEAO4Cg3vWm
// SIG // fyW8kIXdMP0HwHv7FJLAG+ey5MLGhaunQ0/4cSKFFFGc
// SIG // Qo+Wbz4opzw0qFF4AQwWWtxtBSGqaerbgPQV6ttc5R3D
// SIG // vS9NvzBIGdWgM/LaEx72KvyZNcTJQVslQ2Shsghqi50y
// SIG // k0ervOFmDn5WmRJ2Gb3/PeZIRnP7zwwLL3e2RxcvlPQr
// SIG // kOCqjUHS3HBvitTi2M/eckl75vFfiYpxfrwj79KEU6iR
// SIG // XH1QJEMt8rvf1vFi5YFMxeCZAg2XlON+MhmuQyTfmAJo
// SIG // 2cQWk7+wLEx+pUhyMi5iVCHT9SqvicUlr+dyed9QN7Z9
// SIG // 2Fny7Ux7c601UeICINd+5ru5eKGCDKEwggydBgorBgEE
// SIG // AYI3AwMBMYIMjTCCDIkGCSqGSIb3DQEHAqCCDHowggx2
// SIG // AgEDMQ8wDQYJYIZIAWUDBAIBBQAwgd0GCyqGSIb3DQEJ
// SIG // EAEEoIHNBIHKMIHHAgEBBgkrBgEEAaAyAgMwMTANBglg
// SIG // hkgBZQMEAgEFAAQgya3kJF2mJBh+m3BT0uuUkizyLP6u
// SIG // pSQZw1uyHkgdlQUCFDbU/bifY7v1egQT22K1VK7JvrEa
// SIG // GA8yMDE4MDEzMDA0NTExOVqgXaRbMFkxCzAJBgNVBAYT
// SIG // AlNHMR8wHQYDVQQKExZHTU8gR2xvYmFsU2lnbiBQdGUg
// SIG // THRkMSkwJwYDVQQDEyBHbG9iYWxTaWduIFRTQSBmb3Ig
// SIG // QWR2YW5jZWQgLSBHMqCCCMYwggSpMIIDkaADAgECAhIR
// SIG // IQbxD85o8Jv65VsYzY8gAXcwDQYJKoZIhvcNAQELBQAw
// SIG // WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNp
// SIG // Z24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGlt
// SIG // ZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzIwHhcNMTYw
// SIG // NTI0MDAwMDAwWhcNMjcwNjI0MDAwMDAwWjBZMQswCQYD
// SIG // VQQGEwJTRzEfMB0GA1UEChMWR01PIEdsb2JhbFNpZ24g
// SIG // UHRlIEx0ZDEpMCcGA1UEAxMgR2xvYmFsU2lnbiBUU0Eg
// SIG // Zm9yIEFkdmFuY2VkIC0gRzIwggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQC3x5KKKNjzkctQDV3rKUTB
// SIG // glmlymTOvYO1UeWUzG6Amhds3P9i5jZDXgHCDGSNynee
// SIG // 9l13RbleyCTrQTcRZjesyM10m8yz70zifxvOc77Jlp01
// SIG // Hnz3VPds7KAS1q6ZnWPEeF9ZqS4i9cMn2LJbRWMnkP+M
// SIG // sT2ptPMOwPEgZQaJnQMco7BSQYU067zLzlT2Ev6zAYlK
// SIG // pvpUxR/70xzA47+X4z/QG/lAxxvV6yZ8QzDHcPJ4EaqF
// SIG // TqUODQBKOhF3o8ojAYCeyJNWXUbMitjSqgqEhbKJW9Uy
// SIG // zkF7GE5UyqvRUl4S0ySeVvMMj929ko551UGJw6Og5ZH8
// SIG // x2edhzPOcTJzAgMBAAGjggFnMIIBYzAOBgNVHQ8BAf8E
// SIG // BAMCB4AwTAYDVR0gBEUwQzBBBgkrBgEEAaAyAR4wNDAy
// SIG // BggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2ln
// SIG // bi5jb20vcmVwb3NpdG9yeS8wCQYDVR0TBAIwADAWBgNV
// SIG // HSUBAf8EDDAKBggrBgEFBQcDCDBGBgNVHR8EPzA9MDug
// SIG // OaA3hjVodHRwOi8vY3JsLmdsb2JhbHNpZ24uY29tL2dz
// SIG // L2dzdGltZXN0YW1waW5nc2hhMmcyLmNybDBYBggrBgEF
// SIG // BQcBAQRMMEowSAYIKwYBBQUHMAKGPGh0dHA6Ly9zZWN1
// SIG // cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2dzdGltZXN0
// SIG // YW1waW5nc2hhMmcyLmNydDAdBgNVHQ4EFgQULW5u0Y3l
// SIG // A4Du52Ppp9naCzxXIj0wHwYDVR0jBBgwFoAUkiGnSpVd
// SIG // ZLCbtB7mADdH5p1BK0wwDQYJKoZIhvcNAQELBQADggEB
// SIG // AFedU+Td7qUs3uS/YoCiYYZKn5udJCMwm48nhy/6BFnO
// SIG // cIg9RlJ3xJtF0o+9tyEYSdguh/+SLnf5Pwr5oCFcE0/k
// SIG // hCsoh8xcFbkTq+ISVcP4RR5JGc6qdQ8h6O9R/DQsCzIF
// SIG // CbLkFmmt2S5+4fwtSL8a03Q1ATrfMENwZ3o417oRwlb7
// SIG // MUi6W/EhEmLgdomRH0k41FhNeQmKqdpvAtat5NqJfUzZ
// SIG // sTBwQGrbRbpTdpDcS0IWIc/k0liU1FVlm3YDrK4YCLaD
// SIG // EoxBPl+ARWTIidQzF6GhTL0/xJL+f4XyvrGs1/aey84T
// SIG // Ku4CQ0/8/3A1NKMzfM/iDTIkIHwAPqMyEd4wggQVMIIC
// SIG // /aADAgECAgsEAAAAAAExicZQBDANBgkqhkiG9w0BAQsF
// SIG // ADBMMSAwHgYDVQQLExdHbG9iYWxTaWduIFJvb3QgQ0Eg
// SIG // LSBSMzETMBEGA1UEChMKR2xvYmFsU2lnbjETMBEGA1UE
// SIG // AxMKR2xvYmFsU2lnbjAeFw0xMTA4MDIxMDAwMDBaFw0y
// SIG // OTAzMjkxMDAwMDBaMFsxCzAJBgNVBAYTAkJFMRkwFwYD
// SIG // VQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYDVQQDEyhH
// SIG // bG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAtIFNIQTI1
// SIG // NiAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
// SIG // CgKCAQEAqpuOw6sRUSUBtpaU4k/YwQj2RiPZRcWVl1ur
// SIG // Gr/SbFfJMwYfoA/GPH5TSHq/nYeer+7DjEfhQuzj46FK
// SIG // bAwXxKbBuc1b8R5EiY7+C94hWBPuTcjFZwscsrPxNHaR
// SIG // ossHbTfFoEcmAhWkkJGpeZ7X61edK3wi2BTX8QceeCI2
// SIG // a3d5r6/5f45O4bUIMf3q7UtxYowj8QM5j0R5tnYDV56t
// SIG // LwhG3NKMvPSOdM7IaGlRdhGLD10kWxlUPSbMQI2CJxtZ
// SIG // IH1Z9pOAjvgqOP1roEBlH1d2zFuOBE8sqNuEUBNPxtyL
// SIG // ufjdaUyI65x7MCb8eli7WbwUcpKBV7d2ydiACoBuCQID
// SIG // AQABo4HoMIHlMA4GA1UdDwEB/wQEAwIBBjASBgNVHRMB
// SIG // Af8ECDAGAQH/AgEAMB0GA1UdDgQWBBSSIadKlV1ksJu0
// SIG // HuYAN0fmnUErTDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAy
// SIG // BggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2ln
// SIG // bi5jb20vcmVwb3NpdG9yeS8wNgYDVR0fBC8wLTAroCmg
// SIG // J4YlaHR0cDovL2NybC5nbG9iYWxzaWduLm5ldC9yb290
// SIG // LXIzLmNybDAfBgNVHSMEGDAWgBSP8Et/qC5FJK5NUPpj
// SIG // move4t0bvDANBgkqhkiG9w0BAQsFAAOCAQEABFaCSnzQ
// SIG // zsm/NmbRvjWek2yX6AbOMRhZ+WxBX4AuwEIluBjH/NSx
// SIG // N8RooM8oagN0S2OXhXdhO9cv4/W9M6KSfREfnops7yyw
// SIG // 9GKNNnPRFjbxvF7stICYePzSdnno4SGU4B/EouGqZ9uz
// SIG // nHPlQCLPOc7b5neVp7uyy/YZhp2fyNSYBbJxb051rvE9
// SIG // ZGo7Xk5GpipdCJLxo/MddL9iDSOMXCo4ldLA1c3PiNof
// SIG // KLW6gWlkKrWmotVzr9xG2wSukdduxZi61EfEVnSAR3hY
// SIG // jL7vK/3sbL/RlPe/UOB74JD9IBh4GCJdCC6MHKCX8x2Z
// SIG // faOdkdMGRE4EbnocIOM28LZQuTGCArQwggKwAgEBMHEw
// SIG // WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNp
// SIG // Z24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGlt
// SIG // ZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzICEhEhBvEP
// SIG // zmjwm/rlWxjNjyABdzANBglghkgBZQMEAgEFAKCCARQw
// SIG // GgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqG
// SIG // SIb3DQEJBTEPFw0xODAxMzAwNDUxMTlaMC8GCSqGSIb3
// SIG // DQEJBDEiBCA/PMIIBFZ4aX3YNLZv6Gy21d0N97kpmYjz
// SIG // mlKFii809DCBpgYLKoZIhvcNAQkQAgwxgZYwgZMwgZAw
// SIG // gY0EFH1V2OdaVqL8c4JD97hUh1xctSoNMHUwX6RdMFsx
// SIG // CzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWdu
// SIG // IG52LXNhMTEwLwYDVQQDEyhHbG9iYWxTaWduIFRpbWVz
// SIG // dGFtcGluZyBDQSAtIFNIQTI1NiAtIEcyAhIRIQbxD85o
// SIG // 8Jv65VsYzY8gAXcwDQYJKoZIhvcNAQEBBQAEggEATQl7
// SIG // Sm7AI0Ys7+XJwMWbzoMxy+TOZ+YBYYNlSczFpwH88Q95
// SIG // 4UYdgRM/NL68m/Y6loGR5Xuhi2a795GKS2+5/ueq9sk6
// SIG // g1Wx/B1nvFwHhK1aZ/brnjOs7lDRLO9dapjzvxLfY/28
// SIG // IVJhZZzNzdzCf4lqO8IucIxTey32ONOMil9B9wJlQ5vh
// SIG // N9jE2HBHeJhHMKfaEi4pfaSDfoPRms6TYoeSdRcOrWH1
// SIG // bWMLYCWFcz4gsUmty+Jp98kWgQxkoKMUQqaiY3Ja10TX
// SIG // KQGHqXk5Ko5kDrP/bnfo7jSfsJ0jcXR56a+s7an/rFit
// SIG // oMPj28mx0Bbf5wrYlmMBvzAIy3bxKQ==
// SIG // End signature block
